/**************************************************************
 * player.c                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroMail 1.00 - User Database File Maintenance            *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     24-Mar-97 (user.c)                                *
 *          08-Apr-97                                         *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astroml.h"
#include "platform.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 21  /* size of packed data record */
#define IND1SIZE  8  /* size of packed number index record */
#define IND2SIZE 20  /* size of packed empire index record */

/* flags */
#define PF_DELETED 0x01 /* deleted flag */


/* enumerated types *******************************************/


/* player index */
typedef enum {
    P_NONE,   /* read without index */
    P_NUMBER, /* read by number index */
    P_EMPIRE, /* read by empire index */
    P_LAST    /* placeholder */
} pindex;


/* structures *************************************************/


/* hidden player structure */
typedef struct {

    /* record fields */
    char  deleted;   /* deleted and other flags */
    long  number;    /* player number */
    char  empire[16];  /* player empire */

    /* index maintenance fields */
    char  infile;      /* 0=new record, !0=somewhere in file */
    long  pos;         /* record position when last read */
    long  onumber;     /* original player number */
    char  oempire[16]; /* original player empire */

} player;

/* indexes */
typedef struct {   /* player number index */
    long pos,          /* position of record in data file */
         number;       /* player number */
} ind1;
typedef struct {   /* player empire index */
    long pos;          /* position of record in data file */
    char empire[16];     /* player empire */
} ind2;


/* global variables *******************************************/


FILE *player_data, /* player data file */
     *player_ind1, /* player number index file */
     *player_ind2; /* player empire index file */


/* level 3 routines *******************************************/


/* player_readind1() - read a number index record */
result player_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(player_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, player_ind1);
    FREADLONG(i->number, player_ind1);
    return R_OK;
}

/* player_readind2() - read a empire index record */
result player_readind2(ind2 *j)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(player_ind2) ) == EOF)
        return R_EOF;
    j->pos = pos;
    FREADLONGR(j->pos, player_ind2);
    fread(j->empire, 16, 1, player_ind2);
    return R_OK;
}


/* level 2 routines *******************************************/


/* player_findind1() - find a number index record */
long player_findind1(ind1 *i, long number)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(player_ind1); fseek(player_ind1, 0, SEEK_END);
    if( ftell(player_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(player_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(player_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    player_readind1(i);
    while(i->number != number && start <= end)
    {
        if(i->number < number)
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(player_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        player_readind1(i);
    }

    /* return value */
    if(i->number == number)
        return 8 + middle * IND1SIZE;
    fseek(player_ind1, pos, SEEK_SET);
    return 0;
}

/* player_findind2() - find a empire index record */
long player_findind2(ind2 *j, char *empire)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(player_ind2); fseek(player_ind2, 0, SEEK_END);
    if( ftell(player_ind2) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(player_ind2) - 8 ) / IND2SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(player_ind2, 8 + IND2SIZE * middle, SEEK_SET);
    player_readind2(j);
    while( stricmp(j->empire, empire) && start <= end)
    {
        if( stricmp(j->empire, empire) < 0 )
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(player_ind2, 8 + IND2SIZE * middle, SEEK_SET);
        player_readind2(j);
    }

    if( !stricmp(j->empire, empire) )
        return 8 + middle * IND2SIZE;
    fseek(player_ind2, pos, SEEK_SET);
    return 0;
}

/* player_writeind1() - write a number index record */
result player_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, player_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,   player_ind1);
    FWRITELONG(i.number, player_ind1);
    return R_OK;
}

/* player_writeind2() - write a empire index record */
result player_writeind2(ind2 j)
{
    if( fputc(j.pos & 0xff, player_ind2) == EOF ) return R_FILE;
    FWRITELONGR(j.pos, player_ind2);
    fwrite(j.empire, 16, 1, player_ind2);
    return R_OK;
}


/* level 1 routines *******************************************/


/* player_readdata() - read a data record */
result player_readdata(player *p)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(player_data);
    if(( deleted = fgetc(player_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    p->deleted = deleted;
    FREADLONG(p->number, player_data);
    fread(p->empire, 16, 1, player_data);

    /* set index maintenance fields and return */
    p->infile = 1;
    p->pos = pos;
    p->onumber = p->number;
    strcpy(p->oempire, p->empire);
    return R_OK;
}

/* player_writedata() - write a data record */
result player_writedata(player *p)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(player_data);
    if( fputc(p->deleted, player_data) == EOF )
        return R_FILE;

    /* write rest of record */
    FWRITELONG(p->number,      player_data);
    fwrite(p->empire, 16, 1,     player_data);

    /* set index maintenance fields and return */
    p->infile = 1;
    p->pos = pos;
    p->onumber = p->number;
    strcpy(p->oempire, p->empire);
    return R_OK;
}

/* player_sortind1() - single bi-directional pass of sort */
int player_sortind1(void)
{
    long   pos,    /* stored position in index */
           number; /* original player number of index record */
    ind1   i1,     /* first index record to compare */
           i2;     /* second index record to compare */
    result r;      /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(player_ind1);
    if(pos > 8)
    {
        fseek(player_ind1, -IND1SIZE, SEEK_CUR);
        player_readind1(&i1);
        number = i1.number;
    }
    else number = 0;

    /* ensure file is big enough to need sorting */
    fseek(player_ind1, 0, SEEK_END);
    if( ftell(player_ind1) <= 8 + IND1SIZE )
    {
        fseek(player_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(player_ind1, 8, SEEK_SET); player_readind1(&i1);
    while( player_readind1(&i2) == R_OK )
    {
        if(i1.number > i2.number)
        {
            fseek(player_ind1, -2 * IND1SIZE, SEEK_CUR);
            player_writeind1(i2); player_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(player_ind1, -IND1SIZE, SEEK_CUR);
        player_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(player_ind1, -IND1SIZE, SEEK_END);
    while( ftell(player_ind1) > 8 )
    {
        fseek(player_ind1, -IND1SIZE, SEEK_CUR);
        player_readind1(&i1); player_readind1(&i2);
        if(i1.number > i2.number)
        {
            fseek(player_ind1, -2 * IND1SIZE, SEEK_CUR);
            player_writeind1(i2); player_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(player_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        player_findind1(&i1, number);
    else
        fseek(player_ind1, pos, SEEK_SET);
    return r;
}

/* player_sortind2() - single bi-directional pass of sort */
int player_sortind2(void)
{
    long   pos;      /* stored position in index */
    ind2   j1,       /* first index record to compare */
           j2;       /* second index record to compare */
    char   empire[16]; /* empire field of current index record */
    result r;        /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(player_ind2);
    if(pos > 8)
    {
        fseek(player_ind2, -IND2SIZE, SEEK_CUR);
        player_readind2(&j1);
        strcpy(empire, j1.empire);
    }

    /* ensure file is big enough to need sorting */
    fseek(player_ind2, 0, SEEK_END);
    if( ftell(player_ind2) <= 8 + IND2SIZE )
    {
        fseek(player_ind2, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(player_ind2, 8, SEEK_SET); player_readind2(&j1);
    while( player_readind2(&j2) == R_OK )
    {
        if( stricmp(j1.empire, j2.empire) > 0 )
        {
            fseek(player_ind2, -2 * IND2SIZE, SEEK_CUR);
            player_writeind2(j2); player_writeind2(j1);
            r = R_SWAPS;
        }
        fseek(player_ind2, -IND2SIZE, SEEK_CUR);
        player_readind2(&j1);
    }

    /* backward pass of sort */
    fseek(player_ind2, -IND2SIZE, SEEK_END);
    while( ftell(player_ind2) > 8 )
    {
        fseek(player_ind2, -IND2SIZE, SEEK_CUR);
        player_readind2(&j1); player_readind2(&j2);
        if( stricmp(j1.empire, j2.empire) > 0 )
        {
            fseek(player_ind2, -2 * IND2SIZE, SEEK_CUR);
            player_writeind2(j2); player_writeind2(j1);
            r = R_SWAPS;
        }
        fseek(player_ind2, -2 * IND2SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        player_findind2(&j1, empire);
    else
        fseek(player_ind2, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* player_open() - open player files */
result player_open(char *path)
{
    char dataempire[128], /* empire of data file */
         ind1empire[128], /* empire of player number index file */
         ind2empire[128], /* empire of player empire index file */
         dataheader[8], /* data file header */
         ind1header[8], /* index 1 header */
         ind2header[8]; /* index 2 header */

    /* initialise data empires */
    sprintf(dataempire, "%splayer.data", path);
    sprintf(ind1empire, "%splayer1.index", path);
    sprintf(ind2empire, "%splayer2.index", path);

    /* attempt to open files */
    if(( player_data = fopen(dataempire, "r+b") ) == NULL)
        return R_FILE;
    if(( player_ind1 = fopen(ind1empire, "r+b") ) == NULL)
    {
        fclose(player_data);
        return R_FILE;
    }
    if(( player_ind2 = fopen(ind2empire, "r+b") ) == NULL)
    {
        fclose(player_data);
        fclose(player_ind1);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, player_data);
    fread(ind1header, 8, 1, player_ind1);
    fread(ind2header, 8, 1, player_ind2);
    if(( strncmp(dataheader, "ASM10PF", 8) ) ||
       ( strncmp(ind1header, "ASM10P1", 8) ) ||
       ( strncmp(ind2header, "ASM10P2", 8) ))
    {
        fclose(player_data);
        fclose(player_ind1);
        fclose(player_ind2);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* player_create() - create player files */
result player_create(char *path)
{
    char  dataempire[128], /* empire of data file */
          ind1empire[128], /* empire of player number index file */
          ind2empire[128]; /* empire of player empire index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data empires */
    sprintf(dataempire, "%splayer.data", path);
    sprintf(ind1empire, "%splayer1.index", path);
    sprintf(ind2empire, "%splayer2.index", path);

    /* check for files' existence */
    if(( test = fopen(dataempire, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1empire, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1empire, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( player_data = fopen(dataempire, "wb") ) == NULL)
        return R_FILE;
    if(( player_ind1 = fopen(ind1empire, "wb") ) == NULL)
    {
        fclose(player_data); remove(dataempire);
        return R_FILE;
    }
    if(( player_ind2 = fopen(ind2empire, "wb") ) == NULL)
    {
        fclose(player_data); remove(dataempire);
        fclose(player_ind1); remove(ind1empire);
        return R_FILE;
    }

    /* write headers */
    fwrite("ASM10PF", 8, 1, player_data);
    fwrite("ASM10P1", 8, 1, player_ind1);
    fwrite("ASM10P2", 8, 1, player_ind2);

    /* close files and return */
    fclose(player_data);
    fclose(player_ind1);
    fclose(player_ind2);
    return R_OK;
}

/* player_close() - close player files */
result player_close(void)
{
    fclose(player_data);
    fclose(player_ind1);
    fclose(player_ind2);
    return R_OK;
}

/* player_new() - allocate memory for new player record */
player *player_new(void)
{
    return calloc( 1, sizeof(player) );
}

/* player_old() - free memory for new player record */
void player_old(player *p)
{
    free(p);
}

/* player_clear() - clear details from existing player record */
void player_clear(player *p)
{
    memset( p, 0, sizeof(p) );
}

/* player_setnumber() - set the player number */
void player_setnumber(player *p, long number)
{
    p->number = number;
}

/* player_getnumber() - return the player number */
long player_getnumber(player *p)
{
    return p->number;
}

/* player_setempire() - set the player empire */
void player_setempire(player *p, char *empire)
{
    strcpy(p->empire, empire);
}

/* player_getempire() - return the player empire */
char *player_getempire(char *empire, player *p)
{
    return strcpy(empire, p->empire);
}

/* player_write() - write a record */
result player_write(player *p)
{
    ind1   i;  /* number index record */
    ind2   j;  /* empire index record */
    long   p1, /* position of index record 1 */
           p2; /* position of index record 2 */

    /* write new record */
    if(!p->infile)
    {
        if( player_findind1(&i, p->number) ) return R_DUPE;
        if( player_findind2(&j, p->empire) ) return R_DUPE;
        fseek(player_data, 0, SEEK_END);
        fseek(player_ind1, 0, SEEK_END);
        fseek(player_ind2, 0, SEEK_END);
        i.pos = j.pos = ftell(player_data);
        i.number = p->number;
        strcpy(j.empire, p->empire);
        if( player_writedata(p) != R_OK )    return R_FILE;
        if( player_writeind1(i) != R_OK )    return R_CORRUPT;
        player_sortind1();
        if( player_writeind2(j) != R_OK )    return R_CORRUPT;
        player_sortind2();
        return R_OK;
    }

    /* rewrite existing record */
    p1 = player_findind1(&i, p->onumber);
    p2 = player_findind2(&j, p->oempire);
    if( (!p1 || !p2) )                       return R_CORRUPT;
    if( (i.pos != j.pos) )                   return R_CORRUPT;
    fseek(player_data, i.pos, SEEK_SET);
    if( player_writedata(p) != R_OK )        return R_FILE;
    if(i.number != p->number)
    {
        fseek(player_ind1, p1, SEEK_SET);
        if( player_writeind1(i) != R_OK )    return R_CORRUPT;
        player_sortind1();
    }
    if( stricmp(j.empire, p->empire) )
    {
        fseek(player_ind2, p2, SEEK_SET);
        if( player_writeind2(j) != R_OK )    return R_CORRUPT;
        player_sortind2();
    }
    return R_OK;
}

/* player_first() - read the first record */
result player_first(player *p, pindex inum)
{
    ind1   i; /* number index record */
    ind2   j; /* empire index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case P_NONE:
            fseek(player_data, 8, SEEK_SET);
            return player_readdata(p);
        case P_NUMBER:
            fseek(player_ind1, 8, SEEK_SET);
            if(( r = player_readind1(&i) ) != R_OK) return r;
            fseek(player_data, i.pos, SEEK_SET);
            return player_readdata(p);
        case P_EMPIRE:
            fseek(player_ind2, 8, SEEK_SET);
            if(( r = player_readind2(&j) ) != R_OK) return r;
            fseek(player_data, j.pos, SEEK_SET);
            return player_readdata(p);
        default:
            return R_INDEX;
    }
}

/* player_next() - read the next record */
result player_next(player *p, pindex inum)
{
    ind1   i; /* number index record */
    ind2   j; /* empire index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case P_NONE:
            return player_readdata(p);
        case P_NUMBER:
            if(( r = player_readind1(&i) ) != R_OK) return r;
            fseek(player_data, i.pos, SEEK_SET);
            return player_readdata(p);
        case P_EMPIRE:
            if(( r = player_readind2(&j) ) != R_OK) return r;
            fseek(player_data, j.pos, SEEK_SET);
            return player_readdata(p);
        default:
            return R_INDEX;
    }
}

/* player_prev() - read the previous record */
result player_prev(player *p, pindex inum)
{
    ind1   i; /* number index record */
    ind2   j; /* empire index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case P_NONE:
            if(( ftell(player_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(player_data, -2 * DATASIZE, SEEK_CUR);
            return player_readdata(p);
        case P_NUMBER:
            if(( ftell(player_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(player_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = player_readind1(&i) ) != R_OK) return r;
            fseek(player_data, i.pos, SEEK_SET);
            return player_readdata(p);
        case P_EMPIRE:
            if(( ftell(player_ind2) - 2 * IND2SIZE ) < 8)
                return R_EOF;
            fseek(player_ind2, -2 * IND2SIZE, SEEK_CUR);
            if(( r = player_readind2(&j) ) != R_OK) return r;
            fseek(player_data, j.pos, SEEK_SET);
            return player_readdata(p);
        default:
            return R_INDEX;
    }
}

/* player_last() - read the last record */
result player_last(player *p, pindex inum)
{
    ind1   i;   /* number index record */
    ind2   j;   /* empire index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case P_NONE:
            pos = ftell(player_data);
            if( fseek(player_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(player_data) < 8 )
            {
                fseek(player_data, pos, SEEK_SET);
                return R_EOF;
            }
            return player_readdata(p);
        case P_NUMBER:
            pos = ftell(player_ind1);
            if( fseek(player_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(player_ind1) < 8 )
            {
                fseek(player_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = player_readind1(&i) ) != R_OK) return r;
            fseek(player_data, i.pos, SEEK_SET);
            return player_readdata(p);
        case P_EMPIRE:
            pos = ftell(player_ind2);
            if( fseek(player_ind2, -IND2SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(player_ind2) < 8 )
            {
                fseek(player_ind2, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = player_readind2(&j) ) != R_OK) return r;
            fseek(player_data, j.pos, SEEK_SET);
            return player_readdata(p);
        default:
            return R_INDEX;
    }
}

/* player_find() - find a record by number or empire */
result player_find(player *p, char *searchtext)
{
    ind1   i;      /* player number index record */
    ind2   j;      /* player empire index record */
    long   number; /* player number from search text */

    if( sscanf(searchtext, "%ld", &number) == 1 )
    {
        if( player_findind1(&i, number) == 0 ) return R_EOF;
        fseek(player_data, i.pos, SEEK_SET);
        return player_readdata(p);
    }
    if( player_findind2(&j, searchtext) == 0 ) return R_EOF;
    fseek(player_data, j.pos, SEEK_SET);
    return player_readdata(p);

    return R_OK;
}

/* player_delete() - mark a record as deleted */
result player_delete(player *p)
{
    if(p->pos == 0) return R_EOF;
    p->deleted ^= PF_DELETED;
    fseek(player_data, p->pos, SEEK_SET);
    return player_writedata(p);
}

/* player_deleted() - return the deleted status of a record */
int player_deleted(player *p)
{
    if(p->pos == 0) return R_EOF;
    return p->deleted & PF_DELETED;
}

/* player_pack() - pack a close data file */
result player_pack(char *path)
{
    player  *p;             /* player record */
    char   dataempire[128], /* empire of data file */
           ind1empire[128], /* empire of player number index file */
           ind2empire[128], /* empire of player empire index file */
           tempempire[128], /* empire of temporary file */
           dataheader[8]; /* data file header */
    FILE  *player_temp;     /* temporary file */
    ind1   i;             /* number index record */
    ind2   j;             /* empire index record */

    /* initialise data empires */
    sprintf(dataempire, "%splayer.data", path);
    sprintf(tempempire, "%splayer.temp", path);

    /* attempt to open data & temp file */
    if(( player_data = fopen(dataempire, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, player_data);
    if(( strncmp(dataheader, "ASM10PF", 8) ) ||
       ( player_temp = fopen(tempempire, "w+b") ) == NULL)
    {
        fclose(player_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (p = malloc( sizeof(player) )) == NULL )
    {
        fclose(player_data);
        fclose(player_temp);
        remove(tempempire);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( player_readdata(p) == R_OK )
        if( !(p->deleted & PF_DELETED) )
            fwrite(p, sizeof(player), 1, player_temp);
    fclose(player_data);
    remove(dataempire);
    if(( player_data = fopen(dataempire, "w+b") ) == NULL)
    {
        free(p);
        fclose(player_temp);
        remove(tempempire);
        return R_FILE;
    }
    fwrite("ASM10PF", 8, 1, player_data);
    fseek(player_temp, 0, SEEK_SET);
    while( fread(p, sizeof(player), 1, player_temp) == 1 )
        player_writedata(p);
    fclose(player_temp);
    remove(tempempire);

    /* recreate number index */
    sprintf(ind1empire, "%splayer1.index", path); remove(ind1empire);
    if(( player_ind1 = fopen(ind1empire, "w+b") ) == NULL)
    {
        free(p);
        fclose(player_data);
        return R_FILE;
    }
    fwrite("ASM10P1", 8, 1, player_ind1);
    fseek(player_data, 8, SEEK_SET); i.pos = ftell(player_data);
    while( player_readdata(p) == R_OK )
    {
        i.number = p->number;
        player_writeind1(i);
        i.pos = ftell(player_data);
    }
    while( player_sortind1() == R_SWAPS );
    fclose(player_ind1);

    /* recreate empire index */
    sprintf(ind2empire, "%splayer2.index", path); remove(ind2empire);
    if(( player_ind2 = fopen(ind2empire, "w+b") ) == NULL)
    {
        free(p);
        fclose(player_data);
        return R_FILE;
    }
    fwrite("ASM10P2", 8, 1, player_ind2);
    fseek(player_data, 8, SEEK_SET); j.pos = ftell(player_data);
    while( player_readdata(p) == R_OK )
    {
        strcpy(j.empire, p->empire);
        player_writeind2(j);
        j.pos = ftell(player_data);
    }
    while( player_sortind2() == R_SWAPS );
    fclose(player_ind2);

    /* clean up */
    free(p);
    fclose(player_data);
    return R_OK;
}
