/**************************************************************
 * pfleet.c                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroMail 1.00 - Player Fleet Database File Maintenance    *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     24-Mar-97 (user.c)                                *
 *          04-Apr-97                                         *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astroml.h"
#include "platform.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 45  /* size of packed data record */
#define IND1SIZE  8  /* size of packed fleet number index */

/* flags */
#define FF_DELETED 0x01 /* deleted flag */


/* enumerated types *******************************************/


/* pfleet index */
typedef enum {
    F_NONE,    /* read without index */
    F_FLEETNO, /* fleet number index */
    F_LAST     /* placeholder */
} findex;


/* structures *************************************************/


/* hidden pfleet structure */
typedef struct {

    /* record fields */
    char  deleted;    /* deleted and other flags */
    long  fleetno,    /* fleet number */
          user,       /* user number */
          ships;      /* ships in fleet */
    char  origin[16], /* planet of origin */
          dest[16];   /* destination planet */

    /* index maintenance fields */
    char  infile;    /* 0=new record, !0=somewhere in file */
    long  pos,       /* record position when last read */
          ofleetno;  /* original fleet number */

} pfleet;

/* indexes */
typedef struct {   /* pfleet number index */
    long pos,          /* position of record in data file */
         fleetno;      /* pfleet number */
} ind1;


/* global variables *******************************************/


FILE *pfleet_data, /* pfleet data file */
     *pfleet_ind1; /* pfleet number index file */


/* level 3 routines *******************************************/


/* pfleet_readind1() - read a number index record */
result pfleet_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(pfleet_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, pfleet_ind1);
    FREADLONG(i->fleetno, pfleet_ind1);
    return R_OK;
}


/* level 2 routines *******************************************/


/* pfleet_findind1() - find a fleetno index record */
long pfleet_findind1(ind1 *i, long fleetno)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(pfleet_ind1); fseek(pfleet_ind1, 0, SEEK_END);
    if( ftell(pfleet_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(pfleet_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(pfleet_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    pfleet_readind1(i);
    while(i->fleetno != fleetno && start <= end)
    {
        if(i->fleetno < fleetno)
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(pfleet_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        pfleet_readind1(i);
    }

    /* return value */
    if(i->fleetno == fleetno)
        return 8 + middle * IND1SIZE;
    fseek(pfleet_ind1, pos, SEEK_SET);
    return 0;
}

/* pfleet_writeind1() - write a number index record */
result pfleet_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, pfleet_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,    pfleet_ind1);
    FWRITELONG(i.fleetno, pfleet_ind1);
    return R_OK;
}


/* level 1 routines *******************************************/


/* pfleet_readdata() - read a data record */
result pfleet_readdata(pfleet *f)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(pfleet_data);
    if(( deleted = fgetc(pfleet_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    f->deleted = deleted;
    FREADLONG(f->fleetno, pfleet_data);
    FREADLONG(f->user, pfleet_data);
    fread(f->origin, 16, 1, pfleet_data);
    fread(f->dest,   16, 1, pfleet_data);
    FREADLONG(f->ships, pfleet_data);

    /* set index maintenance fields and return */
    f->infile = 1;
    f->pos = pos;
    f->ofleetno = f->fleetno;
    return R_OK;
}

/* pfleet_writedata() - write a data record */
result pfleet_writedata(pfleet *f)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(pfleet_data);
    if( fputc(f->deleted, pfleet_data) == EOF )
        return R_FILE;

    /* write rest of record */
    FWRITELONG(f->fleetno,      pfleet_data);
    FWRITELONG(f->user,         pfleet_data);
    fwrite(f->origin, 16, 1,    pfleet_data);
    fwrite(f->dest, 16, 1,      pfleet_data);
    FWRITELONG(f->ships,        pfleet_data);

    /* set index maintenance fields and return */
    f->infile = 1;
    f->pos = pos;
    f->ofleetno = f->fleetno;
    return R_OK;
}

/* pfleet_sortind1() - single bi-directional pass of sort */
int pfleet_sortind1(void)
{
    long   pos,     /* stored position in index */
           fleetno; /* original fleetno of index record */
    ind1   i1,      /* first index record to compare */
           i2;      /* second index record to compare */
    result r;       /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(pfleet_ind1);
    if(pos > 8)
    {
        fseek(pfleet_ind1, -IND1SIZE, SEEK_CUR);
        pfleet_readind1(&i1);
        fleetno = i1.fleetno;
    }
    else fleetno = 0;

    /* ensure file is big enough to need sorting */
    fseek(pfleet_ind1, 0, SEEK_END);
    if( ftell(pfleet_ind1) <= 8 + IND1SIZE )
    {
        fseek(pfleet_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(pfleet_ind1, 8, SEEK_SET); pfleet_readind1(&i1);
    while( pfleet_readind1(&i2) == R_OK )
    {
        if(i1.fleetno > i2.fleetno)
        {
            fseek(pfleet_ind1, -2 * IND1SIZE, SEEK_CUR);
            pfleet_writeind1(i2); pfleet_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(pfleet_ind1, -IND1SIZE, SEEK_CUR);
        pfleet_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(pfleet_ind1, -IND1SIZE, SEEK_END);
    while( ftell(pfleet_ind1) > 8 )
    {
        fseek(pfleet_ind1, -IND1SIZE, SEEK_CUR);
        pfleet_readind1(&i1); pfleet_readind1(&i2);
        if(i1.fleetno > i2.fleetno)
        {
            fseek(pfleet_ind1, -2 * IND1SIZE, SEEK_CUR);
            pfleet_writeind1(i2); pfleet_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(pfleet_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        pfleet_findind1(&i1, fleetno);
    else
        fseek(pfleet_ind1, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* pfleet_open() - open pfleet files */
result pfleet_open(char *path)
{
    char dataname[128], /* name of data file */
         ind1name[128], /* name of pfleet number index file */
         dataheader[8], /* data file header */
         ind1header[8]; /* index 1 header */

    /* initialise data names */
    sprintf(dataname, "%spfleet.data", path);
    sprintf(ind1name, "%spfleet.index", path);

    /* attempt to open files */
    if(( pfleet_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    if(( pfleet_ind1 = fopen(ind1name, "r+b") ) == NULL)
    {
        fclose(pfleet_data);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, pfleet_data);
    fread(ind1header, 8, 1, pfleet_ind1);
    if(( strncmp(dataheader, "ASM10FF", 8) ) ||
       ( strncmp(ind1header, "ASM10F1", 8) ))
    {
        fclose(pfleet_data);
        fclose(pfleet_ind1);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* pfleet_create() - create pfleet files */
result pfleet_create(char *path)
{
    char  dataname[128], /* name of data file */
          ind1name[128]; /* name of pfleet number index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data names */
    sprintf(dataname, "%spfleet.data", path);
    sprintf(ind1name, "%spfleet.index", path);

    /* check for files' existence */
    if(( test = fopen(dataname, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( pfleet_data = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    if(( pfleet_ind1 = fopen(ind1name, "wb") ) == NULL)
    {
        fclose(pfleet_data); remove(dataname);
        return R_FILE;
    }

    /* write headers */
    fwrite("ASM10FF", 8, 1, pfleet_data);
    fwrite("ASM10F1", 8, 1, pfleet_ind1);

    /* close files and return */
    fclose(pfleet_data);
    fclose(pfleet_ind1);
    return R_OK;
}

/* pfleet_close() - close pfleet files */
result pfleet_close(void)
{
    fclose(pfleet_data);
    fclose(pfleet_ind1);
    return R_OK;
}

/* pfleet_new() - allocate memory for new pfleet record */
pfleet *pfleet_new(void)
{
    return calloc( 1, sizeof(pfleet) );
}

/* pfleet_old() - free memory for new pfleet record */
void pfleet_old(pfleet *f)
{
    free(f);
}

/* pfleet_clear() - clear details from existing pfleet record */
void pfleet_clear(pfleet *f)
{
    memset( f, 0, sizeof(pfleet) );
}

/* pfleet_setfleetno() - set the fleetno */
void pfleet_setfleetno(pfleet *f, long fleetno)
{
    f->fleetno = fleetno;
}

/* pfleet_getfleetno() - return the pfleet fleetno */
long pfleet_getfleetno(pfleet *f)
{
    return f->fleetno;
}

/* pfleet_setuser() - set the user */
void pfleet_setuser(pfleet *f, long user)
{
    f->user = user;
}

/* pfleet_getuser() - return the pfleet user */
long pfleet_getuser(pfleet *f)
{
    return f->user;
}

/* pfleet_setorigin() - set the planet of origin */
char *pfleet_setorigin(pfleet *f, char *origin)
{
    return strcpy(f->origin, origin);
}

/* pfleet_getorigin() - return the planet of origin */
char *pfleet_getorigin(char *origin, pfleet *f)
{
    return strcpy(origin, f->origin);
}

/* pfleet_setdest() - set the planet of dest */
char *pfleet_setdest(pfleet *f, char *dest)
{
    return strcpy(f->dest, dest);
}

/* pfleet_getdest() - return the planet of dest */
char *pfleet_getdest(char *dest, pfleet *f)
{
    return strcpy(dest, f->dest);
}

/* pfleet_setships() - set the ships */
void pfleet_setships(pfleet *f, long ships)
{
    f->ships = ships;
}

/* pfleet_getships() - return the pfleet ships */
long pfleet_getships(pfleet *f)
{
    return f->ships;
}

/* pfleet_write() - write a record */
result pfleet_write(pfleet *f)
{
    ind1   i;  /* fleetno index record */
    long   p1; /* position of index record 1 */
    static
    long   top; /* highest fleet number in file */

    /* write new record */
    if(!f->infile)
    {
        if( pfleet_findind1(&i, f->fleetno) ) return R_DUPE;
        fseek(pfleet_data, 0, SEEK_END);
        fseek(pfleet_ind1, 0, SEEK_END);
        i.pos = ftell(pfleet_data);
        i.fleetno = f->fleetno;
        if( pfleet_writedata(f) != R_OK )    return R_FILE;
        if( pfleet_writeind1(i) != R_OK )    return R_CORRUPT;
        if(i.fleetno <= top || top == 0)
        {
            pfleet_sortind1();
            if(top == 0)
            {
                p1 = ftell(pfleet_ind1);
                fseek(pfleet_ind1, -8, SEEK_END);
                pfleet_readind1(&i);
                top = i.fleetno;
                fseek(pfleet_ind1, p1, SEEK_SET);
            }
        }
        else
            top = i.fleetno;
        return R_OK;
    }

    /* rewrite existing record */
    p1 = pfleet_findind1(&i, f->ofleetno);
    if(!p1)                                  return R_CORRUPT;
    fseek(pfleet_data, i.pos, SEEK_SET);
    if( pfleet_writedata(f) != R_OK )        return R_FILE;
    if(i.fleetno != f->fleetno)
    {
        fseek(pfleet_ind1, p1, SEEK_SET);
        if( pfleet_writeind1(i) != R_OK )    return R_CORRUPT;
        pfleet_sortind1();
    }
    return R_OK;
}

/* pfleet_append() - append a record */
result pfleet_append(pfleet *f)
{
    ind1    i;  /* fleetno index record */

    /* get next fleet number */
    fseek(pfleet_ind1, 0, SEEK_END);
    if( ftell(pfleet_ind1) > 8 )
    {
        fseek(pfleet_ind1, -8, SEEK_CUR);
        pfleet_readind1(&i);
        f->fleetno = ++i.fleetno;
    }
    else f->fleetno = 1;

    /* write new fleet record, no need to sort later */
    fseek(pfleet_data, 0, SEEK_END);
    i.pos = ftell(pfleet_data);
    if( pfleet_writeind1(i) ) return R_FILE;
    if( pfleet_writedata(f) ) return R_FILE;

    return R_OK;
}

/* pfleet_first() - read the first record */
result pfleet_first(pfleet *f, findex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case F_NONE:
            fseek(pfleet_data, 8, SEEK_SET);
            return pfleet_readdata(f);
        case F_FLEETNO:
            fseek(pfleet_ind1, 8, SEEK_SET);
            if(( r = pfleet_readind1(&i) ) != R_OK) return r;
            fseek(pfleet_data, i.pos, SEEK_SET);
            return pfleet_readdata(f);
        default:
            return R_INDEX;
    }
}

/* pfleet_next() - read the next record */
result pfleet_next(pfleet *f, findex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case F_NONE:
            return pfleet_readdata(f);
        case F_FLEETNO:
            if(( r = pfleet_readind1(&i) ) != R_OK) return r;
            fseek(pfleet_data, i.pos, SEEK_SET);
            return pfleet_readdata(f);
        default:
            return R_INDEX;
    }
}

/* pfleet_prev() - read the previous record */
result pfleet_prev(pfleet *f, findex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case F_NONE:
            if(( ftell(pfleet_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(pfleet_data, -2 * DATASIZE, SEEK_CUR);
            return pfleet_readdata(f);
        case F_FLEETNO:
            if(( ftell(pfleet_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(pfleet_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = pfleet_readind1(&i) ) != R_OK) return r;
            fseek(pfleet_data, i.pos, SEEK_SET);
            return pfleet_readdata(f);
        default:
            return R_INDEX;
    }
}

/* pfleet_last() - read the last record */
result pfleet_last(pfleet *f, findex inum)
{
    ind1   i;   /* number index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case F_NONE:
            pos = ftell(pfleet_data);
            if( fseek(pfleet_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(pfleet_data) < 8 )
            {
                fseek(pfleet_data, pos, SEEK_SET);
                return R_EOF;
            }
            return pfleet_readdata(f);
        case F_FLEETNO:
            pos = ftell(pfleet_ind1);
            if( fseek(pfleet_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(pfleet_ind1) < 8 )
            {
                fseek(pfleet_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = pfleet_readind1(&i) ) != R_OK) return r;
            fseek(pfleet_data, i.pos, SEEK_SET);
            return pfleet_readdata(f);
        default:
            return R_INDEX;
    }
}

/* pfleet_find() - find a record by fleetno or name */
result pfleet_find(pfleet *f, long fleetno)
{
    ind1   i;      /* pfleet fleetno index record */

    if( pfleet_findind1(&i, fleetno) == 0 ) return R_EOF;
    fseek(pfleet_data, i.pos, SEEK_SET);
    return pfleet_readdata(f);

    return R_OK;
}

/* pfleet_delete() - mark a record as deleted */
result pfleet_delete(pfleet *f)
{
    if(f->pos == 0) return R_EOF;
    f->deleted ^= FF_DELETED;
    fseek(pfleet_data, f->pos, SEEK_SET);
    return pfleet_writedata(f);
}

/* pfleet_deleted() - return the deleted status of a record */
int pfleet_deleted(pfleet *f)
{
    if(f->pos == 0) return R_EOF;
    return f->deleted & FF_DELETED;
}

/* pfleet_pack() - pack a close data file */
result pfleet_pack(char *path)
{
    pfleet  *f;             /* pfleet record */
    char   dataname[128], /* name of data file */
           ind1name[128], /* name of pfleet number index file */
           tempname[128], /* name of temporary file */
           dataheader[8]; /* data file header */
    FILE  *pfleet_temp;     /* temporary file */
    ind1   i;             /* number index record */

    /* initialise data names */
    sprintf(dataname, "%spfleet.data", path);
    sprintf(tempname, "%spfleet.temp", path);

    /* attempt to open data & temp file */
    if(( pfleet_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, pfleet_data);
    if(( strncmp(dataheader, "ASM10FF", 8) ) ||
       ( pfleet_temp = fopen(tempname, "w+b") ) == NULL)
    {
        fclose(pfleet_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (f = malloc( sizeof(pfleet) )) == NULL )
    {
        fclose(pfleet_data);
        fclose(pfleet_temp);
        remove(tempname);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( pfleet_readdata(f) == R_OK )
        if( !(f->deleted & FF_DELETED) )
            fwrite(f, sizeof(pfleet), 1, pfleet_temp);
    fclose(pfleet_data);
    remove(dataname);
    if(( pfleet_data = fopen(dataname, "w+b") ) == NULL)
    {
        free(f);
        fclose(pfleet_temp);
        remove(tempname);
        return R_FILE;
    }
    fwrite("ASM10FF", 8, 1, pfleet_data);
    fseek(pfleet_temp, 0, SEEK_SET);
    while( fread(f, sizeof(pfleet), 1, pfleet_temp) == 1 )
        pfleet_writedata(f);
    fclose(pfleet_temp);
    remove(tempname);

    /* recreate fleet number index */
    sprintf(ind1name, "%spfleet.index", path);
    remove(ind1name);
    if(( pfleet_ind1 = fopen(ind1name, "w+b") ) == NULL)
    {
        free(f);
        fclose(pfleet_data);
        return R_FILE;
    }
    fwrite("ASM10F1", 8, 1, pfleet_ind1);
    fseek(pfleet_data, 8, SEEK_SET); i.pos = ftell(pfleet_data);
    while( pfleet_readdata(f) == R_OK )
    {
        i.fleetno = f->fleetno;
        pfleet_writeind1(i);
        i.pos = ftell(pfleet_data);
    }
    while( pfleet_sortind1() == R_SWAPS );
    fclose(pfleet_ind1);

    /* clean up */
    free(f);
    fclose(pfleet_data);
    return R_OK;
}
