/**************************************************************
 * joinrq.c                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroMail 1.00 - Join Request Database File Maintenance    *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     24-Mar-97 (user.c)                                *
 *          04-Apr-97 (pfleet.c)                              *
 *          08-Apr-97 (joinrq.c)                              *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astroml.h"
#include "platform.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 25  /* size of packed data record */
#define IND1SIZE  8  /* size of packed fleet number index */

/* flags */
#define JF_DELETED 0x01 /* deleted flag */


/* enumerated types *******************************************/


/* joinrq index */
typedef enum {
    J_NONE,  /* read without index */
    J_TRANS, /* transaction number index */
    J_LAST   /* placeholder */
} jindex;


/* structures *************************************************/


/* hidden joinrq structure */
typedef struct {

    /* record fields */
    char  deleted;    /* deleted and other flags */
    long  trans,      /* transaction number */
          user;       /* user number */
    char  empire[16]; /* requested empire name */

    /* index maintenance fields */
    char  infile;    /* 0=new record, !0=somewhere in file */
    long  pos;       /* record position when last read */
    char  otrans;    /* original transaction number */

} joinrq;

/* indexes */
typedef struct {   /* joinrq number index */
    long pos,          /* position of record in data file */
         trans;        /* joinrq number */
} ind1;


/* global variables *******************************************/


FILE *joinrq_data, /* joinrq data file */
     *joinrq_ind1; /* joinrq number index file */


/* level 3 routines *******************************************/


/* joinrq_readind1() - read a number index record */
result joinrq_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(joinrq_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, joinrq_ind1);
    FREADLONG(i->trans, joinrq_ind1);
    return R_OK;
}


/* level 2 routines *******************************************/


/* joinrq_findind1() - find a trans index record */
long joinrq_findind1(ind1 *i, long trans)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(joinrq_ind1); fseek(joinrq_ind1, 0, SEEK_END);
    if( ftell(joinrq_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(joinrq_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(joinrq_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    joinrq_readind1(i);
    while(i->trans != trans && start <= end)
    {
        if(i->trans < trans)
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(joinrq_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        joinrq_readind1(i);
    }

    /* return value */
    if(i->trans == trans)
        return 8 + middle * IND1SIZE;
    fseek(joinrq_ind1, pos, SEEK_SET);
    return 0;
}

/* joinrq_writeind1() - write a number index record */
result joinrq_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, joinrq_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,  joinrq_ind1);
    FWRITELONG(i.trans, joinrq_ind1);
    return R_OK;
}


/* level 1 routines *******************************************/


/* joinrq_readdata() - read a data record */
result joinrq_readdata(joinrq *j)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(joinrq_data);
    if(( deleted = fgetc(joinrq_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    j->deleted = deleted;
    FREADLONG(j->trans, joinrq_data);
    FREADLONG(j->user, joinrq_data);
    fread(j->empire, 16, 1, joinrq_data);

    /* set index maintenance fields and return */
    j->infile = 1;
    j->pos = pos;
    j->otrans = j->trans;
    return R_OK;
}

/* joinrq_writedata() - write a data record */
result joinrq_writedata(joinrq *j)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(joinrq_data);
    if( fputc(j->deleted, joinrq_data) == EOF )
        return R_FILE;

    /* write rest of record */
    FWRITELONG(j->trans,      joinrq_data);
    FWRITELONG(j->user,         joinrq_data);
    fwrite(j->empire, 16, 1,    joinrq_data);

    /* set index maintenance fields and return */
    j->infile = 1;
    j->pos = pos;
    j->otrans = j->trans;
    return R_OK;
}

/* joinrq_sortind1() - single bi-directional pass of sort */
int joinrq_sortind1(void)
{
    long   pos,     /* stored position in index */
           trans; /* original trans of index record */
    ind1   i1,      /* first index record to compare */
           i2;      /* second index record to compare */
    result r;       /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(joinrq_ind1);
    if(pos > 8)
    {
        fseek(joinrq_ind1, -IND1SIZE, SEEK_CUR);
        joinrq_readind1(&i1);
        trans = i1.trans;
    }
    else trans = 0;

    /* ensure file is big enough to need sorting */
    fseek(joinrq_ind1, 0, SEEK_END);
    if( ftell(joinrq_ind1) <= 8 + IND1SIZE )
    {
        fseek(joinrq_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(joinrq_ind1, 8, SEEK_SET); joinrq_readind1(&i1);
    while( joinrq_readind1(&i2) == R_OK )
    {
        if(i1.trans > i2.trans)
        {
            fseek(joinrq_ind1, -2 * IND1SIZE, SEEK_CUR);
            joinrq_writeind1(i2); joinrq_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(joinrq_ind1, -IND1SIZE, SEEK_CUR);
        joinrq_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(joinrq_ind1, -IND1SIZE, SEEK_END);
    while( ftell(joinrq_ind1) > 8 )
    {
        fseek(joinrq_ind1, -IND1SIZE, SEEK_CUR);
        joinrq_readind1(&i1); joinrq_readind1(&i2);
        if(i1.trans > i2.trans)
        {
            fseek(joinrq_ind1, -2 * IND1SIZE, SEEK_CUR);
            joinrq_writeind1(i2); joinrq_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(joinrq_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        joinrq_findind1(&i1, trans);
    else
        fseek(joinrq_ind1, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* joinrq_open() - open joinrq files */
result joinrq_open(char *path)
{
    char dataname[128], /* name of data file */
         ind1name[128], /* name of joinrq number index file */
         dataheader[8], /* data file header */
         ind1header[8]; /* index 1 header */

    /* initialise data names */
    sprintf(dataname, "%sjoinrq.data", path);
    sprintf(ind1name, "%sjoinrq.index", path);

    /* attempt to open files */
    if(( joinrq_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    if(( joinrq_ind1 = fopen(ind1name, "r+b") ) == NULL)
    {
        fclose(joinrq_data);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, joinrq_data);
    fread(ind1header, 8, 1, joinrq_ind1);
    if(( strncmp(dataheader, "ASM10JF", 8) ) ||
       ( strncmp(ind1header, "ASM10J1", 8) ))
    {
        fclose(joinrq_data);
        fclose(joinrq_ind1);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* joinrq_create() - create joinrq files */
result joinrq_create(char *path)
{
    char  dataname[128], /* name of data file */
          ind1name[128]; /* name of joinrq number index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data names */
    sprintf(dataname, "%sjoinrq.data", path);
    sprintf(ind1name, "%sjoinrq.index", path);

    /* check for files' existence */
    if(( test = fopen(dataname, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( joinrq_data = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    if(( joinrq_ind1 = fopen(ind1name, "wb") ) == NULL)
    {
        fclose(joinrq_data); remove(dataname);
        return R_FILE;
    }

    /* write headers */
    fwrite("ASM10JF", 8, 1, joinrq_data);
    fwrite("ASM10J1", 8, 1, joinrq_ind1);

    /* close files and return */
    fclose(joinrq_data);
    fclose(joinrq_ind1);
    return R_OK;
}

/* joinrq_close() - close joinrq files */
result joinrq_close(void)
{
    fclose(joinrq_data);
    fclose(joinrq_ind1);
    return R_OK;
}

/* joinrq_new() - allocate memory for new joinrq record */
joinrq *joinrq_new(void)
{
    return calloc( 1, sizeof(joinrq) );
}

/* joinrq_old() - free memory for new joinrq record */
void joinrq_old(joinrq *j)
{
    free(j);
}

/* joinrq_clear() - clear details from existing joinrq record */
void joinrq_clear(joinrq *j)
{
    memset( j, 0, sizeof(joinrq) );
}

/* joinrq_settrans() - set the trans */
void joinrq_settrans(joinrq *j, long trans)
{
    j->trans = trans;
}

/* joinrq_gettrans() - return the joinrq trans */
long joinrq_gettrans(joinrq *j)
{
    return j->trans;
}

/* joinrq_setuser() - set the user */
void joinrq_setuser(joinrq *j, long user)
{
    j->user = user;
}

/* joinrq_getuser() - return the joinrq user */
long joinrq_getuser(joinrq *j)
{
    return j->user;
}

/* joinrq_setempire() - set the requested empire name */
char *joinrq_setempire(joinrq *j, char *empire)
{
    return strcpy(j->empire, empire);
}

/* joinrq_getempire() - return the requested empire name */
char *joinrq_getempire(char *empire, joinrq *j)
{
    return strcpy(empire, j->empire);
}

/* joinrq_write() - write a record */
result joinrq_write(joinrq *j)
{
    ind1   i;  /* trans index record */
    long   p1; /* position of index record 1 */

    /* write new record */
    if(!j->infile)
    {
        if( joinrq_findind1(&i, j->trans) ) return R_DUPE;
        fseek(joinrq_data, 0, SEEK_END);
        fseek(joinrq_ind1, 0, SEEK_END);
        i.pos = ftell(joinrq_data);
        i.trans = j->trans;
        if( joinrq_writedata(j) != R_OK )    return R_FILE;
        if( joinrq_writeind1(i) != R_OK )    return R_CORRUPT;
        joinrq_sortind1();
        return R_OK;
    }

    /* rewrite existing record */
    p1 = joinrq_findind1(&i, j->otrans);
    if(!p1)                                  return R_CORRUPT;
    fseek(joinrq_data, i.pos, SEEK_SET);
    if( joinrq_writedata(j) != R_OK )        return R_FILE;
    if(i.trans != j->trans)
    {
        fseek(joinrq_ind1, p1, SEEK_SET);
        if( joinrq_writeind1(i) != R_OK )    return R_CORRUPT;
        joinrq_sortind1();
    }
    return R_OK;
}

/* joinrq_first() - read the first record */
result joinrq_first(joinrq *j, jindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case J_NONE:
            fseek(joinrq_data, 8, SEEK_SET);
            return joinrq_readdata(j);
        case J_TRANS:
            fseek(joinrq_ind1, 8, SEEK_SET);
            if(( r = joinrq_readind1(&i) ) != R_OK) return r;
            fseek(joinrq_data, i.pos, SEEK_SET);
            return joinrq_readdata(j);
        default:
            return R_INDEX;
    }
}

/* joinrq_next() - read the next record */
result joinrq_next(joinrq *j, jindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case J_NONE:
            return joinrq_readdata(j);
        case J_TRANS:
            if(( r = joinrq_readind1(&i) ) != R_OK) return r;
            fseek(joinrq_data, i.pos, SEEK_SET);
            return joinrq_readdata(j);
        default:
            return R_INDEX;
    }
}

/* joinrq_prev() - read the previous record */
result joinrq_prev(joinrq *j, jindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case J_NONE:
            if(( ftell(joinrq_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(joinrq_data, -2 * DATASIZE, SEEK_CUR);
            return joinrq_readdata(j);
        case J_TRANS:
            if(( ftell(joinrq_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(joinrq_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = joinrq_readind1(&i) ) != R_OK) return r;
            fseek(joinrq_data, i.pos, SEEK_SET);
            return joinrq_readdata(j);
        default:
            return R_INDEX;
    }
}

/* joinrq_last() - read the last record */
result joinrq_last(joinrq *j, jindex inum)
{
    ind1   i;   /* number index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case J_NONE:
            pos = ftell(joinrq_data);
            if( fseek(joinrq_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(joinrq_data) < 8 )
            {
                fseek(joinrq_data, pos, SEEK_SET);
                return R_EOF;
            }
            return joinrq_readdata(j);
        case J_TRANS:
            pos = ftell(joinrq_ind1);
            if( fseek(joinrq_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(joinrq_ind1) < 8 )
            {
                fseek(joinrq_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = joinrq_readind1(&i) ) != R_OK) return r;
            fseek(joinrq_data, i.pos, SEEK_SET);
            return joinrq_readdata(j);
        default:
            return R_INDEX;
    }
}

/* joinrq_find() - find a record by trans or name */
result joinrq_find(joinrq *j, long trans)
{
    ind1   i;      /* joinrq trans index record */

    if( joinrq_findind1(&i, trans) == 0 ) return R_EOF;
    fseek(joinrq_data, i.pos, SEEK_SET);
    return joinrq_readdata(j);

    return R_OK;
}

/* joinrq_delete() - mark a record as deleted */
result joinrq_delete(joinrq *j)
{
    if(j->pos == 0) return R_EOF;
    j->deleted ^= JF_DELETED;
    fseek(joinrq_data, j->pos, SEEK_SET);
    return joinrq_writedata(j);
}

/* joinrq_deleted() - return the deleted status of a record */
int joinrq_deleted(joinrq *j)
{
    if(j->pos == 0) return R_EOF;
    return j->deleted & JF_DELETED;
}

/* joinrq_pack() - pack a close data file */
result joinrq_pack(char *path)
{
    joinrq  *j;             /* joinrq record */
    char   dataname[128], /* name of data file */
           ind1name[128], /* name of joinrq number index file */
           tempname[128], /* name of temporary file */
           dataheader[8]; /* data file header */
    FILE  *joinrq_temp;     /* temporary file */
    ind1   i;             /* number index record */

    /* initialise data names */
    sprintf(dataname, "%sjoinrq.data", path);
    sprintf(tempname, "%sjoinrq.temp", path);

    /* attempt to open data & temp file */
    if(( joinrq_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, joinrq_data);
    if(( strncmp(dataheader, "ASM10JF", 8) ) ||
       ( joinrq_temp = fopen(tempname, "w+b") ) == NULL)
    {
        fclose(joinrq_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (j = malloc( sizeof(joinrq) )) == NULL )
    {
        fclose(joinrq_data);
        fclose(joinrq_temp);
        remove(tempname);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( joinrq_readdata(j) == R_OK )
        if( !(j->deleted & JF_DELETED) )
            fwrite(j, sizeof(joinrq), 1, joinrq_temp);
    fclose(joinrq_data);
    remove(dataname);
    if(( joinrq_data = fopen(dataname, "w+b") ) == NULL)
    {
        free(j);
        fclose(joinrq_temp);
        remove(tempname);
        return R_FILE;
    }
    fwrite("ASM10JF", 8, 1, joinrq_data);
    fseek(joinrq_temp, 0, SEEK_SET);
    while( fread(j, sizeof(joinrq), 1, joinrq_temp) == 1 )
        joinrq_writedata(j);
    fclose(joinrq_temp);
    remove(tempname);

    /* recreate fleet number index */
    sprintf(ind1name, "%sjoinrq.index", path);
    remove(ind1name);
    if(( joinrq_ind1 = fopen(ind1name, "w+b") ) == NULL)
    {
        free(j);
        fclose(joinrq_data);
        return R_FILE;
    }
    fwrite("ASM10J1", 8, 1, joinrq_ind1);
    fseek(joinrq_data, 8, SEEK_SET); i.pos = ftell(joinrq_data);
    while( joinrq_readdata(j) == R_OK )
    {
        i.trans = j->trans;
        joinrq_writeind1(i);
        i.pos = ftell(joinrq_data);
    }
    while( joinrq_sortind1() == R_SWAPS );
    fclose(joinrq_ind1);

    /* clean up */
    free(j);
    fclose(joinrq_data);
    return R_OK;
}

