/**************************************************************
 * GAMLST.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroMail 1.00 - Game list module.                         *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     02-Apr-97                                         *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astroml.h"
#include "detail.h"


/* structures *************************************************/


typedef struct {
    char  name[16], /* name of game */
          path[72]; /* path to game files */
    void *next;     /* pointer to next entry */
} game;


/* global variables *******************************************/


game *gamlst_firstg,  /* first game */
     *gamlst_current; /* current game */

extern char pid[36];  /* program ID for gamlst_write() */


/* level 1 routines *******************************************/


/* gamlst_clear() - clear the game list */
void gamlst_clear(void)
{
    game *g; /* game variable */

    while(gamlst_firstg != NULL)
    {
        g = gamlst_firstg;
        gamlst_firstg = gamlst_firstg->next;
        free(g);
    }
}

/* gamlst_add() - add a game to the list */
game *gamlst_add(void)
{
    game *g; /* game variable */

    if( (g = malloc( sizeof(game) )) == NULL) return NULL;
    g->next = gamlst_firstg; gamlst_firstg = g;

    return g;
}


/* level 0 routines *******************************************/


/* gamlst_open() - open the library */
result gamlst_open(char *path, int *line)
{
    FILE   *gamefile;      /* game list file */
    char    filename[128], /* name of game list file */
            buf[1024];     /* line buffer */
    detail *d;             /* detail record */
    game   *g;             /* game record */

    /* attempt to open file */
    sprintf(filename, "%sgamelist.data", path);
    if(( gamefile = fopen(filename, "r") ) == NULL)
        return R_FILE;

    /* read lines */
    *line = 0;
    while( fgets(buf, 1023, gamefile) )
    {
        ++*line;
        if(*buf != '\n' && *buf != ';' && *buf != '\0')
        {
            if(buf[ strlen(buf)-1 ] == '\n')
                buf[ strlen(buf)-1 ] = '\0';
            if( detail_open(&d, buf) != R_OK )
            {
                fclose(gamefile);
                gamlst_clear();
                return R_EOF;
            }
            g = gamlst_add();
            detail_getname(g->name, d);
            if( strlen(buf) < 72 )
                strcpy(g->path, buf);
            else
            {
                fclose(gamefile);
                gamlst_clear();
                return R_BADCMD;
            }
            detail_close(d);
        }
    }

    /* close and clean up */
    fclose(gamefile);
    return R_OK;
}

/* gamlst_close() - close the game list */
void gamlst_close(void)
{
    gamlst_clear();
}

/* gamlst_first() - select first game in list */
result gamlst_first(void)
{
    if(gamlst_firstg == NULL)
        return R_EOF;
    gamlst_current = gamlst_firstg;
    return R_OK;
}

/* gamlst_next() - select next game from list */
result gamlst_next(void)
{
    if(gamlst_current->next == NULL)
        return R_EOF;
    gamlst_current = gamlst_current->next;
    return R_OK;
}

/* gamlst_find() - attempt to find a named game */
result gamlst_find(char *name)
{
    game *g; /* used to flick through games */

    g = gamlst_current; gamlst_current = gamlst_firstg;
    while(gamlst_current != NULL)
        if( !stricmp(gamlst_current->name, name) )
            return R_OK;
        else
            gamlst_current = gamlst_current->next;
    gamlst_current = g;

    return R_EOF;
}

/* gamlst_get...() - 'get' functions */
char *gamlst_getname(char *name)
    { return strcpy(name, gamlst_current->name); }
char *gamlst_getpath(char *path)
    { return strcpy(path, gamlst_current->path); }

/* gamlst_set...() - 'set' functions */
char *gamlst_setname(char *name)
    { return strcpy(gamlst_current->name, name); }
char *gamlst_setpath(char *path)
    { return strcpy(gamlst_current->path, path); }

/* gamlst_write() - write a new game list */
result gamlst_write(char *path)
{
    FILE   *gamefile;      /* game list file */
    char    filename[128]; /* name of game list file */
    game   *g;             /* game record */

    /* attempt to open file */
    sprintf(filename, "%sgamelist.data", path);
    if(( gamefile = fopen(filename, "w") ) == NULL)
        return R_FILE;

    /* write header */
    fprintf(gamefile, "; AstroWar Game List File\n"
                      "; Generated by %s\n\n", pid);

    /* write lines */
    g = gamlst_firstg;
    while(g != NULL)
    {
        fprintf(gamefile, "%s\n", g->path);
        g = g->next;
    }

    /* finish */
    fclose(gamefile);
    return R_OK;
}

/* gamlst_new() - add a new game */
result gamlst_new(void)
{
    game *g;

    if(( g = gamlst_add()) == NULL) return R_MEMORY;
    gamlst_current = g;

    return R_OK;
}

/* gamlst_old() - delete an old game */
result gamlst_old(void)
{
    game *g, /* temp game pointer */
         *n; /* pointer to game after g */

    if(gamlst_current == NULL || gamlst_first == NULL)
        return R_EOF;
    if(gamlst_current == gamlst_firstg)
    {
        gamlst_firstg = gamlst_firstg->next;
        free(gamlst_current);
        gamlst_current = gamlst_firstg;
        return R_OK;
    }
    g = gamlst_firstg;
    while(g->next != gamlst_current || g->next != NULL)
        g = g->next;
    if( (n = g->next) == NULL ) return R_EOF;
    g->next = n->next; free(n);
    return R_OK;
}
