/**************************************************************
 * config.c                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroMail 1.00 - configuration file maintenance            *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     28-Mar-97 - AstroMail 1.00 config.c               *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astroml.h"


/* structures *************************************************/


typedef struct {
    int direct:1; /* direct responses as default */
} flags;

typedef struct {
    fido f;     /* a fidonet address */
    void *next; /* pointer to the next address */
} fidolist;

typedef struct {
    char      astname[36], /* name to answer to */
              netmail[72], /* netmail directory */
              logfile[72]; /* name of log file */
    flags     f;           /* some flags */
    int       rattr,       /* routed message attributes */
              dattr;       /* direct message attributes */
    fidolist *first,       /* first aka */
             *current;     /* current aka */
} config;


/* global variables ***************************************************/


extern char pid[36]; /* program ID */
config config_rec;   /* configuration record */


/* level 1 routines ***************************************************/


/* config_intname() - interpret 'name' config item */
void config_intname(void)
{
    char *tok; /* name token */

    tok = strtok(NULL, "\n");
    if( strlen(tok) > 35 )
    {
        strncpy(config_rec.astname, tok, 35);
        config_rec.astname[35] = '\0';
    }
    else strcpy(config_rec.astname, tok);
}

/* config_intnetmail() - interpret 'log file' config item */
void config_intnetmail(void)
{
    char *tok; /* log filename token */

    tok = strtok(NULL, "\n");
    if( strlen(tok) > 71 )
    {
        strncpy(config_rec.netmail, tok, 71);
        config_rec.netmail[71] = '\0';
    }
    else strcpy(config_rec.netmail, tok);
}

/* config_intlogfile() - interpret 'log file' config item */
void config_intlogfile(void)
{
    char *tok; /* log filename token */

    tok = strtok(NULL, "\n");
    if( strlen(tok) > 71 )
    {
        strncpy(config_rec.logfile, tok, 71);
        config_rec.logfile[71] = '\0';
    }
    else strcpy(config_rec.logfile, tok);
}

/* config_introuted/direct() - interpret 'routed'/'direct' config */
void config_introuted(void) { config_rec.f.direct = 0; }
void config_intdirect(void) { config_rec.f.direct = 1; }

/* config_intattr() - interpret 'attr' config item */
int config_intattr(void)
{
    char *tok;  /* attr token */
    int   dir;  /* routed(0) or direct(1)? */
    short attr; /* stored attribute */

    /* check validity of command */
    tok = strtok(NULL, " \n");
    if( !stricmp(tok, "direct") )
        dir = 1;
    else if( !stricmp(tok, "routed") )
        dir = 0;
    else return R_BADCMD;

    /* check specific attributes */
    attr = 0;
    while(( tok = strtok(NULL, " \n") ) != NULL)
             if( !stricmp(tok, "cra") ) attr |= A_CRA;
        else if( !stricmp(tok, "att") ) attr |= A_ATT;
        else if( !stricmp(tok, "hld") ) attr |= A_HLD;
        else if( !stricmp(tok, "frq") ) attr |= A_FRQ;
        else if( !stricmp(tok, "upd") ) attr |= A_UPD;
        else if( !stricmp(tok, "kil") ) attr |= A_KIL;
        else if( !stricmp(tok, "pvt") ) attr |= A_PVT;
        else if( !stricmp(tok, "loc") ) attr |= A_LOC;
        else if( !stricmp(tok, "dir") ) attr |= A_DIR;
        else if( !stricmp(tok, "era") ) attr |= A_ERA;
        else if( !stricmp(tok, "rcv") ) attr |= A_RCV;
        else if( !stricmp(tok, "snt") ) attr |= A_SNT;
        else return R_BADCMD;

    /* set appropriate attribute word */
    if(dir)
        config_rec.dattr = attr;
    else
        config_rec.rattr = attr;

    return R_OK;
}

/* config_intaddress() - interpret 'address' config item */
int config_intaddress(void)
{
    char     *tok;  /* token pointing to address */
    fido      addr; /* address */
    fidolist *f;    /* pointer to address in list */

    while(( tok = strtok(NULL, " \n") ) != NULL)
        if( sscanf(tok, "%d:%d/%d.%d",
            &addr.zone, &addr.net, &addr.node, &addr.point)
            == 4 )
        {
            f = malloc( sizeof(fidolist) );
            f->f = addr; f->next = config_rec.first;
            config_rec.first = f;
        }
        else return R_BADCMD;
    return R_OK;
}

/* config_attrstr() - return attribute string */
char *config_attrstr(char *buf, int attr)
{
    *buf = '\0';
    if(attr & A_CRA) strcat(buf, "cra ");
    if(attr & A_ATT) strcat(buf, "att ");
    if(attr & A_HLD) strcat(buf, "hld ");
    if(attr & A_FRQ) strcat(buf, "frq ");
    if(attr & A_UPD) strcat(buf, "upd ");
    if(attr & A_KIL) strcat(buf, "kil ");
    if(attr & A_PVT) strcat(buf, "pvt ");
    if(attr & A_LOC) strcat(buf, "loc ");
    if(attr & A_DIR) strcat(buf, "dir ");
    if(attr & A_ERA) strcat(buf, "era ");
    if(attr & A_RCV) strcat(buf, "rcv ");
    if(attr & A_SNT) strcat(buf, "snt");
    if( strlen(buf) > 3 ) buf[ strlen(buf) - 1 ] = '\0';
    return buf;
}


/* level 0 routines ***************************************************/


/* config_open() - intialise the library */
result config_open(char *path)
{
    strcpy(config_rec.astname, "AstroMail");
    sprintf(config_rec.logfile, "%sastroml.log", path);
    config_rec.f.direct = 0;
    config_rec.rattr = A_PVT | A_LOC;
    config_rec.dattr = A_PVT | A_LOC | A_DIR;
    config_rec.first = NULL;
    return R_OK;
}

/* config_close() - clean up and close */
void config_close(void)
{
    fidolist *f, /* current fido address */
             *p; /* next fido address */

    f = config_rec.first;
    while(f != NULL)
    {
        p = f;
        f = f->next;
        free(p);
    }
}

/* config_read() - open (and read) the configuration file */
result config_read(char *path, int *line)
{
    char      filename[128]; /* full filename of config file */
    FILE     *config_file;   /* config file handle */
    char      buf[1024],     /* buffer for line input */
             *tok;           /* pointer to token in buf */
    fidolist *f,             /* current fido address */
             *p;             /* next fido address */

    /* attempt to open configuration file */
    sprintf(filename, "%sastroml.data", path);
    if(( config_file = fopen(filename, "r") ) == NULL )
        return R_FILE;
    f = config_rec.first;
    while(f != NULL)
    {
        p = f;
        f = f->next;
        free(p);
    }
    config_rec.first = NULL;

    /* read and interpret config lines */
    *line = 0;
    while( fgets(buf, 1023, config_file) )
    {
        ++*line;
        if(*buf != '\n' && *buf != ';' && *buf != '\0')
        {
            if( strchr(buf, '\n') == NULL ) strcat(buf, "\n");
            tok = strtok(buf, " \n");
            if( !stricmp(tok, "name") )
                config_intname();
            else if( !stricmp(tok, "netmail") )
                config_intnetmail();
            else if( !stricmp(tok, "logfile") )
                config_intlogfile();
            else if( !stricmp(tok, "direct") )
                config_intdirect();
            else if( !stricmp(tok, "routed") )
                config_introuted();
            else if( !stricmp(tok, "attr") )
            {
                if( config_intattr() != R_OK )
                    { fclose(config_file); return R_BADCMD; }
            }
            else if( !stricmp(tok, "address") )
            {
                if( config_intaddress() != R_OK )
                    { fclose(config_file); return R_BADCMD; }
            }
            else
            {
                fclose(config_file);
                return R_BADCMD;
            }
        }
    }

    /* close and clean up */
    fclose(config_file);
    return R_OK;
}

/* config_write() - write the config to a file */
result config_write(char *path)
{
    char      filename[128], /* full filename of config file */
              attrstr[48];   /* attribute string */
    FILE     *config_file;   /* config file handle */
    fidolist *f;             /* pointer to address line */

    /* attempt to open configuration file */
    sprintf(filename, "%sastroml.data", path);
    if(( config_file = fopen(filename, "w") ) == NULL )
        return R_FILE;

    /* output title */
    fprintf(config_file,
        ";   AstroMail Configuration File %sastroml.data\n"
        ";   automatically generated by %s\n", path, pid);

    /* output name */
    fprintf(config_file,
        "\n; name <username>\n"
          ";   sets AstroMail to answer to <username>.\n");
    fprintf(config_file, "\nname %s\n", config_rec.astname);

    /* output netmail directory */
    fprintf(config_file,
        "\n; netmail <netdir>\n"
          ";   sets the *.MSG netmail area to <netdir>.\n");
    fprintf(config_file, "\nnetmail %s\n", config_rec.netmail);

    /* output logfile name */
    fprintf(config_file,
        "\n; logfile <logname>\n"
          ";   sets the log output file to <logname>.  "
              "No trailing slash, please.");
    fprintf(config_file, "\nlogfile %s\n", config_rec.logfile);

    /* output direct */
    fprintf(config_file,
        "\n; direct\n;   sends replies direct as default.\n");
    fputc('\n', config_file);
    if(!config_rec.f.direct) fputc(';', config_file);
    fprintf(config_file, "direct\n");

    /* output routed */
    fprintf(config_file,
        "\n; routed\n;   sends replies routed as default.\n");
    fputc('\n', config_file);
    if(config_rec.f.direct) fputc(';', config_file);
    fprintf(config_file, "routed\n");

    /* output attr */
    fprintf(config_file,
        "\n; attr <routed|direct> flag...\n"
          ";   sets attributes for routed and direct replies:\n"
          ";     cra = crash,\n"
          ";     att = attached file,\n"
          ";     hld = hold,\n"
          ";     frq = file request,\n"
          ";     upd = file update request,\n"
          ";     kil = kill,\n"
          ";     pvt = private,\n"
          ";     loc = local,\n"
          ";     dir = direct,\n"
          ";     era = erase file when sent,\n"
          ";     rcv = received,\n"
          ";     snt = sent.\n");
    fprintf( config_file, "\nattr routed %s\n",
        config_attrstr(attrstr, config_rec.rattr) );
    fprintf( config_file, "attr direct %s\n",
        config_attrstr(attrstr, config_rec.dattr) );

    /* output address(es) */
    fprintf(config_file,
        "\n; address <fido>...\n"
          ";    specifies one or more fidonet addresses.\n"
          ";    multiple address lines may be used, and each\n"
          ";    address line may have more than one address.\n"
          ";    NOTE: all addresses must be full 4D. Don't leave\n"
          ";    off the zone/point numbers.\n");
    fputc('\n', config_file);
    f = config_rec.first;
    while(f != NULL)
    {
        fprintf(config_file, "address %d:%d/%d.%d\n",
            f->f.zone, f->f.net, f->f.node, f->f.point);
        f = f->next;
    }

    /* close file */
    fclose(config_file);
    return R_OK;
}

/* config_set...() - set field values */
char *config_setname(char *astname)
    { return strcpy(config_rec.astname, astname); }
char *config_setnetmail(char *netmail)
    { return strcpy(config_rec.netmail, netmail); }
char *config_setlogfile(char *logfile)
    { return strcpy(config_rec.logfile, logfile); }
int config_setudirect(int direct)
    { return config_rec.f.direct = direct; }
int config_setrattr(int rattr) { return config_rec.rattr = rattr; }
int config_setdattr(int dattr) { return config_rec.dattr = dattr; }

/* config_get...() - get field values */
char *config_getname(char *astname)
    { return strcpy(astname, config_rec.astname); }
char *config_getnetmail(char *netmail)
    { return strcpy(netmail, config_rec.netmail); }
char *config_getlogfile(char *logfile)
    { return strcpy(logfile, config_rec.logfile); }
int config_getudirect(void) { return config_rec.f.direct; }
int config_getrattr(void)   { return config_rec.rattr; }
int config_getdattr(void)   { return config_rec.dattr; }

/* config_firstaka() - return first aka */
result config_firstaka(fido *f)
{
    if( config_rec.first == NULL ) return R_EOF;
    config_rec.current = config_rec.first;
    *f = config_rec.current->f;
    return R_OK;
}

/* config_nextaka() - return the next aka */
result config_nextaka(fido *f)
{
    if( config_rec.current->next == NULL ) return R_EOF;
    config_rec.current = config_rec.current->next;
    *f = config_rec.current->f;
    return R_OK;
}

/* config_findaka() - find an AKA in the list */
result config_findaka(fido n)
{
    fidolist *f; /* used to flick through addresses */

    f = config_rec.first;
    while(f != NULL)
        if(f->f.zone  == n.zone  &&
           f->f.net   == n.net   &&
           f->f.node  == n.node  &&
           f->f.point == n.point)
            return R_OK;
        else
            f = f->next;
    return R_EOF;
}

/* config_addaka() - add an AKA to the list */
result config_addaka(fido n)
{
    fidolist *f; /* pointer to new fido address */

    if( (f = malloc( sizeof(fidolist) )) == NULL )
        return R_MEMORY;
    f->f = n; f->next = config_rec.first;
    config_rec.first = f;
    return R_OK;
}

/* config_deleteaka() - delete an AKA from the list */
result config_deleteaka(fido n)
{
    fidolist *f, /* used to flick through addresses */
             *p; /* ditto, address previous to 'f' */

    /* initialise */
    p = NULL;
    f = config_rec.first;

    /* search & delete */
    while(f != NULL)
    {
        if(f->f.zone  == n.zone  ||
           f->f.net   == n.net   ||
           f->f.node  == n.node  ||
           f->f.point == n.point)
        {
            if(p == NULL)
                config_rec.first = f->next;
            else
                p->next = f->next;
            free(f);
            return R_OK;
        }
        p = f;
        f = f->next;
    }

    /* not found - return EOF */
    return R_EOF;
}
