/**********************************************************************
 * PLINFO.C                          Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * Ash 1.00 - Planetary Information Screen                            *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     20-Jul-97                                                 *
 **********************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ash.h"
#include "forecs.h"
#include "fleets.h"
#include "planet.h"
#include "detail.h"
#include "orders.h"


/* level 3 routines ***************************************************/


/* calcdist() - calculate distance between two planets */
int calcdist(pos p1, pos p2)
{
    if( abs(p1.x - p2.x) > abs(p1.y - p2.y) )
        return abs(p1.x - p2.x);
    else
        return abs(p1.y - p2.y);
}


/* level 2 routines ***************************************************/


/* plinfo_shell() - display shell of the screen */
void plinfo_shell(void)
{
    int count;

    /* command line */
    printf("\x1b[1;1H-Planetary-View-----------------");
    printf("\x1b[2;1H>");

    /* map */
    printf("\x1b[3;1H+------------------------------+");
    for(count = 4; count <= 18; ++count)
        printf("\x1b[%d;1H|..............................|", count);
    printf("\x1b[19;1H+------------------------------+");

    /* obvious details */
    printf("\x1b[20;1HPlanet name:");
    printf("\x1b[21;1HPosition:");
    printf("\x1b[22;1HProduction:");
    printf("\x1b[23;1HOwner:");
    printf("\x1b[24;1HShips in orbit:");

    /* planet list/approaching fleets headers */
    printf("\x1b[1;34H-Planets--------------------------------------");
    printf("\x1b[10;34H-Approaching-Fleets---------------------------");

    /* forecasts */
    printf("\x1b[19;34H-Forecasts------------------------------------");
    printf("\x1b[20;34H This turn");
    printf("\x1b[21;34H Next turn");
    printf("\x1b[22;34H 2 turns");
    printf("\x1b[23;34H 3 turns");
    printf("\x1b[24;34H Long term");
}

/* plinfo_obvious() - display obvious data */
void plinfo_obvious(planet *p)
{
    char name[16],
         coords[14],
         owner[16];
    pos  pp;

    /* planet name */
    planet_getname(name, p);
    printf("\x1b[20;18H%15s", name);

    /* coordinates */
    pp = planet_getpos(p);
    sprintf(coords, "%d,%d", pp.x, pp.y);
    printf("\x1b[21;20H%13s", coords);

    /* production */
    printf( "\x1b[22;28H%5d", planet_getprod(p) );

    /* owner */
    planet_getowner(owner, p);
    printf("\x1b[23;18H%15s", owner);

    /* ships in orbit */
    printf( "\x1b[24;23H%10ld", planet_getships(p) );
}

/* plinfo_fleets() - display details of approaching fleets */
void plinfo_fleets(planet *p, int first)
{
    char   name[16],
           origin[16],
           dest[16];
    long   number,
           ships;
    int    dist,
           count;
    fleet *f = fleets_new();
    result r;

    /* initialise */
    planet_getname(name, p);

    /* loop */
    for(count = 11; count <= 18; ++count)
    {
        do {
            if(first)
                r = fleets_first(f, F_NUMBER);
            else
                r = fleets_next(f, F_NUMBER);
            first = 0;
            if( !fleets_deleted(f) )
                fleets_getdest(dest, f);
            else
                strcpy(dest, "");
        } while(!r && strcmp(name, dest) );
        if(!r)
        {
            number = fleets_getnumber(f);
            fleets_getorigin(origin, f);
            ships = fleets_getships(f);
            dist = fleets_getdist(f);
            printf("\x1b[%d;34H %10ld %-15s %10lds %5d+", count, number,
                origin, ships, dist);
        }
        else printf("\x1b[%d;34H%46s", count, "");
    }

    /* clean up */
    fleets_old(f);
}

/* plinfo_map() - display map on the screen */
void plinfo_map(planet *p)
{
    pos     pp,
            mp;
    char    temp[10],
            name[16];
    planet *m = planet_new();
    result  r;
    int     count;

    /* initialise */
    pp = planet_getpos(p);

    /* display map border */
    printf("\x1b[3;2H[%d]", pp.x - 7);
    sprintf(temp, "[%d]", pp.x + 7);
    printf("\x1b[3;%ldH%s", 32 - strlen(temp), temp);
    sprintf(temp, "%d", pp.y + 7);
    for(count = 0; count < strlen(temp); ++count)
        printf("\x1b[%d;1H%c", 4 + count, temp[count]);
    sprintf(temp, "%d", pp.y - 7);
    for(count = 0; count < strlen(temp); ++count)
        printf("\x1b[%ld;1H%c", 19 - strlen(temp) + count, temp[count]);

    /* display planets */
    r = planet_first(m, P_NONE);
    while(!r)
    {
        mp = planet_getpos(m);
        if( calcdist(mp, pp) <= 7 )
        {
            planet_getname(name, m);
            printf("\x1b[%d;%dH%.2s", 11 - (mp.y - pp.y), 2 * (mp.x -
                pp.x) + 16, name);
        }
        r = planet_next(m, P_NONE);
    }

    /* clean up */
    planet_old(m);
}

/* plinfo_planets() - display planet list */
void plinfo_planets(planet *p, int first, char *ename)
{
    char    name[16],
            owner[16],
            mine;
    pos     pp,
            lp;
    int     count,
            dist;
    long    ships;
    planet *l = planet_new();
    result  r;

    /* initialise */
    pp = planet_getpos(p);

    /* loop */
    for(count = 2; count <= 9; ++count)
    {
        do {
            if(first)
                r = planet_first(l, P_NAME);
            else
                r = planet_next(l, P_NAME);
            first = 0;
            if( planet_deleted(l) )
                lp.x = pp.x + 8;
            else
                lp = planet_getpos(l);
        } while( !r && calcdist(pp, lp) > 7 );
        if(!r)
        {
            planet_getname(name, l);
            planet_getowner(owner, l);
            dist = calcdist(lp, pp);
            ships = planet_getships(l);
            if( !stricmp(ename, owner) ) mine = '*'; else mine = ' ';
            printf("\x1b[%d;34H%c%-15s %6d,%6d %4d %10ld", count, mine,
                name, lp.x, lp.y, dist, ships);
        }
        else printf("\x1b[%d;34H%47s", count, "");
    }

    /* clean up */
    planet_old(l);
}

/* plinfo_forecasts() - display forecasts */
void plinfo_forecasts(planet *p, detail *d, char *ename)
{
    forecs *z = forecs_new();
    fleet  *f = fleets_new();
    order  *o = orders_new();
    result  r;
    int     speed,
            dist,
            count,
            prod,
            mine;
    char    dest[16],
            name[16],
            owner[16],
            origin[16],
            mchar;
    long    arrivals[5],
            minships,
            sent,
            diff,
            ships;

    /* initialise */
    speed = detail_getspeed(d);
    planet_getname(name, p);
    planet_getowner(owner, p);
    if( strcmp(ename, owner) ) minships = 2; else minships = 1;
    for(count = 0; count < 5; ++count) arrivals[count] = 0;
    sent = 0;

    /* display yesterday's forecast */
    if( !forecs_find(z, name) )
    {
        ships = forecs_getships(z);
        diff = planet_getships(p) - ships;
        if( stricmp(owner, ename) ) mchar = ' '; else mchar = '*';
        if(diff < 0)
            printf("\x1b[20;57H%10ld- %10ld%c", -diff, ships, mchar);
        else if(diff > 0)
            printf("\x1b[20;57H%10ld+ %10ld%c", diff, ships, mchar);
        else
            printf("\x1b[20;57H%12s%10ld%c", "", ships, mchar);
    }
    else printf("\x1b[20;57H%23s", "");

    /* calculate & print arrivals */
    r = fleets_first(f, F_NONE);
    while(!r)
    {
        fleets_getdest(dest, f);
        ships = fleets_getships(f);
        if( !strcmp(name, dest) && ships >= minships &&
            !fleets_deleted(f) )
        {
            dist = fleets_getdist(f);
            if(dist <= speed)
                arrivals[1] += fleets_getships(f);
            else if(dist <= 2 * speed)
                arrivals[2] += fleets_getships(f);
            else if(dist <= 3 * speed)
                arrivals[3] += fleets_getships(f);
            else
                arrivals[4] += fleets_getships(f);
        }
        r = fleets_next(f, F_NONE);
    }
    for(count = 1; count <= 4; ++count)
        printf("\x1b[%d;45H%10ld+", 20 + count, arrivals[count]);

    /* calculate & print ship totals */
    if( strcmp(owner, "neutral") )
        prod = planet_getprod(p);
    else
        prod = 0;
    mine = !strcmp(owner, ename);
    ships = planet_getships(p);
    for(count = 1; count <= 4; ++count)
    {
        ships += prod;
        if(mine)
            ships += arrivals[count];
        else
            ships -= arrivals[count];
        if(ships < 0)
        {
            mine = 1;
            ships = -ships;
            prod = planet_getprod(p);
        }
        if(mine && count == 1)
        {
            r = orders_first(o);
            while(!r)
            {
                if( orders_gettype(o) == 'S' )
                {
                    orders_getorigin(origin, o);
                    if( !strcmp(origin, name) )
                        sent += orders_getships(o);
                }
                r = orders_next(o);
            }
            ships -= sent;
        }
        if(mine) mchar = '*'; else mchar = ' ';
        printf("\x1b[%d;69H%10ld%c", 20 + count, ships, mchar);
    }
    printf("\x1b[%d;57H%10ld-", 21, sent);

    /* clean up */
    orders_old(o);
    fleets_old(f);
    forecs_old(z);
}

/* plinfo_command() - accept command */
command plinfo_command(char *param)
{
    char    commandstr[33],
            commandword[33],
           *space;
    command c;

    /* input command */
    printf("\x1b[2;1H%32s\x1b[2;1H>", "");
    gets(commandstr);
    printf("\x1b[1;1H-Planetary-View-----------------");

    /* extract command */
    strcpy(commandword, commandstr);
    space = strchr(commandword, ' ');
    if(space) *space = '\0';

    /* extract parameters */
    space = strchr(commandstr, ' ');
    if(space) strcpy(param, &space[1]); else strcpy(param, "");

    /* interpret command */
    if( !*commandword )
        c = C_NONE;
    else if(!strnicmp( commandword, "exit", strlen(commandword) ))
        c = C_EXIT;
    else if(!strnicmp(commandword, "view", strlen(commandword) ))
        c = C_VIEW;
    else if(!strnicmp(commandword, "more", strlen(commandword) ))
        c = C_MORE;
    else if(!strnicmp(commandword, "first", strlen(commandword) ))
        c = C_FIRST;
    else if(!strnicmp(commandword, "send", strlen(commandword) ))
        c = C_SEND;
    else if(!strnicmp(commandword, "map", strlen(commandword) ))
        c = C_MAP;
    else if(!strnicmp(commandword, "scout", strlen(commandword) ))
        c = C_SCOUT;
    else
        c = C_NONE;

    return c;
}

/* plinfo_error() - display error message */
void plinfo_error(void)
{
    printf("\x1b[1;27HERROR");
}


/* level 1 routines ***************************************************/


/* plinfo_view() - view a single planet */
result plinfo_view(char *name, char *ename, detail *d)
{
  planet 
    *p = planet_new(),
    *s = planet_new();
  order  *o = orders_new();
  result  r = R_OK;
  char
    param[33],
    dest[33],
    owner[16];
  command c;
  long    ships;

  if( planet_find(p, name) )
    r = R_FILE;
  else
    {
      planet_getname(name, p);
      plinfo_shell();
      plinfo_obvious(p);
      plinfo_forecasts(p, d, ename);
      plinfo_fleets(p, 1);
      plinfo_map(p);
      plinfo_planets(p, 1, ename);
      do {
	c = plinfo_command(param);
	switch(c)
	  {
	  case C_VIEW:
	    strcpy(name, param);
	    if( planet_find(p, name) )
	      {
		planet_getname(name, p);
		plinfo_error();
	      }
	    else
	      {
		planet_getname(name, p);
		plinfo_shell();
		plinfo_obvious(p);
		plinfo_forecasts(p, d, ename);
		plinfo_fleets(p, 1);
		plinfo_map(p);
		plinfo_planets(p, 1, ename);
	      }
	    break;
	  case C_EXIT:
	    r = R_EOF;
	    break;
	  case C_MORE:
	    if(!strnicmp( param, "fleets", strlen(param) ))
	      plinfo_fleets(p, 0);
	    else if(!strnicmp( param, "planets", strlen(param) ))
	      plinfo_planets(p, 0, ename);
	    break;
	  case C_FIRST:
	    if(!strnicmp( param, "fleets", strlen(param) ))
	      plinfo_fleets(p, 1);
	    else if(!strnicmp( param, "planets", strlen(param)
			       ))
	      plinfo_planets(p, 1, ename);
	    break;
	  case C_SEND:
	    if( sscanf(param, "%ld%s", &ships, dest) != 2 )
	      plinfo_error();
	    else if( planet_find(s, dest) )
	      plinfo_error();
	    else if(calcdist( planet_getpos(p), planet_getpos(s) ) 
		    > detail_getmaxdist( d ))
	      plinfo_error();
	    else
	      {
		orders_settype(o, 'S');
		orders_setempire(o, ename);
		orders_setorigin(o, name);
		orders_setdest(o, dest);
		orders_setships(o, ships);
		orders_write(o);
		plinfo_forecasts(p, d, ename);
		plinfo_fleets(p, 1);
	      }
	    break;
	  case C_MAP:
	    orders_settype(o, 'M');
	    orders_setempire(o, ename);
	    orders_setcentre(o, name);
	    orders_write(o);
	    break;
	  case C_SCOUT:
	    orders_settype(o, 'S');
	    orders_setempire(o, ename);
	    orders_setorigin(o, name);
	    orders_setships(o, 1);
	    if( strcmp(param, "*") )
	      {
		if( planet_find(s, param) )
		  plinfo_error();
		else if(calcdist( planet_getpos(s), planet_getpos(p) ) 
			> detail_getmaxdist( d ))
		  plinfo_error();
		else
		  {
		    orders_setdest(o, param);
		    orders_write(o);
		  }
	      }
	    else
	      {
		r = planet_first(s, P_NONE);
		while(!r)
		  {
		    planet_getowner(owner, s);
		    if( stricmp(owner, ename) &&
			calcdist( planet_getpos(p), planet_getpos(s) ) 
			<= detail_getmaxdist(d) )
		      {
			planet_getname(param, s);
			orders_setdest(o, param);
			orders_write(o);
		      }
		    r = planet_next(s, P_NONE);
		  }
		r = R_OK;
		plinfo_planets(p, 1, ename);
	      }
	    plinfo_forecasts(p, d, ename);
	    plinfo_fleets(p, 1);
	    break;
	  default:
	    plinfo_error();
	  }
      } while(!r);
    }

  /* clean up */
  planet_old(p);
  planet_old(s);
  orders_old(o);
  if(r == R_EOF) r = R_OK;
  return r;
}

/* level 0 routines ***************************************************/


/* main routine */
result plinfo_main(char *name, char *ename, detail *d)
{
    result r;

    printf("\x1b[2J");
    if((r = plinfo_view(name, ename, d) ))
      printf("Planet %s unknown\n", name);
    else
      printf("\x1b[2J");
    return r;
}
