/**********************************************************************
 * ANNOUNCE.C                        Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * Announce 1.00 - main program                                       *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     01-Sep-97                                                 *
 **********************************************************************/


/* included headers ***************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "announce.h"
#include "orders.h"
#include "gamlst.h"
#include "config.h"
#include "scrlog.h"
#include "textfl.h"


/* enumerated types ***************************************************/


typedef enum {
  S_NONE, /* no service selected */
  S_POST, /* turn input selected */
  S_HELP, /* program help selected */
  S_ERROR /* error on command line */
} service;


/* global variables ***************************************************/


char pid[36] = PROGNAME;


/* level 2 routines ***************************************************/


/* announce_open() - open files for announcement */
result announce_open(char **text, char *path, char *game, char *file)
{
  result
    r;            /* result of read */
  int
    line;         /* line on which config read failed */
  char
    gamepath[72], /* path to game files */
    logfile[128]; /* name of log file */

  /* attempt to load Astrowar's config file */
  config_open(path);
  switch( r = config_read(path, &line) )
    {
    case R_FILE:
      scrlog_cannotreadconfig(path);
      return r;
    case R_BADCMD:
      scrlog_badcommandinconfig(path, line);
      return r;
    case R_OK:
      config_getlogfile(logfile);
      if( ( r = scrlog_openlog(logfile, 0xffff) )
	  != R_OK)
	{
	  scrlog_cannotopenlog(logfile);
	  goto closeconfig;
	}
      scrlog_post();
      scrlog_configread(path);
      break;
    default:
      scrlog_badreturncode(r, "config_read");
      return r;
    }

  /* attempt to open game list */
  switch( r = gamlst_open(path, &line) )
    {
    case R_FILE:
      scrlog_cannotopengamelist(path);
      goto closeconfig;
    case R_EOF:
      scrlog_badfileingamelist(path, line);
      goto closeconfig;
    case R_BADCMD:
      scrlog_badlineingamelist(path, line);
      goto closeconfig;
    case R_OK:
      scrlog_gamelistread(path);
      break;
    default:
      scrlog_badreturncode(r, "gamlst_open");
      goto closeconfig;
    }

  /* attempt to find game */
  if( gamlst_find(game) )
    {
      scrlog_gamenotfound(game);
      goto closegamlst;
    }
  gamlst_getpath(gamepath);

  /* open orders file */
  switch( r = orders_open(gamepath) )
    {
    case R_OK:
      scrlog_fileopened(gamepath, "orders.data");
      break;
    case R_FILE:
      if(!( r = orders_create(gamepath) ))
	if(!( r = orders_open(gamepath) ))
	  {
	    scrlog_fileopened(gamepath, "orders.data");
	    break;
	  }
      scrlog_cannotopenfile(gamepath, "orders.data");
      goto closegamlst;
    case R_HEADER:
      scrlog_badheaderinfile(gamepath, "orders.data");
      goto closegamlst;
    default:
      scrlog_badreturncode(r, "orders_open");
      goto closegamlst;
    }

  /* read text file */
  if(!( *text = textfl_read(file) ))
    {
      scrlog_cannotopenfile("", file);
      goto closeorders;
    }
  scrlog_fileopened("", file);

  /* everything went OK */
  return R_OK;

  /* deal with errors */
closeorders:
  orders_close();
closegamlst:
  gamlst_close();
closeconfig:
  config_close();
  return r;
}

/* announce_write() - write the announcement message */
void announce_write(char *text)
{
  order *o = orders_new(); /* order record for message */

  orders_settype(o, 'W');
  orders_setempire(o, "");
  orders_setrecip(o, "*");
  orders_settext(o, text);
  orders_write(o);
  scrlog_messageposted();

  orders_old(o);
}

/* announce_close() - close files after posting */
void announce_close(char *text)
{
  textfl_old(text);
  orders_close();
  gamlst_close();
  config_close();
  scrlog_finished();
}


/* level 1 routines ***************************************************/


/* announce_args() - parse command line */
service announce_args(char *path, char *game, char *file, int argc, 
		      char **argv)
{
  service s; /* service to return */
  char *env; /* pointer to environment variable */
  int argi;  /* current input parameter (argv) */

  /* initialise */
  strcpy(path, "");
  strcpy(file, "");
  if(( env = getenv("ASTROWAR") ) != NULL) strcpy(path, env);
  argi = 1; s = S_NONE;

  /* loop through parameters */
  while(argi < argc)
    {
      if( !stricmp(argv[argi], "post") && s == S_NONE )
	s = S_POST;
      else if( !stricmp(argv[argi], "?") )
	s = S_HELP;
      else if( !stricmp(argv[argi], "-path") )
        {
	  if(++argi < argc)
	    strcpy(path, argv[argi]);
	  else
	    s = S_ERROR;
        }
      else if( !stricmp(argv[argi], "-game") )
        {
	  if(++argi < argc)
	    strcpy( game, argv[argi]);
	  else
	    s = S_ERROR;
        }
      else if(s == S_POST && !*file)
	strcpy( file, argv[argi] );
      else s = S_ERROR;
      argi++;
    }

  /* fill rest of output parameters and return */
  if(!*file) strcpy(file, "message.txt");
  return s;
}

/* announce_drop() - drop semaphore */
result announce_drop(char *path)
{
    char  filename[128]; /* full name of file */
    FILE *sem;           /* file handle for semaphore */

    sprintf(filename, "%sastrowar.sem", path);
    if(( sem = fopen(filename, "r") ) != NULL)
    {
        fclose(sem);
        return R_EXISTS;
    }
    if(( sem = fopen(filename, "w") ) == NULL) return R_FILE;
    fclose(sem);

    return R_OK;
}

/* announce_post() - post an announcement */
result announce_post(char *path, char *game, char *file)
{
  result r;
  char *text;

  if(!( r = announce_open(&text, path, game, file) ))
    {
      announce_write(text);
      announce_close(text);
    }

  return r;
}

/* announce_lift() - lift a semaphore */
void announce_lift(char *path)
{
    char filename[128]; /* full name of file */

    sprintf(filename, "%sastrowar.sem", path);
    remove(filename);
}


/* level 0 routines ***************************************************/


/* main program */
int main(int argc, char **argv)
{
  service s;       /* service requested on command line */
  result r = R_OK; /* returned to calling process */
  char
    file[128],     /* file to post */
    game[16],      /* process-specific parameters */
    path[72];      /* path to configs, semaphores etc. */

  scrlog_open();
  s = announce_args(path, game, file, argc, argv);
  if(( r = announce_drop(path) ) == R_OK)
    {
      switch(s)
        {
	case S_POST:
	  r = announce_post(path, game, file);
	  break;
	case S_ERROR: case S_NONE:
	  r = R_BADCMD;
	  scrlog_error();
	  break;
	default:
	  r = R_OK;
	  scrlog_help();
        }
      announce_lift(path);
    }
  else
    scrlog_running();
  scrlog_close();

  return r;
}
