/***********************************************************
        Copyright 1991 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

static char rcs_id[]="$Header: /afs/andrew.cmu.edu/system/src/local/depot/016/RCS/update.c,v 4.7 1992/02/12 18:04:23 ww0r Exp $";


/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/time.h>
#include <errno.h>
#ifdef sun4
#include <vfork.h>
#endif sun4
#include <sys/wait.h>

#include "globals.h"
#include "depot.h"
#include "DepotDBStruct.h"
#include "DepotDB.h"
#include "DepotDBCommandStruct.h"
#include "DepotDBCommand.h"
#include "PreferenceStruct.h"
#include "Preference.h"

#ifdef ibm032
extern int errno;
#endif /* ibm032 */

static void Update_Remove();
static void Update_CleanDir();

Boolean Update_Noop(entryp, path)
     ENTRY *entryp;
     char *path;
{
  struct stat stb;
  SOURCE *sourcep;
  Boolean NoOpFileExists;

  dbgprint(F_TRACE, (stderr, "Update_Noop\n"));

  sourcep = entryp->sourcelist + (entryp->nsources - 1);
  if (Depot_TrustTargetDir && (sourcep->status & S_TRUSTTARGET))
    {
      dbgprint(F_TRUSTTARGET, (stderr, "NOOP %s - Trusting targetdir!\n", path));
      VerboseMessage((stdout, "NOOP %s\n", path));
    }
  else
    {
      if (lstat(path,&stb) < 0)
	{
	  if (errno == ENOENT) NoOpFileExists = FALSE;
	  else {FatalError(E_LSTATFAILED, (stderr, "Could not lstat %s; errno =%d\n", path, errno));}
	}
      else
	NoOpFileExists = TRUE;
      if (!NoOpFileExists)
	{
	  WarningError((stderr, "Warning: File %s not found\n", path));
	  VerboseMessage((stdout, "Warning: NOOP %s - File not found\n", path));
	}
      else
	{
	  VerboseMessage((stdout, "NOOP %s\n", path));
	}
    }

  dbgprint(F_TRACE, (stderr, "Update_Noop done\n"));
  return TRUE;
}


Boolean Update_Copy(entryp, dest, src)
     ENTRY *entryp;
     char *dest, *src;
{
  int dp, sp;
  int nread;
  struct stat stbsrc, stbdest;
  int destuid, destgid;
#ifdef USE_UTIME
   time_t desttime[2];
#else /* USE_UTIME */
  struct timeval desttime[2];
#endif /* USE_UTIME */
  char *destdate;
  SOURCE *sourcep;
  char buffer[MAXBSIZE];
  Boolean DestFileExists, DestFileDeleteable;
  char linksrc[MAXPATHLEN], srcpath[MAXPATHLEN];
  int cc;

  dbgprint(F_TRACE, (stderr, "Update_Copy - being implemented\n"));

  sourcep = entryp->sourcelist + (entryp->nsources - 1);
  if (Depot_TrustTargetDir && (sourcep->status & S_TRUSTTARGET))
    {
      dbgprint(F_TRUSTTARGET, (stderr, "COPY %s %s - Trusting targetdir!\n", src, dest));
    }
  else
    {
      if (lstat(dest,&stbdest) < 0)
	{
	  if (errno == ENOENT) DestFileExists = FALSE;
	  else {FatalError(E_LSTATFAILED, (stderr, "Could not lstat %s; errno =%d\n", dest, errno));}
	}
      else
	DestFileExists = TRUE;
      
      if (DestFileExists)
	{
	  DestFileDeleteable = ((Depot_DeleteUnReferenced == TRUE) || DepotDB_AntiqueEntry(entryp));
	  if (!DestFileDeleteable)
	    {
	      FatalError(E_BADDESTFILE, (stderr, "COPY %s %s : %s not writable\n", src, dest, dest));
	    }
	  else
	    Update_Remove(dest);	
	}

      VerboseMessage((stdout, "COPY %s %s\n", src, dest));
      if ( src[0] == '/' ) /* absolute path */
	(void)strcpy(srcpath, src);
      else /* relative path */
	(void)sprintf(srcpath, "%s/%s", Depot_TargetPath, src);

      if (lstat(srcpath, &stbsrc) < 0)
	{ FatalError(E_LSTATFAILED, (stderr, "Could not lstat %s; errno = %d\n", srcpath, errno)); }

      if ((stbsrc.st_mode & S_IFMT) == S_IFLNK)
	{
	  if ((cc = readlink(srcpath, linksrc, MAXPATHLEN-1)) <= 0)
	    { FatalError(E_RDLINKFAILED, (stderr, "Could not read the link %s; errno = %d\n", srcpath, errno)); }
	  else
	    {
	      linksrc[cc] = '\0';
	      if (symlink(linksrc, dest) < 0)
		{ FatalError(E_LINKFAILED, (stderr, "Could not make symlink from %s->%s, errno = %d\n", dest, linksrc, errno)); }
	    }
	}
      else
	{
	  /* copy file over */
	  if ((sp = open(srcpath, O_RDONLY)) < 0)
	    { FatalError(E_OPENFAILED, (stderr, "COPY %s %s : Could not open %s to read\n", src, dest, srcpath)); }
	  if ((dp = open(dest, O_WRONLY|O_CREAT|O_TRUNC, stbsrc.st_mode & 0755)) < 0)
	    { FatalError(E_OPENFAILED, (stderr, "COPY %s %s : Could not open %s to write\n", src, dest, dest)); }
      
	  while ( (nread = read(sp, buffer, MAXBSIZE)) > 0 )
	    {
	      if (write(dp, buffer, nread) != nread)
		{ FatalError(E_WRITEFAILED, (stderr, "COPY %s %s : write to %s failed!\n", src, dest, dest)); }
	    }
      
	  (void)close(sp); (void)close(dp);

	  /* if suid or sgid, set owner/group and mode bits */
	  if (stbsrc.st_mode & (S_ISUID | S_ISGID))
	    {
	      destuid = (stbsrc.st_mode & S_ISUID) ?  stbsrc.st_uid : -1 ;
	      destgid = (stbsrc.st_mode & S_ISGID) ?  stbsrc.st_gid : -1 ;
	      VerboseMessage((stdout, "CHOWN %s %d %d\n", dest, destuid, destgid));
	      if (chown(dest, destuid, destgid) < 0)
		{ DepotError(DE_CHOWNFAILED, (stderr, "CHOWN %s %d %d failed, errno = %d\n", dest, destuid, destgid, errno)); }
	      else
		{
		  VerboseMessage((stdout, "CHMOD %s %o\n", dest, stbsrc.st_mode & 07755));
		  if (chmod(dest, (stbsrc.st_mode & 07755)) < 0)
		    { DepotError(DE_CHMODFAILED, (stderr, "CHMOD %s %o failed, errno = %s\n", dest, stbsrc.st_mode & 07755, errno)); }
		}
	    }

	  /* set mod time */
#ifdef USE_UTIME
	  desttime[0] = desttime[1] = (long)(stbsrc.st_mtime);
	  destdate = ctime(desttime[1]); destdate[24] = '\0';
	  VerboseMessage((stdout, "UTIME %s %s\n", dest, destdate));

	  if (utime(dest, desttime) < 0)
	    { DepotError(DE_UTIMEFAILED, (stderr, "UTIME %s %s failed, errno = %s\n", dest, destdate, errno)); }

#else /* USE_UTIME */
	  desttime[0].tv_sec = desttime[1].tv_sec = (long)(stbsrc.st_mtime);
	  desttime[0].tv_usec = desttime[1].tv_usec = 0;
	  destdate = ctime((long *)&(desttime[1].tv_sec)); destdate[24] = '\0';
	  VerboseMessage((stdout, "UTIMES %s %s\n", dest, destdate));

	  if (utimes(dest, desttime) < 0)
	    { DepotError(DE_UTIMESFAILED, (stderr, "UTIMES %s %s failed, errno = %s\n", dest, destdate, errno)); }

#endif /* USE_UTIME */
	}
    }
  dbgprint(F_TRACE, (stderr, "Update_Copy done\n"));
  return TRUE;
}


/*
 * Boolean Update_Link(entryp, dest, src, depth)
 * depth indicates the depth of the dest from the top of the destination tree.
 */
Boolean Update_Link(entryp, dest, src, depth)
     ENTRY *entryp;
     char *dest, *src;
     unsigned depth;
{
  struct stat stb;
  SOURCE *sourcep;
  Boolean DestFileExists, DestFileDeleteable;
  char srcpath[MAXPATHLEN];
  register unsigned i;
  register char *cp;

  dbgprint(F_TRACE, (stderr, "Update_Link\n"));

  sourcep = entryp->sourcelist + (entryp->nsources - 1);
  if (Depot_TrustTargetDir && (sourcep->status & S_TRUSTTARGET) && (sourcep->update_spec & U_LINK))
    {
      dbgprint(F_TRUSTTARGET, (stderr, "LINK %s %s - Trusting targetdir!\n", src, dest));
    }
  else
    {
      if (lstat(dest,&stb) < 0)
	{
	  if (errno == ENOENT) DestFileExists = FALSE;
	  else {FatalError(E_LSTATFAILED, (stderr, "Could not lstat %s; errno =%d\n", dest, errno));}
	}
      else
	DestFileExists = TRUE;

      /* if src is relative, prefix src with a string of ../s depth minus 1 deep to get sourcepath */
      if ( src[0] == '/' )
	{ /* absolute path */
	  (void)strcpy(srcpath, "");
	}
      else
	{ /* relative path */
	  cp = srcpath; i = 1;
	  while ( i < depth )
	    { *cp++ = '.'; *cp++ = '.'; *cp++ = '/'; i++; }
	  *cp = '\0';
	}
      (void)strcat(srcpath, src);
      
      DestFileDeleteable = ((Depot_DeleteUnReferenced == TRUE) || DepotDB_AntiqueEntry(entryp));
      if (!DestFileExists || DestFileDeleteable)
	{
	  Update_Remove(dest);
	  VerboseMessage((stdout, "LINK %s %s\n", src, dest));
	  if (symlink(srcpath, dest) < 0)
	    {
	      FatalError(E_LINKFAILED, (stderr, "Could not make symlink from %s->%s, errno = %d\n", dest, srcpath, errno));
	    }
	}
    }

  dbgprint(F_TRACE, (stderr, "Update_Link done\n"));
  return TRUE;
}


/*
 * Boolean Update_Map(entryp, dest, src, depth)
 * depth indicates the depth of the dest from the top of the destination tree.
 */
Boolean Update_Map(entryp, dest, src, depth)
     ENTRY *entryp;
     char *dest, *src;
     unsigned depth;
{
  int MapCommandType;
  Boolean ret_val;

  MapCommandType = GetMapCommand((entryp->sourcelist+(entryp->nsources-1))->collection_name, preference);
  switch (MapCommandType)
    {
    case U_COPY:
      ret_val = Update_Copy(entryp, dest, src);
      break;
    case U_LINK:
    default:
      ret_val = Update_Link(entryp, dest, src, depth);
      break;
    }
  return ret_val;
}


Boolean Update_MakeDir(entryp, path)
     ENTRY *entryp;
     char *path;
{
  struct stat stb;
  SOURCE *sourcep;

  dbgprint(F_TRACE, (stderr, "Update_MakeDir\n"));

  sourcep = entryp->sourcelist + (entryp->nsources - 1);
  if (Depot_TrustTargetDir && (sourcep->status & S_TRUSTTARGET))
    {
      dbgprint(F_TRUSTTARGET, (stderr, "MKDIR %s - Trusting targetdir!\n", path));
    }
  else
    {
      if (lstat(path,&stb) < 0)
	{
	  if (errno != ENOENT)
	    {
	      FatalError(E_LSTATFAILED, (stderr, "Could not lstat %s; errno =%d\n", path, errno));
	    }
	}
      else if ((stb.st_mode & S_IFMT) == S_IFDIR)
	{
	  VerboseMessage((stdout, "DIRECTORY %s\n", path));
	  Update_CleanDir(entryp, path);
	  dbgprint(F_TRACE, (stderr, "Update_MakeDir - being implemented\n"));
	  return TRUE;
	}
      else
	{
	  Update_Remove(path);
	}
      VerboseMessage((stdout, "MKDIR %s\n", path));
      if (mkdir(path, 0755) < 0)
	{
	  FatalError(E_MKDIRFAILED, (stderr, "Could not make directory %s\n", path));
	}
    }

  dbgprint(F_TRACE, (stderr, "Update_MakeDir - being implemented\n"));
  return TRUE;
}


void Update_ExecRCFile(commandfilep)
     COMMANDFILE *commandfilep;
{
  char **av;
  struct stat stb;
  int pid;
  union wait ExecStatus;

  dbgprint(F_TRACE, (stderr, "Update_ExecRCFile\n"));

  if (Depot_TrustTargetDir && (commandfilep->status & S_TRUSTTARGET))
    {
      dbgprint(F_TRUSTTARGET, (stderr, "EXEC %s - Trusting targetdir!\n", commandfilep->label));
    }
  else
    {
      if (commandfilep->command == NULL)
	{
	  FatalError(E_NOEXECFILE, (stderr, "No executable file found for label %s\n", commandfilep->label));
	}
      /* expand any "%t"s in the command to the targetdir */
      av = DepotDB_Command_ExpandMagic(commandfilep->command);

      if (stat(av[0], &stb) < 0)
	{ FatalError(E_STATFAILED, (stderr, "Could not stat %s; errno =%d\n", av[0], errno)); }
      if (!(stb.st_mode & 0111))
	{ /* no exec permission */
	  FatalError(E_BADEXECFILE, (stderr, "Could not execute %s; permission denied\n", av[0]));
	}

      VerboseMessage((stdout, "EXEC %s\n", commandfilep->label));
  
      pid = vfork();
      if ( pid < 0 )
	{
	  FatalError(E_VFORKFAILED, (stderr, "Could not execute %s; vfork failed\n", av[0]));
	}
      else if ( pid == 0 )
	{ /* child */
	  execv(av[0], av);
	  _exit(0);      /* fluff */
	}
      wait(&ExecStatus);

      if (ExecStatus.w_retcode != 0)
	{
	  FatalError(E_EXECFAILED, (stderr, "Execution of %s failed with exit status = %d\n", av[0], ExecStatus.w_retcode));
	}

      strarrfree(av);
    }
  dbgprint(F_TRACE, (stderr, "Update_ExecRCFile done\n"));
  return;
}


/*
 * static void Update_Remove(path)
 * Removes any file/dir/symlink represented by path - Handle with care!!
 */
static void Update_Remove(path)
     char *path;
{
  char newpath[MAXPATHLEN];
  register struct direct *de;
  register DIR *dp;
  struct stat stb;

  if (lstat(path,&stb) < 0)
    {
      if (errno == ENOENT) return;
      else
	{
	  FatalError(E_LSTATFAILED, (stderr, "Could not lstat %s; errno =%d\n", path, errno));
	}
    }

  if ((stb.st_mode & S_IFMT) != S_IFDIR)
    {
      VerboseMessage((stdout, "REMOVE %s\n", path));
      if (unlink(path) < 0)
	{
	  FatalError(E_UNLINKFAILED, (stderr, "Could not unlink %s; errno =%d\n", path, errno));
	}
      else return;
    }

  if ((dp = opendir(path)) == NULL)
    {
      FatalError(E_OPENDIRFAILED, (stderr, "Could not open directory %s\n", path));
    }

  while ((de = readdir(dp)) != NULL)
    {
      if ( (strcmp(de->d_name, ".") != 0) && (strcmp(de->d_name, "..") != 0))
	{
	  (void)sprintf(newpath, "%s/%s", path, de->d_name);
	  Update_Remove(newpath);
	}
    }
  (void)closedir(dp);
  VerboseMessage((stdout, "REMOVE DIRECTORY %s\n", path));
  if (rmdir(path) < 0)
    {
      FatalError(E_RMDIRFAILED, (stderr, "Could not remove directory %s; errno =%d\n", path, errno));
    }
  else return;
}



/*
 * static void Update_CleanDir(entryp, path)
 * removes all unreferenced files other than depot under path
 */
static void Update_CleanDir(entryp, path)
     ENTRY *entryp;
     char *path;
{
  register ENTRY *ep;
  char newpath[MAXPATHLEN];
  register struct direct *de;
  register DIR *dp;
  Boolean FoundReference;

  if ((dp = opendir(path)) == NULL)
    {
      if (errno == ENOENT) return;
      if (errno == ENOTDIR) {
	FatalError(E_NOTDIR, 
		   (stderr, "Update_CleanDir: %s is not a directory\n", path));
      } else {
	FatalError(E_OPENDIRFAILED, (stderr, "Could not open directory %s\n", path));
      }
    }

  while ((de = readdir(dp)) != NULL)
    {
      if ( (strcmp(de->d_name, ".") != 0)
	  && (strcmp(de->d_name, "..") != 0))
	{
	  /* KLUDGEY - write a routine to do this in DepotDBUtil.c? */
	  ep = ENTRY_child(entryp); FoundReference = FALSE;
	  while ( (ep!= NULL) && !FoundReference)
	    {
	      if (ep->name == NULL)
		{
		  FatalError(E_UNNAMEDENTRY, (stderr, "Entry with no name found in database tree\n"));
		}
	      else if (strcmp(ep->name, de->d_name) == 0)
		FoundReference = TRUE;
	      ep = ENTRY_sibling(ep);
	    }
	  if (!FoundReference)
	    {
	      (void)sprintf(newpath, "%s/%s", path, de->d_name);
	      Update_Remove(newpath);
	    }
	}
    }
  (void)closedir(dp);

  return;
}
/*
 * $Log: update.c,v $
 * Revision 4.7  1992/02/12  18:04:23  ww0r
 * made rcs headers visible in the compiled programs
 *
 * Revision 4.6  1992/02/12  16:35:25  ww0r
 * add support for utime(3)
 *
 * Revision 4.5  1992/02/03  22:06:20  ww0r
 * return proper value for Update_MakeDir
 *
 * Revision 4.4  1992/01/18  19:07:12  ww0r
 * fixed ambiguous 'if/else' combo
 *
 * Revision 4.3  1992/01/18  18:45:46  ww0r
 * added ibm032 support
 *
 * Revision 4.2  1992/01/18  18:14:44  ww0r
 * removed some verbose messages to compenstate for slow scrolling on
 * console on some machines
 *
 * removed a stat from Update_CleanDir. The function of the stat is taken
 * care of in the opendir call.
 *
 * Revision 4.1  1991/10/08  21:53:56  dl2n
 * add CMU copyright
 *
 * Revision 4.0  1991/09/25  17:36:39  sohan
 * Release 4
 *
 * Revision 4.0  1991/09/25  16:08:52  sohan
 * Version 4.0
 *
 */
/* $Source: /afs/andrew.cmu.edu/system/src/local/depot/016/RCS/update.c,v $ */
