/******************************************************************************
 *                                                                            *
 * File:   readtab .c         Version  1.12            Date: 1997-05-01       *
 *                                                                            *
 * Copyright (C) 1993-1997 by kostis@acm.org (Kosta Kostis)                   *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *     1997-05-01: KK V1.12                                                   *
 *        - new release of package                                            *
 *     1996-06-01: KK V1.11                                                   *
 *        - new release of package, change of email address                   *
 *     1995-07-01: KK V1.10                                                   *
 *        - fixed bug in bstrcmp (must be ushort instead of int)              *
 *        - fixed bug in ReadCodeTable (Code must be ushort instead of uint)  *
 *     1994-03-31: KK V1.00                                                   *
 *        - some support for ISO 10646                                        *
 *     1994-03-04: KK V0.97                                                   *
 *        - new release of the package                                        *
 *     1994-01-18: KK V0.96                                                   *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "os-stuff.h"
#include "taberr.h"
#include "tab.h"

#include "readtab.h"

/******************************************************************************

	Function:
		find string position in sorted array using binary search

	Parameters:
		char	*Key		name to find
		char	*Names	[]	search table
		int	MaxTableEntries	max. number of entries in table

	Returns:
		MaxTableEntries		if code is not found
		ushort	Code		code found

	Note:
		the searched array *must* be sorted alphabetically

 *****************************************************************************/

static	ushort	bstrcmp
(
	char	*Key,
	char	*Names		[],
	ushort	MaxTableEntries
)
{
	ushort	LoIndex	= 0 ;
	ushort	HiIndex = MaxTableEntries - 1 ;
	ushort	MiIndex ;
	int	rc ;

	/**********************************************************************

		Because MiIndex never reaches HiIndex, this index must
		be checked first.

	 *********************************************************************/

	if (strcmp (Key, Names [HiIndex]) == 0)
		return (HiIndex) ;

	do
	{
		/**************************************************************

			we either find the string or at least we know which
			half of the array to search

		 *************************************************************/

		MiIndex = LoIndex + ((HiIndex - LoIndex) / 2) ;
		rc = strcmp (Key, Names [MiIndex]) ;
		if (rc == 0)
			return (MiIndex) ;

		/**************************************************************

			divide the number of strings searched by 2

		 *************************************************************/

		if (rc < 0)
			HiIndex = MiIndex ;
		else
			LoIndex = MiIndex ;
	}
	while (HiIndex - LoIndex > 1) ;

	/**********************************************************************

		if the string is not found in the array (illegal name)
		return an illegal value...

	 *********************************************************************/

	return (MaxTableEntries) ;
}

/******************************************************************************

	Function:
		delete trailing white space in a string

	Parameters:
		char    *String         string to be modified

	Returns:
		char    *String         if the pointer is valid
		NULL                    if String == (char *) NULL

 *****************************************************************************/

static	char    *strdeltw
(
	char    *String
)
{
	char    *Help ;

	/**********************************************************************

		is the parameter valid?

	 *********************************************************************/

	if (String == (char *) NULL)
		return (String) ;

	/**********************************************************************

		get the address of the end of the string

	 *********************************************************************/

	Help = & String [strlen (String)] ;

	/**********************************************************************

		if the string isn't empty, delete trailing white space

	 *********************************************************************/

	if (Help != String)
	{
		--Help ;
		if (isspace ((uchar) *Help))
		{
			while (isspace ((uchar) *Help))
			{
				if (Help == String)
				{
					*String = '\0' ;
					return (String) ;
				}
				--Help ;
			}
			*++Help = '\0' ;
		}
	}

	return (String) ;
}

/******************************************************************************

	Function:
		initialize character encoding description table

	Parameters:
		char    *Names  []      table to be initialized
		ushort	*Codes		codes, if used
		ushort	MaxTableEntries	number of entries in table

	Returns:
		nothing

 *****************************************************************************/

static	void	InitCodeTable
(
	char	*Names	[],
	ushort	*Codes,
	ushort	MaxTableEntries
)
{
	ushort	i ;

	for (i = 0 ; i < MaxTableEntries ; ++i)
	{
		Names [i] = (char *) NULL ;
		if (Codes != (ushort *) NULL)
			Codes [i] = BAD ;
	}
}

/******************************************************************************

	Function:
		free character encoding description table

	Parameters:
		char    *Table  []      table to be freed
		ushort	MaxTableEntries	number of entries in table

	Returns:
		nothing

 *****************************************************************************/

void	FreeCodeTable
(
	char    *Names	[],
	ushort	MaxTableEntries
)
{
	ushort	i ;

	for (i = 0 ; i < MaxTableEntries ; ++i)
	{
		if (Names [i] != (char *) NULL)
		{
			free (Names [i]) ;
			Names [i] = (char *) NULL ;
		}
	}
}

/******************************************************************************

	Function:
		read character encoding description table from a file

	Parameters:
		char    *FileName       file to be read
		char    *Names	[]	table to be filled
		ushort	*Codes
		ushort	MaxTableEntries	max. number of table entries
		char    *Encoding       name of the character encoding
		char    *Comment []     comment for character encoding

	Returns:
		ERR_NONE                if all goes well
		ERR_OPEN                if unable to open file
		ERR_READ                if unable to read file

	Note:
		if Codes != NULL the Names [] *must* be sorted

 *****************************************************************************/

int	ReadCodeTable
(
	char	*FileName,
	char	*Names		[],
	ushort	*Codes,
	ushort	MaxTableEntries,
	char	*Encoding,
	char	*Comment	[]
)
{
	FILE    *Fin    = (FILE *) NULL ;
	int     i ;
	int     rc ;
	int     ch ;
	char    *Help ;
	char    ISOname [MAX_ISO_NAME_LENGTH + 1] ;
	char    Buffer  [MAX_COMMENT_LENGTH + 1] ;
	ushort	Code ;
	ushort	Line ;

	/**********************************************************************

		initialize code table(s) (all pointers point to NULL)

	 *********************************************************************/

	InitCodeTable (Names, Codes, MaxTableEntries) ;

	/**********************************************************************

		open code description file

	 *********************************************************************/

	Fin = fopen (FileName, "r") ;
	if (Fin == (FILE *) NULL)
	{
		fprintf (stderr, ERR_MSG_OPEN, PRGNAM, FileName) ;
		return (ERR_OPEN) ;
	}

	/**********************************************************************

		read character encoding name & strip off trailing white space

	 *********************************************************************/

	strcpy (Encoding, "") ;
	if (fgets (Encoding, MAX_ENC_NAME_LENGTH, Fin) == (char *) NULL)
	{
		fclose (Fin) ;
		fprintf (stderr, ERR_MSG_CODE, PRGNAM, FileName, 1) ;
		return (ERR_CODE) ;
	}

	strdeltw (Encoding) ;

	/**********************************************************************

		read character encoding comments & strip off trailing white spc

	 *********************************************************************/

	for (i = 0 ; i < MAX_COMMENTS ; ++i)
	{
		if (fgets (Buffer, MAX_COMMENT_LENGTH, Fin) == (char *) NULL)
		{
			fclose (Fin) ;
			fprintf (stderr, ERR_MSG_CODE,
				PRGNAM, FileName, i + 2) ;
			return (ERR_CODE) ;
		}

		strdeltw (Buffer) ;
		Comment [i] = strdup (Buffer) ;
	}

	/**********************************************************************

		read code description file line by line

	 *********************************************************************/

	for (Line = 0 ; feof (Fin) == 0 ; ++Line)
	{
		/**************************************************************

			first get the code number (index)

		 *************************************************************/

		rc = fscanf (Fin, "%hx", &Code) ;
		if (feof (Fin))
		{
			fclose (Fin) ;
			return (ERR_NONE) ;
		}

		/**************************************************************

			is the code valid?

		 *************************************************************/

		if
		(
			(rc != 1) ||
			(
				(Code >= MaxTableEntries) &&
				(Codes == (ushort *) NULL)
			)
		)
		{
			fclose (Fin) ;
			FreeCodeTable (Names, MaxTableEntries) ;
			fprintf (stderr, ERR_MSG_CODE,
				PRGNAM, FileName, Line + 5) ;
			return (ERR_CODE) ;
		}

		/**************************************************************

			skip first tab - if eoln is read, skip line

		 *************************************************************/

		ch = fgetc (Fin) ;
		if (ch == '\n')
			continue ;

		if (ch != '\t')
		{
			fclose (Fin) ;
			FreeCodeTable (Names, MaxTableEntries) ;
			fprintf (stderr, ERR_MSG_CODE,
				PRGNAM, FileName, Line + 5) ;
			return (ERR_CODE) ;
		}

		/**************************************************************

			skip comment characters - if eoln is read, skip line

		 *************************************************************/

		do
		{
			ch = fgetc (Fin) ;
		}
		while ((ch != '\t') && (ch != '\n') && (ch != EOF)) ;

		if (ch == '\n')
			continue ;

		if (ch == EOF)
		{
			fclose (Fin) ;
			FreeCodeTable (Names, MaxTableEntries) ;
			fprintf (stderr, ERR_MSG_CODE,
				PRGNAM, FileName, Line + 5) ;
			return (ERR_CODE) ;
		}

		/**************************************************************

			now read character description (rest of line)

		 *************************************************************/

		if (fgets (ISOname, MAX_ISO_NAME_LENGTH, Fin) != (char *) NULL)
		{
			/******************************************************

				prevent code redfinition

			 *****************************************************/

			if
			(
				(Codes == (ushort *) NULL) &&
				(Names [Code] != (char *) NULL)
			)
			{
				fclose (Fin) ;
				FreeCodeTable (Names, MaxTableEntries) ;
				fprintf (stderr, ERR_MSG_REDEFINED,
					PRGNAM, FileName, Code, Line) ;
				return (ERR_REDEFINED) ;
			}

			/******************************************************

				cut off comment strings (in parenthesis)

			 *****************************************************/

			Help = strchr (ISOname, '(') ;
			if (Help != (char *) NULL)
				*Help = '\0' ;

			/******************************************************

				ignore case (convert to upper case)
				change HYPHEN-MINUS and LOW LINE to SPACE

			 *****************************************************/

			strupr (ISOname) ;      /* not ANSI-C, but also fine */

			/******************************************************

				cut off trailing white space

			 *****************************************************/

			strdeltw (ISOname) ;

			/******************************************************

				skip this one, if ISOname is "empty"

			 *****************************************************/

			if (strlen (ISOname) == 0)
				continue ;

			/******************************************************

				allocate memory for the string

			 *****************************************************/

			Help = strdup (ISOname) ;
			if (Help == (char *) NULL)
			{
				fclose (Fin) ;
				FreeCodeTable (Names, MaxTableEntries) ;
				fprintf (stderr, ERR_MSG_MEMORY,
					PRGNAM, FileName, Line) ;
				return (ERR_MEMORY) ;
			}

			/******************************************************

				finally store description string in table

			 *****************************************************/

			if (Codes == (ushort *) NULL)
			{
				Names [Code] = Help ;
			}
			else
			{
				Names [Line] = Help ;
				Codes [Line] = Code ;
			}
		}
	}

	fclose (Fin) ;

	return (ERR_NONE) ;
}

/******************************************************************************

	Function:
		dump character encoding description table

	Parameters:
		FILE	*Fout		output file
		char	*FileName	table filename
		char	*Names	[]	table to be dumped
		ushort	*Codes		codes, if used
		ushort	MaxTableEntries	number of entries in table

	Returns:
		nothing

 *****************************************************************************/

void	DumpCodeTable
(
	FILE	*Fout,
	char	*FileName,
	char	*Names		[],
	ushort	*Codes,
	ushort	MaxTableEntries
)
{
	ushort	i ;

	fprintf (Fout, "\nFile: %s\n\n", FileName) ;

	for (i = 0 ; i < MaxTableEntries ; ++i)
	{
		if (Names [i] != (char *) NULL)
		{
			if (Codes == (ushort *) NULL)
			{
				fprintf (Fout, "%04X: \'%s\'\n",
					i, Names [i]) ;
			}
			else
			{
				fprintf (Fout, "%04X: \'%s\'\n",
					Codes [i], Names [i]) ;
			}
		}
	}
}

/******************************************************************************

	Function:
		find character code in character description table

	Parameters:
		char	*Key		code to find
		char	*Names	[]	search table
		ushort	*Codes
		ushort	MaxTableEntries	max. number of entries in table

	Returns:
		BAD			if code is not found
		ushort	Code		code found

	Note:
		if Codes != NULL the Names [] *must* be sorted

 *****************************************************************************/

ushort	FindCode
(
	char    *Key,
	char	*Names		[],
	ushort	*Codes,
	ushort	MaxTableEntries
)
{
	ushort	i ;
	ushort	Pos ;

	/**********************************************************************

		make sure the parameters are valid

	 *********************************************************************/

	if (Key == (char *) NULL)
		return (BAD) ;

	if (strlen (Key) == 0)
		return (BAD) ;

	/**********************************************************************

		now step through the destination code descriptions

	 *********************************************************************/

	if (Codes == (ushort *) NULL)
	{
		for (i = 0 ; i < MaxTableEntries ; ++i)
			if (Names [i] != (char *) NULL)
				if (strcmp (Key, Names [i]) == 0)
					return (i) ;
	}
	else
	{
		Pos = (ushort) bstrcmp (Key, Names, (int) MaxTableEntries) ;
		if (Pos != MaxTableEntries)
			return (Codes [Pos]) ;
	}

	return (BAD) ;
}
