;;; charset-konvers.el --- conversion of text

;; Copyright (C) 1995 Karl Brodowsky

;; Author: Karl Brodowsky <bk1@ix.urz.uni-heidelberg.de>
;; Version: 0.500
;; Keywords: i18n, tex, wp

;; This file was written for GNU Emacs version 19

;; It is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; It is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.	If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; created May 1995
;; by Karl Brodowsky
;; using tables provided by Kosta Kostis
;;
;; It provides conversion for different character sets
;;
;; konvers-iso-to-os2 converts region from ISO-8859-1 to cp850
;; konvers-iso-to-nxt converts region from ISO-8859-1 to nextstep encoding
;; konvers-os2-to-iso converts region from cp850 to ISO-8859-1
;; konvers-nxt-to-iso converts region from nextstep encoding to ISO-8859-1
;;
;; more conversions might be added later.

;;; Code:

(require 'iso-konvers)


(defvar konvers-iso-to-os2-alist nil
  "alist describing the replacements for conversion iso to os2
more precisely: iso8859.1 (short iso) to cp850 (short os2)")

(setq konvers-iso-to-os2-alist
      '(
	("\240" . "\377")	; NO-BREAK SPACE
	("\241" . "\255")	; INVERTED EXCLAMATION MARK
	("\242" . "\275")	; CENT SIGN
	("\243" . "\234")	; POUND SIGN
	("\244" . "\317")	; CURRENCY SIGN
	("\245" . "\276")	; YEN SIGN
	("\246" . "\335")	; BROKEN BAR
	("\247" . "\365")	; SECTION SIGN
	("\250" . "\371")	; DIAERESIS
	("\251" . "\270")	; COPYRIGHT SIGN
	("\252" . "\246")	; FEMININE ORDINAL INDICATOR
	("\253" . "\256")	; LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
	("\254" . "\252")	; NOT SIGN
	("\255" . "\360")	; SOFT HYPHEN
	("\256" . "\251")	; REGISTERED SIGN
	("\257" . "\356")	; MACRON
	("\260" . "\370")	; DEGREE SIGN
	("\261" . "\361")	; PLUS-MINUS SIGN
	("\262" . "\375")	; SUPERSCRIPT TWO
	("\263" . "\374")	; SUPERSCRIPT THREE
	("\264" . "\357")	; ACUTE ACCENT
	("\265" . "\346")	; MICRO SIGN
	("\266" . "\364")	; PILCROW SIGN
	("\267" . "\372")	; MIDDLE DOT
	("\270" . "\367")	; CEDILLA
	("\271" . "\373")	; SUPERSCRIPT ONE
	("\272" . "\247")	; MASCULINE ORDINAL INDICATOR
	("\273" . "\257")	; RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
	("\274" . "\254")	; VULGAR FRACTION ONE QUARTER
	("\275" . "\253")	; VULGAR FRACTION ONE HALF
	("\276" . "\363")	; VULGAR FRACTION THREE QUARTERS
	("\277" . "\250")	; INVERTED QUESTION MARK
	("\300" . "\267")	; LATIN CAPITAL LETTER A WITH GRAVE
	("\301" . "\265")	; LATIN CAPITAL LETTER A WITH ACUTE
	("\302" . "\266")	; LATIN CAPITAL LETTER A WITH CIRCUMFLEX
	("\303" . "\307")	; LATIN CAPITAL LETTER A WITH TILDE
	("\304" . "\216")	; LATIN CAPITAL LETTER A WITH DIAERESIS
	("\305" . "\217")	; LATIN CAPITAL LETTER A WITH RING ABOVE
	("\306" . "\222")	; LATIN CAPITAL LIGATURE AE
	("\307" . "\200")	; LATIN CAPITAL LETTER C WITH CEDILLA
	("\310" . "\324")	; LATIN CAPITAL LETTER E WITH GRAVE
	("\311" . "\220")	; LATIN CAPITAL LETTER E WITH ACUTE
	("\312" . "\322")	; LATIN CAPITAL LETTER E WITH CIRCUMFLEX
	("\313" . "\323")	; LATIN CAPITAL LETTER E WITH DIAERESIS
	("\314" . "\336")	; LATIN CAPITAL LETTER I WITH GRAVE
	("\315" . "\326")	; LATIN CAPITAL LETTER I WITH ACUTE
	("\316" . "\327")	; LATIN CAPITAL LETTER I WITH CIRCUMFLEX
	("\317" . "\330")	; LATIN CAPITAL LETTER I WITH DIAERESIS
	("\320" . "\321")	; LATIN CAPITAL LETTER ETH
	("\321" . "\245")	; LATIN CAPITAL LETTER N WITH TILDE
	("\322" . "\343")	; LATIN CAPITAL LETTER O WITH GRAVE
	("\323" . "\340")	; LATIN CAPITAL LETTER O WITH ACUTE
	("\324" . "\342")	; LATIN CAPITAL LETTER O WITH CIRCUMFLEX
	("\325" . "\345")	; LATIN CAPITAL LETTER O WITH TILDE
	("\326" . "\231")	; LATIN CAPITAL LETTER O WITH DIAERESIS
	("\327" . "\236")	; MULTIPLICATION SIGN
	("\330" . "\235")	; LATIN CAPITAL LETTER O WITH STROKE
	("\331" . "\353")	; LATIN CAPITAL LETTER U WITH GRAVE
	("\332" . "\351")	; LATIN CAPITAL LETTER U WITH ACUTE
	("\333" . "\352")	; LATIN CAPITAL LETTER U WITH CIRCUMFLEX
	("\334" . "\232")	; LATIN CAPITAL LETTER U WITH DIAERESIS
	("\335" . "\355")	; LATIN CAPITAL LETTER Y WITH ACUTE
	("\336" . "\350")	; LATIN CAPITAL LETTER THORN
	("\337" . "\341")	; LATIN SMALL LETTER SHARP S
	("\340" . "\205")	; LATIN SMALL LETTER A WITH GRAVE
	("\341" . "\240")	; LATIN SMALL LETTER A WITH ACUTE
	("\342" . "\203")	; LATIN SMALL LETTER A WITH CIRCUMFLEX
	("\343" . "\306")	; LATIN SMALL LETTER A WITH TILDE
	("\344" . "\204")	; LATIN SMALL LETTER A WITH DIAERESIS
	("\345" . "\206")	; LATIN SMALL LETTER A WITH RING ABOVE
	("\346" . "\221")	; LATIN SMALL LIGATURE AE
	("\347" . "\207")	; LATIN SMALL LETTER C WITH CEDILLA
	("\350" . "\212")	; LATIN SMALL LETTER E WITH GRAVE
	("\351" . "\202")	; LATIN SMALL LETTER E WITH ACUTE
	("\352" . "\210")	; LATIN SMALL LETTER E WITH CIRCUMFLEX
	("\353" . "\211")	; LATIN SMALL LETTER E WITH DIAERESIS
	("\354" . "\215")	; LATIN SMALL LETTER I WITH GRAVE
	("\355" . "\241")	; LATIN SMALL LETTER I WITH ACUTE
	("\356" . "\214")	; LATIN SMALL LETTER I WITH CIRCUMFLEX
	("\357" . "\213")	; LATIN SMALL LETTER I WITH DIAERESIS
	("\360" . "\320")	; LATIN SMALL LETTER ETH
	("\361" . "\244")	; LATIN SMALL LETTER N WITH TILDE
	("\362" . "\225")	; LATIN SMALL LETTER O WITH GRAVE
	("\363" . "\242")	; LATIN SMALL LETTER O WITH ACUTE
	("\364" . "\223")	; LATIN SMALL LETTER O WITH CIRCUMFLEX
	("\365" . "\344")	; LATIN SMALL LETTER O WITH TILDE
	("\366" . "\224")	; LATIN SMALL LETTER O WITH DIAERESIS
	("\367" . "\366")	; DIVISION SIGN
	("\370" . "\233")	; LATIN SMALL LETTER O WITH STROKE
	("\371" . "\227")	; LATIN SMALL LETTER U WITH GRAVE
	("\372" . "\243")	; LATIN SMALL LETTER U WITH ACUTE
	("\373" . "\226")	; LATIN SMALL LETTER U WITH CIRCUMFLEX
	("\374" . "\201")	; LATIN SMALL LETTER U WITH DIAERESIS
	("\375" . "\354")	; LATIN SMALL LETTER Y WITH ACUTE
	("\376" . "\347")	; LATIN SMALL LETTER THORN
	("\377" . "\230")	; LATIN SMALL LETTER Y WITH DIAERESIS
))		; konvers-iso-to-os2-alist


(defun konvers-iso-to-os2 (from to)
  "Transforms region from iso to os2"
  (interactive "*r")
  (konvers-replace-string-set from to konvers-iso-to-os2-alist))

(defun konvers-buffer-iso-to-os2 (&optional arg)
  "convert full buffer from iso to os2
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-iso-to-os2 (konvers-point-min-or-behind-header arg) (point-max)))

(defun konvers-os2-to-iso (from to)
  "Transforms region from os2 to iso"
  (interactive "*r")
  (konvers-replace-string-set from to (konvers-anti-set konvers-iso-to-os2-alist)))

(defun konvers-buffer-os2-to-iso (&optional arg)
  "convert full buffer from os2 to iso
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-os2-to-iso (konvers-point-min-or-behind-header arg) (point-max)))


(defvar konvers-iso-to-nxt-alist nil
  "alist describing the replacements for conversion iso to nxt
more precisely: iso8859.1 (short iso) to nextstep.enc (short nxt)")

(setq konvers-iso-to-nxt-alist
      '(
	("\240" . "\200")	; NO-BREAK SPACE
	("\244" . "\250")	; CURRENCY SIGN
	("\246" . "\265")	; BROKEN BAR
	("\250" . "\310")	; DIAERESIS
	("\251" . "\240")	; COPYRIGHT SIGN
	("\252" . "\343")	; FEMININE ORDINAL INDICATOR
	("\254" . "\276")	; NOT SIGN
	("\256" . "\260")	; REGISTERED SIGN
	("\257" . "\305")	; MACRON
	("\260" . "\312")	; DEGREE SIGN
	("\261" . "\321")	; PLUS-MINUS SIGN
	("\262" . "\311")	; SUPERSCRIPT TWO
	("\263" . "\314")	; SUPERSCRIPT THREE
	("\264" . "\302")	; ACUTE ACCENT
	("\265" . "\235")	; MICRO SIGN
	("\267" . "\264")	; MIDDLE DOT
	("\270" . "\313")	; CEDILLA
	("\271" . "\300")	; SUPERSCRIPT ONE
	("\272" . "\353")	; MASCULINE ORDINAL INDICATOR
	("\274" . "\322")	; VULGAR FRACTION ONE QUARTER
	("\275" . "\323")	; VULGAR FRACTION ONE HALF
	("\276" . "\324")	; VULGAR FRACTION THREE QUARTERS
	("\300" . "\201")	; LATIN CAPITAL LETTER A WITH GRAVE
	("\301" . "\202")	; LATIN CAPITAL LETTER A WITH ACUTE
	("\302" . "\203")	; LATIN CAPITAL LETTER A WITH CIRCUMFLEX
	("\303" . "\204")	; LATIN CAPITAL LETTER A WITH TILDE
	("\304" . "\205")	; LATIN CAPITAL LETTER A WITH DIAERESIS
	("\305" . "\206")	; LATIN CAPITAL LETTER A WITH RING ABOVE
	("\306" . "\341")	; LATIN CAPITAL LIGATURE AE
	("\307" . "\207")	; LATIN CAPITAL LETTER C WITH CEDILLA
	("\310" . "\210")	; LATIN CAPITAL LETTER E WITH GRAVE
	("\311" . "\211")	; LATIN CAPITAL LETTER E WITH ACUTE
	("\312" . "\212")	; LATIN CAPITAL LETTER E WITH CIRCUMFLEX
	("\313" . "\213")	; LATIN CAPITAL LETTER E WITH DIAERESIS
	("\314" . "\214")	; LATIN CAPITAL LETTER I WITH GRAVE
	("\315" . "\215")	; LATIN CAPITAL LETTER I WITH ACUTE
	("\316" . "\216")	; LATIN CAPITAL LETTER I WITH CIRCUMFLEX
	("\317" . "\217")	; LATIN CAPITAL LETTER I WITH DIAERESIS
	("\320" . "\220")	; LATIN CAPITAL LETTER ETH
	("\321" . "\221")	; LATIN CAPITAL LETTER N WITH TILDE
	("\322" . "\222")	; LATIN CAPITAL LETTER O WITH GRAVE
	("\323" . "\223")	; LATIN CAPITAL LETTER O WITH ACUTE
	("\324" . "\224")	; LATIN CAPITAL LETTER O WITH CIRCUMFLEX
	("\325" . "\225")	; LATIN CAPITAL LETTER O WITH TILDE
	("\326" . "\226")	; LATIN CAPITAL LETTER O WITH DIAERESIS
	("\327" . "\236")	; MULTIPLICATION SIGN
	("\330" . "\351")	; LATIN CAPITAL LETTER O WITH STROKE
	("\331" . "\227")	; LATIN CAPITAL LETTER U WITH GRAVE
	("\332" . "\230")	; LATIN CAPITAL LETTER U WITH ACUTE
	("\333" . "\231")	; LATIN CAPITAL LETTER U WITH CIRCUMFLEX
	("\334" . "\232")	; LATIN CAPITAL LETTER U WITH DIAERESIS
	("\335" . "\233")	; LATIN CAPITAL LETTER Y WITH ACUTE
	("\336" . "\234")	; LATIN CAPITAL LETTER THORN
	("\337" . "\373")	; LATIN SMALL LETTER SHARP S
	("\340" . "\325")	; LATIN SMALL LETTER A WITH GRAVE
	("\341" . "\326")	; LATIN SMALL LETTER A WITH ACUTE
	("\342" . "\327")	; LATIN SMALL LETTER A WITH CIRCUMFLEX
	("\343" . "\330")	; LATIN SMALL LETTER A WITH TILDE
	("\344" . "\331")	; LATIN SMALL LETTER A WITH DIAERESIS
	("\345" . "\332")	; LATIN SMALL LETTER A WITH RING ABOVE
	("\346" . "\361")	; LATIN SMALL LIGATURE AE
	("\347" . "\333")	; LATIN SMALL LETTER C WITH CEDILLA
	("\350" . "\334")	; LATIN SMALL LETTER E WITH GRAVE
	("\351" . "\335")	; LATIN SMALL LETTER E WITH ACUTE
	("\352" . "\336")	; LATIN SMALL LETTER E WITH CIRCUMFLEX
	("\353" . "\337")	; LATIN SMALL LETTER E WITH DIAERESIS
	("\354" . "\340")	; LATIN SMALL LETTER I WITH GRAVE
	("\355" . "\342")	; LATIN SMALL LETTER I WITH ACUTE
	("\356" . "\344")	; LATIN SMALL LETTER I WITH CIRCUMFLEX
	("\357" . "\345")	; LATIN SMALL LETTER I WITH DIAERESIS
	("\360" . "\346")	; LATIN SMALL LETTER ETH
	("\361" . "\347")	; LATIN SMALL LETTER N WITH TILDE
	("\362" . "\354")	; LATIN SMALL LETTER O WITH GRAVE
	("\363" . "\355")	; LATIN SMALL LETTER O WITH ACUTE
	("\364" . "\356")	; LATIN SMALL LETTER O WITH CIRCUMFLEX
	("\365" . "\357")	; LATIN SMALL LETTER O WITH TILDE
	("\366" . "\360")	; LATIN SMALL LETTER O WITH DIAERESIS
	("\367" . "\237")	; DIVISION SIGN
	("\370" . "\371")	; LATIN SMALL LETTER O WITH STROKE
	("\371" . "\362")	; LATIN SMALL LETTER U WITH GRAVE
	("\372" . "\363")	; LATIN SMALL LETTER U WITH ACUTE
	("\373" . "\364")	; LATIN SMALL LETTER U WITH CIRCUMFLEX
	("\374" . "\366")	; LATIN SMALL LETTER U WITH DIAERESIS
	("\375" . "\367")	; LATIN SMALL LETTER Y WITH ACUTE
	("\376" . "\374")	; LATIN SMALL LETTER THORN
	("\377" . "\375")	; LATIN SMALL LETTER Y WITH DIAERESIS
))		; konvers-iso-to-nxt-alist


(defun konvers-iso-to-nxt (from to)
  "Transforms region from iso to nxt"
  (interactive "*r")
  (konvers-replace-string-set from to konvers-iso-to-nxt-alist))

(defun konvers-buffer-iso-to-nxt (&optional arg)
  "convert full buffer from iso to nxt
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-iso-to-nxt (konvers-point-min-or-behind-header arg) (point-max)))

(defun konvers-nxt-to-iso (from to)
  "Transforms region from nxt to iso"
  (interactive "*r")
  (konvers-replace-string-set from to (konvers-anti-set konvers-iso-to-nxt-alist)))

(defun konvers-buffer-nxt-to-iso (&optional arg)
  "convert full buffer from nxt to iso
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-nxt-to-iso (konvers-point-min-or-behind-header arg) (point-max)))

(provide 'charset-konvers)

;;; charset-konvers.el ends here
