/******************************************************************************
 *                                                                            *
 * File:   transce8.c         Version  1.12            Date: 1997-05-01       *
 *                                                                            *
 * Copyright (C) 1993-1997 by kostis@acm.org (Kosta Kostis)                   *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - Translate Character Encoding 8-bit                                *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *     1997-05-01: KK V1.12                                                   *
 +        - new release of the package                                        *
 *     1996-06-01: KK V1.01                                                   *
 *        - changed email address                                             *
 *     1995-07-01: KK V1.00                                                   *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "datatype.h"
#include "os-stuff.h"
#include "tab.h"

#include "transce8.h"

#ifdef	NO_STRUPR
	#include "strupr.c"
#endif

#ifdef	USE_UTIMBUF
	#include "bin/utimbuf.h"
#endif

/******************************************************************************

	global variables

 *****************************************************************************/

static	ushort	BadChar = BAD_CHAR ;		/* default BAD character     */
static	char	*source = (char *) NULL ;	/* source encoding name      */
static	char	*target = (char *) NULL ;	/* target encoding name      */

/******************************************************************************

	Function:
		display copyright, author and program information

	Parameters:
		none

	Returns:
		nothing

 *****************************************************************************/

static  void    Hello (void)
{
	fprintf (stderr, MSG_COPYRIGHT) ;
	fprintf (stderr, MSG_VERSION, PRGNAM, VERSION) ;
	if ((source != (char *) NULL) && (target != (char *) NULL))
		fprintf (stderr, MSG_FROM_TO, source, target) ;
}

/******************************************************************************

	Function:
		load table from file

	Parameters:
		char	*Encoding
		ushort	*bintab
		char	*Path

	Returns:

 *****************************************************************************/

static	int	LoadTable
(
	char	*Encoding,
	ushort	*bintab
)
{
	FILE	*Fin ;
	char	FileName	[PATHNAME_MAX + 1] ;
	char	*Help ;

	Help = getenv (ENV_TRANS) ;
	if (Help == (char *) NULL)
		Help = "" ;

	sprintf (FileName, EXT_BINARY, Help, CHR_DIR, Encoding) ;

	Fin = fopen (FileName, "rb") ;
	if (Fin == (FILE *) NULL)
	{
		fprintf (stderr, ERR_MSG_TAB, PRGNAM, FileName) ;
		return (ERR_TAB) ;
	}

	fread (bintab, sizeof (ushort), MAX_TABLE_ENTRIES, Fin) ;	
	fclose (Fin) ;

	return (ERR_NONE) ;
}

/******************************************************************************

	Function:
		search ISO 10646 in table giving index

	Parameters:
		ushort	ISOcode ;

	Returns:
		BAD			if code can not be found
		ushort			target code

 *****************************************************************************/

static	ushort GetTarget
(
	ushort	ISOcode,
	ushort	*bintab
)
{
	ushort	i ;

	for (i = 0 ; i < MAX_TABLE_ENTRIES ; ++i)
		if (bintab [i] == ISOcode)
			return (i) ;

	return (BAD) ;
}

/******************************************************************************

	Function:
		copy input to output doing conversion

	Parameters:
		FILE    *Src            source file
		FILE    *Dst            destination file
		ushort	*srcbin		source encoding
		ushort	*trgbin		target encoding

	Returns:
		ERR_NONE                if all goes well
		ERR_BAD                 if characters have been untranslatable

 *****************************************************************************/

static  int     TransCE8
(
	FILE    *Src,
	FILE    *Dst,
	ushort	*srcbin,
	ushort	*trgbin
)
{
	int     ch ;			/* character read               */
	ushort	newch ;			/* translated character         */
	int     codes ;			/* bad code flag                */

	codes = ERR_NONE ;		/* assume no bad code           */

	while ((ch = fgetc (Src)) != EOF)	/* read character       */
	{
		newch = srcbin [ch] ;
		if (newch != BAD)
			newch = GetTarget (newch, trgbin) ;
		if (newch != BAD)
		{
			fputc (newch, Dst) ;	/* output character     */
		}
		else
		{
			codes = ERR_BAD ;
			fputc (BadChar, Dst) ;
		}
	}

	return (codes) ;
}

/*****************************************************************************/

static	char	*InpName ;	/* input file name                           */
static	char	*OutName ;	/* output file name                          */

/******************************************************************************

	ScanFlags () return codes

 *****************************************************************************/

#define	SCN_OK		0	/* parameters are all valid                  */
#define	SCN_UNKNOWN	1	/* unknown parameter used                    */
#define	SCN_BADARGS	2	/* illegal number of params                  */

/******************************************************************************

	Function:
		parse parameters and set flags and variables accordingly

	Parameters:
		int     argc    number of parameters
		char    *argv[] argument strings

	Returns:
		SCN_OK          if all parameters are valid
		SCN_UNKNOWN     if any of the parameters is unknown
		SCN_BADARGS     if the number of parameters is wrong

 *****************************************************************************/

static  int     ScanFlags
(
	int	argc,
	char	*argv	[]
)
{
	int	i ;		/* index through the parameters              */
	int	parms = 0 ;	/* yet no parameter scanned                  */

	/**********************************************************************

		set defaults / initialize

	**********************************************************************/

	InpName = (char *) NULL ;
	OutName = (char *) NULL ;

	for (i = 1 ; i < argc ; ++i)    /* scan command line                 */
	{
		/**************************************************************

			is it a flag/switch? (*NIX style)

		 *************************************************************/

		if (*argv [i] == '-')
		{
			/******************************************************

				check out which flag it is

			 *****************************************************/

			switch (argv [i][1])
			{
				case	'B':	/* change BAD character      */
				case	'b':

					BadChar = (int) argv [i][2] ;
					break ;

				case	'S':	/* set source encoding name  */
				case	's':

					source = & (argv [i][2]) ;
					break ;

				case	'T':	/* set target encoding name  */
				case	't':

					target = & (argv [i][2]) ;
					break ;

				default:        /* Uh! This flag's unknown!  */

					return (SCN_UNKNOWN) ;
			}
		}
		else
		{
			/******************************************************

				if it's no flag it must be a filename

			 *****************************************************/

			++parms ;       /* check for multiple filenames      */
			switch (parms)
			{
				case	1:

					InpName = argv [i] ;
					break ;

				case	2:

					OutName = argv [i] ;
					break ;

				default :

					return (SCN_BADARGS) ;
			}
		}
	}

	/**********************************************************************

		have encoding names been given?

	 *********************************************************************/

	if ((source == (char *) NULL) || (target == (char *) NULL))
		return (SCN_BADARGS) ;

	/**********************************************************************

		everything's been fine - let's quit!

	 *********************************************************************/

	return (SCN_OK) ;
}

/*****************************************************************************/

int     main
(
	int     argc,
	char    *argv   []
)
{
	FILE    *SRCfile = stdin ;      /* Source default is stdin           */
	FILE    *DSTfile = stdout ;     /* Dest. default is stdout           */
	char    *TmpName = (char *) NULL ;
	char    *Help ;
	int     codes ;                 /* bad code flag                     */
	int     syntax ;                /* return code for ScanFlags ()      */
	ushort	srcbin	[MAX_TABLE_ENTRIES] ;	/* source encoding table     */
	ushort	trgbin	[MAX_TABLE_ENTRIES] ;	/* target encoding table     */

#ifdef	USE_UTIMBUF
	struct  stat    f_stat ;
	struct  utimbuf f_date ;
#endif

	/**********************************************************************

		parse parameters/arguments

	**********************************************************************/

	syntax = ScanFlags (argc, argv) ;
	if (syntax != SCN_OK)
	{
		/**************************************************************

			display syntax error message and quit

		**************************************************************/

		Hello () ;
		fprintf (stderr, ERR_MSG_SYNTAX, PRGNAM) ;

		return (ERR_SYNTAX) ;
	}

	/**********************************************************************

		load tables

	**********************************************************************/

	if ((codes = LoadTable (source, srcbin)) != ERR_NONE)
		return (codes) ;

	if ((codes = LoadTable (target, trgbin)) != ERR_NONE)
		return (codes) ;

	/**********************************************************************

		check for output file - create if necessary

	**********************************************************************/

	if (OutName != (char *) NULL)
	{
		/**************************************************************

			file type must be binary not to trash Waffle U*IX
			style files for UUCP

		**************************************************************/

		DSTfile = fopen (OutName, "wb") ;
		if (DSTfile == (FILE *) NULL)
		{
			Hello () ;
			fprintf (stderr, ERR_MSG_DST, PRGNAM, OutName) ;

			return (ERR_DST) ;
		}
	}

	/**********************************************************************

		check for input file - open if necessary

	**********************************************************************/

	if (InpName != (char *) NULL)
	{       
		SRCfile = fopen (InpName, "rb") ;
		if (SRCfile == (FILE *) NULL)
		{
			Hello () ;
			fprintf (stderr, ERR_MSG_SRC, PRGNAM, InpName) ;
			if (DSTfile != stdout)
				fclose (DSTfile) ;

			return (ERR_SRC) ;
		}

		/**************************************************************

			if no output file name is supplied, we have to create
			a temporary file which gets renamed afterwards

			preserve original file date - if possible

		**************************************************************/

		if (OutName == (char *) NULL)
		{
#ifdef	USE_UTIMBUF
			stat (InpName, &f_stat) ;
			f_date.actime  = f_stat.st_atime ;
			f_date.modtime = f_stat.st_mtime ;
#endif

			TmpName = tmpnam ((char *) NULL) ;
			Help = strrchr (TmpName, '/') ;
			if (Help != (char *) NULL)
				TmpName = ++Help ;
			DSTfile = fopen (TmpName, "wb") ;
			if (DSTfile == (FILE *) NULL)
			{
				fclose (SRCfile) ;
				return (ERR_DST) ;
			}
		}
	}

	/**********************************************************************

		now do the actual conversion

	**********************************************************************/

	codes = TransCE8 (SRCfile, DSTfile, srcbin, trgbin) ;

	/**********************************************************************

		close files if opened

	**********************************************************************/

	if (DSTfile != stdout)
		fclose (DSTfile) ;
	if (SRCfile != stdin)
		fclose (SRCfile) ;

	/**********************************************************************

		if there is a temporary file, rename it to the original name

	**********************************************************************/

	if (TmpName != (char *) NULL)
	{
		remove (InpName) ;
		rename (TmpName, InpName) ;
#ifdef	USE_UTIMBUF
		utime (InpName, &f_date) ;
#endif
	}

	return (codes) ;
}
