/******************************************************************************
 *                                                                            *
 * File:   encode  .c         Version  1.30            Date: 1996-01-08       *
 *                                                                            *
 * Copyright (C) 1993-1996 by Kosta Kostis - this is freeware!                *
 * Written by kostis@acm.org (Kosta Kostis)                                   *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - convert 8-bit text/binary to quoted-printables or base64          *
 *        - add/correct MIME-header                                           *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *    1996-01-08: KK V1.30                                                    *
 *        - updated Copyright message and email address                       *
 *        - changed int to ushort for tables                                  *
 *    1994-01-04: KK V1.21                                                    *
 *        - added Codepage 853, 862 and 864                                   *
 *    1993-08-02: KK V1.20                                                    *
 *        - using correct Codepages now instead of 801-810                    *
 *    1993-06-11: KK V1.11                                                    *
 *        - added Codepages 801-810 and 819 (ISO 8859-x)                      *
 *    1993-05-29: KK V1.10                                                    *
 *        - support for ISO 8859-1/2/5/7/9 character sets                     *
 *    1993-03-21: KK V1.03                                                    *
 *        - changes due to changes in mime.c                                  *
 *    1993-03-05: KK V1.02                                                    *
 *        - included translation table                                        *
 *    1993-03-01: KK V1.01                                                    *
 *        - cosmetic changes                                                  *
 *    1993-02-01: KK V1.00                                                    *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "rfc.h"
#include "strccmp.h"
#include "readline.h"
#include "rstatic.h"
#include "mime.h"
#include "patch.h"
#include "binary.h"
#include "qp.h"
#include "base64.h"
#include "codepage.h"

#include "encode.h"

/******************************************************************************

	include IBM Code Page y to ISO 8859-x conversion here

******************************************************************************/

#define BAD             160
#define CHARNUM         256

#include "850xiso.tab"
#include "852xiso.tab"
#include "853xiso.tab"
#include "857xiso.tab"
#include "862xiso.tab"
#include "864xiso.tab"
#include "866xiso.tab"
#include "869xiso.tab"

ushort	*c_ibmxiso	= (ushort *) NULL ;

/******************************************************************************

	Function:
		display program message

	Parameter:
		none

	Returns:
		nothing

******************************************************************************/

static  void    Hello (void)
{
	fprintf (stderr, MSG_COPYRIGHT) ;
	fprintf (stderr, MSG_AUTHOR) ;
	fprintf (stderr, MSG_VERSION, PRGNAM) ;
}

/******************************************************************************

	Function:
		copy header and interpret/modify MIME-field C-T-E

	Parameters:
		FILE    *fin            file being read
		FILE    *fout           file being written

	Returns:
		MIME_CODE_ERROR

		MIME_CODE_NONE
		MIME_CODE_UNKNOWN

		MIME_CODE_7BIT
		MIME_CODE_8BIT
		MIME_CODE_BINARY

		MIME_CODE_QP
		MIME_CODE_BASE64

******************************************************************************/

static  int     CopyHeader
(
	FILE    *fin,
	FILE    *fout,
	int     DestEncoding
)
{
	char    Line            [RFC822_LINE_LENGTH + 1] ;
	int     Status ;
	int     Encoding        = MIME_CODE_NOHEAD ;    /* no tag */
	char    *EncType ;

	do
	{
		Status = ReadLine (fin, Line, RFC822_LINE_LENGTH) ;
		if (Status == READLINE_LONG)
			return (MIME_CODE_ERROR) ;

		if (*Line != '\0')
		{
			EncType = strccmp (Line, MIME_TAG_ENCODING) ;
			if (EncType != (char *) NULL)
			{
				Encoding = MIME_TagToCode (EncType) ;

				switch (DestEncoding)
				{
					case    MIME_CODE_QP:

						sprintf (Line,
							MIME_HEAD_ENCODING,
							MIME_TAG_ENCODING,
							MIME_TAG_QP) ;
						break ;

					case    MIME_CODE_BASE64:

						sprintf (Line,
							MIME_HEAD_ENCODING,
							MIME_TAG_ENCODING,
							MIME_TAG_BASE64) ;
						break ;
				}
			}

			if
			(
				(Encoding == MIME_CODE_NOHEAD) &&
				(
					(strccmp (Line, RFC822_HEAD_DATE)
						!= (char *) NULL) ||
					(strccmp (Line, RFC822_HEAD_FROM)
						!= (char *) NULL) ||
					(strccmp (Line, RFC822_HEAD_TO)
						!= (char *) NULL) ||
					(strccmp (Line, RFC822_HEAD_SUBJECT)
						!= (char *) NULL)
				)
			)
			{
				Encoding = MIME_CODE_UNKNOWN ;
			}

			fprintf (fout, "%s\n", Line) ;
		}
	}
	while ((*Line != '\0') && (Status == READLINE_OK)) ;

	return (Encoding) ;
}

/*****************************************************************************/

int     main
(
	int     argc,
	char    *argv []
)
{
	int     DestEncoding    = MIME_CODE_8BIT ;      /* default */
	int     Encoding ;
	int     Status ;
	char    *EncodingParm ;
	char    *Translate ;                            /* translation flag */
	char    Default [2]     = { MIME_FLAG_8BIT, '\0' } ;
	int     cp ;
	int     iso8859 ;

	/**********************************************************************

		check syntax first

	**********************************************************************/

	if (argc > 2)
	{
		Hello () ;
		fprintf (stderr, ERR_MSG_SYNTAX, PRGNAM,
			MIME_FLAG_BASE64, MIME_FLAG_QP,
			MIME_FLAG_7BIT, MIME_FLAG_8BIT,
			MIME_FLAG_BINARY) ;
		return (ERR_SYNTAX) ;
	}

	if (argc == 2)
	{
		EncodingParm = argv [1] ;
	}
	else
	{
		EncodingParm = rstatic (STATIC_MIME_CTE) ;
		if (EncodingParm == (char *) NULL)
			EncodingParm = Default ;
		else
			EncodingParm = strdup (EncodingParm) ;
	}

	/**********************************************************************

		parse parameter

	**********************************************************************/

	switch (toupper (*EncodingParm))
	{
		case    MIME_FLAG_BASE64:

			DestEncoding = MIME_CODE_BASE64 ;
			break ;

		case    MIME_FLAG_QP:

			DestEncoding = MIME_CODE_QP ;
			break ;

		case    MIME_FLAG_7BIT:

			DestEncoding = MIME_CODE_7BIT ;
			break ;

		case    MIME_FLAG_8BIT:

			DestEncoding = MIME_CODE_8BIT ;
			break ;

		case    MIME_FLAG_BINARY:

			DestEncoding = MIME_CODE_BINARY ;
			break ;

		default:

			Hello () ;
			fprintf (stderr, ERR_MSG_SYNTAX, PRGNAM,
				MIME_FLAG_BASE64, MIME_FLAG_QP,
				MIME_FLAG_7BIT, MIME_FLAG_8BIT,
				MIME_FLAG_BINARY) ;
			return (ERR_SYNTAX) ;
	}

	/**********************************************************************

		determine current Code Page (and ISO 8859 equivalent)

	**********************************************************************/

	cp = CodePage () ;
	switch (cp)
	{
		case    437:
		case    850:
			c_ibmxiso = c_850xiso ;
			iso8859 = 1 ;
			break ;

		case    852:
			c_ibmxiso = c_852xiso ;
			iso8859 = 2 ;
			break ;

		case    853:
			c_ibmxiso = c_853xiso ;
			iso8859 = 3 ;
			break ;

		case    857:
			c_ibmxiso = c_857xiso ;
			iso8859 = 9 ;
			break ;

		case    862:
			c_ibmxiso = c_862xiso ;
			iso8859 = 8 ;
			break ;

		case    864:
			c_ibmxiso = c_864xiso ;
			iso8859 = 6 ;
			break ;

		case    866:
			c_ibmxiso = c_866xiso ;
			iso8859 = 5 ;
			break ;

		case    869:
			c_ibmxiso = c_869xiso ;
			iso8859 = 7 ;
			break ;

		case	819:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 1 ;
			break ;

		case	912:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 2 ;
			break ;

		case	913:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 3 ;
			break ;

		case	914:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 4 ;
			break ;

		case	915:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 5 ;
			break ;

		case	1089:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 7 ;
			break ;

		case	813:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 7 ;
			break ;

		case	916:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 8 ;
			break ;

		case	920:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 9 ;
			break ;

		case	919:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 10 ;
			break ;

		default:
			c_ibmxiso = (int *) NULL ;
			iso8859 = 0 ;
	}

	/**********************************************************************

		shall we use the internal character translation?

	**********************************************************************/

	Translate = rstatic (STATIC_MIME_IBMxISO) ;
	if (Translate != (char *) NULL)
		if (strcmp (Translate, VAL_NO) == 0)
			c_ibmxiso = (int *) NULL ;

	/**********************************************************************

		first we copy and interpret the original header

	**********************************************************************/

	Encoding = CopyHeader (stdin, stdout, DestEncoding) ;
	if (Encoding == MIME_CODE_ERROR)
	{
		Encoding        = MIME_CODE_8BIT ;
		DestEncoding    = MIME_CODE_8BIT ;
	}

	if (Encoding == MIME_CODE_UNKNOWN)
	{
		Encoding        = MIME_CODE_8BIT ;
		MIME_WriteHeader (stdout, DestEncoding, iso8859) ;
	}

	switch (DestEncoding)
	{
		case    MIME_CODE_7BIT:
		case    MIME_CODE_8BIT:
		case    MIME_CODE_BINARY:

			fputc ('\n', stdout) ;
			Status = CopyBinary (stdin, stdout, c_ibmxiso) ;
			break ;

		case    MIME_CODE_QP:

			if (Encoding == MIME_CODE_NOHEAD)
				Encoding = MIME_CODE_8BIT ;

			fprintf (stdout,
				"X-Remark: %s converted to %s\n\n",
				MIME_CodeToTag (Encoding), MIME_TAG_QP) ;
			Status = Copy8BitToQP (stdin, stdout, c_ibmxiso) ;
			break ;

		case    MIME_CODE_BASE64:

			if (Encoding == MIME_CODE_NOHEAD)
				Encoding = MIME_CODE_8BIT ;

			fprintf (stdout,
				"X-Remark: %s converted to %s\n\n",
				MIME_CodeToTag (Encoding), MIME_TAG_BASE64) ;
			Status = Copy8BitToBase64 (stdin, stdout, c_ibmxiso) ;
			break ;
	}

	if (Status)
		fprintf (stderr, ERR_MSG_WRITE, PRGNAM) ;

	return (Status) ;
}
