/******************************************************************************
 *                                                                            *
 * File:   transhtm.c         Version  1.13            Date: 1997-06-14       *
 *                                                                            *
 * Copyright (C) 1997 by kostis@acm.org (Kosta Kostis)                        *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - this program creates a HTML table using a cedf                    *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *     1997-06-14: KK V1.13                                                   *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>

/******************************************************************************

	General Program Definitions

 *****************************************************************************/

#include "transhtm.h"
#include "taberr.h"
#include "tab.h"

/******************************************************************************

	String Support Functions

 *****************************************************************************/

#ifdef	NO_STRDUP
	#include "strdup.c"
#endif

#ifdef	NO_STRUPR
	#include "strupr.c"
#endif

/******************************************************************************

	Character Encoding Description Table Functions

 *****************************************************************************/

#include "loadtab.h"
#include "readtab.h"

/******************************************************************************

	Function:
		check if character description is a control char

	Parameters:
		character description string

	Returns:
		0	if not a control char

 *****************************************************************************/

static	int	IsCtrl
(
	char	*Desc
)
{
	int	c ;

	static	char	*ctrl	[] =
	{
		"NUL", "SOH", "STX", "ETX", "EOT", "ENQ", "ACK", "BEL",
		"BS" , "HT" , "LF" , "VT" , "FF" , "CR" , "SO" , "SI" ,
		"DLE", "DC1", "DC2", "DC3", "DC4", "NAK", "SYN", "ETB",
		"CAN", "EM" , "SUB", "ESC", "FS" , "GS" , "RS" , "US" ,
		"DEL", ""
	} ;

	if (Desc == (char *) NULL)
		return (1) ;

	if (*Desc == '\0')
		return (1) ;

	if (strncmp (Desc, "NC", 2) == 0)
		return (1) ;

	for (c = 0 ; *ctrl [c] != '\0' ; ++c)
		if (strcmp (Desc, ctrl [c]) == 0)
			return (1) ;

	return (0) ;
}

/******************************************************************************

	Function:
		display string and make it HTML conformant

	Parameters:
		FILE		*Fout		output file
		char		*str		output string

	Returns:
		nothing

 *****************************************************************************/

static	void	DispHTML
(
	FILE	*Fout,
	char	*string
)
{
	char	*cur ;

	if (string == (char *) NULL)
		return ;

	for (cur = string ; *cur != '\0' ; ++cur)
	{
		switch (*cur)
		{
			case	'<':
				fprintf (Fout, "&lt;") ;
				break ;

			case	'>':
				fprintf (Fout, "&gt;") ;
				break ;

			case	'\"':
				fprintf (Fout, "&quot;") ;
				break ;

			case	'&':
				fprintf (Fout, "&amp;") ;
				break ;

			default:

				fprintf (Fout, "%c", *cur) ;
		}
	}
}

/******************************************************************************

	Function:
		display character encoding as HTML table

	Parameters:
		FILE		*Fout		output file
		char		**SrcTable	character encoding descr.
		ushort		entries		number of entries
		ushort		*bintab
		char		*SrcName	name of character encoding
		char		**SrcComment	comments

	Returns:
		nothing

 *****************************************************************************/

static	void	DumpHTML
(
	FILE	*Fout,
	char	**SrcTable,
	ushort	entries,
	ushort	*bintab,
	char	*SrcName,
	char	**SrcComment
)
{
	int	i ;

	fprintf (Fout, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n") ;
	fprintf (Fout, "<!-- generated by %s %s -->\n", PRGNAM, VERSION) ;
	fprintf (Fout, "<HTML>\n") ;

	fprintf (Fout, "\t<HEAD>\n") ;

	fprintf (Fout, "\t\t<META HTTP-EQUIV=\"Content-Type\" ") ;
	fprintf (Fout, "CONTENT=\"text/html; charset=UTF-8\">\n") ;

	fprintf (Fout, "\t\t<TITLE>") ;
	DispHTML (Fout, SrcName) ;
	fprintf (Fout, "</TITLE>\n") ;
	fprintf (Fout, "\t</HEAD>\n") ;

	fprintf (Fout, "\t<BODY>\n") ;
	fprintf (Fout, "\t\t<H1>") ;
	DispHTML (Fout, SrcName) ;
	fprintf (Fout, "</H1>\n") ;
	fprintf (Fout, "\t\t<H2>") ;
	DispHTML (Fout, SrcComment [0]) ;
	fprintf (Fout, "</H2>\n") ;

	fprintf (Fout, "\t\t<TABLE BORDER CELLPADDING=\"2\">\n") ;
	fprintf (Fout, "\t\t<TR>\n") ;
	fprintf (Fout, "\t\t\t<TD><B>Hex</B></TD>\n") ;
	fprintf (Fout, "\t\t\t<TD><B>Dec</B></TD>\n") ;
	fprintf (Fout, "\t\t\t<TD><B>Chr</B></TD>\n") ;
	fprintf (Fout, "\t\t\t<TD><B>ISO/IEC 10646-1:1993(E) Character Name</B></TD>\n") ;
	fprintf (Fout, "\t\t</TR>\n") ;

	for (i = 0 ; i < entries ; ++i)
	{
		if (IsCtrl (SrcTable [i]) == 0)
		{
			fprintf (Fout, "\t\t<TR>\n") ;
			fprintf (Fout, "\t\t\t<TD>%02X</TD>\n", i) ;
			fprintf (Fout, "\t\t\t<TD><P ALIGN=Right>%u</TD>\n", i) ;
			fprintf (Fout, "\t\t\t<TD><P ALIGN=Center>&#%u;</TD>\n",
				bintab [i]) ;
			fprintf (Fout, "\t\t\t<TD>%s</TD>\n", SrcTable [i]) ;
		}
	}

	fprintf (Fout, "\t\t</TABLE>\n") ;

	fprintf (Fout, "\t</BODY>\n") ;

	fprintf (Fout, "</HTML>\n") ;
}


/******************************************************************************

	Function:
		display copyright, author and program information

	Parameters:
		none

	Returns:
		nothing

 *****************************************************************************/

void	Hello (void)
{
	fprintf (stderr, MSG_COPYRIGHT) ;
	fprintf (stderr, MSG_VERSION, PRGNAM, VERSION) ;
}

/*****************************************************************************/

int     main
(
	int	argc,
	char	*argv		[]
)
{
	int	rc ;
	char	*SrcTable	[MAX_TABLE_ENTRIES] ;
	char	SrcName		[MAX_ENC_NAME_LENGTH + 1] ;
	char	*SrcComment	[MAX_COMMENTS] ;
	ushort	bintab		[MAX_TABLE_ENTRIES] ;	/* src enc table */

	/**********************************************************************

		parse parameters/arguments

	 *********************************************************************/

	if (argc != 2)
	{
		/**************************************************************

			display syntax error message and quit

		 *************************************************************/

		Hello () ;
		fprintf (stderr, ERR_MSG_SYNTAX, PRGNAM) ;
		return (ERR_SYNTAX) ;
	}

	/**********************************************************************

		try to read character encoding description files

	 *********************************************************************/

	rc = ReadCodeTable (argv [1], SrcTable, (ushort *) NULL,
		MAX_TABLE_ENTRIES, SrcName, SrcComment) ;
	if (rc != ERR_NONE)
		return (rc) ;
	rc = LoadTable (argv [1], bintab) ;
	if (rc != ERR_NONE)
		return (rc) ;

	DumpHTML (stdout, SrcTable, MAX_TABLE_ENTRIES,
		bintab, SrcName, SrcComment) ;

	/**********************************************************************

		free memory allocated by ReadCodeTable () and quit

	 *********************************************************************/

	FreeCodeTable (SrcTable, MAX_TABLE_ENTRIES) ;

	return (ERR_NONE) ;
}
