/******************************************************************************
 *                                                                            *
 * File:   comptran.c         Version  1.12            Date: 1997-05-01       *
 *                                                                            *
 * Copyright (C) 1993-1997 by kostis@acm.org (Kosta Kostis)                   *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *     1997-05-01: KK V1.12                                                   *
 *        - new release of the package                                        *
 *     1996-06-01: KK V1.11                                                   *
 *        - changed Win's format to Unicode A Format                          *
 *     1995-07-01: KK V1.10                                                   *
 *        - fixed bug in DumpCodeTrans (must be unsigned *short* int)         *
 *     1994-03-31: KK V1.00                                                   *
 *        - some extensions for ISO 10646                                     *
 *        - output format changes                                             *
 *     1994-03-04: KK V0.97                                                   *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "datatype.h"
#include "os-stuff.h"

#include "tab.h"
#include "scanflag.h"
#include "readtab.h"

#include "head_c.h"
#include "head_h.h"
#include "head_tab.h"
#include "head_u.h"

#include "comptran.h"

/******************************************************************************

	Function:
		initialize translation table

	Parameters:
		ushort	*Table          table to be initialized
		ushort	MaxTableEntries	number of entries in table

	Returns:
		nothing

 *****************************************************************************/

static	void    InitCodeTrans
(
	ushort	*Table,
	ushort	MaxTableEntries
)
{
	ushort	i ;

	for (i = 0 ; i < MaxTableEntries ; ++i)
		Table [i] = BAD ;
}

/******************************************************************************

	Function:
		compute translation table

	Parameters:
		char	*Src	[]	source table
		ushort	SrcMax		number of entries in source table
		char	*Dst	[]	dest. table
		ushort	DstMax		number of entries in dest. table
		ushort	*DstCodes
		ushort	*Table		trans table

	Returns:
		nothing

 *****************************************************************************/

void	TransTable
(
	char    *Src            [],
	ushort	SrcMax,
	char    *Dst            [],
	ushort	DstMax,
	ushort	*DstCodes,
	ushort	*Table
)
{
	ushort	i ;
	ushort	Code ;

	/**********************************************************************

		intitialize translation table (all translations BAD)

	 *********************************************************************/

	InitCodeTrans (Table, SrcMax) ;

	/**********************************************************************

		build translation table "the hard way"

	 *********************************************************************/

	for (i = 0 ; i < SrcMax ; ++i)
	{
		Code = FindCode (Src [i], Dst, DstCodes, DstMax) ;
		if (Code != BAD)
			Table [i] = Code ;
	}
}

/******************************************************************************

	Function:
		dump translation table in Unicode A format

	Parameters:
		FILE    *Fout           output file
		char	*Src	[]	source character name table
		ushort	*Table          translation table

	Returns:
		nothing

 *****************************************************************************/

static	void	DumpUnicodeATrans
(
	FILE	*Fout,
	char	*Src		[],
	ushort	*Table
)
{
	ushort	Code ;

	/**********************************************************************

		step through the translation table

	 *********************************************************************/

	for (Code = 0 ; Code < MAX_TABLE_ENTRIES ; ++Code)
	{
		/**************************************************************

			display translation value or BAD

		 *************************************************************/

		if
		(
			(Table [Code] != BAD) &&
			(strncmp (Src [Code], STR_NC, strlen (STR_NC)) != 0)
		)
		{
			fprintf (Fout, "0x%02X\t0x%04X\t#\t%s\n",
				Code, Table [Code], Src [Code]) ;
		}
	}
}

/******************************************************************************

	Function:
		dump translation table (C int array)

	Parameters:
		FILE    *Fout           output file
		ushort	*Table          translation table
		char    *TableName      array name for the table

	Returns:
		nothing

 *****************************************************************************/

static	void	DumpCodeTrans
(
	FILE	*Fout,
	ushort	*Table,
	char	*TableName
)
{
	ushort	Code ;
	int	Column  = 0 ;
	int	Start	= 0 ;

	/**********************************************************************

		print declation line including array name

	 *********************************************************************/

	fprintf (Fout,
		"static\tunsigned short int\tc_%s\t[CHARNUM] =\t\t/* %02X-%02X */\n{\n",
		TableName, 0, MAX_TABLE_ENTRIES - 1) ;

	/**********************************************************************

		now step through the translation table

	 *********************************************************************/

	for (Code = 0 ; Code < MAX_TABLE_ENTRIES ; ++Code)
	{
		/**************************************************************

			at the start of a line insert a tab - keep current code

		 *************************************************************/

		if (Column == 0)
			Start = Code ;

		/**************************************************************

			display translation value or BAD

		 *************************************************************/

		if (Table [Code] != BAD)
			fprintf (Fout, "0x%04X", Table [Code]) ;
		else
			fprintf (Fout, "   BAD") ;

		/**************************************************************

			at the end of a line insert a tab, too

		 *************************************************************/

		if (Code != MAX_TABLE_ENTRIES - 1)
			fputc (',', Fout) ;

		if (Column < MAX_CODES_PER_LINE - 1)
			fputc (' ', Fout) ;
		else
			fputc ('\t', Fout) ;

		/**************************************************************

			add the codes displayed, too

		 *************************************************************/

		if (++Column >= MAX_CODES_PER_LINE)
		{
			Column = 0 ;
			fprintf (Fout, "/* %02X-%02X */\n",
				Start, Start + MAX_CODES_PER_LINE - 1) ;
		}
	}

	/**********************************************************************

		finish array declaration

	 *********************************************************************/

	fprintf (Fout, "} ;\n") ;
}

/******************************************************************************

	Function:
		dump complete character encoding translation program

	Parameters:
		char    *Src    []      source character encoding
		char    *Dst    []      destination character encoding
		ushort	*DstCodes
		char    *TableName      name for the table
		ushort	*Table          translation table
		ushort	MaxTableEntries
		char    *SrcName        src character encoding name
		char    *DstName        dst character encoding name
		char    *SrcComment []  src character encoding comment
		char    *DstComment []  dst character encoding comment
		int	TransMake	flag for table only

	Returns:
		nothing

 *****************************************************************************/

void	DumpProg
(
	char	*Src		[],
	char	*Dst		[],
	ushort	*DstCodes,
	char	*TableName,
	ushort	*Table,
	ushort	MaxTableEntries,
	char	*SrcName,
	char	*DstName,
	char	*SrcComment	[],
	char	*DstComment	[],
	int	TransMake
)
{
	char    FileName        [PATHNAME_MAX + 1] ;
	FILE    *Fout ;
	ushort	NoBreakSpace ;
	char	*Help ;

	/**********************************************************************

		create translation table file

	 *********************************************************************/

	switch (TransMake)
	{
		case	TRANS_MAKE_NOTHING:

			return ;

		case	TRANS_MAKE_UNICODE_A:

			sprintf (FileName, EXT_UNICODE_A, TableName) ;
			Fout = fopen (FileName, "w") ;
			if (Fout == (FILE *) NULL)
				return ;

			fprintf (Fout, HEADER_UNICODE_A,
				SrcName, PRGNAM, VERSION,
				SrcName, SrcComment [0],
				SrcComment [1], SrcComment [2],
				DstName, DstComment [0],
				DstComment [1], DstComment [2]) ;
			DumpUnicodeATrans (Fout, Src, Table) ;
			fclose (Fout) ;
			break ;

		case	TRANS_MAKE_ALL:
		case	TRANS_MAKE_TABLE:

			sprintf (FileName, EXT_TABLE, TableName) ;
			Fout = fopen (FileName, "w") ;
			if (Fout == (FILE *) NULL)
				return ;

			fprintf (Fout, HEADER_TAB, TableName,
				SrcName, SrcComment [0],
				SrcComment [1], SrcComment [2],
				DstName, DstComment [0],
				DstComment [1], DstComment [2]) ;
			DumpCodeTrans (Fout, Table, TableName) ;
			fclose (Fout) ;
			break ;

		case	TRANS_MAKE_BINARY:

			Help = getenv (ENV_TRANS) ;
			if (Help == (char *) NULL)
				Help = "" ;

			sprintf (FileName, EXT_BINARY, Help, CHR_DIR, TableName) ;
			Fout = fopen (FileName, "wb") ;
			if (Fout == (FILE *) NULL)
				return ;

			fwrite (Table, sizeof (ushort), MaxTableEntries, Fout) ;
			fclose (Fout) ;
			return ;
	}

	if (TransMake != TRANS_MAKE_ALL)
		return ;

	/**********************************************************************

		create C main program

	 *********************************************************************/

	sprintf (FileName, EXT_C, TableName) ;
	Fout = fopen (FileName, "w") ;
	if (Fout == (FILE *) NULL)
		return ;

	fprintf (Fout, HEADER_C, TableName,
		SrcName, SrcComment [0], SrcComment [1], SrcComment [2],
		DstName, DstComment [0], DstComment [1], DstComment [2],
		TableName, TableName) ;
	fclose (Fout) ;

	/**********************************************************************

		create C header file

	 *********************************************************************/

	sprintf (FileName, EXT_H, TableName) ;
	Fout = fopen (FileName, "w") ;
	if (Fout == (FILE *) NULL)
		return ;

	NoBreakSpace = FindCode (ISO_NO_BREAK_SPACE,
		Dst, DstCodes, MaxTableEntries) ;
	if (NoBreakSpace == BAD)
		NoBreakSpace = DEF_NO_BREAK_SPACE ;

	fprintf (Fout, HEADER_H, TableName, MaxTableEntries,
		NoBreakSpace, TableName, TableName,
		SrcName, SrcComment [0], SrcComment [1], SrcComment [2],
		DstName, DstComment [0], DstComment [1], DstComment [2]) ;
	fclose (Fout) ;
}
