/******************************************************************************
 *                                                                            *
 * File:   transibm.c         Version  1.20            Date: 1996-01-08       *
 *                                                                            *
 * Copyright (C) 1992-1996 by Kosta Kostis - this is freeware!                *
 * Written by kostis@acm.org (Kosta Kostis)                                   *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - this program converts one 8 bit character code to another         *
 *          depending on the currently active Code Page the appropriate       *
 *          conversion table is choosen.                                      *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Note:                                                                      *
 *        - this is just the generic program body. definitions like the       *
 *          actual conversion tables and so on must be supplied.              *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *    1996-01-08: KK V1.20                                                    *
 *        - added env var CP for Codepage detection                           *
 *    1994-01-04: KK V1.10                                                    *
 *        - added Codepages 853, 862 and 864                                  *
 *    1993-06-11: KK V1.01                                                    *
 *        - accept other Codepages                                            *
 *    1993-05-25: KK V1.00                                                    *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "transibm.h"
#include "codepage.h"

/*****************************************************************************/

static	int	BadChar = BAD_CHAR ;	/* default BAD character             */

/******************************************************************************

	Function:
		display copyright, author and program information

	Parameters:
		none

	Returns:
		nothing

******************************************************************************/

static  void    TransHello (void)
{
	fprintf (stderr, MSG_COPYRIGHT) ;
	fprintf (stderr, MSG_AUTHOR) ;
	fprintf (stderr, MSG_VERSION, PRGNAM, VERSION) ;
}

/******************************************************************************

	Function:
		copy input to output doing conversion

	Parameters:
		FILE	*src		source file
		FILE	*dest		destination file

	Returns:
		ERR_NONE		if all goes well
		ERR_BAD			if characters have been untranslatable

******************************************************************************/

static  int	Trans (FILE *src, FILE *dest)
{
	int     ch ;                    /* character read               */
	int	newch ;			/* translated character		*/
	int     codes ;                 /* bad code flag                */
	int	cp ;			/* Code Page currently in use   */

	cp = CodePage () ;
	codes = ERR_NONE ;		/* assume no bad code           */

	while ((ch = fgetc (src)) != EOF)       /* read character       */
	{
		switch (cp)
		{
			case	437:
			case	850:
				newch = TRANS_C_850 [ch] ;
				break ;

			case	852:
				newch = TRANS_C_852 [ch] ;
				break ;

			case	853:
				newch = TRANS_C_853 [ch] ;
				break ;

			case	857:
				newch = TRANS_C_857 [ch] ;
				break ;

			case	862:
				newch = TRANS_C_862 [ch] ;
				break ;

			case	864:
				newch = TRANS_C_864 [ch] ;
				break ;

			case	866:
				newch = TRANS_C_866 [ch] ;
				break ;

			case	869:
				newch = TRANS_C_869 [ch] ;
				break ;

			default:
				newch = ch ;	/* don't translate      */
		}

		if (newch != BAD)
		{
			fputc (newch, dest) ;	/* output character     */
		}
		else
		{
			codes = ERR_BAD ;
			fputc (BadChar, dest) ;
		}
	}

	return (codes) ;
}

/*****************************************************************************/

static	char	*InpName ;	/* input file name                           */
static	char	*OutName ;	/* output file name                          */

/******************************************************************************

	ScanFlags () return codes

******************************************************************************/

#define SCN_OK          0       /* parameters are all valid                  */
#define SCN_UNKNOWN     1       /* unknown parameter used                    */
#define SCN_BADARGS     2       /* illegal number of params                  */

/******************************************************************************

	Function:
		parse parameters and set flags and variables accordingly

	Parameters:
		command line parameters

	Returns:
		SCN_OK		if all parameters are valid
		SCN_UNKNOWN	if any of the parameters is unknown
		SCN_BADARGS	if the number of parameters is wrong

******************************************************************************/

static  int     ScanFlags (int argc, char *argv [])
{
	int     i ;             /* index through the parameters              */
	int     parms = 0 ;     /* yet no parameter scanned                  */

	/**********************************************************************

		set defaults / initialize

	**********************************************************************/

	InpName = (char *) NULL ;
	OutName = (char *) NULL ;       

	for (i = 1 ; i < argc ; ++i)    /* scan command line                 */
	{
		/**************************************************************

			is it a flag/switch? (*NIX style)

		**************************************************************/

		if (*argv [i] == '-')
		{
			/******************************************************

				check out which flag it is

			******************************************************/

			switch (argv [i][1])
			{
				case 'B':	/* change BAD character      */
				case 'b':

					BadChar = (int) argv [i][2] ;
					break ;

				default:	/* Uh! This flag's unknown!  */

					return (SCN_UNKNOWN) ;
			}
		}
		else
		{
			/******************************************************

				if it's no flag it must be a filename

			******************************************************/

			++parms ;       /* check for multiple filenames      */
			switch (parms)
			{
				case 1:

					InpName = argv [i] ;
					break ;

				case 2:

					OutName = argv [i] ;
					break ;

				default :

					return (SCN_BADARGS) ;
			}
		}
	}

	/**********************************************************************

		everything has been fine - let's quit!

	**********************************************************************/

	return (SCN_OK) ;
}

/******************************************************************************

	Function:

	Parameters:

	Returns:

******************************************************************************/

int     main (int argc, char *argv [])
{
	FILE    *SRCfile = stdin ;      /* Source default is stdin           */
	FILE    *DSTfile = stdout ;     /* Dest. default is stdout           */
	char	*TmpName = (char *) NULL ;
	int     codes ;                 /* bad code flag                     */
	int     syntax ;                /* return code for ScanFlags ()      */

	/**********************************************************************

		parse parameters/arguments

	**********************************************************************/

	syntax = ScanFlags (argc, argv) ;
	if (syntax != SCN_OK)
	{
		/**************************************************************

			display syntax error message and quit

		**************************************************************/

		TransHello () ;
		fprintf (stderr, ERR_MSG_SYNTAX, PRGNAM) ;

		return (ERR_SYNTAX) ;
	}

	/**********************************************************************

		check for output file - create if necessary

	**********************************************************************/

	if (OutName != (char *) NULL)
	{
		/**************************************************************

			file type must be binary not to trash Waffle U*IX
			style files for UUCP

		**************************************************************/

		DSTfile = fopen (OutName, "wb") ;
		if (DSTfile == (FILE *) NULL)
		{
			TransHello () ;
			fprintf (stderr, ERR_MSG_DST, PRGNAM, OutName) ;

			return (ERR_DST) ;
		}
	}

	/**********************************************************************

		check for input file - open if necessary

	**********************************************************************/

	if (InpName != (char *) NULL)
	{       
		SRCfile = fopen (InpName, "rb") ;
		if (SRCfile == (FILE *) NULL)
		{
			TransHello () ;
			fprintf (stderr, ERR_MSG_SRC, PRGNAM, InpName) ;
			if (DSTfile != stdout)
				fclose (DSTfile) ;

			return (ERR_SRC) ;
		}

		/**************************************************************

			if no output file name is supplied, we have to create
			a temporary file which gets renamed afterwards

		**************************************************************/

		if (OutName == (char *) NULL)
		{
			TmpName = tmpnam ((char *) NULL) ;
			DSTfile = fopen (TmpName, "wb") ;
			if (DSTfile == (FILE *) NULL)
			{
				fclose (SRCfile) ;
				return (ERR_DST) ;
			}
		}
	}

	/**********************************************************************

		now do the actual conversion

	**********************************************************************/

	codes = Trans (SRCfile, DSTfile) ;

	/**********************************************************************

		close files if opened

	**********************************************************************/

	if (DSTfile != stdout)
		fclose (DSTfile) ;
	if (SRCfile != stdin)
		fclose (SRCfile) ;

	/**********************************************************************

		if there is a temporary file, rename it to the original name

	**********************************************************************/

	if (TmpName != (char *) NULL)
	{
		remove (InpName) ;
		rename (TmpName, InpName) ;
	}

	return (codes) ;
}
