/******************************************************************************
 *                                                                            *
 * File:   qp      .c         Version  1.10            Date: 1996-01-08       *
 *                                                                            *
 * Copyright (C) 1993-1996 by Kosta Kostis - this is freeware!                *
 * Written by kostis@acm.org (Kosta Kostis)                                   *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - convert from/to quoted-printables and 8-bit text                  *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *    1996-01-08: KK V1.10                                                    *
 *        - updated Copyright message and email address                       *
 *        - changed int to ushort for tables                                  *
 *    1993-05-29: KK V1.02                                                    *
 *        - now accepting/deleting "=" at the end of the line for long lines  *
 *    1993-03-05: KK V1.01                                                    *
 *        - added translation mechanism                                       *
 *    1993-02-01: KK V1.00                                                    *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "qp.h"

/******************************************************************************

	basic QP functions

 *****************************************************************************/

/******************************************************************************

	Function:
		check whether a character is a valid hex digit

	Parameters:
		int     ch      character to be checked

	Returns:
		0..15           value of the hex digit
		QP_CODE_INVALID invalid hex digit

 *****************************************************************************/

static  int     ishex
(
	int     ch
)
{
	static  char    *diff ;
	static  char    hexdigits [] =
	{
		'0', '1', '2', '3', '4', '5', '6', '7',         /* 0-7 */
		'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'          /* 8-F */
	} ;

	/**********************************************************************

		convert lower case to upper case

	 *********************************************************************/

	ch = toupper (ch) ;

	/**********************************************************************

		I want this function to be independend of the character
		set used. There's more than ASCII (e. g. EBCDIC)

		I think efficiency is not important here

	 *********************************************************************/

	diff = strchr (hexdigits, ch) ;
	if (diff != (char *) NULL)
		return (diff - hexdigits) ;

	return (QP_CODE_INVALID) ;              /* invalid digit   */
}

/******************************************************************************

	Function:
		convert quoted-printable to 8bit

	Parameters:
		FILE    *fin    file being read
		FILE    *fout   file being written
		ushort	*table  pointer to translation table

	Returns:
		QP_OK           if all goes well
		QP_WRITE        if there's a write error

 *****************************************************************************/

int     CopyQPTo8Bit
(
	FILE    *fin,
	FILE    *fout,
	ushort	*table
)
{
	int     ch ;
	int     ch2 ;
	int     msn ;   /* most  significant nibble */
	int     lsn ;   /* least significant nibble */

	while ((ch = fgetc (fin)) != EOF)
	{
		if (ch != QP_CODE_INTRO)
		{
			if (fputc (ch, fout) != ch)
				return (QP_WRITE) ;
		}
		else
		{
			/******************************************************

				we have read the QP_CODE_INTRO char
				now we have to check for a valid
				two nibble hex value

			 *****************************************************/

			ch  = fgetc (fin) ;
			if (ch == '\n')
				continue ;

			msn = ishex (ch) ;
			if (msn == QP_CODE_INVALID)
			{
				/**********************************************

					we have to restore two characters
					to the output stream:

					QP_CODE_INTRO   read before
					ch              invalid hex digit

				 *********************************************/

				if (fputc (QP_CODE_INTRO, fout) != QP_CODE_INTRO)
					return (QP_WRITE) ;
				if (fputc (ch, fout) != ch)
					return (QP_WRITE) ;
			}
			else
			{
				/**********************************************

					we have read the QP_CODE_INTRO char
					and one valid hex digit so far
					now let's check for the second digit

				 *********************************************/

				ch2 = fgetc (fin) ;
				lsn = ishex (ch2) ;
				if (lsn == QP_CODE_INVALID)
				{
					/**************************************

						unfortunately the third
						character is invalid.
						Restore CP_CODE_INTRO,
						ch and ch2, the last
						characters read

					 *************************************/

					if (fputc (QP_CODE_INTRO, fout) != QP_CODE_INTRO)
						return (QP_WRITE) ;
					if (fputc (ch, fout) != ch)
						return (QP_WRITE) ;
					if (fputc (ch2, fout) != ch2)
						return (QP_WRITE) ;
				}
				else
				{
					/**************************************

						a fine QP has been read, now
						let's create the 8-bit octet

					 *************************************/

					ch = (msn << 4) | lsn ;
					if (table != (ushort *) NULL)
						ch = (int) table [ch] ;
					if (fputc (ch, fout) != ch)
						return (QP_WRITE) ;
				}
			}
		}
	}

	return (QP_OK) ;
}

/******************************************************************************

	Function:
		convert 8bit to quoted-printables

	Parameters:
		FILE    *fin    file being read
		FILE    *fout   file being written
		ushort	*table  pointer to translation table

	Returns:
		QP_OK           if all goes well
		QP_WRITE        if there's a write error

 *****************************************************************************/

int     Copy8BitToQP
(
	FILE    *fin,
	FILE    *fout,
	ushort	*table
)
{
	int     ch ;
/*	int	lastch = '\0' ; */

	while ((ch = fgetc (fin)) != EOF)
	{
		/**************************************************************

			if a translation table is supplied, use it

		 *************************************************************/

		if (table != (ushort *) NULL)
			ch = (int) table [ch] ;

		/**************************************************************

			shall the character be encoded as a quoted printable?

		 *************************************************************/

		if
		(
			(ch == QP_CODE_INTRO)
			|| (ch > 0x7F)
/*			|| ((ch == 'F') && (lastch == '\n')) */
		)
		{
			if (fprintf (fout, "%c%02X", QP_CODE_INTRO, ch) == EOF)
				return (QP_WRITE) ;
		}
		else
		{
			if (fputc (ch, fout) != ch)
				return (QP_WRITE) ;
		}

/*		lastch = ch ; */
	}

	return (QP_OK) ;
}
