/******************************************************************************
 *                                                                            *
 * File:   mime    .c         Version  1.10            Date: 1996-01-08       *
 *                                                                            *
 * Copyright (C) 1993-1996 by Kosta Kostis - this is freeware!                *
 * Written by kostis@acm.org (Kosta Kostis)                                   *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - handle MIME tags and codes                                        *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *    1996-01-08: KK V1.10                                                    *
 *        - updated Copyright message and email address                       *
 *    1993-05-23: KK V1.10                                                    *
 *        - support for ISO 8859-1/2/5/7/9 character sets                     *
 *    1993-03-21: KK V1.01                                                    *
 *        - internal changes to enhance flexability                           *
 *    1993-02-01: KK V1.00                                                    *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>

#include "strccmp.h"
#include "mime.h"

/******************************************************************************

	RFC1521: known MIME Content-Transfer-Encoding tags

******************************************************************************/

static  char    *MIME_Tags [] =
{
	/**********************************************************************

		do not change order or edit mime.h also

	**********************************************************************/

	MIME_TAG_7BIT,          /*  0 : MIME_CODE_7BIT          */
	MIME_TAG_8BIT,          /*  1 : MIME_CODE_8BIT          */
	MIME_TAG_BINARY,        /*  2 : MIME_CODE_BINARY        */
	MIME_TAG_QP,            /*  3 : MIME_CODE_QP            */
	MIME_TAG_BASE64,        /*  4 : MIME_CODE_BASE64        */
	""                      /* End of Table Marker          */
} ;

/******************************************************************************

	Function:
		Convert MIME code to MIME tag string

	Parameters:
		int     Code            MIME code

	Returns:
		NULL                    if code is invalid
		char    *               MIME tag string

******************************************************************************/

char    *MIME_CodeToTag
(
	int     Code
)
{
	if ((Code < MIME_CODE_FIRST) || (Code > MIME_CODE_LAST))
		return ((char *) NULL) ;

	return (MIME_Tags [Code]) ;
}

/******************************************************************************

	Function:
		Convert MIME tag string to MIME code

	Parameters:
		char    *Tag            MIME tag string

	Returns:
		MIME_CODE_UNKNOWN       if tag is unknown or if valid:

		MIME_CODE_7BIT
		MIME_CODE_8BIT
		MIME_CODE_BINARY
		MIME_CODE_QP
		MIME_CODE_BASE64

******************************************************************************/

int     MIME_TagToCode
(
	char    *Tag
)
{
	int     i ;

	for (i = 0 ; *MIME_Tags [i] != '\0' ; ++i)
	{
		if (strccmp (MIME_Tags [i], Tag) != (char *) NULL)
			return (i) ;
	}

	return (MIME_CODE_UNKNOWN) ;
}

/******************************************************************************

	Function:
		write MIME standard header for plain 8bit/7bit text
		using either ISO 8859-1 or US ASCII to a file

	Parameters:
		FILE    *outf           output file
		int     Encoding        encoding used
		int	iso8859		ISO 8859-x charset used

	Returns:
		MIME_OK                 if all goes well
		MIME_WRITE              if there's a write error

******************************************************************************/

int     MIME_WriteHeader
(
	FILE    *fout,
	int     Encoding,
	int	iso8859
)
{
	char    *CharSet ;

	if ((iso8859 < 1) || (iso8859 > 10))
		iso8859 = 1 ;		// this *is* the default

	if (fprintf (fout, MIME_HEAD_VERSION,
		MIME_TAG_VERSION, MIME_TAG_VERNUM) == EOF)
		return (MIME_WRITE) ;

	if (fputc ('\n', fout) != '\n')
		return (MIME_WRITE) ;

	if (Encoding == MIME_CODE_7BIT)
		CharSet = MIME_TAG_PT_US_ASCII ;
	else
		CharSet = MIME_TAG_PT_ISO_8859 ;

	if (fprintf (fout, MIME_HEAD_TYPE,
		MIME_TAG_TYPE, MIME_TAG_PLAINTEXT,
		MIME_TAG_CHARSET, CharSet) == EOF)
		return (MIME_WRITE) ;

	if (Encoding != MIME_CODE_7BIT)
		if (fprintf (fout, "%d", iso8859) == EOF)
			return (MIME_WRITE) ;

	if (fputc ('\n', fout) != '\n')
		return (MIME_WRITE) ;

	if (fprintf (fout, MIME_HEAD_ENCODING,
		MIME_TAG_ENCODING, MIME_CodeToTag (Encoding)) == EOF)
		return (MIME_WRITE) ;

	if (fputc ('\n', fout) != '\n')
		return (MIME_WRITE) ;

	return (MIME_OK) ;
}
