;;; konvers.el --- conversion of text

;; Copyright (C) 1993 Karl Brodowsky

;; Author: Karl Brodowsky <bk1@ix.urz.uni-heidelberg.de>
;; Version: 0.911
;; Keywords: i18n, tex, wp

;; This file was written for GNU Emacs version 19

;; It is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; It is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.	If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; created April 1993
;; last changed 4. May 1995 (large portions moved to iso-konvers.el)
;; (borrows heavily from Xavier Leroy's latin.el, July 1990)
;; (Version 0.820: completed using TeX-book and tables provided
;; by Kosta Kostis and Markus Kuhn)
;; (Errors in first alpha version had been detected by Uwe Bonnes)
;;
;; WARNING:  THIS FILE CONTAINS 8-BIT CHARACTERS.  
;; MAKE SURE IT HAS BEEN TRANSFERRED IN BINARY MODE!!!
;; DO NOT CONVERT THIS FILE.
;;
;; ACHTUNG:  DIESE DATEI ENTHLT 8-BIT-ZEICHEN.
;; STELLEN SIE SICHER, DASZ SIE BINR BERTRAGEN WORDEN IST.
;; DIESE DATEI VERTRGT KEINE KONVERTIERUNG.
;;
;; It provides conversion for
;; Mime-quoted-printable (=E4=F6=FC=C4=D6=DC)  --> iso
;;
;; and reverse for Mail and News.  In addition iso-konvers.el
;; is loaded by konvers.el providing conversion to and from
;; other 7-Bit-transcriptions based on ISO-8859-1.
;; See iso-konvers.el for more details.
;;
;; By enabling conversion in both directions these .el-files
;; should help you using the 8-bit characters wherever you
;; want and using the 7-bit-characters wherever you must.
;; Just write what you want, convert it to 7 bit and mail it
;; or TeX it with a bad TeX implementation.  Conversion
;; is reversible under certain conditions on your text file.
;; This will be true in practice, but you will always be able to
;; construct something to break the conversion.
;;
;; Key sequences beginning with ctrl-x 7 are used:
;;
;; These are done by iso-konvers.el:
;;
;; C-x 7 i       konvers-german-tex-to-iso		; converts region 
;; C-x 7 I       konvers-buffer-german-tex-to-iso	; converts buffer from German TeX to Iso
;; C-u C-x 7 I   konvers-buffer-german-tex-to-iso	; excludes header
;; C-x 7 j       konvers-seven-bit-tex-to-iso		; converts region
;; C-x 7 J       konvers-buffer-seven-bit-tex-to-iso	; converts buffer from TeX to Iso
;; C-u C-x 7 J   konvers-buffer-seven-bit-tex-to-iso	; excludes header
;; C-x 7 g       konvers-iso-to-german-tex		; converts region
;; C-x 7 G       konvers-buffer-iso-to-german-tex	; converts buffer to German TeX
;; C-u C-x 7 G   konvers-buffer-iso-to-german-tex	; excludes header
;; C-x 7 t       konvers-iso-to-seven-bit-tex		; converts region
;; C-x 7 T       konvers-buffer-iso-to-seven-bit-tex	; converts buffer to TeX
;; C-u C-x 7 T   konvers-buffer-iso-to-seven-bit-tex	; excludes header
;; C-x 7 k       konvers-7bit-html-to-8bit-html		; converts region
;; C-x 7 K       konvers-buffer-7bit-html-to-8bit-html	; converts buffer from TeX to Iso
;; C-u C-x 7 K   konvers-buffer-7bit-html-to-8bit-html	; excludes header
;; C-x 7 l       konvers-8bit-html-to-7bit-html		; converts region
;; C-x 7 L       konvers-buffer-8bit-html-to-7bit-html	; converts buffer to TeX
;; C-u C-x 7 L   konvers-buffer-8bit-html-to-7bit-html	; excludes header
;; C-x 7 h   no-seven-type-mode				; turns off all seven-bit typing minor modes 
;; C-x 7 f   german-tex-type-mode			; toggles minor mode for typing German TeX
;; C-x 7 s   seven-bit-tex-type-mode			; toggles minor mode for typing seven bit TeX
;;
;; These are done here in konvers.el:
;;
;; C-x 7 m       konvers-iso-to-mime			; converts region
;; C-x 7 M       konvers-buffer-iso-to-mime		; converts buffer from quoted printable to Iso
;; C-u C-x 7 M   konvers-buffer-iso-to-mime		; excludes header
;; C-x 7 C-m     konvers-insert-quoted-printable-header	; inserts header for Mime-Mail (quoted-printable)
;; C-x 7 n       konvers-mime-to-iso			; converts region
;; C-x 7 N       konvers-buffer-mime-to-iso		; converts buffer from quoted printable to Iso
;; C-u C-x 7 N   konvers-buffer-mime-to-iso		; excludes header
;; C-x 7 C-n     konvers-insert-eight-bit-mime-header	; inserts header for Mime-Mail/News (8-Bit)
;; C-x 7 o       konvers-iso-to-krisable		; converts region
;; C-x 7 O       konvers-header-line-iso-to-krisable    ; converts header line
;; C-x 7 p       konvers-krisable-to-iso                ; converts region
;; C-x 7 P       konvers-header-line-krisable-to-iso    ; converts header line
;;
;; To be memorized easily:
;; C-x 7 <small letter> always converts region
;; C-x 7 <capital letter> without prefix arg converts whole buffer
;; C-u C-x 7 <capital letter> converts buffer excluding header
;; C-x 7 <predecessor of small letter> switches to corresponding
;;   type-minor-mode, only works with  C-x 7 h  ,  C-x 7 f  and  C-x 7 s.
;;
;; b/B means to bibtex (implemented in bib-konvers.el)
;; g/G means to German TeX
;; i/I means to ISO from TeX, understanding German TeX
;; j/J means to ISO from TeX ignoring German TeX
;; k/K means to 8bit from 7bit (HTML)
;; l/L means to 7bit from 8bit (HTML)
;; m/M means to Mime quoted printable
;; n/N means from Mime quoted printable to ISO
;; t/T means to TeX

;;; Code:

(require 'iso-konvers)

(defun konvers-hex-digit-value (x)
  "Calculates the value of a hex digit for konvers-mime-to-iso"
  (if (< x 64) (- x 48) (- x 55)))

;; konvers-hex-digit-value ist only needed by konvers-mime-to-iso.

(defun konvers-mime-to-iso (from to)
  "Converts region from quoted-printable representation to ISO Latin
This is necessary for some mail messages that are encode in this fashion"
  (interactive "*r")
  (let ((case-replace nil)
	(case-fold-search nil)
	(pos nil))
    (save-excursion
      (save-restriction
	(goto-char from)
	(while 
	    (setq pos 
		  (search-forward-regexp 
		   "=[0-9A-F][0-9A-F]\\|=\n" to t))
	  ; find patterns =\n and =<Hexdigit><Hexdigit>
	  (if (= (char-after (- pos 1)) 10)
	      (replace-match "" t t)
	    ; =\n is replaced by nothing (oversized lines)
	    (replace-match (char-to-string
			    (+ (* 16 (konvers-hex-digit-value 
				      (char-after (- pos 2))))
			       (konvers-hex-digit-value
				(char-after (- pos 1))))) t t)
	    ; =?? is replaced by the characters corresponding to
	    ; that hex-digit
	    ))))))



(defun konvers-hex-digit (x)
  "Calculates a string containing a hex digit of x;
only needed for konvers-iso-to-mime"
  (char-to-string 
   (cond
    ((and (<= 0 x) (<= x 9)) (+ x 48))
    ((< x 16) (+ x 55))
    (t 36))))

(defun konvers-hex-transcript (char)
  "Calculates a string =<hexdigit><hexdigit> for CHAR
only needed for konvers-iso-to-mime"
  (concat "=" (konvers-hex-digit (/ char 16))
	  (konvers-hex-digit (% char 16))))

(defun konvers-iso-to-mime (from to &optional arg)
  "Converts region from ISO Latin 1 to quoted-printable representation.
This is necessary for sending mails on seven-bit-lines.
If ARG ist non-nil, handling of long lines is ommitted"
  (interactive "*r")
  (let ((case-replace nil)
	(case-fold-search nil)
	(pos nil)
	(char nil)
	(replace-string nil))
    (save-excursion
      (save-restriction
	(narrow-to-region from to)
	(goto-char (point-min))
	(while 
	    (setq pos (search-forward-regexp "." (point-max) t))
	  (setq char (char-after (- pos 1)))
	  (cond
	   ((or (= char ?=) (> char ?~))
	    (replace-match (konvers-hex-transcript char)))))
		; replaced all suspicious characters now
	(goto-char (point-min))
	(while 
	    (setq pos (search-forward-regexp "[\000-\011\013-\040]+$" (point-max) t))
	  (setq replace-string "")
	  (while (progn
		   (setq char (char-after (setq pos (- pos 1))))
		   (or (and (<= 0 char) (< char 10))
		       (and (< 10 char) (<= char 32))))
	    (setq replace-string (concat (konvers-hex-transcript char) replace-string)))
	  (replace-match replace-string))
		; replaced white space at end of line
	(or arg
	    (progn
	      (goto-char from)
	      (replace-regexp 
	       "^Fr"
	       "=46r")
	      (goto-char from)
	      (replace-regexp 
	       "\\\(.................................................................[^=\n][^=\n]\\\)\\\(.\\\)"
	       "\\1=\n\\2") 
		; long lines have to be split
	      ))))))




(defun konvers-buffer-mime-to-iso (&optional arg)
  "convert full buffer from quoted-printable representation to ISO Latin.
iff ARG is non-nil ommit header.
This is necessary for some mail messages that are encoded in this fashion"
  (interactive "*P")
  (konvers-mime-to-iso (konvers-point-min-or-behind-header arg) (point-max)))

(defun konvers-buffer-iso-to-mime (&optional arg)
  "convert full buffer from ISO Latin 1 to quoted-printable representation.
iff ARG is non-nil ommit header.
This is necessary for sending mal messages through a seven bit line"
  (interactive "*P")
  (konvers-iso-to-mime (konvers-point-min-or-behind-header arg) (point-max)))


(defun konvers-iso-to-krisable (from to)
  "convert region from ISO Latin1 to header-line-quoted-printable representation."
  (interactive "*r")
  (let ((case-replace nil)
	(case-fold-search nil))
    (save-excursion
      (save-restriction
	(narrow-to-region from to)
	(goto-char (point-max))
	(and (re-search-backward "[\200-\377]" (point-min) t)
	     (progn (konvers-iso-to-mime (point-min) (point-max) t)
		    (goto-char (point-min))
		    (insert "=?ISO-8859-1?Q?")
		(goto-char (point-max))
		(insert "?=")
		(while (search-backward "_" (point-min) t)
		  (replace-match "=5F"))
		(goto-char (point-min))
		(while (search-forward " " (point-max) t)
		  (replace-match "_"))))))))

(defun konvers-header-line-iso-to-krisable ()
  "convert header-line from ISO Latin 1 to quoted-printable representation."
  (interactive "*")
  (let
      ((case-replace nil)
       (case-fold-search nil)
       (position (point))
       (header-end (konvers-point-min-or-behind-header t))
       (from ()) 
       (to ()))
    (and (< position header-end)
	 (progn (beginning-of-line)
		(re-search-forward "^[a-zA-Z0-9-]+: " header-end t))
	 (progn (setq from (point))
		(end-of-line)
		(setq to (point))
		(konvers-iso-to-krisable from to)))))

(defun konvers-krisable-to-iso (from to)
  "convert region from header-line-quoted-printable representation to iso."
  (interactive "*r")
  (let ((sub-from nil)
	(sub-to nil)
	(case-replace nil)
	(case-fold-search nil))
    (save-excursion
      (save-restriction
	(narrow-to-region from to)
	(goto-char (point-min))
	(while (search-forward "=?ISO-8859-1?Q?" (point-max) t)
	  (progn (replace-match "")
		 (setq sub-from (point))
		 (search-forward "?=")
		 (replace-match "")
		 (setq sub-to (point))
		 (while (search-backward "_" sub-from t)
		   (replace-match " "))
		 (konvers-mime-to-iso sub-from sub-to)))))))
	

(defun konvers-header-line-krisable-to-iso ()
  "convert header-line from quoted-printable to ISO"
  (interactive "*")
  (let
      ((case-replace nil)
       (case-fold-search nil)
       (position (point))
       (header-end (konvers-point-min-or-behind-header t))
       (from ()) 
       (to ()))
    (and (< position header-end)
	 (progn (beginning-of-line)
		(re-search-forward "^[a-zA-Z0-9-]+: " header-end t))
	 (progn (setq from (point))
		(end-of-line)
		(setq to (point))
		(konvers-krisable-to-iso from to)))))



(defun konvers-insert-mime-version ()
  "inserts the line \"`Mime-Version: 1.0\\n\"'"
  (interactive "*")
  (insert "Mime-Version: 1.0\n"))

(defun konvers-insert-content-type-iso ()
  "inserts the line \"`Content-Type: text/plain; charset=ISO-8859-1\\n\"'"
  (interactive "*")
  (insert "Content-Type: text/plain; charset=ISO-8859-1\n"))

(defun konvers-insert-transfer-enc-eight-bit ()
  "inserts the line \"`Content-Transfer-Encoding: 8bit\\n\"'"
  (interactive "*")
  (insert "Content-Transfer-Encoding: 8bit\n"))

(defun konvers-insert-transfer-enc-quoted-printable ()
  "inserts the line \"`Content-Transfer-Encoding: quoted-printable\\n\"'"
  (interactive "*")
  (insert "Content-Transfer-Encoding: quoted-printable\n"))

(defun konvers-insert-quoted-printable-header ()
  "inserts the three Mime-Mail header lines for quoted printable."
  (interactive "*")
  (konvers-insert-mime-version)
  (konvers-insert-content-type-iso)
  (konvers-insert-transfer-enc-quoted-printable))

(defun konvers-insert-eight-bit-mime-header ()
  "inserts the three Mime-Mail header lines for eight-bit transmission."
  (interactive "*")
  (konvers-insert-mime-version)
  (konvers-insert-content-type-iso)
  (konvers-insert-transfer-enc-eight-bit))


;(define-key ctl-x-map [?7 ?i] 'konvers-german-tex-to-iso)		; done in iso-konvers.el
;(define-key ctl-x-map [?7 ?I] 'konvers-buffer-german-tex-to-iso)
;(define-key ctl-x-map [?7 ?h] 'no-seven-type-mode)
;(define-key ctl-x-map [?7 ?j] 'konvers-seven-bit-tex-to-iso)
;(define-key ctl-x-map [?7 ?J] 'konvers-buffer-seven-bit-tex-to-iso)
;(define-key ctl-x-map [?7 ?g] 'konvers-iso-to-german-tex)
;(define-key ctl-x-map [?7 ?G] 'konvers-buffer-iso-to-german-tex)
;(define-key ctl-x-map [?7 ?f] 'german-tex-type-mode)
;(define-key ctl-x-map [?7 ?t] 'konvers-iso-to-seven-bit-tex)
;(define-key ctl-x-map [?7 ?T] 'konvers-buffer-iso-to-seven-bit-tex)
;(define-key ctl-x-map [?7 ?s] 'seven-bit-tex-type-mode)
;(define-key ctl-x-map [?7 ?k] 'konvers-7bit-html-to-8bit-html)
;(define-key ctl-x-map [?7 ?K] 'konvers-buffer-7bit-html-to-8bit-html)
;(define-key ctl-x-map [?7 ?l] 'konvers-8bit-html-to-7bit-html)
;(define-key ctl-x-map [?7 ?L] 'konvers-buffer-8bit-html-to-7bit-html)	; done in iso-konvers.el
(define-key ctl-x-map [?7 ?m] 'konvers-iso-to-mime)
(define-key ctl-x-map [?7 ?M] 'konvers-buffer-iso-to-mime)
(define-key ctl-x-map [?7 ?\C-M] 'konvers-insert-quoted-printable-header)
(define-key ctl-x-map [?7 ?n] 'konvers-mime-to-iso)
(define-key ctl-x-map [?7 ?N] 'konvers-buffer-mime-to-iso)
(define-key ctl-x-map [?7 ?\C-N] 'konvers-insert-eight-bit-mime-header)
(define-key ctl-x-map [?7 ?o] 'konvers-iso-to-krisable)
(define-key ctl-x-map [?7 ?O] 'konvers-header-line-iso-to-krisable)
(define-key ctl-x-map [?7 ?p] 'konvers-krisable-to-iso)
(define-key ctl-x-map [?7 ?P] 'konvers-header-line-krisable-to-iso)


;; reserved:  e/E asc-to-iso	   (implemented in asc2iso.el)
;; reserved:  a/A iso-to-asc	   (not implemented)
;; reserved:  b/B iso-to-bibtex	   (implemented in bib-konvers.el)

(provide 'konvers)

;;; konvers.el ends here
