;;; iso-konvers.el --- conversion of text

;; Copyright (C) 1993 Karl Brodowsky

;; Author: Karl Brodowsky <bk1@ix.urz.uni-heidelberg.de>
;; Version: 0.911
;; Keywords: i18n, tex, wp

;; This file was written for GNU Emacs version 19

;; It is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; It is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.	If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; created May 1995 by splitting konvers.el
;; last changed 5. May 1995
;; (borrows heavily from Xavier Leroy's latin.el, July 1990)
;; (Version 0.820: completed using TeX-book and tables provided
;; by Kosta Kostis and Markus Kuhn)
;; (Errors in first alpha version had been detected by Uwe Bonnes)
;;
;; WARNING:  THIS FILE CONTAINS 8-BIT CHARACTERS.  
;; MAKE SURE IT HAS BEEN TRANSFERRED IN BINARY MODE!!!
;; DO NOT CONVERT THIS FILE.
;;
;; ACHTUNG:  DIESE DATEI ENTHLT 8-BIT-ZEICHEN.
;; STELLEN SIE SICHER, DASZ SIE BINR BERTRAGEN WORDEN IST.
;; DIESE DATEI VERTRGT KEINE KONVERTIERUNG.
;;
;; It provides conversion for different character sets
;;
;; ISO-8859-1	  --> German	TeX ("a"o"u"A"O"U...)
;; ISO-8859-1	  --> seven bit TeX (\"a\"o\"u\"A\"O\"U...)
;; ISO-8859-1     --> html          (&auml;&ouml;&uuml;....)
;; 
;; and reverse and (since version 0.830)
;;
;; TeX can be taught to understand 8-bit characters.  
;; This should be done whenever possible.  Other solutions
;; are outdated.
;;
;; By enabling conversion in both directions this .el-file
;; should help you using the 8-bit characters wherever you
;; want and using the 7-bit-characters wherever you must.
;; Just write what you want, convert it to 7 bit and mail it
;; or to TeX it with a bad TeX implementation.	Conversion
;; is reversible under certain conditions on your text file.
;;
;; Key sequences beginning with ctrl-x 7 are used:
;;
;; C-x 7 i       konvers-german-tex-to-iso		; converts region 
;; C-x 7 I       konvers-buffer-german-tex-to-iso	; converts buffer from German TeX to ISO-8859-1
;; C-u C-x 7 I   konvers-buffer-german-tex-to-iso	; excludes header
;; C-x 7 j       konvers-seven-bit-tex-to-iso		; converts region
;; C-x 7 J       konvers-buffer-seven-bit-tex-to-iso	; converts buffer from TeX to ISO-8859-1
;; C-u C-x 7 J   konvers-buffer-seven-bit-tex-to-iso	; excludes header
;; C-x 7 g       konvers-iso-to-german-tex		; converts region
;; C-x 7 G       konvers-buffer-iso-to-german-tex	; converts buffer to German TeX
;; C-u C-x 7 G   konvers-buffer-iso-to-german-tex	; excludes header
;; C-x 7 t       konvers-iso-to-seven-bit-tex		; converts region
;; C-x 7 T       konvers-buffer-iso-to-seven-bit-tex	; converts buffer to TeX
;; C-u C-x 7 T   konvers-buffer-iso-to-seven-bit-tex	; excludes header
;; C-x 7 k       konvers-7bit-html-to-8bit-html		; converts region
;; C-x 7 K       konvers-buffer-7bit-html-to-8bit-html	; converts buffer from TeX to ISO-8859-1
;; C-u C-x 7 K   konvers-buffer-7bit-html-to-8bit-html	; excludes header
;; C-x 7 l       konvers-8bit-html-to-7bit-html		; converts region
;; C-x 7 L       konvers-buffer-8bit-html-to-7bit-html	; converts buffer to TeX
;; C-u C-x 7 L   konvers-buffer-8bit-html-to-7bit-html	; excludes header
;;
;; where German TeX means using "a"o"u"s... instead of \"a\"o\"u\ss{}... 
;; C-x 7 i and C-x 7 I also understand non German TeX.
;;
;; C-x 7 h   no-seven-type-mode				; turns off all seven-bit typing minor modes 
;; C-x 7 f   german-tex-type-mode			; toggles minor mode for typing German TeX
;; C-x 7 s   seven-bit-tex-type-mode			; toggles minor mode for typing seven bit TeX
;;
;; These minor modes are independend of TeX/LaTeX mode and can also be
;; used for typing letters or articles or other documents using one of
;; the two TeX conventions.
;;
;; To be memorized easily:
;; C-x 7 <small letter> always converts region
;; C-x 7 <capital letter> without prefix arg converts whole buffer
;; C-u C-x 7 <capital letter> converts buffer excluding header
;; C-x 7 <predecessor of small letter> switches to corresponding
;;   type-minor-mode, only works with  C-x 7 h  ,  C-x 7 f  and  C-x 7 s.
;;
;; b/B means to bibtex (implemented in bib-konvers.el)
;; g/G means to German TeX
;; i/I means to ISO-8859-1 from TeX, understanding German TeX
;; j/J means to ISO-8859-1 from TeX ignoring German TeX
;; k/K means to 8bit from 7bit (HTML)
;; l/L means to 7bit from 8bit (HTML)
;; t/T means to TeX

;;; Code:

(defvar konvers-iso-to-seven-bit-tex-alist nil
  "alist describing the replacements for conversion ISO-8859-1 to seven bit TeX")

(setq konvers-iso-to-seven-bit-tex-alist
      '(
	("" . "\\iso160")	; NO-BREAK SPACE
	("" . "!\'")		; INVERTED EXCLAMATION MARK
	("" . "\\cents{}")	; CENT SIGN
	("" . "{\\it \\$}")	; POUND SIGN
	("" . "\\iso164")	; CURRENCY SIGN
	("" . "\\iso165")	; YEN SIGN
	("" . "\\iso166")	; BROKEN BAR
	("" . "\\S{}")		; SECTION SIGN
	("" . "\\\"\\ ")	; DIAERESIS
	("" . "\\copyright{}") ; COPYRIGHT SIGN
	("" . "\\iso170")	; FEMININE ORDINAL INDICATOR
	("" . "\\flqq{}")	; LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
	("" . "\\neg{}")	; NOT SIGN
	("" . "\\-")		; SOFT HYPHEN
	("" . "\\iso174")	; REGISTERED SIGN
	("" . "\\=\\ ")	; MACRON
	("" . "^\\circ{}")	; DEGREE SIGN
	("" . "\\pm{}")	; PLUS-MINUS SIGN
	("" . "^2")		; SUPERSCRIPT TWO
	("" . "^3")		; SUPERSCRIPT THREE
	("" . "\\'\\ ")	; ACUTE ACCENT
	("" . "\\mu{}")	; MICRO SIGN
	("" . "\\P{}")		; PILCROW SIGN
	("" . "\\cdot{}")	; MIDDLE DOT
	("" . "\\C\\ ")	; CEDILLA
	("" . "^1")		; SUPERSCRIPT ONE
	("" . "\\iso186")	; MASCULINE ORDINAL INDICATOR
	("" . "\\frqq")	; RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
	("" . "{1\\over 4}")	; VULGAR FRACTION ONE QUARTER
	("" . "{1\\over 2}")	; VULGAR FRACTION ONE HALF
	("" . "{3\\over 4}")	; VULGAR FRACTION THREE QUARTERS
	("" . "?\'")		; INVERTED QUESTION MARK
	("" . "\\`A")		; LATIN CAPITAL LETTER A WITH GRAVE
	("" . "\\'A")		; LATIN CAPITAL LETTER A WITH ACUTE
	("" . "\\^A")		; LATIN CAPITAL LETTER A WITH CIRCUMFLEX
	("" . "\\~A")		; LATIN CAPITAL LETTER A WITH TILDE
	("" . "\\\"A")		; LATIN CAPITAL LETTER A WITH DIAERESIS
	("" . "\\AA{}")	; LATIN CAPITAL LETTER A WITH RING ABOVE
	("" . "\\AE{}")	; LATIN CAPITAL LIGATURE AE
	("" . "\\c{C}")	; LATIN CAPITAL LETTER C WITH CEDILLA
	("" . "\\`E")		; LATIN CAPITAL LETTER E WITH GRAVE
	("" . "\\'E")		; LATIN CAPITAL LETTER E WITH ACUTE
	("" . "\\^E")		; LATIN CAPITAL LETTER E WITH CIRCUMFLEX
	("" . "\\\"E")		; LATIN CAPITAL LETTER E WITH DIAERESIS
	("" . "\\`I")		; LATIN CAPITAL LETTER I WITH GRAVE
	("" . "\\'I")		; LATIN CAPITAL LETTER I WITH ACUTE
	("" . "\\^I")		; LATIN CAPITAL LETTER I WITH CIRCUMFLEX
	("" . "\\\"I")		; LATIN CAPITAL LETTER I WITH DIAERESIS
	("" . "\\iso208")	; LATIN CAPITAL LETTER ETH
	("" . "\\~N")		; LATIN CAPITAL LETTER N WITH TILDE
	("" . "\\`O")		; LATIN CAPITAL LETTER O WITH GRAVE
	("" . "\\'O")		; LATIN CAPITAL LETTER O WITH ACUTE
	("" . "\\^O")		; LATIN CAPITAL LETTER O WITH CIRCUMFLEX
	("" . "\\~O")		; LATIN CAPITAL LETTER O WITH TILDE
	("" . "\\\"O")		; LATIN CAPITAL LETTER O WITH DIAERESIS
	("" . "\\times{}")	; MULTIPLICATION SIGN
	("" . "\\O{}")		; LATIN CAPITAL LETTER O WITH STROKE
	("" . "\\`U")		; LATIN CAPITAL LETTER U WITH GRAVE
	("" . "\\'U")		; LATIN CAPITAL LETTER U WITH ACUTE
	("" . "\\^U")		; LATIN CAPITAL LETTER U WITH CIRCUMFLEX
	("" . "\\\"U")		; LATIN CAPITAL LETTER U WITH DIAERESIS
	("" . "\\'Y")		; LATIN CAPITAL LETTER Y WITH ACUTE
	("" . "\\iso222")	; LATIN CAPITAL LETTER THORN
	("" . "\\ss{}")	; LATIN SMALL LETTER SHARP S
	("" . "\\`a")		; LATIN SMALL LETTER A WITH GRAVE
	("" . "\\'a")		; LATIN SMALL LETTER A WITH ACUTE
	("" . "\\^a")		; LATIN SMALL LETTER A WITH CIRCUMFLEX
	("" . "\\~a")		; LATIN SMALL LETTER A WITH TILDE
	("" . "\\\"a")		; LATIN SMALL LETTER A WITH DIAERESIS
	("" . "\\aa{}")	; LATIN SMALL LETTER A WITH RING ABOVE
	("" . "\\ae{}")	; LATIN SMALL LIGATURE AE
	("" . "\\c{c}")	; LATIN SMALL LETTER C WITH CEDILLA
	("" . "\\`e")		; LATIN SMALL LETTER E WITH GRAVE
	("" . "\\'e")		; LATIN SMALL LETTER E WITH ACUTE
	("" . "\\^e")		; LATIN SMALL LETTER E WITH CIRCUMFLEX
	("" . "\\\"e")		; LATIN SMALL LETTER E WITH DIAERESIS
	("" . "\\`{\\i}")	; LATIN SMALL LETTER I WITH GRAVE
	("" . "\\'{\\i}")	; LATIN SMALL LETTER I WITH ACUTE
	("" . "\\^{\\i}")	; LATIN SMALL LETTER I WITH CIRCUMFLEX
	("" . "\\\"{\\i}")	; LATIN SMALL LETTER I WITH DIAERESIS
	("" . "\\iso240")	; LATIN SMALL LETTER ETH
	("" . "\\~n")		; LATIN SMALL LETTER N WITH TILDE
	("" . "\\`o")		; LATIN SMALL LETTER O WITH GRAVE
	("" . "\\'o")		; LATIN SMALL LETTER O WITH ACUTE
	("" . "\\^o")		; LATIN SMALL LETTER O WITH CIRCUMFLEX
	("" . "\\~o")		; LATIN SMALL LETTER O WITH TILDE
	("" . "\\\"o")		; LATIN SMALL LETTER O WITH DIAERESIS
	("" . "\\div{}")	; DIVISION SIGN
	("" . "\\o{}")		; LATIN SMALL LETTER O WITH STROKE
	("" . "\\`u")		; LATIN SMALL LETTER U WITH GRAVE
	("" . "\\'u")		; LATIN SMALL LETTER U WITH ACUTE
	("" . "\\^u")		; LATIN SMALL LETTER U WITH CIRCUMFLEX
	("" . "\\\"u")		; LATIN SMALL LETTER U WITH DIAERESIS
	("" . "\\'y")		; LATIN SMALL LETTER Y WITH ACUTE
	("" . "\\iso254")	; LATIN SMALL LETTER THORN
	("" . "\\\"y")		; LATIN SMALL LETTER Y WITH DIAERESIS
	))		 ; konvers-iso-to-seven-bit-tex-alist

(let ((i 256))											       
  (while (> i 128) 
    (setq i (- i 1))
    (if (null (assoc (char-to-string i) konvers-iso-to-seven-bit-tex-alist))
	(setq konvers-iso-to-seven-bit-tex-alist
	      (cons (cons (char-to-string i) 
			  (format "\\iso%03d" i))
		    konvers-iso-to-seven-bit-tex-alist)))))

;; fill in missing eight-bit characters which are replaced by \iso<3-digit-decimal-number>


(defvar konvers-seven-bit-tex-to-iso-alist nil
  "alist describing the replacements for conversion from seven bit TeX to ISO-8859-1")

(setq konvers-seven-bit-tex-to-iso-alist 
      '(
	("!\'" . "")		; INVERTED EXCLAMATION MARK
	("?\'" . "")		; INVERTED QUESTION MARK
	("\\'\\ " . "")	; ACUTE ACCENT
	("\\-" . "")		; SOFT HYPHEN
	("\\=\\ " . "")	; MACRON
	("\\C\\ " . "")	; CEDILLA
	("\\P{}" . "")		; PILCROW SIGN
	("\\S{}" . "")		; SECTION SIGN
	("\\cdot{}" . "")	; MIDDLE DOT
	("\\cents{}" . "")	; CENT SIGN
	("\\copyright{}" . "") ; COPYRIGHT SIGN
	("\\flqq{}" . "")	; LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
	("\\frqq" . "")	; RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
;	("\\iso160" . "")	; NO-BREAK SPACE
;	("\\iso164" . "")	; CURRENCY SIGN
;	("\\iso165" . "")	; YEN SIGN
;	("\\iso166" . "")	; BROKEN BAR
;	("\\iso170" . "")	; FEMININE ORDINAL INDICATOR
;	("\\iso174" . "")	; REGISTERED SIGN
;	("\\iso186" . "")	; MASCULINE ORDINAL INDICATOR
	("\\mu{}" . "")	; MICRO SIGN
	("\\neg{}" . "")	; NOT SIGN
	("\\pm{}" . "")	; PLUS-MINUS SIGN
	("^1" . "")		; SUPERSCRIPT ONE
	("^2" . "")		; SUPERSCRIPT TWO
	("^3" . "")		; SUPERSCRIPT THREE
	("^\\circ{}" . "")	; DEGREE SIGN
	("{1\\over 2}" . "")	; VULGAR FRACTION ONE HALF
	("{1\\over 4}" . "")	; VULGAR FRACTION ONE QUARTER
	("{3\\over 4" . "")	; VULGAR FRACTION THREE QUARTERS
	("{\\it \\$}" . "")	; POUND SIGN
	("\\'A" . "")		; LATIN CAPITAL LETTER A WITH ACUTE
	("\\'E" . "")		; LATIN CAPITAL LETTER E WITH ACUTE
	("\\'I" . "")		; LATIN CAPITAL LETTER I WITH ACUTE
	("\\'O" . "")		; LATIN CAPITAL LETTER O WITH ACUTE
	("\\'U" . "")		; LATIN CAPITAL LETTER U WITH ACUTE
	("\\'Y" . "")		; LATIN CAPITAL LETTER Y WITH ACUTE
	("\\'a" . "")		; LATIN SMALL LETTER A WITH ACUTE
	("\\'e" . "")		; LATIN SMALL LETTER E WITH ACUTE
	("\\'o" . "")		; LATIN SMALL LETTER O WITH ACUTE
	("\\'u" . "")		; LATIN SMALL LETTER U WITH ACUTE
	("\\'y" . "")		; LATIN SMALL LETTER Y WITH ACUTE
	("\\'{\\i}" . "")	; LATIN SMALL LETTER I WITH ACUTE
	("\\'{a}" . "")
	("\\'{e}" . "")
	("\\'{o}" . "")
	("\\'{u}" . "")
	("\\AA{}" . "")	; LATIN CAPITAL LETTER A WITH RING ABOVE
	("\\AE{}" . "")	; LATIN CAPITAL LIGATURE AE
	("\\O{}" . "")		; LATIN CAPITAL LETTER O WITH STROKE
	("\\\"A" . "")		; LATIN CAPITAL LETTER A WITH DIAERESIS
	("\\\"E" . "")		; LATIN CAPITAL LETTER E WITH DIAERESIS
	("\\\"I" . "")		; LATIN CAPITAL LETTER I WITH DIAERESIS
	("\\\"O" . "")		; LATIN CAPITAL LETTER O WITH DIAERESIS
	("\\\"U" . "")		; LATIN CAPITAL LETTER U WITH DIAERESIS
	("\\\"\\ " . "")	; DIAERESIS
	("\\\"a" . "")		; LATIN SMALL LETTER A WITH DIAERESIS
	("\\\"e" . "")		; LATIN SMALL LETTER E WITH DIAERESIS
	("\\\"o" . "")		; LATIN SMALL LETTER O WITH DIAERESIS
	("\\\"u" . "")		; LATIN SMALL LETTER U WITH DIAERESIS
	("\\\"y" . "")		; LATIN SMALL LETTER Y WITH DIAERESIS
	("\\\"{\\i}" . "")	; LATIN SMALL LETTER I WITH DIAERESIS
	("\\\"{a}" . "")
	("\\\"{e}" . "")
	("\\\"{o}" . "")
	("\\\"{u}" . "")
	("\\^A" . "")		; LATIN CAPITAL LETTER A WITH CIRCUMFLEX
	("\\^E" . "")		; LATIN CAPITAL LETTER E WITH CIRCUMFLEX
	("\\^I" . "")		; LATIN CAPITAL LETTER I WITH CIRCUMFLEX
	("\\^O" . "")		; LATIN CAPITAL LETTER O WITH CIRCUMFLEX
	("\\^U" . "")		; LATIN CAPITAL LETTER U WITH CIRCUMFLEX
	("\\^a" . "")		; LATIN SMALL LETTER A WITH CIRCUMFLEX
	("\\^e" . "")		; LATIN SMALL LETTER E WITH CIRCUMFLEX
	("\\^o" . "")		; LATIN SMALL LETTER O WITH CIRCUMFLEX
	("\\^u" . "")		; LATIN SMALL LETTER U WITH CIRCUMFLEX
	("\\^{\\i}" . "")	; LATIN SMALL LETTER I WITH CIRCUMFLEX
	("\\^{a}" . "")
	("\\^{e}" . "")
	("\\^{o}" . "")
	("\\^{u}" . "")
	("\\`A" . "")		; LATIN CAPITAL LETTER A WITH GRAVE
	("\\`E" . "")		; LATIN CAPITAL LETTER E WITH GRAVE
	("\\`I" . "")		; LATIN CAPITAL LETTER I WITH GRAVE
	("\\`O" . "")		; LATIN CAPITAL LETTER O WITH GRAVE
	("\\`U" . "")		; LATIN CAPITAL LETTER U WITH GRAVE
	("\\`a" . "")		; LATIN SMALL LETTER A WITH GRAVE
	("\\`e" . "")		; LATIN SMALL LETTER E WITH GRAVE
	("\\`o" . "")		; LATIN SMALL LETTER O WITH GRAVE
	("\\`u" . "")		; LATIN SMALL LETTER U WITH GRAVE
	("\\`{\\i}" . "")	; LATIN SMALL LETTER I WITH GRAVE
	("\\`{a}" . "")
	("\\`{e}" . "")
	("\\`{o}" . "")
	("\\`{u}" . "")
	("\\aa{}" . "")	; LATIN SMALL LETTER A WITH RING ABOVE
	("\\ae{}" . "")	; LATIN SMALL LIGATURE AE
	("\\c{C}" . "")	; LATIN CAPITAL LETTER C WITH CEDILLA
	("\\c{c}" . "")	; LATIN SMALL LETTER C WITH CEDILLA
	("\\c{}C" . "")
	("\\c{}c" . "")
	("\\div{}" . "")	; DIVISION SIGN
	("\\iso208" . "")	; LATIN CAPITAL LETTER ETH
	("\\iso222" . "")	; LATIN CAPITAL LETTER THORN
	("\\iso240" . "")	; LATIN SMALL LETTER ETH
	("\\iso254" . "")	; LATIN SMALL LETTER THORN
	("\\o{}" . "")		; LATIN SMALL LETTER O WITH STROKE
	("\\ss{}" . "")	; LATIN SMALL LETTER SHARP S
	("\\times{}" . "")	; MULTIPLICATION SIGN
	("\\~A" . "")		; LATIN CAPITAL LETTER A WITH TILDE
	("\\~N" . "")		; LATIN CAPITAL LETTER N WITH TILDE
	("\\~O" . "")		; LATIN CAPITAL LETTER O WITH TILDE
	("\\~a" . "")		; LATIN SMALL LETTER A WITH TILDE
	("\\~n" . "")		; LATIN SMALL LETTER N WITH TILDE
	("\\~o" . "")		; LATIN SMALL LETTER O WITH TILDE
	("{\\ss}" . "")
	))		; konvers-seven-bit-tex-to-iso-alist

(let ((i 256))											       
  (while (> i 128) 
    (setq i (- i 1))
    (setq konvers-seven-bit-tex-to-iso-alist
	  (cons (cons (format "\\iso%d" i)
		      (char-to-string i))
		konvers-seven-bit-tex-to-iso-alist))))


(defvar konvers-iso-to-german-tex-alist nil
  "alist describing the replacements")

(setq konvers-iso-to-german-tex-alist
      (append
       '(
	 ("" . "\"A")
	 ("" . "\"E")
	 ("" . "\"I")
	 ("" . "\"O")
	 ("" . "\"U")
	 ("" . "\"s")
	 ("" . "\"a")
	 ("" . "\"e")
	 ("" . "\"i")
	 ("" . "\"o")
	 ("" . "\"u"))
       konvers-iso-to-seven-bit-tex-alist)) ; konvers-iso-to-german-tex-alist

(defvar konvers-german-tex-to-iso-alist nil
  "alist describing the replacements")

(setq konvers-german-tex-to-iso-alist
      (append
       konvers-seven-bit-tex-to-iso-alist
       '(
	 ("\"s" . "")
	 ("\\3" . "")
	 ("\"a" . "")
	 ("\"A" . "")
	 ("\"e" . "")
	 ("\"E" . "")
	 ("\"i" . "")
	 ("\"I" . "")
	 ("\"o" . "")
	 ("\"O" . "")
	 ("\"u" . "")
	 ("\"U" . ""))))		; konvers-german-tex-to-iso-alist


(defvar konvers-8bit-html-to-7bit-html-alist nil
  "alist describing the replacements for conversion ISO-8859-1 to seven bit HTML")

(setq konvers-8bit-html-to-7bit-html-alist
      '(
        ("" . "&nbsp;")        ; NO-BREAK SPACE
        ("" . "&iexcl;")       ; INVERTED EXCLAMATION MARK
        ("" . "&cent;")        ; CENT SIGN
        ("" . "&pound;")       ; POUND SIGN
        ("" . "&curren;")      ; CURRENCY SIGN
        ("" . "&yen;")         ; YEN SIGN
        ("" . "&brvbar;")      ; BROKEN BAR
        ("" . "&sect;")        ; SECTION SIGN
        ("" . "&uml;")         ; DIAERESIS
        ("" . "&copy;")        ; COPYRIGHT SIGN
        ("" . "&ordf;")        ; FEMININE ORDINAL INDICATOR
        ("" . "&laquo;")       ; LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        ("" . "&not;")         ; NOT SIGN
        ("" . "&shy;")         ; SOFT HYPHEN
        ("" . "&reg;")         ; REGISTERED SIGN
        ("" . "&hibar;")       ; MACRON
        ("" . "&deg;")         ; DEGREE SIGN
        ("" . "&plusmn;")      ; PLUS-MINUS SIGN
        ("" . "&sup2;")        ; SUPERSCRIPT TWO
        ("" . "&sup3;")        ; SUPERSCRIPT THREE
        ("" . "&acute;")       ; ACUTE ACCENT
        ("" . "&micro;")       ; MICRO SIGN
        ("" . "&para;")        ; PILCROW SIGN
        ("" . "&middot;")      ; MIDDLE DOT
        ("" . "&cedil;")       ; CEDILLA
        ("" . "&sup1;")        ; SUPERSCRIPT ONE
        ("" . "&ordm;")        ; MASCULINE ORDINAL INDICATOR
        ("" . "&raquo;")       ; RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        ("" . "&frac14;")      ; VULGAR FRACTION ONE QUARTER
        ("" . "&frac12;")      ; VULGAR FRACTION ONE HALF
        ("" . "&frac34;")      ; VULGAR FRACTION THREE QUARTERS
        ("" . "&iquest;")      ; INVERTED QUESTION MARK
        ("" . "&Agrave;")      ; LATIN CAPITAL LETTER A WITH GRAVE
        ("" . "&Aacute;")      ; LATIN CAPITAL LETTER A WITH ACUTE
        ("" . "&Acirc;")       ; LATIN CAPITAL LETTER A WITH CIRCUMFLEX
        ("" . "&Atilde;")      ; LATIN CAPITAL LETTER A WITH TILDE
        ("" . "&Auml;")        ; LATIN CAPITAL LETTER A WITH DIAERESIS
        ("" . "&Aring;")       ; LATIN CAPITAL LETTER A WITH RING ABOVE
        ("" . "&AElig;")       ; LATIN CAPITAL LIGATURE AE
        ("" . "&Ccedil;")      ; LATIN CAPITAL LETTER C WITH CEDILLA
        ("" . "&Egrave;")      ; LATIN CAPITAL LETTER E WITH GRAVE
        ("" . "&Eacute;")      ; LATIN CAPITAL LETTER E WITH ACUTE
        ("" . "&Ecirc;")       ; LATIN CAPITAL LETTER E WITH CIRCUMFLEX
        ("" . "&Euml;")        ; LATIN CAPITAL LETTER E WITH DIAERESIS
        ("" . "&Igrave;")      ; LATIN CAPITAL LETTER I WITH GRAVE
        ("" . "&Iacute;")      ; LATIN CAPITAL LETTER I WITH ACUTE
        ("" . "&Icirc;")       ; LATIN CAPITAL LETTER I WITH CIRCUMFLEX
        ("" . "&Iuml;")        ; LATIN CAPITAL LETTER I WITH DIAERESIS
        ("" . "&ETH;")         ; LATIN CAPITAL LETTER ETH
        ("" . "&Ntilde;")      ; LATIN CAPITAL LETTER N WITH TILDE
        ("" . "&Ograve;")      ; LATIN CAPITAL LETTER O WITH GRAVE
        ("" . "&Oacute;")      ; LATIN CAPITAL LETTER O WITH ACUTE
        ("" . "&Ocirc;")       ; LATIN CAPITAL LETTER O WITH CIRCUMFLEX
        ("" . "&Otilde;")      ; LATIN CAPITAL LETTER O WITH TILDE
        ("" . "&Ouml;")        ; LATIN CAPITAL LETTER O WITH DIAERESIS
        ("" . "&times;")       ; MULTIPLICATION SIGN
        ("" . "&Oslash;")      ; LATIN CAPITAL LETTER O WITH STROKE
        ("" . "&Ugrave;")      ; LATIN CAPITAL LETTER U WITH GRAVE
        ("" . "&Uacute;")      ; LATIN CAPITAL LETTER U WITH ACUTE
        ("" . "&Ucirc;")       ; LATIN CAPITAL LETTER U WITH CIRCUMFLEX
        ("" . "&Uuml;")        ; LATIN CAPITAL LETTER U WITH DIAERESIS
        ("" . "&Yacute;")      ; LATIN CAPITAL LETTER Y WITH ACUTE
        ("" . "&THORN;")       ; LATIN CAPITAL LETTER THORN
        ("" . "&szlig;")       ; LATIN SMALL LETTER SHARP S
        ("" . "&agrave;")      ; LATIN SMALL LETTER A WITH GRAVE
        ("" . "&aacute;")      ; LATIN SMALL LETTER A WITH ACUTE
        ("" . "&acirc;")       ; LATIN SMALL LETTER A WITH CIRCUMFLEX
        ("" . "&atilde;")      ; LATIN SMALL LETTER A WITH TILDE
        ("" . "&auml;")        ; LATIN SMALL LETTER A WITH DIAERESIS
        ("" . "&aring;")       ; LATIN SMALL LETTER A WITH RING ABOVE
        ("" . "&aelig;")       ; LATIN SMALL LIGATURE AE
        ("" . "&ccedil;")      ; LATIN SMALL LETTER C WITH CEDILLA
        ("" . "&egrave;")      ; LATIN SMALL LETTER E WITH GRAVE
        ("" . "&eacute;")      ; LATIN SMALL LETTER E WITH ACUTE
        ("" . "&ecirc;")       ; LATIN SMALL LETTER E WITH CIRCUMFLEX
        ("" . "&euml;")        ; LATIN SMALL LETTER E WITH DIAERESIS
        ("" . "&igrave;")      ; LATIN SMALL LETTER I WITH GRAVE
        ("" . "&iacute;")      ; LATIN SMALL LETTER I WITH ACUTE
        ("" . "&icirc;")       ; LATIN SMALL LETTER I WITH CIRCUMFLEX
        ("" . "&iuml;")        ; LATIN SMALL LETTER I WITH DIAERESIS
        ("" . "&eth;")         ; LATIN SMALL LETTER ETH
        ("" . "&ntilde;")      ; LATIN SMALL LETTER N WITH TILDE
        ("" . "&ograve;")      ; LATIN SMALL LETTER O WITH GRAVE
        ("" . "&oacute;")      ; LATIN SMALL LETTER O WITH ACUTE
        ("" . "&ocirc;")       ; LATIN SMALL LETTER O WITH CIRCUMFLEX
        ("" . "&otilde;")      ; LATIN SMALL LETTER O WITH TILDE
        ("" . "&ouml;")        ; LATIN SMALL LETTER O WITH DIAERESIS
        ("" . "&divide;")      ; DIVISION SIGN
        ("" . "&oslash;")      ; LATIN SMALL LETTER O WITH STROKE
        ("" . "&ugrave;")      ; LATIN SMALL LETTER U WITH GRAVE
        ("" . "&uacute;")      ; LATIN SMALL LETTER U WITH ACUTE
        ("" . "&ucirc;")       ; LATIN SMALL LETTER U WITH CIRCUMFLEX
        ("" . "&uuml;")        ; LATIN SMALL LETTER U WITH DIAERESIS
        ("" . "&yacute;")      ; LATIN SMALL LETTER Y WITH ACUTE
        ("" . "&thorn;")       ; LATIN SMALL LETTER THORN
        ("" . "&yuml;")        ; LATIN SMALL LETTER Y WITH DIAERESIS
	))	      	; konvers-8bit-html-to-7bit-html-alist

(let ((i 256))											       
  (while (> i 128) 
    (setq i (- i 1))
    (if (null (assoc (char-to-string i) konvers-8bit-html-to-7bit-html-alist))
	(setq konvers-8bit-html-to-7bit-html-alist
	      (cons (cons (char-to-string i) 
			  (format "&#%d" i))
		    konvers-8bit-html-to-7bit-html-alist)))))


(defvar konvers-7bit-html-to-8bit-html-alist nil
  "alist describing the replacements for conversion from seven bit TeX to ISO-8859-1")

(setq konvers-7bit-html-to-8bit-html-alist 
      '(
        ("&nbsp;" . "")        ; NO-BREAK SPACE
        ("&iexcl;" . "")       ; INVERTED EXCLAMATION MARK
        ("&cent;" . "")        ; CENT SIGN
        ("&pound;" . "")       ; POUND SIGN
        ("&curren;" . "")      ; CURRENCY SIGN
        ("&yen;" . "")         ; YEN SIGN
        ("&brvbar;" . "")      ; BROKEN BAR
        ("&brkbar;" . "")      ; BROKEN BAR (second form)
        ("&sect;" . "")        ; SECTION SIGN
        ("&uml;" . "")         ; DIAERESIS
        ("&copy;" . "")        ; COPYRIGHT SIGN
        ("&ordf;" . "")        ; FEMININE ORDINAL INDICATOR
        ("&laquo;" . "")       ; LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        ("&not;" . "")         ; NOT SIGN
        ("&shy;" . "")         ; SOFT HYPHEN
        ("&reg;" . "")         ; REGISTERED SIGN
        ("&hibar;" . "")       ; MACRON
        ("&deg;" . "")         ; DEGREE SIGN
        ("&plusmn;" . "")      ; PLUS-MINUS SIGN
        ("&sup2;" . "")        ; SUPERSCRIPT TWO
        ("&sup3;" . "")        ; SUPERSCRIPT THREE
        ("&acute;" . "")       ; ACUTE ACCENT
        ("&micro;" . "")       ; MICRO SIGN
        ("&para;" . "")        ; PILCROW SIGN
        ("&middot;" . "")      ; MIDDLE DOT
        ("&cedil;" . "")       ; CEDILLA
        ("&sup1;" . "")        ; SUPERSCRIPT ONE
        ("&ordm;" . "")        ; MASCULINE ORDINAL INDICATOR
        ("&raquo;" . "")       ; RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        ("&frac14;" . "")      ; VULGAR FRACTION ONE QUARTER
        ("&frac12;" . "")      ; VULGAR FRACTION ONE HALF
        ("&frac34;" . "")      ; VULGAR FRACTION THREE QUARTERS
        ("&iquest;" . "")      ; INVERTED QUESTION MARK
        ("&Agrave;" . "")      ; LATIN CAPITAL LETTER A WITH GRAVE
        ("&Aacute;" . "")      ; LATIN CAPITAL LETTER A WITH ACUTE
        ("&Acirc;" . "")       ; LATIN CAPITAL LETTER A WITH CIRCUMFLEX
        ("&Atilde;" . "")      ; LATIN CAPITAL LETTER A WITH TILDE
        ("&Auml;" . "")        ; LATIN CAPITAL LETTER A WITH DIAERESIS
        ("&Aring;" . "")       ; LATIN CAPITAL LETTER A WITH RING ABOVE
        ("&AElig;" . "")       ; LATIN CAPITAL LIGATURE AE
        ("&Ccedil;" . "")      ; LATIN CAPITAL LETTER C WITH CEDILLA
        ("&Egrave;" . "")      ; LATIN CAPITAL LETTER E WITH GRAVE
        ("&Eacute;" . "")      ; LATIN CAPITAL LETTER E WITH ACUTE
        ("&Ecirc;" . "")       ; LATIN CAPITAL LETTER E WITH CIRCUMFLEX
        ("&Euml;" . "")        ; LATIN CAPITAL LETTER E WITH DIAERESIS
        ("&Igrave;" . "")      ; LATIN CAPITAL LETTER I WITH GRAVE
        ("&Iacute;" . "")      ; LATIN CAPITAL LETTER I WITH ACUTE
        ("&Icirc;" . "")       ; LATIN CAPITAL LETTER I WITH CIRCUMFLEX
        ("&Iuml;" . "")        ; LATIN CAPITAL LETTER I WITH DIAERESIS
        ("&ETH;" . "")         ; LATIN CAPITAL LETTER ETH
        ("&Dstrok;" . "")      ; LATIN CAPITAL LETTER ETH (second form)
        ("&Ntilde;" . "")      ; LATIN CAPITAL LETTER N WITH TILDE
        ("&Ograve;" . "")      ; LATIN CAPITAL LETTER O WITH GRAVE
        ("&Oacute;" . "")      ; LATIN CAPITAL LETTER O WITH ACUTE
        ("&Ocirc;" . "")       ; LATIN CAPITAL LETTER O WITH CIRCUMFLEX
        ("&Otilde;" . "")      ; LATIN CAPITAL LETTER O WITH TILDE
        ("&Ouml;" . "")        ; LATIN CAPITAL LETTER O WITH DIAERESIS
        ("&times;" . "")       ; MULTIPLICATION SIGN
        ("&Oslash;" . "")      ; LATIN CAPITAL LETTER O WITH STROKE
        ("&Ugrave;" . "")      ; LATIN CAPITAL LETTER U WITH GRAVE
        ("&Uacute;" . "")      ; LATIN CAPITAL LETTER U WITH ACUTE
        ("&Ucirc;" . "")       ; LATIN CAPITAL LETTER U WITH CIRCUMFLEX
        ("&Uuml;" . "")        ; LATIN CAPITAL LETTER U WITH DIAERESIS
        ("&Yacute;" . "")      ; LATIN CAPITAL LETTER Y WITH ACUTE
        ("&THORN;" . "")       ; LATIN CAPITAL LETTER THORN
        ("&szlig;" . "")       ; LATIN SMALL LETTER SHARP S
        ("&agrave;" . "")      ; LATIN SMALL LETTER A WITH GRAVE
        ("&aacute;" . "")      ; LATIN SMALL LETTER A WITH ACUTE
        ("&acirc;" . "")       ; LATIN SMALL LETTER A WITH CIRCUMFLEX
        ("&atilde;" . "")      ; LATIN SMALL LETTER A WITH TILDE
        ("&auml;" . "")        ; LATIN SMALL LETTER A WITH DIAERESIS
        ("&aring;" . "")       ; LATIN SMALL LETTER A WITH RING ABOVE
        ("&aelig;" . "")       ; LATIN SMALL LIGATURE AE
        ("&ccedil;" . "")      ; LATIN SMALL LETTER C WITH CEDILLA
        ("&egrave;" . "")      ; LATIN SMALL LETTER E WITH GRAVE
        ("&eacute;" . "")      ; LATIN SMALL LETTER E WITH ACUTE
        ("&ecirc;" . "")       ; LATIN SMALL LETTER E WITH CIRCUMFLEX
        ("&euml;" . "")        ; LATIN SMALL LETTER E WITH DIAERESIS
        ("&igrave;" . "")      ; LATIN SMALL LETTER I WITH GRAVE
        ("&iacute;" . "")      ; LATIN SMALL LETTER I WITH ACUTE
        ("&icirc;" . "")       ; LATIN SMALL LETTER I WITH CIRCUMFLEX
        ("&iuml;" . "")        ; LATIN SMALL LETTER I WITH DIAERESIS
        ("&eth;" . "")         ; LATIN SMALL LETTER ETH
        ("&ntilde;" . "")      ; LATIN SMALL LETTER N WITH TILDE
        ("&ograve;" . "")      ; LATIN SMALL LETTER O WITH GRAVE
        ("&oacute;" . "")      ; LATIN SMALL LETTER O WITH ACUTE
        ("&ocirc;" . "")       ; LATIN SMALL LETTER O WITH CIRCUMFLEX
        ("&otilde;" . "")      ; LATIN SMALL LETTER O WITH TILDE
        ("&ouml;" . "")        ; LATIN SMALL LETTER O WITH DIAERESIS
        ("&divide;" . "")      ; DIVISION SIGN
        ("&oslash;" . "")      ; LATIN SMALL LETTER O WITH STROKE
        ("&ugrave;" . "")      ; LATIN SMALL LETTER U WITH GRAVE
        ("&uacute;" . "")      ; LATIN SMALL LETTER U WITH ACUTE
        ("&ucirc;" . "")       ; LATIN SMALL LETTER U WITH CIRCUMFLEX
        ("&uuml;" . "")        ; LATIN SMALL LETTER U WITH DIAERESIS
        ("&yacute;" . "")      ; LATIN SMALL LETTER Y WITH ACUTE
        ("&thorn;" . "")       ; LATIN SMALL LETTER THORN
        ("&yuml;" . "")        ; LATIN SMALL LETTER Y WITH DIAERESIS
	))   		; konvers-7bit-html-to-8bit-html-alist

(let ((i 256))											       
  (while (> i 128) 
    (setq i (- i 1))
    (setq konvers-7bit-html-to-8bit-html-alist
	  (cons (cons (format "&#%d;" i)
		      (char-to-string i))
		konvers-7bit-html-to-8bit-html-alist))))


(defvar konvers-seven-bit-type-alist nil
  "holds an conversion alist describing current seven bit typing mode")

(make-variable-buffer-local 'konvers-seven-bit-type-alist)

(defun konvers-seven-bit-self-insert-command (&optional arg)
  "Inserts the seven bit replacement of the character you type.
Do insertion ARG times if prefix arg was used.
Which seven bit replacement is being used depends on which of the
available minor modes for seven bit typing has been chosen"
  (interactive "*p")
  (let ((keys (this-command-keys))
	(last-key nil)                ; the last key typed
	(ins nil))                    ; sequence to be inserted
    (setq last-key (aref keys (- (length keys) 1)))
    (if (and (numberp last-key)
	     (boundp 'konvers-seven-bit-type-alist)
	     (listp konvers-seven-bit-type-alist))
	(let  ((8bit (char-to-string last-key)))
	  (if (not arg) (setq arg 1))
	  (setq ins (assoc 8bit konvers-seven-bit-type-alist))))
    ;; if a sequence of seven-bit-chars has been found insert it,
    ;; otherwise do something like self-insert-command would have
    ;; done.
    (if ins (while (> arg 0) (insert (cdr ins)) (setq arg (- arg 1)))
      (insert-char last-key arg))))


(defun konvers-make-seven-bit-type-map (alist)
  "produces a keymap for a seven-bit-type-mode (minor-mode)
This keymap does the conversion described by alist on the fly when
typing 8-bit-keys.  non-interactive, only to be used when loading
konvers and similar programs."
  (let ((new-map (make-sparse-keymap)))
    (mapcar (function 
	     (lambda (pair)
	       (define-key new-map
		 (make-vector 1 (string-to-char (car pair)))
		 'konvers-seven-bit-self-insert-command)))
	    alist)
    new-map))


(defvar seven-bit-tex-type-map nil
  "minor mode key map for seven-bit-tex-type-mode")

(setq seven-bit-tex-type-map (konvers-make-seven-bit-type-map 
			      konvers-iso-to-seven-bit-tex-alist))

;; build a keymap defining the keys of iso-8-bit-characters with macros
;; that insert exactly the substitution the conversion would produce.


(defvar german-tex-type-map nil
  "minor mode key map for german-tex-type-mode")

(setq german-tex-type-map (konvers-make-seven-bit-type-map
			   konvers-iso-to-german-tex-alist))

;; build a keymap defining the keys of iso-8-bit-characters with macros
;; that insert exactly the substitution the conversion would produce.


(defun konvers-replace-string-set (from to set)
  "Given an A-list of strings, substitute (in the given region) any string
which is the car of an element of the A-list by the cdr of the same element."
  (let ((case-replace nil)
	(case-fold-search nil)
	(regexp (mapconcat (function (lambda (x) (regexp-quote (car x))))
			   set "\\|")))
    (save-excursion
      (save-restriction
	(narrow-to-region from to)
	(goto-char from)
	(while (re-search-forward regexp nil t)
	  (replace-match
	   (cdr (assoc (buffer-substring (match-beginning 0) (match-end 0))
		       set))
	   t t))))))

;; The function above was taken from Xavier Leroy's latin.el.

(defun konvers-anti-set (liste)
  "inverts a set for use with conversion"
  (mapcar '(lambda (x) (cons (cdr x) (car x))) liste))


(defun konvers-point-min-or-behind-header (&optional arg)
  "Return position of beginning of buffer or behind header
If ARG is nil or omitted just return point-min.
If ARG ist non nil return first position after mail or news header,
where the end of the header is recognized by a blank line
or by a line containing exactly
--text follows this line--"
  (interactive "P")
  (if arg
    (save-excursion
      (goto-char (point-min))
      (re-search-forward
       "^\\\(--text follows this line--\\\)?$"
       (point-max) t)
      (point))
    (point-min)))


(defun konvers-iso-to-seven-bit-tex (from to)
  "Transforms ISO-8859-1 letters to seven bit TeX commands."
  (interactive "*r")
  (konvers-replace-string-set from to konvers-iso-to-seven-bit-tex-alist))

(defun konvers-seven-bit-tex-to-iso (from to)
  "Transforms TeX accents (\'e, ...) to ISO-8859-1 accented characters."
  (interactive "*r")
  (konvers-replace-string-set from to konvers-seven-bit-tex-to-iso-alist))

(defun konvers-iso-to-german-tex (from to)
  "Transforms ISO-8859-1 letters to TeX commands using convention for German TeX"
  (interactive "*r")
  (konvers-replace-string-set from to konvers-iso-to-german-tex-alist))

(defun konvers-german-tex-to-iso (from to)
  "Transforms TeX commands to ISO-8859-1 letters using convention for German TeX"
  (interactive "*r")
  (konvers-replace-string-set from to konvers-german-tex-to-iso-alist))


(defun konvers-8bit-html-to-7bit-html (from to)
  "Transforms ISO-8859-1 letters to seven bit HTML commands."
  (interactive "*r")
  (konvers-replace-string-set from to konvers-8bit-html-to-7bit-html-alist))

(defun konvers-7bit-html-to-8bit-html (from to)
  "Transforms HTML accents to ISO-8859-1 accented characters."
  (interactive "*r")
  (konvers-replace-string-set from to konvers-7bit-html-to-8bit-html-alist))


(defun konvers-buffer-iso-to-german-tex (&optional arg)
  "convert full buffer from ISO-8859-1 to German TeX
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-iso-to-german-tex (konvers-point-min-or-behind-header arg) (point-max)))

(defun konvers-buffer-german-tex-to-iso (&optional arg)
  "convert full buffer from German TeX to ISO-8859-1
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-german-tex-to-iso (konvers-point-min-or-behind-header arg) (point-max)))

(defun konvers-buffer-iso-to-seven-bit-tex (&optional arg)
  "convert full buffer from ISO-8859-1 to seven-bit-TeX
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-iso-to-seven-bit-tex (konvers-point-min-or-behind-header arg) (point-max)))

(defun konvers-buffer-seven-bit-tex-to-iso (&optional arg)
  "convert full buffer from seven-bit-TeX to ISO-8859-1 
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-seven-bit-tex-to-iso (konvers-point-min-or-behind-header arg) (point-max)))


(defun konvers-buffer-8bit-html-to-7bit-html (&optional arg)
  "convert full buffer from ISO-8859-1 to seven-bit-html
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-8bit-html-to-7bit-html (konvers-point-min-or-behind-header arg) (point-max)))

(defun konvers-buffer-7bit-html-to-8bit-html (&optional arg)
  "convert full buffer from seven-bit-html to ISO-8859-1 
iff ARG is non-nil ommit header."
  (interactive "*P")
  (konvers-7bit-html-to-8bit-html (konvers-point-min-or-behind-header arg) (point-max)))


(defvar seven-bit-tex-type-mode nil
  "Non-nil iff eight-bit characters should insert seven-bit sequences.
If non-nil eight-bit non-ascii characters do not selfinsert but insert
a seven bit sequence as replacement which is taken from TeX where
possible and  which has the form \\iso<decimal number> otherwise.")

(make-variable-buffer-local 'seven-bit-tex-type-mode)

(defvar german-tex-type-mode nil
  "Non-nil iff eight-bit characters should insert seven-bit sequences.
If non-nil eight-bit non-ascii characters do not selfinsert but insert
a seven bit sequence as replacement which is taken from TeX preferring
the shortcuts defined in german.sty where possible and	which has the
form \\iso<decimal number> otherwise.")

(make-variable-buffer-local 'german-tex-type-mode)


(defun seven-bit-tex-type-mode (&optional arg)
  "Toggle seven-bit-tex-type-mode, turn off german-tex-type-mode.
With arg, turn seven-bit-tex-type-mode on iff arg is positive.
In seven-bit-tex-type-mode typing of ISO-8859-1-characters results in inserting
the corresponding seven Bit TeX sequences."
  (interactive "P")
  (let ((temp-seven-bit-tex-type-mode
	(if (null arg) (not seven-bit-tex-type-mode)
	  (> (prefix-numeric-value arg) 0))))
    (no-seven-type-mode)
	; turn off all minor modes that cannot coexist with
	; seven-bit-tex-type-mode
    (setq seven-bit-tex-type-mode temp-seven-bit-tex-type-mode)
    (if seven-bit-tex-type-mode 
	(setq konvers-seven-bit-type-alist 
	      konvers-iso-to-seven-bit-tex-alist))
  (set-buffer-modified-p (buffer-modified-p)))) ; No-op, but updates mode line.


(defun german-tex-type-mode (&optional arg)
  "Toggle german-tex-type-mode, turn off seven-bit-tex-type-mode.
With arg, turn german-tex-type-mode on iff arg is positive.
In german-tex-type-mode typing of ISO-8859-1-characters results in inserting
the corresponding German TeX sequences."
  (interactive "P")
  (let ((temp-german-tex-type-mode
	(if (null arg) (not german-tex-type-mode)
	  (> (prefix-numeric-value arg) 0))))
    (no-seven-type-mode)
	; turn off all minor modes that cannot coexist with
	; seven-bit-tex-type-mode
    (setq german-tex-type-mode temp-german-tex-type-mode)
    (if german-tex-type-mode 
	(setq konvers-seven-bit-type-alist 
	      konvers-iso-to-german-tex-alist))
  (set-buffer-modified-p (buffer-modified-p)))) ;No-op, but updates mode line.


(defun no-seven-type-mode ()
  "Turn off all seven-bit-type minor modes that are known."
  (interactive)
  (if (boundp 'seven-bit-tex-type-mode) (set 'seven-bit-tex-type-mode nil))
  (if (boundp 'seven-bit-asc-type-mode) (set 'seven-bit-asc-type-mode nil))
  (if (boundp 'german-tex-type-mode) (set 'german-tex-type-mode nil))
  (if (boundp 'bibtex-type-mode) (set 'bibtex-type-mode nil))
  	; use set and not setq because the variables may not be bound
	; add new lines here if another seven-bit-type minor mode is
	; introduced to turn it off.
  (if (boundp 'seven-bit-type-map) (set 'seven-bit-type-map nil))
  (set-buffer-modified-p (buffer-modified-p))) ;No-op, but updates mode line.


(or (assq 'seven-bit-tex-type-mode minor-mode-alist)
    (setq minor-mode-alist 
	  (cons '(seven-bit-tex-type-mode " t7" t) 
		minor-mode-alist)
	  minor-mode-map-alist
	  (cons (cons 'seven-bit-tex-type-mode
		      seven-bit-tex-type-map)
		minor-mode-map-alist)))


(or (assq 'german-tex-type-mode minor-mode-alist)
    (setq minor-mode-alist 
	  (cons '(german-tex-type-mode " gt7" t) 
		minor-mode-alist)
	  minor-mode-map-alist
	  (cons (cons 'german-tex-type-mode
		      german-tex-type-map)
		minor-mode-map-alist)))


(define-key ctl-x-map [?7 ?i] 'konvers-german-tex-to-iso)
(define-key ctl-x-map [?7 ?I] 'konvers-buffer-german-tex-to-iso)
(define-key ctl-x-map [?7 ?h] 'no-seven-type-mode)
(define-key ctl-x-map [?7 ?j] 'konvers-seven-bit-tex-to-iso)
(define-key ctl-x-map [?7 ?J] 'konvers-buffer-seven-bit-tex-to-iso)
(define-key ctl-x-map [?7 ?g] 'konvers-iso-to-german-tex)
(define-key ctl-x-map [?7 ?G] 'konvers-buffer-iso-to-german-tex)
(define-key ctl-x-map [?7 ?f] 'german-tex-type-mode)
(define-key ctl-x-map [?7 ?t] 'konvers-iso-to-seven-bit-tex)
(define-key ctl-x-map [?7 ?T] 'konvers-buffer-iso-to-seven-bit-tex)
(define-key ctl-x-map [?7 ?s] 'seven-bit-tex-type-mode)
(define-key ctl-x-map [?7 ?k] 'konvers-7bit-html-to-8bit-html)
(define-key ctl-x-map [?7 ?K] 'konvers-buffer-7bit-html-to-8bit-html)
(define-key ctl-x-map [?7 ?l] 'konvers-8bit-html-to-7bit-html)
(define-key ctl-x-map [?7 ?L] 'konvers-buffer-8bit-html-to-7bit-html)
;(define-key ctl-x-map [?7 ?m] 'konvers-iso-to-mime)				; done in konvers.el
;(define-key ctl-x-map [?7 ?M] 'konvers-buffer-iso-to-mime)
;(define-key ctl-x-map [?7 ?\C-M] 'konvers-insert-quoted-printable-header)
;(define-key ctl-x-map [?7 ?n] 'konvers-mime-to-iso)
;(define-key ctl-x-map [?7 ?N] 'konvers-buffer-mime-to-iso)
;(define-key ctl-x-map [?7 ?\C-N] 'konvers-insert-eight-bit-mime-header)
;(define-key ctl-x-map [?7 ?o] 'konvers-iso-to-krisable)
;(define-key ctl-x-map [?7 ?O] 'konvers-header-line-iso-to-krisable)
;(define-key ctl-x-map [?7 ?p] 'konvers-krisable-to-iso)
;(define-key ctl-x-map [?7 ?P] 'konvers-header-line-krisable-to-iso)		; done in konvers.el


;; reserved:  e/E asc-to-iso	   (implemented in asc2iso.el)
;; reserved:  a/A iso-to-asc	   (not implemented)
;; reserved:  b/B iso-to-bibtex	   (implemented in bib-konvers.el)

(provide 'iso-konvers)
(provide 'xxx-konvers)

;;; iso-konvers.el ends here
