
#ifndef lint
static char Sccs_Id[]="@(#)cthread.c	1.1    8/1/94";
#endif

#define USAGE \
"cthread [-a][-o ofile][-r rows][-c cols][-d delim][-p pattern] file [file ...]"

/*
 * where
 *        -a          puts program in Automatic mode immediately
 *        -o ofile    write contents to ofile, default is stdout
 *        -c columns  max screen columns
 *        -r rows     max screen rows
 *        -d delim    defines "delim" as the digest message delimiter
 *        -p pattern  defines "pattern" as the pattern matching string
 */

/*----------------------------------------------------------------------*
*                            Program Thread                            *
*                            July 18, 1992                             *
* -------------------------------------------------------------------- *
* This program has been placed in the public domain.  The author will  *
* accept no responsibility or liability for the software or its use.   *
* Legal issues aside, I'd be happy to receive any comments or sugges-  *
* tions concerning the software.                                       *
* -------------------------------------------------------------------- *
* AUTHOR:  Tom Kaltenbach                                              *
*          429 Woodsong Lane                                           *
*          Rochester, New York 14612                                   *
*          Email:  tom@kalten.bach1.sai.com                            *
* -------------------------------------------------------------------- *
*                                                                      *
* SUMMARY:                                                             *
*                                                                      *
*     THREAD is designed to search back issues of the Homebrew digest  *
* and extract those messages making up a given "thread" of conversa-   *
* tion.  It is also useful to extract all discussion on a particular   *
* subject.  The user supplies key words or phrases for the search, and *
* logical operators (AND, OR, NOT) for combining the key words.  Case  *
* sensitivity of key words is optional.  THREAD searches through a set *
* of digest files input by the user (wildcard filenames are okay)      *
* until it finds a message that meets the key word specifications.     *
* The message is then displayed on the screen, and the user is allowed *
* to read it and decide whether to skip that message (and continue     *
* looking), keep that message (write it to the output file), quit the  *
* program, or enable automatic processing.  If automatic processing is *
* enabled, the user is no longer prompted, and every message meeting   *
* the key word specification is written to the output file.            *
*                                                                      *
* -------------------------------------------------------------------- *
*                                                                      *
* SYSTEM REQUIREMENTS:                                                 *
*                                                                      *
*     THREAD is written in Turbo Pascal 5.5/6.0 for the IBM PC and     *
* compatible 80x86 microcomputers running the MS-DOS (or compatible)   *
* operating system, version 3.x or higher.  The digest issues should   *
* be present as plain, standard ASCII files on the computer's hard     *
* disk.  NOTE:  each line of a standard ASCII file must end in a CR-LF *
* (carriage return - linefeed);  files transferred from non-DOS        *
* computers sometimes end in only a LF.  Such files must be converted  *
* to standard MS-DOS ASCII format.                                     *
*                                                                      *
* -------------------------------------------------------------------- *
*                                                                      *
*  REVISION HISTORY:                                                   *
*                                                                      *
*   Ver    Date/Author/Comments                                        *
*  -----  ---------------------------------------------------          *
*   1.0    18 Jul 1992 -- T. Kaltenbach -- First working version       *
*                                                                      *
*   1.1    20 Jul 1992 -- T. Kaltenbach -- Rewrote section handling    *
*          the logical operators combining the search fields to solve  *
*          problems with complex searches.  Enhanced the user inter-   *
*          face somewhat.                                              *
*                                                                      *
*   1.2    22 Jul 1992 -- T. Kaltenbach -- Improved handling of input  *
*          files, added header to output that records the search para- *
*          meters, etc.  Added procedure-level documentation.          *
*                                                                      *
*   1.3    01 Aug 1992 -- Rick Larson -- Updated Pascal code to C      *
*          and ported to various UNIX OSs (SunOS, HP-UX, Ultrix).      *
*          To compile on a Sun try:                                    *
*          /usr/5bin/cc thread.c -o thread -lcurses                    *
*                 -or-                                                 *
*          cc thread.c -o thread -lcurses -ltermcap                    *
*          To compile on an HP try:                                    *
*          cc thread.c -o thread -lcurses                              *
*          To compile on Ultrix try:                                   *
*          cc thread.c -o thread -lcurses -ltermcap                    *
*          To compile on XENIX try:                                    *
*          cc thread.c -o thread -lcurses -ltermcap                    *
*   1.4    18 Aug 1992 -- Rick Larson -- fixed case problem            *
*                                                                      *
*   2.0    25 Jul 1994 -- John T. Pearson --                           *
*          o BECOMES CTHREAD as it diverges from the DOS version...    *
*            WHATEVER BUGS MAY APPEAR IN THIS VERSION ARE PROBABLY     *
*            CAUSED BY ME.                                             *
*          o convert to case insensitive searches CONSTANTLY           *
*          o user interface changes: pattern matching sentence for     *
*            input; regular expressions allowed; single char inputs w/o*
*            <RETURN>; highlight every matching line;                  *
*          o allow digest files to be compressed if platform supports  *
*          o command line changes: fix "D" delim bug; add auto flag;   *
*            add pattern arg                                           *
*          o remove the awkward "off by one" indexing (probably a      *
*            Pascal [1 based] to C [0 based] conversion issue?)        *
*          o Fix Delimiter bug; it would find match anywhere. Force it *
*            to match first part of line (eliminated false positives)  *
*----------------------------------------------------------------------*/

#include <unistd.h>

#ifdef TRUE
#undef TRUE
#undef FALSE
#endif

#if defined(XOPEN)
#include <cursesX.h>
#else
#include <curses.h>
#endif

#ifndef M_XENIX
#include <sysexits.h>
#endif

#include <string.h>
#include <ctype.h>
#include <signal.h>

#include "cthread.h"

static NoFiles = 0, FileCnt = 0;
static MatchingMsgs = 0;
static char *CurrentFile = NULL;
static SearchRec FileList[1024];		/* some big array */
static char Pattern[NUMCOLS+1];

static BuffType Buffer;
static TxtBuffer TxtInBuf, TxtOutBuf;
static TargetType Target;
static TargetIndex TargIdx;
static char SrcSpec[80], OutName[80];
static short I, J, K, Error, BuffPtr, MaxLines, NumFields;
static boolean Success, Done, Found, Automatic, FilesLeft;
static char Line_[MAX_LINE], Delim[MAX_LINE];
static int DelimLen;
static FILE *InFile, *OutFile;
static char Answer;
static char InFile_NAME[_FNSIZE];
static char *ProgName;
static char Greeting[MAX_LINE];
static char EC = '\010';            /* Default erase character ^H */
static char KC = '\025';            /* Default kill character ^U */
static WINDOW *Win;

static short NumRows = NUMROWS;
static short NumCols = NUMCOLS;

static char compressed = 0;

/*----------------------------------------------------------------------*
*                        Procedure GetUserInput                        *
*                            July 18, 1992                             *
* -------------------------------------------------------------------- *
* AUTHOR:  Tom Kaltenbach                                              *
*          429 Woodsong Lane                                           *
*          Rochester, New York 14612                                   *
*          Email:  tom@kalten.bach1.sai.com                            *
* -------------------------------------------------------------------- *
*                                                                      *
* SUMMARY:                                                             *
*                                                                      *
*    This procedure prompts the user for the necessary input para-     *
* meters.  It performs SOME error-checking on the input parameters and *
* filenames.                                                           *
*                                                                      *
* Input includes input files to scan, optional output filename,        *
* scanning pattern, logical operator (A, O, N)                         *
*                                                                      *
* 07-27-94     John T. Pearson - Butchered to accept pattern strings.  *
*              All bugs should be considered to have been introduced   *
*              by me.                                                  *
*----------------------------------------------------------------------*/

void GetUserInput()
{
  scrollok(Win, TRUE);

  /*
  ** Get input and output files if not passed on command line
  */

  if (!CurrentFile) {
    do {
      addstr("Enter source filespec (wild *not* ok):\n");
      GetString(SrcSpec, sizeof(SrcSpec));
    } while(FindFirst(SrcSpec) != 0);
  }

  if (!*OutName) {
    do {
      addstr("Enter name for output file:\n");
      GetString(OutName, sizeof(OutName));
    } while(*OutName == '\0');
  }

  clear();
  refresh();

  /*------------------------------------------
  Get target string(s) and logical operators
  ------------------------------------------*/

  I = 0;

  if(*Pattern != '\0') {
    ParsePattern(Pattern,Target,&I); 
  }
  else {
    Done = FALSE;
    while (!Done && I < MAXFIELDS) {
      printw("Pattern is \"target op target op target ... end\"\n");
      printw("Logical ops: [A]nd (&+), [O]r (|), [N]ot (!~), [E]nd (.)\n\n");
      printw("Enter pattern string (always case insensitive, ? for help):\n");
      GetString(Pattern,NumCols);
  
      if(*Pattern == '\0') {			/* user entered blank	*/
        if (I == 0) {				/*  loop back		*/
          addstr("Error -- blank field entered\n");
          refresh();
        }
        else
          Done = TRUE;
        continue;
      }
  
      if(*Pattern == '?') {			/* user asked for help	*/
        ShowHelp();
        continue;
      }
  
      Done = ParsePattern(Pattern,Target,&I);	/* parse sentence into	*/
  						/*  targets and logicals*/
    }  /*** of while not(Done) ... ***/
  }

  NumFields = I;
  strcpy(Target[NumFields].Logical, "OR");	/* last logical is OR	*/
						/*  as a safety		*/
  scrollok(Win, FALSE);
}  /*** of procedure GetUserInput ***/

/**/

/* module: ShowHelp()

	Purpose: Display user help to the pattern input.

	Syntax:	void ShowHelp()

	John T. Pearson
*/

void ShowHelp()
{
  clear();
  printw("Enter one or more lines of patterns.  A pattern is a list\n");
  printw("of fields made up of targets separated by zero or more\n");
  printw("logical operations.  Each field is separated by at least\n");
  printw("one space.  Terminate the input with \"end\", \"e\", or \".\"\n");
  printw("in the last logical operator field.\n");

#if defined(RE_PATTERNS)
  printw("\nUNIX(tm) style regular expressions are allowed.\n");
#endif

  printw("\nExample: mill and grain not miller end\n");
  printw("Example: mill & grain ! miller .\n");
  printw("Press return to continue: ");
  refresh();
  getch();
  clear();
  refresh();
  return;
}


/*----------------------------------------------------------------------*
*                          Procedure ReadMsg                           *
*                            July 18, 1992                             *
* -------------------------------------------------------------------- *
* AUTHOR:  Tom Kaltenbach                                              *
*          429 Woodsong Lane                                           *
*          Rochester, New York 14612                                   *
*          Email:  tom@kalten.bach1.sai.com                            *
* -------------------------------------------------------------------- *
*                                                                      *
* SUMMARY:                                                             *
*                                                                      *
*    This procedure reads a message from the current input file.  It   *
* extracts a single message by searching for the delimeter between     *
* messages.  If EOF is reached on the input file, the next file in the *
* series is opened.  If no more files are left, a flag is set and the  *
* program terminates.                                                  *
*                                                                      *
* 07-29-94     John T. Pearson - Changed strstr(Buffer,Delim) to       *
*              strncmp(Buffer,Delim,DelimLen) to force Delim match only*
*              at start of the line (eliminates many false positives)  *
*----------------------------------------------------------------------*/

void ReadMsg(Found, Error)
boolean *Found;
short *Error;
{

  /*--------------------------------------------------------------
  Read message into Buffer until delimiter found, EOF reached, or
  buffer overflows
  ---------------------------------------------------------------*/

  /*** Clear Error flag ***/

  *Error = 0;

  /*** Make sure there are files left ***/

  if (!FilesLeft) {
    *Error = 1;   /*** Set flag for normal program termination ***/
    goto _LEndProc;
  }

  /*** Read next message into Buffer ***/

_LReadMsg:

  BuffPtr = 0;
  *Found = FALSE;
  while (!*Found && !P_eof(InFile) && BuffPtr < MaxLines && FilesLeft) {
    fgets(Buffer[BuffPtr], BUFFER_LEN, InFile);
    if(strncmp(Buffer[BuffPtr], Delim, DelimLen) == 0) {
      *Found = TRUE;
      break;
    }
    BuffPtr++;
  }

  /*** If EOF detected, get next file ready for reading ***/

  if (P_eof(InFile)) {
    if (InFile != NULL)
      CloseFile(InFile);
    InFile = NULL;
    if (!FindNext()) {
      InFile = OpenFile(CurrentFile);

      scrollok (Win, TRUE);                /* allow window scrolling */
      printw("Scanning input file:  %s\n", CurrentFile);
      refresh();
      scrollok (Win, FALSE);                /* inhibit window scrolling */

      /*** Search for delimiter ***/

      *Line_ = '\0';
      *Found = FALSE;
      while (!P_eof(InFile) && !*Found) {
        fgets(Line_, MAX_LINE, InFile);
        if (strncmp(Line_, Delim, DelimLen) == 0)
          *Found = TRUE;
      }

      if (P_eof(InFile)) {
        addstr("Error!  EOF reached searching for first delimiter in the\n");
        printw("file %s\n", CurrentFile);
        refresh();
        if (InFile != NULL)
          CloseFile(InFile);
        *Error = 2;   /*** Set flag to terminate program with error ***/
        goto _LEndProc;
      }
    }  /*** of "if (!FindNext()) ... else ..." ***/

    else
      FilesLeft = FALSE;

    if (BuffPtr == 0 && FilesLeft)
      goto _LReadMsg;

  }  /*** of "if EOF(InFile) ..." ***/

  /*** Warn user if buffer full ***/

  if (BuffPtr == MaxLines) {
    printw(
     "Warning: message longer than buffer limit of %d lines.  Message split.\n",
      MaxLines);
    addstr("Press RETURN to continue: ");

    do {
      GetString(&Answer, 1);
      Answer = toupper(Answer);
    } while (Answer != '\r');
  }

_LEndProc: ;

}  /*** of procedure ReadMsg ***/


/*----------------------------------------------------------------------*
*                          Procedure FindTarg                          *
*                            July 18, 1992                             *
* -------------------------------------------------------------------- *
* AUTHOR:  Tom Kaltenbach                                              *
*          429 Woodsong Lane                                           *
*          Rochester, New York 14612                                   *
*          Email:  tom@kalten.bach1.sai.com                            *
* -------------------------------------------------------------------- *
*                                                                      *
* SUMMARY:                                                             *
*                                                                      *
*    This procedure searches the message buffer for a single user-     *
* supplied search fields.  If search field is case-insensitive, the    *
* strings are converted to uppercase prior to searching.               *
*                                                                      *
* 07-28-94     John T. Pearson - Change TargIdx to capture truth value *
*              of what lines contain a pattern match,                  *
*              so all matching lines can be manipulated.               *
*              Constant caseless comparison.                           *
*              Track largest line number matching.                     *
*----------------------------------------------------------------------*/

boolean FindTarg(TargPtr,maxMatch)
short TargPtr;
int   *maxMatch;
{
  boolean found = FALSE;
  char UpperStr[MAX_LINE], TempStr[MAX_LINE];
  char *p;

  /*** For case-insensitive match, change to target to upper case ***/

  Upper(UpperStr, Target[TargPtr].Str);

#if defined(RE_PATTERNS)
  if((p = re_comp(UpperStr)) != NULL) {		/* if pattern matching	*/
    fprintf(stderr,"re_comp failed: %s\n",p);
    thread_exit(1000);
  }
#endif

  /*** Scan buffer for target string number <TargPtr> ***/

  for(J=0; J < BuffPtr; ++J) {
    Upper(TempStr, Buffer[J]);

#if defined(RE_PATTERNS)
    if(re_exec(TempStr) == 1) {			/* regular expressions	*/
#else
    if (strstr(TempStr, UpperStr)) {		/* straight substring	*/
#endif
      TargIdx[J] = TRUE;
      found = TRUE;
      *maxMatch = max(*maxMatch,J);
    }

  }

  return found;

}  /*** of procedure ScanTarg ***/


/*----------------------------------------------------------------------*
*                       Procedure DisplayBuffer                        *
*                            July 18, 1992                             *
* -------------------------------------------------------------------- *
* AUTHOR:  Tom Kaltenbach                                              *
*          429 Woodsong Lane                                           *
*          Rochester, New York 14612                                   *
*          Email:  tom@kalten.bach1.sai.com                            *
* -------------------------------------------------------------------- *
*                                                                      *
* SUMMARY:                                                             *
*                                                                      *
*    This procedure displays the current message on the screen, and    *
* allows the user to read it and decide whether to skip that message   *
* (and continue looking), keep that message (write it to the output    *
* file), quit the program, or enable automatic processing.             *
*                                                                      *
* 07-27-94     John T. Pearson - Add [R]edisplay option to redisplay   *
*              message from line 1 immediately                         *
*              Zap tabs in output...                                   *
*              Eliminate the line numbers (?)                          *
*----------------------------------------------------------------------*/

#define PAGE_PROMPT \
"[S]kip, [K]eep, [A]uto (keep all), [Q]uit, [B]ack up, [R]edisplay, [M]ore:"

#define PAGE_COMMANDS	"SKAQBRM "		/* note SPACE == MORE	*/

void DisplayBuffer(maxMatch)
int maxMatch;
{
  int line;

  Error = 0;

  for (I = 0; I < BuffPtr; I++) {		/* trim off trailing	*/
    chop(Buffer[I]);				/*  spaces from Buffer	*/
  }						/*  lines		*/

  /*** Display message for user to review ***/

  Answer = 'Z';
  I = 0;
  J = 0;

  while (Answer != 'Q' && Answer != 'A' && Answer != 'K' && Answer != 'S' &&
         !Automatic) {
    clear();
    mvprintw(0,0,"Extracted from file:  %s", CurrentFile);

    if (I > BuffPtr) {				/* we have rolled over	*/
      J = 0;					/*  and are to redisplay*/
      I = 0;
    }
    J = I + NumRows;				/* J is the last line to*/
    if (J > BuffPtr)				/*  display on screen	*/
      J = BuffPtr;				/*  this page		*/

    line = 1;

    for (K = I; K < J; K++) {			/* display each line	*/

      strtran(Buffer[K]," ","\t");		/* zap tabs		*/

      if(TargIdx[K]) {				/* highlight lines that	*/
        standout();				/*  have match targets	*/
        mvprintw(line,0,"%-*.*s", NumCols, NumCols, Buffer[K]);
        standend();
      }
      else
        mvprintw(line,0,"%-*.*s", NumCols, NumCols, Buffer[K]);

      if (strlen(Buffer[K]) > NumCols)		/* mark extended lines	*/
        addch('+');

      ++line;
    } /* for (K ... display each line */

    mvprintw(NumRows+1, 0, "================================================");
    mvprintw(NumRows+2, 0, "%d more lines",BuffPtr-J);
    if(maxMatch >= K) {
      standout();
      printw(" >> more matching lines later in message <<");
      standend();
    }
    mvprintw(NumRows+3, 0, PAGE_PROMPT);

    do {
      GetString(&Answer, 1);
      Answer = toupper(Answer);
    } while (!strchr(PAGE_COMMANDS,Answer));

    if(Answer == 'R') {				/* if user wants to 	*/
      I = 0;					/*  redisplay msg, do so*/
    }
    else if(Answer == 'B') {			/* user wants to back	*/
      if(I > NumRows)				/*  up one page		*/
        I -= NumRows;
      else
        I = 0;
    }
    else					/* else proceed to next	*/
      I += NumRows;				/*  page		*/

    if (Answer != 'A')
      continue;

    Automatic = TRUE;
    clear();
    addstr("Automatic processing enabled\n");
    refresh();
  }

  if (Automatic || Answer == 'K') {

        /*** Add header info to output file ***/
    if (!OutFile) {

      OutFile = fopen(OutName, "w");
      if (OutFile == NULL) {
        fprintf(stderr, "%s(%d):  cannot open output file %s\n", ProgName,
                        __LINE__, OutName);
        thread_exit(FileNotFound);
      }

      setvbuf(OutFile, TxtOutBuf, _IOFBF, sizeof(TxtBuffer));

      fprintf(OutFile,"\nOutput from %s\n\n",Greeting);
      fprintf(OutFile, "Search terms:\n");
      fprintf(OutFile, "   ");
      for (I = 0; I < NumFields-1; I++) {
        fprintf(OutFile, "\"%s\"", Target[I].Str);
        fprintf(OutFile, " %s ", Target[I].Logical);
      }
      fprintf(OutFile, "\"%s\"", Target[NumFields-1].Str);
      fprintf(OutFile,"\n%s\n", Delim);
    }

    fprintf(OutFile, "%sExtracted from file:  %s\n%s",
                      OUTFILE_DELIM,CurrentFile,OUTFILE_DELIM);

    for (I = 0; I < BuffPtr; I++)
      fprintf(OutFile, "%s\n", Buffer[I]);
    fprintf(OutFile, "%s\n", Delim);
    return;
  }

  if (Answer != 'Q') {
    clear();
    refresh();
    return;
  }

  clear();
  Error = 2;
  thread_exit(Error);
}


/*** ------------------------------------------------------------ ***/


main(argc, argv)
int argc;
char *argv[];
{
  char *TEMP;
  int  maxMatch;

  thread_init(argc, argv);

  printw("\n%s\n\n",Greeting);
  refresh();

  /*--------------------------------------------------------
  Allocate dynamic memory for message buffer:  if <MAX_BUFF>
  lines aren't available to us, take what we can get
  --------------------------------------------------------*/

  I = 0;
  while (I < MAX_BUFF) {
 
    if ((Buffer[I] = (char *)malloc(BUFFER_LEN)) == NULL) {
        fprintf(stderr,
            "%s: cannot allocate any more message buffers, limping along\n",
            ProgName);
        break;
    }

    I++;
  }

  MaxLines = I;

  /*** Prompt user for filenames ***/

  GetUserInput();

  /*----------------------------------------
  Open first file and find first delimiter
  -----------------------------------------*/

  InFile = OpenFile(CurrentFile);

  printw("\nScanning input file:  %s\n", CurrentFile);
  refresh();

  /*** Search for delimiter ***/

  *Line_ = '\0';
  Found = FALSE;
  while (!P_eof(InFile) && !Found) {
    fgets(Line_, MAX_LINE, InFile);
    if (strncmp(Line_, Delim, DelimLen) == 0)
      Found = TRUE;
  }

  if (P_eof(InFile)) {
    printw("Error!  EOF reached searching for first delimiter in the\n");
    printw("file %s\n", CurrentFile);
    refresh();
    if (InFile != NULL)
      CloseFile(InFile);
    Error = 2;
  }

  while (FilesLeft && Error == 0) {
    /*** Read next message in current file ***/

    ReadMsg(&Found, &Error);

    /*** Check for read errors ***/

    if (Error > 0)
      thread_exit(Error);

    /*** ------------ ***/

    Success = FALSE;
    for (I = 0; I < MAX_BUFF; I++)
      TargIdx[I] = FALSE;

    maxMatch = -1;

    for (I = 0; I < NumFields; I++) {
      Found = FindTarg(I,&maxMatch);
      if(I > 0) {
        if (!strcmp(Target[I-1].Logical, THREAD_AND))
          Success = (Success && Found);
        else if (!strcmp(Target[I-1].Logical, THREAD_OR))
          Success = (Success || Found);
        else if (!strcmp(Target[I-1].Logical, THREAD_NOT))
          Success = (Success && !Found);
      }
      else
        Success = Found;
    }

    /*** ------------ ***/

    if (Success) {
      DisplayBuffer(maxMatch);
      ++MatchingMsgs;
    }

  }  /*** of "while(FilesLeft) do..." ***/


  thread_exit(EX_OK);
  /*NOTREACHED*/

}  /*** of program Thread ***/


thread_exit(status)
int status;
{
  if (Error < 2) {
    clear();
    printw("\nFile search complete -- program terminated normally.\n");
    printw("%d matching messages.\n",MatchingMsgs);
  }
    
  refresh();
  endwin();

  if (OutFile != NULL)
    fclose(OutFile);
  if (InFile != NULL)
    CloseFile(InFile);

  exit(status);
}

/*
 * the following help get thread running on no DOS environments.
 */


thread_init(argc, argv)
int argc;
char *argv[];
{
  register char *p;
  register int c;
  int errflg = 0;
  long maxFiles = sizeof(FileList)/sizeof(FileList[0]);
  extern int getopt(), opterr, optind;
  extern char *optarg;

  OutFile        = NULL;
  InFile         = NULL;
  CurrentFile    = NULL;
  InFile_NAME[0] = '\0';
  OutName[0]     = '\0';
  Error          = 0;
  FilesLeft      = TRUE;
  Automatic      = FALSE;

  strcpy(Delim, "------------");

  if ((p = strrchr(argv[0], '/')))
    p++;
  else
    p = argv[0];

  ProgName = p;

  while ((c = getopt (argc, argv, "ar:c:o:d:p:")) != EOF) {
    switch(c) {
      case 'a':
        Automatic = TRUE;
        break;
      case 'o':
        strcpy(OutName, optarg);	/* hopefully optart wont be too big */
        break;
      case 'r':
        NumRows = atoi(optarg);
        break;
      case 'c':
        NumCols = atoi(optarg);
        break;
      case 'd':
        strcpy(Delim, optarg);
        break;
      case 'p':
        strcpy(Pattern,optarg);
        break;
      case '?':
        errflg++;
        break;
    }
  }

  if (errflg) {
    fprintf(stderr, "Usage:\n %s\n", USAGE);
    exit(1);
  }

  for (; optind < argc; optind++) {
    if (optind > maxFiles) {
      fprintf(stderr, "%s: Too many files specified, recompile %s.c\n",
                                        ProgName, ProgName);
      exit(1);
    }
    strcpy(FileList[NoFiles].Name, argv[optind]);
    NoFiles++;
  }

  if(NoFiles > 0) {
    CurrentFile = FileList[0].Name;
    FileCnt = 1;
  }

  sprintf(Greeting,"%s v%s by Tom Kaltenbach & John Pearson -- Modified %s",
          ProgName, Ver, LastMod);

  DelimLen = strlen(Delim);
  Win = initscr();
  raw();
  noecho();
  EC = erasechar();
  KC = killchar();

#ifdef getmaxyx
  getmaxyx(Win, NumRows, NumCols);
#endif

  NumRows -= 4;   /* account for headers and footers */
  NumCols -= 1;   /* account for margins */
}


FindFirst(s)
char *s;
{
  int stat;

  FileCnt = 1;
  strcpy(FileList[0].Name, s);
  CurrentFile = FileList[0].Name;

  if((stat = access(CurrentFile,0)) != 0)
    printw("Cannot access file '%s'\n",CurrentFile);

  return stat;
}

FindNext()
{
  if (FileCnt < NoFiles) {
    CurrentFile = FileList[FileCnt].Name;
    FileCnt++;
    return 0;
  }
  return 1;
}

/* End. */




/* this is mine :-), efl */
/*
** 07-29-94     John T. Pearson - Made one char inputs not require a <RETURN>,
**              use curses highlighting to highlight input buffer length.
**              Fix bug by NULL terminating regardless of there the <RETURN>
**              happened.
*/

GetString(cmdstr, size)
char *cmdstr;       /* pointer to a buffer to fill */
unsigned int size;  /* number of characters allowed in buffer */
{
  char *p;
  int c;

  refresh();
  p = cmdstr;

  if(size == 1) {				/* single stroke inputs	*/
    *p++ = getch();				/*  require no <RETURN>	*/
  }
  else {					/* otherwise, it's a	*/
    while ((c = getch()) != '\n') {		/*  string input up to	*/
      if (c == '\r')				/*  <RETURN>		*/
        break;

      if (c == EC) {  		/* erase character like ^H */
        if (p > cmdstr) {
          addstr("\b \b");
          p--;
        }
      }

      else if (c == KC)   		/* kill character like ^U */
        while (p > cmdstr) {
          addstr("\b \b");
          p--;
        }
      else if (c == '\027') { 	 /* erase word character ^W */
        while (p > cmdstr) {
          addstr("\b \b");
          p--;
          if (*p == ' ')
            break;
        }
      }
      else if ((p < cmdstr + size)) {
        addch(c);
        *p++ = c;
      }

      refresh();
    }
  }

  addch('\n');
  refresh();

  *p = '\0';
}

/*
 * The following functions were swipped from the Pascal to C (p2c)
 * library.  I hope this is OK.
 *
 * The code is from FSF's p2c.  Below is the copyleft found in the file.
 * Hopefully this covers the copyright stuff.
 */
 
 
/* Run-time library for use with "p2c", the Pascal to C translator */
 
/* "p2c"  Copyright (C) 1989, 1990, 1991 Free Software Foundation.
 * By Dave Gillespie, daveg@csvax.cs.caltech.edu.  Version --VERSION--.
 * This file may be copied, modified, etc. in any way.  It is not restricted
 * by the licence agreement accompanying p2c itself.
 */
 
 
/* Check if at end of file, using Pascal "eof" semantics.  End-of-file for
   stdin is broken; remove the special case for it to be broken in a
   different way. */
 
int P_eof(f)
FILE *f;
{
  register int ch;
 
  if (feof(f))
    return 1;
  if (f == stdin)
    return 0;   /* not safe to look-ahead on the keyboard! */
  ch = getc(f);
  if (ch == EOF)
    return 1;
  ungetc(ch, f);
  return 0;
}
 

/**/

/* module: ParsePattern()

	Purpose:Parse user input pattern matching string into internal
		search codes.

	Syntax:	ParsePattern(pattern,target,i)
		char      *pattern;
		TargetRec target[];
		short     *i;

	John Pearson
*/

ParsePattern(pattern,target,i)
char      *pattern;
TargetRec target[];
short     *i;
{
  char   *p;					/* tokens in string	*/
  short  targetCount = *i;			/* # targets seen	*/
  char   done = FALSE;				/* see END, E or . ???	*/
  static logical = 0;				/* expecting logical op?*/
						/*  1st token => target	*/
  p = strtok(pattern," ");			/* pick off first token	*/

  if(p == NULL && pattern != NULL)		/* only one, it's token	*/
    p = pattern;

  while(!done && p != NULL) {
    if(logical) {				/* we're expecting a	*/
      switch(*p) {				/*  logical operator	*/
        case 'A':
        case 'a':
        case '&':
        case '+':
          strcpy(target[targetCount].Logical,THREAD_AND);
          break;
        case 'O':
        case 'o':
        case '|':
          strcpy(target[targetCount].Logical,THREAD_OR);
          break;
        case 'N':
        case 'n':
        case '!':
        case '~':
          strcpy(target[targetCount].Logical,THREAD_NOT);
          break;
        case 'E':				/* END...put in OR	*/
        case 'e':				/*  for logical op as	*/
        case '.':				/*  a safety		*/
          strcpy(target[targetCount].Logical,THREAD_OR);
          done = TRUE;
          break;
      }
      logical = FALSE;				/* next token should be	*/
      ++targetCount;				/*  another target	*/
    }
    else {					/* expecting target	*/
      strtran(p," ","_");			/* transliterate	*/
      strcpy(target[targetCount].Str,p);	/*  underscore to space	*/
      logical = TRUE;				/*  and copy target in	*/
    }						/* next should be logic	*/

    p = strtok(NULL," ");			/* pick apart next piece*/
  } /* while not done */

  *i = targetCount;				/* update # targets	*/
						/*  seen		*/
  return done;
} /* end ParseInput() */

/**/

/* module: OpenFile()

	Purpose:Opens the current digest file for processing.

	Syntax:	OpenFile(CurrentFile)
		char *CurrentFile;

	Return:	FILE * for open file.

	Notes:	Checks for compressed digest files if compiled into thread,
		and expands them appropriately.

	John Pearson
*/


FILE *OpenFile(CurrentFile)
char *CurrentFile;
{
  char compFile[_FNSIZE];			/* compressed file name	*/
  char compCmd[_FNSIZE];			/* compress command line*/
  char *p;

#if defined(COMPRESS)
  compressed = 0;				/* reset compression	*/
						/*  flag for next file	*/
  if(access(CurrentFile,0) != 0) {		/* if filename given	*/
    strcpy(compFile,CurrentFile);		/*  doesn't exist, check*/
    strcat(compFile,COMP_STRING);		/*  for a compressed	*/
    if(access(compFile,0) == 0)			/*  version.		*/
      compressed = 1;
  }						/* else if filename	*/
  else if((p = strstr(CurrentFile,COMP_STRING)) != NULL) {
    strcpy(compFile,CurrentFile);		/*  given contains the	*/
    *p = '\0';					/*  compression string	*/
    compressed = 1;				/*  assume it is comp'd	*/
  }

  if(compressed) {				/* if compressed,	*/
    strcpy(compCmd,DECOMP_PROG);		/*  uncompress it first	*/
    strcat(compCmd,compFile);
    strcat(compCmd," > ");
    strcat(compCmd,CurrentFile);
    printw("Decompressing %s...\n",compFile);
    refresh();
    system(compCmd);
  }
#endif /* defined(COMPRESS) */

  strcpy(InFile_NAME, CurrentFile);
    
  InFile = fopen(InFile_NAME, "r");

  if (InFile == NULL) {
    fprintf(stderr, "%s(%d):  cannot open input file '%s'\n",
                     ProgName, __LINE__, InFile_NAME);
    thread_exit(FileNotFound);
  }

  setvbuf(InFile, TxtInBuf, _IOFBF, sizeof(TxtBuffer));

  return InFile;
}

/**/

/* module:CloseFile()

	Purpose:Close current input file, handling previously uncompressed
		files if necessary.

	Syntax:	CloseFile(InFile)
		FILE *InFile;

	John Pearson
*/

void CloseFile(InFile)
FILE *InFile;
{
  fclose(InFile);
  InFile = NULL;

  if(compressed && *CurrentFile != '\0') {	/* previous file	*/
    remove(CurrentFile);			/*  was decompressed?	*/
  }

  return;
}
