/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_get_statisticscounter.c
 * @brief command to test GET /statistics-counter/$SLUG
 * @author Martin Schanzenbach
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State of a "GET statistics-counter" CMD.
 */
struct GetStatisticsCounterState
{

  /**
   * Handle for a "GET statistics-counter" request.
   */
  struct TALER_MERCHANT_StatisticsCounterGetHandle *scgh;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Base URL of the merchant serving the request.
   */
  const char *merchant_url;

  /**
   * Slug of the statistic to get.
   */
  const char *slug;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

  /**
   * Expected bucket size.
   */
  uint64_t buckets_length;

  /**
   * Expected intervals size.
   */
  uint64_t intervals_length;

};


/**
 * Callback for a GET /statistics-counter operation.
 *
 * @param cls closure for this function
 * @param scgr response details
 */
static void
get_statisticscounter_cb (void *cls,
                          const struct
                          TALER_MERCHANT_StatisticsCounterGetResponse *scgr)
{
  struct GetStatisticsCounterState *scs = cls;
  const struct TALER_MERCHANT_HttpResponse *hr = &scgr->hr;

  scs->scgh = NULL;
  if (scs->http_status != hr->http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                hr->http_status,
                (int) hr->ec,
                TALER_TESTING_interpreter_get_current_label (scs->is));
    TALER_TESTING_interpreter_fail (scs->is);
    return;
  }
  switch (hr->http_status)
  {
  case MHD_HTTP_OK:
    {
      if (scgr->details.ok.buckets_length != scs->buckets_length)
      {
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Length of buckets found does not match (Got %llu, expected %llu)\n",
                    (unsigned long long) scgr->details.ok.buckets_length,
                    (unsigned long long) scs->buckets_length);
        TALER_TESTING_interpreter_fail (scs->is);
        return;
      }
      if (scgr->details.ok.intervals_length != scs->intervals_length)
      {
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Length of intervals found does not match (Got %llu, expected %llu)\n",
                    (unsigned long long) scgr->details.ok.intervals_length,
                    (unsigned long long) scs->intervals_length);
        TALER_TESTING_interpreter_fail (scs->is);
        return;
      }
    }
    break;
  case MHD_HTTP_UNAUTHORIZED:
    break;
  case MHD_HTTP_NOT_FOUND:
    /* instance does not exist */
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unhandled HTTP status %u (%d).\n",
                hr->http_status,
                hr->ec);
  }
  TALER_TESTING_interpreter_next (scs->is);
}


/**
 * Run the "GET /products" CMD.
 *
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
get_statisticscounter_run (void *cls,
                           const struct TALER_TESTING_Command *cmd,
                           struct TALER_TESTING_Interpreter *is)
{
  struct GetStatisticsCounterState *scs = cls;

  scs->is = is;
  scs->scgh = TALER_MERCHANT_statistic_counter_get (
    TALER_TESTING_interpreter_get_context (is),
    scs->merchant_url,
    scs->slug,
    TALER_MERCHANT_STATISTICS_ALL,
    &get_statisticscounter_cb,
    scs);
  GNUNET_assert (NULL != scs->scgh);
}


/**
 * Free the state of a "GET statistics-counter" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
get_statisticscounter_cleanup (void *cls,
                               const struct TALER_TESTING_Command *cmd)
{
  struct GetStatisticsCounterState *scs = cls;

  if (NULL != scs->scgh)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "GET /statistics-counter operation did not complete\n");
    TALER_MERCHANT_statistic_counter_get_cancel (scs->scgh);
  }
  GNUNET_free (scs);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_get_statisticscounter (const char *label,
                                                  const char *merchant_url,
                                                  const char *slug,
                                                  uint64_t
                                                  expected_buckets_length,
                                                  uint64_t
                                                  expected_intervals_length,
                                                  unsigned int http_status)
{
  struct GetStatisticsCounterState *scs;

  scs = GNUNET_new (struct GetStatisticsCounterState);
  scs->merchant_url = merchant_url;
  scs->slug = slug;
  scs->buckets_length = expected_buckets_length;
  scs->intervals_length = expected_intervals_length;
  scs->http_status = http_status;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = scs,
      .label = label,
      .run = &get_statisticscounter_run,
      .cleanup = &get_statisticscounter_cleanup
    };

    return cmd;
  }
}


/* end of testing_api_cmd_get_statisticscounter.c */
