--
-- This file is part of TALER
-- Copyright (C) 2024 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_legitimization_measures(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(legitimization_measure_serial_id INT8 GENERATED BY DEFAULT AS IDENTITY'
      ',access_token BYTEA NOT NULL CHECK (LENGTH(access_token)=32)'
      ',start_time INT8 NOT NULL'
      ',jmeasures TEXT NOT NULL'
      ',display_priority INT4 NOT NULL' -- DEAD?
      ',is_finished BOOL NOT NULL DEFAULT(FALSE)'
    ') %s ;'
    ,'legitimization_measures'
    ,'PARTITION BY HASH (access_token)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'List of required legitimizations by account'
    ,'legitimization_measures'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'unique ID for this legitimization process at the exchange'
    ,'legitimization_measure_serial_id'
    ,'legitimization_measures'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'foreign key linking the entry to the kyc_targets table, NOT a primary key (multiple legitimizations are possible per account)'
    ,'access_token'
    ,'legitimization_measures'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the measure was triggered (by decision or rule)'
    ,'start_time'
    ,'legitimization_measures'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'JSON object of type LegitimizationMeasures with KYC/AML measures for the account encoded'
    ,'jmeasures'
    ,'legitimization_measures'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Display priority of the rule that triggered this measure; if in the meantime another rule also triggers, the measure is only replaced if the new rule has a higher display priority; probably not really useful, as right now there is only ever one set of legitimization_measures active at any time, might be removed in the future'
    ,'display_priority'
    ,'legitimization_measures'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Set to TRUE if this set of measures was processed; used to avoid indexing measures that are done'
    ,'is_finished'
    ,'legitimization_measures'
    ,partition_suffix
  );
END
$$;



CREATE FUNCTION constrain_table_legitimization_measures(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'legitimization_measures';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);

  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_serial_id_key'
    ' UNIQUE (legitimization_measure_serial_id)');
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_access_token'
    ' ON ' || table_name ||
    ' (access_token)'
    ' WHERE NOT is_finished' ||
    ';'
  );
END
$$;


CREATE FUNCTION foreign_table_legitimization_measures()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'legitimization_measures';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_key_access_token'
    ' FOREIGN KEY (access_token)'
    ' REFERENCES wire_targets (access_token)'
    '  ON DELETE CASCADE');
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('legitimization_measures'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('legitimization_measures'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('legitimization_measures'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
