/*  Motti -- a strategy game
    Copyright (C) 1999 Free Software Foundation

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
#include <config.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#ifdef HAVE_LIBPTHREAD
#include <pthread.h>
#endif

#include "thread.h"
#include "xinit.h"
#include "xwin.h"
#include "map.h"
#include "occupy.h"

static void set_input_state (struct x_info_struct *, const int);
static int check_uses_def_mode (struct x_info_struct *);
static void resolve_map_mouse (struct x_info_struct *, const XEvent);
static void disable_inputs (struct x_info_struct *);

static void
set_input_state (x_info, uses_def_mode)
     struct x_info_struct *x_info;
     const int uses_def_mode;
{
#ifdef HAVE_LIBPTHREAD
  if (x_info->active)
    {
#endif
      XSelectInput (x_info->display, x_info->but_win[ATT], ButtonPressMask);
      XSelectInput (x_info->display, x_info->but_win[DEF], ButtonPressMask);
      XSelectInput (x_info->display, x_info->but_win[GUE], ButtonPressMask);
      XSelectInput (x_info->display, x_info->map_win, ExposureMask |
		    ButtonPressMask);
      if (uses_def_mode)
	XSetWindowBorderWidth (x_info->display, x_info->but_win[DEF],
			       x_info->db_tab.sel_width);
#ifdef HAVE_LIBPTHREAD
    }
  else /* Passive, just display events.  */
    {
      XSelectInput (x_info->display, x_info->but_win[ATT], 0);
      XSelectInput (x_info->display, x_info->but_win[DEF], 0);
      XSelectInput (x_info->display, x_info->but_win[GUE], 0);
      XSelectInput (x_info->display, x_info->map_win, ExposureMask);
      XSetWindowBackgroundPixmap (x_info->display, x_info->but_win[ATT],
				  x_info->d_att_pix[6]);
      XSetWindowBackgroundPixmap (x_info->display, x_info->but_win[DEF],
				  x_info->d_def_pix);
      XSetWindowBackgroundPixmap (x_info->display, x_info->but_win[GUE],
				  x_info->d_gue_pix);
      if (uses_def_mode)
	{
	  XSetWindowBorderWidth (x_info->display, x_info->but_win[ATT], 1);
	  XSetWindowBorderWidth (x_info->display, x_info->but_win[DEF], 1);
	  XSetWindowBorderWidth (x_info->display, x_info->but_win[GUE], 1);
	}

      XClearWindow (x_info->display, x_info->but_win[DEF]);
      XClearWindow (x_info->display, x_info->but_win[GUE]);
    }
#endif	/* HAVE_LIBPTHREAD */
}

static int
check_uses_def_mode (x_info)
     struct x_info_struct *x_info;
{
  register int i;
  for (i = 0; i < 5; i++)
    if (x_info->db_tab.mouse[i] == act)
      return 1;
  return 0;
}

static void
resolve_map_mouse (x_info, report)
     struct x_info_struct *x_info;
     const XEvent report;
{
  enum mouse_action button_action;
  /* Unfortunately we can't be sure of the internal representation of
     button field, therefore this is unnecessarily ugly.  */
  switch (report.xbutton.button)
    {
    case Button1:
      button_action = x_info->db_tab.mouse[0];
      break;
    case Button2:
      button_action = x_info->db_tab.mouse[1];
      break;
    case Button3:
      button_action = x_info->db_tab.mouse[2];
      break;
    case Button4:
      button_action = x_info->db_tab.mouse[3];
      break;
    case Button5:
      button_action = x_info->db_tab.mouse[4];
      break;
    }
  switch (button_action)
    {
      Action map_events;
      Coord coord;
    case mark:
      coord = win_pos2map_coord (x_info, report.xbutton.x, report.xbutton.y);
      if (toggle_cross (coord))
	{
#ifdef HAVE_LIBPTHREAD
	  expose_map (x_info->conn_num, coord.x, coord.y, 0, 0);
	  expose_turnwin (x_info->conn_num);
#endif
	  draw_map (x_info, report.xbutton.x, report.xbutton.y, 0, 0);
	  draw_turn_win (x_info);
	}
      break;
    case sel:
      select_next ();
      break;
    case act:
      action (game_map.sel_mode, &map_events);
      draw_effects (x_info, map_events);
      break;
    case last_place:
      /* Yet unimplemented.  */
    case attack:
      action (MODE_ATT, &map_events);
      draw_effects (x_info, map_events);
      break;
    case defend:
      action (MODE_DEF, &map_events);
      draw_effects (x_info, map_events);
      break;
    case guerilla:
      action (MODE_GUE, &map_events);
      draw_effects (x_info, map_events);
      break;
    case reset:
      clear_crosses ();
      map_events.type = EVENT_REFRESH;
      map_events.loc = NULL;
      map_events.count = 0;
      draw_effects (x_info, map_events);
      break;
      /* Smart action: do guerilla if possible, then defend and then attack.
       */
    case smart:
      if (game_map.modes & MODE_GUE)
	action (MODE_GUE, &map_events);
      else if (game_map.modes & MODE_DEF)
	action (MODE_DEF, &map_events);
      else
	action (MODE_ATT, &map_events);
      draw_effects (x_info, map_events);
      break;
      /* Half smart: do guerilla if possible, then defend.  Better for trigger
	 happy people with 3-button mouse than smart.  */
    case halfsmart:
      if (game_map.modes & MODE_GUE)
	action (MODE_GUE, &map_events);
      else if (game_map.modes & MODE_DEF)
	action (MODE_DEF, &map_events);
      else
	break;
      draw_effects (x_info, map_events);
      break;
    }
}

static void
disable_inputs (x_info)
     struct x_info_struct *x_info;
{
  create_end_win (x_info);
  XSelectInput (x_info->display, x_info->but_win[ATT], 0);
  XSelectInput (x_info->display, x_info->but_win[DEF], 0);
  XSelectInput (x_info->display, x_info->but_win[GUE], 0);
  XSelectInput (x_info->display, x_info->turn_win, ExposureMask);
  XSelectInput (x_info->display, x_info->map_win, ExposureMask);
  XSelectInput (x_info->display, x_info->endgame_win, ExposureMask |
		ButtonReleaseMask);
  XMapWindow (x_info->display, x_info->endgame_win);
}

extern void
event_loop (x_info)
     struct x_info_struct *x_info;
{
  char last_modes = 0;	/* Used to update buttons.  */
  int uses_def_mode;
#ifdef HAVE_LIBPTHREAD
  int last_active;
#endif
  uses_def_mode = check_uses_def_mode (x_info);

  x_info->last_sel = MODE_ATT;
#ifdef HAVE_LIBPTHREAD
  x_info->active = 1<<(game_map.turn-1) &
    connect_mask[x_info->conn_num] ? 1 : 0;
  last_active = x_info->active;
#endif
  XSelectInput (x_info->display, x_info->main_win, 0);
  XSelectInput (x_info->display, x_info->turn_win, ExposureMask);

  XMapWindow (x_info->display, x_info->map_win);
  XMapWindow (x_info->display, x_info->but_win[ATT]);
  XMapWindow (x_info->display, x_info->but_win[DEF]);
  XMapWindow (x_info->display, x_info->but_win[GUE]);
  XMapWindow (x_info->display, x_info->turn_win);
  XMapWindow (x_info->display, x_info->main_win);

  set_input_state (x_info, uses_def_mode);
#ifdef HAVE_LIBPTHREAD
  if (x_info->active)
#endif
    update_buttons (x_info, uses_def_mode, &last_modes);
#ifdef HAVE_LIBPTHREAD
  pthread_mutex_unlock (&(x_info->init_lock));
  release_read_lock ();
#endif

  while (1)
    {
      XEvent report;

#ifdef HAVE_LIBPTHREAD
      get_read_lock ();
      x_info->active = 1<<(game_map.turn-1) & connect_mask[x_info->conn_num] ?
	1 : 0;
      release_read_lock ();
      if (last_active != x_info->active)
	{
	  set_input_state (x_info, uses_def_mode);
	  last_active = x_info->active;
	  if (x_info->active)
	    {
	      last_modes = 0;	/* Draw all windows, since they were all
				   previously disabled.  */
	      update_buttons (x_info, uses_def_mode, &last_modes);
	    }
	}
#endif

      XNextEvent (x_info->display, &report);

      switch (report.type)
	{
	  Action map_events;

	case Expose:
	  if (report.xexpose.window == x_info->map_win)
	    {
#ifdef HAVE_LIBPTHREAD
	      get_read_lock ();
#endif
	      draw_map (x_info, report.xexpose.x, report.xexpose.y,
			report.xexpose.width, report.xexpose.height);
#ifdef HAVE_LIBPTHREAD
	      release_read_lock ();
#endif
	    }
	  else if (report.xexpose.window == x_info->turn_win)
	    {
#ifdef HAVE_LIBPTHREAD
	      get_read_lock ();
#endif
	      draw_turn_win (x_info);
#ifdef HAVE_LIBPTHREAD
	      release_read_lock ();
#endif
	    }
	  else if (report.xexpose.window == x_info->endgame_win)
	    draw_end_win (x_info);
	  break;

	  /* No read locks here, since these can happen only to the active
	     player.  Only the active player wants write locks.  */
	case ButtonPress:
	  if (!report.xbutton.same_screen) break;
	  if (report.xbutton.window == x_info->but_win[ATT])
	    {
	      action (MODE_ATT, &map_events);
	      draw_effects (x_info, map_events);
	      if (game_map.players_left == 1)
		{
#ifdef HAVE_LIBPTHREAD
		  signal_end_game (x_info->conn_num);
#endif
		  disable_inputs (x_info);
		}
	    }
	  else if (report.xbutton.window == x_info->but_win[DEF])
	    {
	      action (MODE_DEF, &map_events);
	      draw_effects (x_info, map_events);
	      /* Nobody can win by defending.  In real life, nobody can
		 win by attacking, either.  */
	    }
	  else if (report.xbutton.window == x_info->but_win[GUE])
	    {
	      action (MODE_GUE, &map_events);
	      draw_effects (x_info, map_events);
	      if (game_map.players_left == 1)
		{
#ifdef HAVE_LIBPTHREAD
		  signal_end_game (x_info->conn_num);
#endif
		  disable_inputs (x_info);
		}
	    }
	  else if (report.xbutton.window == x_info->map_win)
	    {
	      resolve_map_mouse (x_info, report);
	      if (game_map.players_left == 1)
		{
#ifdef HAVE_LIBPTHREAD
		  /* It's possible to attack by clicking on the map window
		     too.  */
		  signal_end_game (x_info->conn_num);
#endif
		  disable_inputs (x_info);
		}
	    }

	  update_buttons (x_info, uses_def_mode, &last_modes);
	  break;
	case ButtonRelease:	/* This happens only from endgame window.  */
	  XCloseDisplay (x_info->display);
#ifdef HAVE_LIBPTHREAD
	  XCloseDisplay (x_info->common_display);
	  pthread_exit (NULL);
#else
	  exit (0);
#endif
#ifdef HAVE_LIBPTHREAD
	case ClientMessage:
	  switch (report.xclient.data.b[0])
	    {
	    case 0:
	      get_read_lock ();
	      XSetWindowBackgroundPixmap
		(x_info->display, x_info->but_win[ATT],
		 x_info->d_att_pix[(game_map.modes & MODE_ATT)
				  ? 6 - game_map.n_att : 0]);
	      release_read_lock ();
	      XClearWindow (x_info->display, x_info->but_win[ATT]);
	      break;
	    case 1:
	      disable_inputs (x_info);
	      break;
	    }
	  break;
#endif
	}
    }
}
