// 2007-02-04  Edward Smith-Rowland <3dw4rd@verizon.net>
//
// Copyright (C) 2007, 2009 Free Software Foundation, Inc.
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3, or (at your option)
// any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING3.  If not see
// <http://www.gnu.org/licenses/>.

//  ellint_3


//  Compare against values generated by the GNU Scientific Library.
//  The GSL can be found on the web: http://www.gnu.org/software/gsl/

#include <tr1/cmath>
#if defined(__TEST_DEBUG)
#include <iostream>
#define VERIFY(A) \
if (!(A)) \
  { \
    std::cout << "line " << __LINE__ \
      << "  max_abs_frac = " << max_abs_frac \
      << std::endl; \
  }
#else
#include <testsuite_hooks.h>
#endif
#include "../testcase.h"


// Test data for k=-0.90000000000000002, nu=0.0000000000000000.
testcase_ellint_3<double> data001[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17525427376115027, -0.90000000000000002, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35492464591297446, -0.90000000000000002, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.54388221416157134, -0.90000000000000002, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.74797400423532523, -0.90000000000000002, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.97463898451966458, -0.90000000000000002, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.2334463254523440, -0.90000000000000002, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.5355247765594910, -0.90000000000000002, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.8882928567775124, -0.90000000000000002, 0.0000000000000000, 
          1.3962634015954636 },
  { 2.2805491384227703, -0.90000000000000002, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.0000000000000000.
template <typename Tp>
void test001()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data001)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data001[i].k), Tp(data001[i].nu),
                   Tp(data001[i].phi));
      const Tp f0 = data001[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.10000000000000001.
testcase_ellint_3<double> data002[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17507714233254659, -0.90000000000000002, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35350932904326521, -0.90000000000000002, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.53911129989870998, -0.90000000000000002, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.73666644254508429, -0.90000000000000002, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.95250736612100184, -0.90000000000000002, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.1950199550905594, -0.90000000000000002, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.4741687286340848, -0.90000000000000002, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.7968678183506059, -0.90000000000000002, 0.10000000000000001, 
          1.3962634015954636 },
  { 2.1537868513875287, -0.90000000000000002, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.10000000000000001.
template <typename Tp>
void test002()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data002)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data002[i].k), Tp(data002[i].nu),
                   Tp(data002[i].phi));
      const Tp f0 = data002[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.20000000000000001.
testcase_ellint_3<double> data003[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17490065089140930, -0.90000000000000002, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.35211377590661436, -0.90000000000000002, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.53448220334204122, -0.90000000000000002, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.72591368943179613, -0.90000000000000002, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.93192539780038763, -0.90000000000000002, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.1600809679692683, -0.90000000000000002, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.4195407225882508, -0.90000000000000002, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.7168966476424528, -0.90000000000000002, 0.20000000000000001, 
          1.3962634015954636 },
  { 2.0443194576468890, -0.90000000000000002, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.20000000000000001.
template <typename Tp>
void test003()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data003)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data003[i].k), Tp(data003[i].nu),
                   Tp(data003[i].phi));
      const Tp f0 = data003[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.29999999999999999.
testcase_ellint_3<double> data004[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17472479532647534, -0.90000000000000002, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.35073750187374114, -0.90000000000000002, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.52998766129466979, -0.90000000000000002, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.71566993548699587, -0.90000000000000002, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.91271517762560195, -0.90000000000000002, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.1281241199843370, -0.90000000000000002, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.3704929576917448, -0.90000000000000002, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.6461981511487715, -0.90000000000000002, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.9486280260314426, -0.90000000000000002, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.29999999999999999.
template <typename Tp>
void test004()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data004)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data004[i].k), Tp(data004[i].nu),
                   Tp(data004[i].phi));
      const Tp f0 = data004[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.40000000000000002.
testcase_ellint_3<double> data005[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17454957156468839, -0.90000000000000002, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34938003933330430, -0.90000000000000002, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.52562093533067455, -0.90000000000000002, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.70589461324915703, -0.90000000000000002, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.89472658511942849, -0.90000000000000002, 0.40000000000000002, 
          0.87266462599716477 },
  { 1.0987419542323440, -0.90000000000000002, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.3261349565496301, -0.90000000000000002, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.5831293909853767, -0.90000000000000002, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.8641114227238349, -0.90000000000000002, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.40000000000000002.
template <typename Tp>
void test005()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data005)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data005[i].k), Tp(data005[i].nu),
                   Tp(data005[i].phi));
      const Tp f0 = data005[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.50000000000000000.
testcase_ellint_3<double> data006[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17437497557073336, -0.90000000000000002, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34804093691586013, -0.90000000000000002, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.52137576320372914, -0.90000000000000002, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.69655163996912284, -0.90000000000000002, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.87783188683054236, -0.90000000000000002, 0.50000000000000000, 
          0.87266462599716477 },
  { 1.0716015959755185, -0.90000000000000002, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.2857636916026747, -0.90000000000000002, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.5264263913252365, -0.90000000000000002, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.7888013241937861, -0.90000000000000002, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.50000000000000000.
template <typename Tp>
void test006()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data006)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data006[i].k), Tp(data006[i].nu),
                   Tp(data006[i].phi));
      const Tp f0 = data006[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.59999999999999998.
testcase_ellint_3<double> data007[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17420100334657815, -0.90000000000000002, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34671975876122157, -0.90000000000000002, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.51724631570707968, -0.90000000000000002, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.68760879113743056, -0.90000000000000002, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.86192157779698364, -0.90000000000000002, 0.59999999999999998, 
          0.87266462599716477 },
  { 1.0464279696166354, -0.90000000000000002, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.2488156247094004, -0.90000000000000002, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.4750988777188474, -0.90000000000000002, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.7211781128919523, -0.90000000000000002, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.59999999999999998.
template <typename Tp>
void test007()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data007)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data007[i].k), Tp(data007[i].nu),
                   Tp(data007[i].phi));
      const Tp f0 = data007[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.69999999999999996.
testcase_ellint_3<double> data008[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17402765093102210, -0.90000000000000002, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34541608382635131, -0.90000000000000002, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.51322715827061705, -0.90000000000000002, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.67903717872440306, -0.90000000000000002, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.84690113601682671, -0.90000000000000002, 0.69999999999999996, 
          0.87266462599716477 },
  { 1.0229914311548418, -0.90000000000000002, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.2148329639709381, -0.90000000000000002, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.4283586501307806, -0.90000000000000002, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.6600480747670938, -0.90000000000000002, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.69999999999999996.
template <typename Tp>
void test008()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data008)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data008[i].k), Tp(data008[i].nu),
                   Tp(data008[i].phi));
      const Tp f0 = data008[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.80000000000000004.
testcase_ellint_3<double> data009[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17385491439925149, -0.90000000000000002, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34412950523113928, -0.90000000000000002, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.50931321668729612, -0.90000000000000002, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.67081081392296349, -0.90000000000000002, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.83268846097293259, -0.90000000000000002, 0.80000000000000004, 
          0.87266462599716477 },
  { 1.0010985015814027, -0.90000000000000002, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.1834394045489678, -0.90000000000000002, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.3855695891683188, -0.90000000000000002, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.6044591960982204, -0.90000000000000002, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.80000000000000004.
template <typename Tp>
void test009()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data009)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data009[i].k), Tp(data009[i].nu),
                   Tp(data009[i].phi));
      const Tp f0 = data009[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.90000000000000002, nu=0.90000000000000002.
testcase_ellint_3<double> data010[] = {
  { -0.0000000000000000, -0.90000000000000002, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17368278986240138, -0.90000000000000002, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.34285962963961397, -0.90000000000000002, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.50549974644993323, -0.90000000000000002, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.66290623857720898, -0.90000000000000002, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.81921183128847164, -0.90000000000000002, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.98058481956066390, -0.90000000000000002, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.1543223520473567, -0.90000000000000002, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.3462119782292938, -0.90000000000000002, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.5536420236310946, -0.90000000000000002, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.90000000000000002, nu=0.90000000000000002.
template <typename Tp>
void test010()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data010)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data010[i].k), Tp(data010[i].nu),
                   Tp(data010[i].phi));
      const Tp f0 = data010[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.0000000000000000.
testcase_ellint_3<double> data011[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17510154241338902, -0.80000000000000004, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35365068839779390, -0.80000000000000004, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.53926804409084561, -0.80000000000000004, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.73587926028070383, -0.80000000000000004, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.94770942970071170, -0.80000000000000004, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.1789022995388239, -0.80000000000000004, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.4323027881876009, -0.80000000000000004, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.7069629739121674, -0.80000000000000004, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.9953027776647296, -0.80000000000000004, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.0000000000000000.
template <typename Tp>
void test011()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data011)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data011[i].k), Tp(data011[i].nu),
                   Tp(data011[i].phi));
      const Tp f0 = data011[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.10000000000000001.
testcase_ellint_3<double> data012[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17492468824017166, -0.80000000000000004, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35224443521476911, -0.80000000000000004, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.53456851853226961, -0.80000000000000004, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.72488875602364944, -0.80000000000000004, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.92661354274638952, -0.80000000000000004, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.1432651144499077, -0.80000000000000004, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.3774479927211429, -0.80000000000000004, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.6287092337196041, -0.80000000000000004, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.8910755418379521, -0.80000000000000004, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.10000000000000001.
template <typename Tp>
void test012()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data012)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data012[i].k), Tp(data012[i].nu),
                   Tp(data012[i].phi));
      const Tp f0 = data012[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.20000000000000001.
testcase_ellint_3<double> data013[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17474847286224943, -0.80000000000000004, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.35085779529084682, -0.80000000000000004, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.53000829263059157, -0.80000000000000004, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.71443466027453406, -0.80000000000000004, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.90698196872715420, -0.80000000000000004, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.1108198200558581, -0.80000000000000004, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.3284988909963957, -0.80000000000000004, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.5600369318140328, -0.80000000000000004, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.8007226661734588, -0.80000000000000004, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.20000000000000001.
template <typename Tp>
void test013()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data013)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data013[i].k), Tp(data013[i].nu),
                   Tp(data013[i].phi));
      const Tp f0 = data013[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.29999999999999999.
testcase_ellint_3<double> data014[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17457289217669891, -0.80000000000000004, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34949028801501258, -0.80000000000000004, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.52558024362769318, -0.80000000000000004, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.70447281740094914, -0.80000000000000004, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.88864745641528986, -0.80000000000000004, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0811075819341465, -0.80000000000000004, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.2844589654082377, -0.80000000000000004, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.4991461361277849, -0.80000000000000004, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.7214611048717301, -0.80000000000000004, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.29999999999999999.
template <typename Tp>
void test014()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data014)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data014[i].k), Tp(data014[i].nu),
                   Tp(data014[i].phi));
      const Tp f0 = data014[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.40000000000000002.
testcase_ellint_3<double> data015[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17439794211872178, -0.80000000000000004, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34814144964568972, -0.80000000000000004, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.52127776285273075, -0.80000000000000004, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.69496411438966599, -0.80000000000000004, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.87146878427509589, -0.80000000000000004, 0.40000000000000002, 
          0.87266462599716477 },
  { 1.0537579024937762, -0.80000000000000004, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.2445534387922637, -0.80000000000000004, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.4446769766361993, -0.80000000000000004, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.6512267838651289, -0.80000000000000004, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.40000000000000002.
template <typename Tp>
void test015()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data015)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data015[i].k), Tp(data015[i].nu),
                   Tp(data015[i].phi));
      const Tp f0 = data015[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.50000000000000000.
testcase_ellint_3<double> data016[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17422361866118047, -0.80000000000000004, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34681083254170475, -0.80000000000000004, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.51709470815494440, -0.80000000000000004, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.68587375344080259, -0.80000000000000004, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.85532571852810624, -0.80000000000000004, 0.50000000000000000, 
          0.87266462599716477 },
  { 1.0284677391874906, -0.80000000000000004, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.2081693942686225, -0.80000000000000004, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.3955803006426311, -0.80000000000000004, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.5884528947755532, -0.80000000000000004, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.50000000000000000.
template <typename Tp>
void test016()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data016)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data016[i].k), Tp(data016[i].nu),
                   Tp(data016[i].phi));
      const Tp f0 = data016[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.59999999999999998.
testcase_ellint_3<double> data017[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17404991781414092, -0.80000000000000004, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34549800443625167, -0.80000000000000004, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.51302536167001556, -0.80000000000000004, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.67717065003912258, -0.80000000000000004, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.84011512421134416, -0.80000000000000004, 0.59999999999999998, 
          0.87266462599716477 },
  { 1.0049863847088742, -0.80000000000000004, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.1748145941898918, -0.80000000000000004, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.3510319699755071, -0.80000000000000004, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.5319262547427865, -0.80000000000000004, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.59999999999999998.
template <typename Tp>
void test017()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data017)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data017[i].k), Tp(data017[i].nu),
                   Tp(data017[i].phi));
      const Tp f0 = data017[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.69999999999999996.
testcase_ellint_3<double> data018[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17387683562442202, -0.80000000000000004, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34420254775101611, -0.80000000000000004, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50906439222143685, -0.80000000000000004, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.66882693152688433, -0.80000000000000004, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.82574792844091316, -0.80000000000000004, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.98310431309490953, -0.80000000000000004, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.1440884535113258, -0.80000000000000004, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.3103743938952537, -0.80000000000000004, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.4806912324625332, -0.80000000000000004, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.69999999999999996.
template <typename Tp>
void test018()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data018)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data018[i].k), Tp(data018[i].nu),
                   Tp(data018[i].phi));
      const Tp f0 = data018[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.80000000000000004.
testcase_ellint_3<double> data019[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17370436817515206, -0.80000000000000004, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34292405894783395, -0.80000000000000004, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.50520682176250087, -0.80000000000000004, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.66081751679736189, -0.80000000000000004, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.81214672249355102, -0.80000000000000004, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.96264481387685574, -0.80000000000000004, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.1156611352656258, -0.80000000000000004, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.2730756225143889, -0.80000000000000004, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.4339837018309474, -0.80000000000000004, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.80000000000000004.
template <typename Tp>
void test019()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data019)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data019[i].k), Tp(data019[i].nu),
                   Tp(data019[i].phi));
      const Tp f0 = data019[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.80000000000000004, nu=0.90000000000000002.
testcase_ellint_3<double> data020[] = {
  { -0.0000000000000000, -0.80000000000000004, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17353251158533153, -0.80000000000000004, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.34166214791545768, -0.80000000000000004, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.50144799535130580, -0.80000000000000004, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.65311976193814447, -0.80000000000000004, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.79924384892320866, -0.80000000000000004, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.94345762353365625, -0.80000000000000004, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.0892582069219159, -0.80000000000000004, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.2387000876610268, -0.80000000000000004, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.3911845406776222, -0.80000000000000004, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.80000000000000004, nu=0.90000000000000002.
template <typename Tp>
void test020()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data020)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data020[i].k), Tp(data020[i].nu),
                   Tp(data020[i].phi));
      const Tp f0 = data020[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.0000000000000000.
testcase_ellint_3<double> data021[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17496737466916720, -0.69999999999999996, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35254687535677925, -0.69999999999999996, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.53536740275997130, -0.69999999999999996, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.72603797651684465, -0.69999999999999996, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.92698296348313458, -0.69999999999999996, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.1400447527693316, -0.69999999999999996, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.3657668117194071, -0.69999999999999996, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.6024686895959159, -0.69999999999999996, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.8456939983747236, -0.69999999999999996, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.0000000000000000.
template <typename Tp>
void test021()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data021)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data021[i].k), Tp(data021[i].nu),
                   Tp(data021[i].phi));
      const Tp f0 = data021[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.10000000000000001.
testcase_ellint_3<double> data022[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17479076384884681, -0.69999999999999996, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35114844900396364, -0.69999999999999996, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.53072776947527012, -0.69999999999999996, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.71530198262386246, -0.69999999999999996, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.90666760677828306, -0.69999999999999996, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.1063366517438080, -0.69999999999999996, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.3149477243092147, -0.69999999999999996, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.5314886725038925, -0.69999999999999996, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.7528050171757608, -0.69999999999999996, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.10000000000000001.
template <typename Tp>
void test022()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data022)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data022[i].k), Tp(data022[i].nu),
                   Tp(data022[i].phi));
      const Tp f0 = data022[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.20000000000000001.
testcase_ellint_3<double> data023[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17461479077791472, -0.69999999999999996, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34976950621407538, -0.69999999999999996, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.52622533231350188, -0.69999999999999996, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.70508774017895226, -0.69999999999999996, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.88775302531730294, -0.69999999999999996, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0756195476149006, -0.69999999999999996, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.2695349716654372, -0.69999999999999996, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.4690814617070540, -0.69999999999999996, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.6721098780092147, -0.69999999999999996, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.20000000000000001.
template <typename Tp>
void test023()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data023)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data023[i].k), Tp(data023[i].nu),
                   Tp(data023[i].phi));
      const Tp f0 = data023[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.29999999999999999.
testcase_ellint_3<double> data024[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17443945136076172, -0.69999999999999996, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34840956983535287, -0.69999999999999996, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.52185308551329179, -0.69999999999999996, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.69535240431168266, -0.69999999999999996, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.87007983473964923, -0.69999999999999996, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0474657975577066, -0.69999999999999996, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.2286225419931889, -0.69999999999999996, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.4136490671013271, -0.69999999999999996, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.6011813647733213, -0.69999999999999996, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.29999999999999999.
template <typename Tp>
void test024()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data024)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data024[i].k), Tp(data024[i].nu),
                   Tp(data024[i].phi));
      const Tp f0 = data024[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.40000000000000002.
testcase_ellint_3<double> data025[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17426474153983226, -0.69999999999999996, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34706817945773732, -0.69999999999999996, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.51760452851738159, -0.69999999999999996, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.68605801534722766, -0.69999999999999996, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.85351339387296532, -0.69999999999999996, 0.40000000000000002, 
          0.87266462599716477 },
  { 1.0215297967969537, -0.69999999999999996, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1915051074460528, -0.69999999999999996, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.3639821911744707, -0.69999999999999996, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.5382162002954762, -0.69999999999999996, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.40000000000000002.
template <typename Tp>
void test025()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data025)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data025[i].k), Tp(data025[i].nu),
                   Tp(data025[i].phi));
      const Tp f0 = data025[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.50000000000000000.
testcase_ellint_3<double> data026[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17409065729516093, -0.69999999999999996, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34574489064986091, -0.69999999999999996, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.51347361925579793, -0.69999999999999996, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.67717079489579290, -0.69999999999999996, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.83793902055292280, -0.69999999999999996, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.99752863545289705, -0.69999999999999996, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.1576240080401499, -0.69999999999999996, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.3191464023923762, -0.69999999999999996, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.4818433192178544, -0.69999999999999996, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.50000000000000000.
template <typename Tp>
void test026()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data026)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data026[i].k), Tp(data026[i].nu),
                   Tp(data026[i].phi));
      const Tp f0 = data026[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.59999999999999998.
testcase_ellint_3<double> data027[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17391719464391611, -0.69999999999999996, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34443927423869031, -0.69999999999999996, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50945473266486074, -0.69999999999999996, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.66866056326513823, -0.69999999999999996, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.82325830002337352, -0.69999999999999996, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.97522808245669357, -0.69999999999999996, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.1265300613705282, -0.69999999999999996, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.2784066076152003, -0.69999999999999996, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.4309994736080540, -0.69999999999999996, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.59999999999999998.
template <typename Tp>
void test027()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data027)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data027[i].k), Tp(data027[i].nu),
                   Tp(data027[i].phi));
      const Tp f0 = data027[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.69999999999999996.
testcase_ellint_3<double> data028[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17374434963995028, -0.69999999999999996, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34315091562900674, -0.69999999999999996, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50554262375653358, -0.69999999999999996, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.66050025406305812, -0.69999999999999996, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.80938620118847404, -0.69999999999999996, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.95443223855852144, -0.69999999999999996, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0978573207128302, -0.69999999999999996, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.2411754575007123, -0.69999999999999996, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.3848459188329196, -0.69999999999999996, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.69999999999999996.
template <typename Tp>
void test028()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data028)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data028[i].k), Tp(data028[i].nu),
                   Tp(data028[i].phi));
      const Tp f0 = data028[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.80000000000000004.
testcase_ellint_3<double> data029[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17357211837335737, -0.69999999999999996, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34187941416012108, -0.69999999999999996, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.50173239465478270, -0.69999999999999996, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.65266550725988315, -0.69999999999999996, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.79624879865249298, -0.69999999999999996, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.93497577043296920, -0.69999999999999996, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0713041566930748, -0.69999999999999996, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.2069772023255652, -0.69999999999999996, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.3427110650397533, -0.69999999999999996, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.80000000000000004.
template <typename Tp>
void test029()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data029)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data029[i].k), Tp(data029[i].nu),
                   Tp(data029[i].phi));
      const Tp f0 = data029[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.69999999999999996, nu=0.90000000000000002.
testcase_ellint_3<double> data030[] = {
  { -0.0000000000000000, -0.69999999999999996, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17340049697003634, -0.69999999999999996, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.34062438249741556, -0.69999999999999996, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49801946510076878, -0.69999999999999996, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.64513432604750487, -0.69999999999999996, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.78378145487573758, -0.69999999999999996, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.91671799500854634, -0.69999999999999996, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.0466193579463123, -0.69999999999999996, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.1754218079199146, -0.69999999999999996, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.3040500499695911, -0.69999999999999996, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.69999999999999996, nu=0.90000000000000002.
template <typename Tp>
void test030()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data030)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data030[i].k), Tp(data030[i].nu),
                   Tp(data030[i].phi));
      const Tp f0 = data030[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.0000000000000000.
testcase_ellint_3<double> data031[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17485154362988362, -0.59999999999999998, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35160509865544326, -0.59999999999999998, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.53210652578446160, -0.59999999999999998, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.71805304664485670, -0.59999999999999998, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.91082759030195970, -0.59999999999999998, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.1112333229323361, -0.59999999999999998, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.3191461190365270, -0.59999999999999998, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.5332022105084773, -0.59999999999999998, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.7507538029157526, -0.59999999999999998, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.0000000000000000.
template <typename Tp>
void test031()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data031)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data031[i].k), Tp(data031[i].nu),
                   Tp(data031[i].phi));
      const Tp f0 = data031[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.10000000000000001.
testcase_ellint_3<double> data032[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17467514275022014, -0.59999999999999998, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35021333086258255, -0.59999999999999998, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52751664092962713, -0.59999999999999998, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.70752126971957885, -0.59999999999999998, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.89111058756112871, -0.59999999999999998, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0789241202877768, -0.59999999999999998, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.2710800210399946, -0.59999999999999998, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.4669060574440276, -0.59999999999999998, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.6648615773343014, -0.59999999999999998, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.10000000000000001.
template <typename Tp>
void test032()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data032)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data032[i].k), Tp(data032[i].nu),
                   Tp(data032[i].phi));
      const Tp f0 = data032[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.20000000000000001.
testcase_ellint_3<double> data033[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17449937871800653, -0.59999999999999998, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34884093647346553, -0.59999999999999998, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.52306221119844110, -0.59999999999999998, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.69749955678982223, -0.59999999999999998, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.87274610682416853, -0.59999999999999998, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0494620540750792, -0.59999999999999998, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.2280847305507339, -0.59999999999999998, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.4085436279696886, -0.59999999999999998, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.5901418016279374, -0.59999999999999998, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.20000000000000001.
template <typename Tp>
void test033()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data033)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data033[i].k), Tp(data033[i].nu),
                   Tp(data033[i].phi));
      const Tp f0 = data033[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.29999999999999999.
testcase_ellint_3<double> data034[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17432424744393935, -0.59999999999999998, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34748744127146447, -0.59999999999999998, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51873632743924847, -0.59999999999999998, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.68794610396313127, -0.59999999999999998, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.85558070175468726, -0.59999999999999998, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0224416343605653, -0.59999999999999998, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1893144457936788, -0.59999999999999998, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.3566435377982575, -0.59999999999999998, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.5243814243493585, -0.59999999999999998, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.29999999999999999.
template <typename Tp>
void test034()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data034)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data034[i].k), Tp(data034[i].nu),
                   Tp(data034[i].phi));
      const Tp f0 = data034[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.40000000000000002.
testcase_ellint_3<double> data035[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17414974487670720, -0.59999999999999998, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34615238767335027, -0.59999999999999998, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.51453257838108579, -0.59999999999999998, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.67882386787534410, -0.59999999999999998, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.83948470233173578, -0.59999999999999998, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.99753496200073977, -0.59999999999999998, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1541101404388487, -0.59999999999999998, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.3100911323398814, -0.59999999999999998, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.4659345278069984, -0.59999999999999998, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.40000000000000002.
template <typename Tp>
void test035()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data035)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data035[i].k), Tp(data035[i].nu),
                   Tp(data035[i].phi));
      const Tp f0 = data035[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.50000000000000000.
testcase_ellint_3<double> data036[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17397586700252810, -0.59999999999999998, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34483533397138516, -0.59999999999999998, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.51044500461706499, -0.59999999999999998, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.67009988034712675, -0.59999999999999998, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.82434762375735193, -0.59999999999999998, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.97447346702798998, -0.59999999999999998, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.1219494000522143, -0.59999999999999998, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.2680242605954486, -0.59999999999999998, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.4135484285693078, -0.59999999999999998, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.50000000000000000.
template <typename Tp>
void test036()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data036)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data036[i].k), Tp(data036[i].nu),
                   Tp(data036[i].phi));
      const Tp f0 = data036[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.59999999999999998.
testcase_ellint_3<double> data037[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17380260984469356, -0.59999999999999998, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34353585361777839, -0.59999999999999998, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50646805774321402, -0.59999999999999998, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.66174468108625517, -0.59999999999999998, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.81007462280278408, -0.59999999999999998, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.95303466945718729, -0.59999999999999998, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0924118588677503, -0.59999999999999998, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.2297640574847937, -0.59999999999999998, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.3662507535812816, -0.59999999999999998, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.59999999999999998.
template <typename Tp>
void test037()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data037)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data037[i].k), Tp(data037[i].nu),
                   Tp(data037[i].phi));
      const Tp f0 = data037[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.69999999999999996.
testcase_ellint_3<double> data038[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17362996946312009, -0.59999999999999998, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34225353454870588, -0.59999999999999998, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50259656397799546, -0.59999999999999998, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.65373184496628944, -0.59999999999999998, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.79658372884056439, -0.59999999999999998, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.93303240100245421, -0.59999999999999998, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0651547944716557, -0.59999999999999998, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1947676204853441, -0.59999999999999998, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.3232737468822811, -0.59999999999999998, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.69999999999999996.
template <typename Tp>
void test038()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data038)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data038[i].k), Tp(data038[i].nu),
                   Tp(data038[i].phi));
      const Tp f0 = data038[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.80000000000000004.
testcase_ellint_3<double> data039[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17345794195390687, -0.59999999999999998, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34098797854531027, -0.59999999999999998, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49882569168826230, -0.59999999999999998, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.64603758566475511, -0.59999999999999998, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.78380365594769730, -0.59999999999999998, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.91430946255611190, -0.59999999999999998, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0398955217270607, -0.59999999999999998, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.1625948314277676, -0.59999999999999998, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.2840021261752192, -0.59999999999999998, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.80000000000000004.
template <typename Tp>
void test039()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data039)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data039[i].k), Tp(data039[i].nu),
                   Tp(data039[i].phi));
      const Tp f0 = data039[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.59999999999999998, nu=0.90000000000000002.
testcase_ellint_3<double> data040[] = {
  { -0.0000000000000000, -0.59999999999999998, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17328652344890033, -0.59999999999999998, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33973880062929018, -0.59999999999999998, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49515092233122765, -0.59999999999999998, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.63864042139737043, -0.59999999999999998, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.77167205646538850, -0.59999999999999998, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.89673202848034383, -0.59999999999999998, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.0163984492661304, -0.59999999999999998, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.1328845785162431, -0.59999999999999998, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.2479362973851875, -0.59999999999999998, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.59999999999999998, nu=0.90000000000000002.
template <typename Tp>
void test040()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data040)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data040[i].k), Tp(data040[i].nu),
                   Tp(data040[i].phi));
      const Tp f0 = data040[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.0000000000000000.
testcase_ellint_3<double> data041[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17475385514035785, -0.50000000000000000, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35081868470101585, -0.50000000000000000, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52942862705190585, -0.50000000000000000, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.71164727562630326, -0.50000000000000000, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.89824523594227768, -0.50000000000000000, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0895506700518851, -0.50000000000000000, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2853005857432933, -0.50000000000000000, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4845545520549484, -0.50000000000000000, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.6857503548125963, -0.50000000000000000, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.0000000000000000.
template <typename Tp>
void test041()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data041)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data041[i].k), Tp(data041[i].nu),
                   Tp(data041[i].phi));
      const Tp f0 = data041[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.10000000000000001.
testcase_ellint_3<double> data042[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17457763120814676, -0.50000000000000000, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34943246340849154, -0.50000000000000000, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52487937869610801, -0.50000000000000000, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.70127785096388395, -0.50000000000000000, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.87898815988624479, -0.50000000000000000, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0582764576094172, -0.50000000000000000, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.2391936844060207, -0.50000000000000000, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.4214793542995841, -0.50000000000000000, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.6045524936084892, -0.50000000000000000, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.10000000000000001.
template <typename Tp>
void test042()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data042)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data042[i].k), Tp(data042[i].nu),
                   Tp(data042[i].phi));
      const Tp f0 = data042[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.20000000000000001.
testcase_ellint_3<double> data043[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17440204336345433, -0.50000000000000000, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34806552388338824, -0.50000000000000000, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.52046416757129821, -0.50000000000000000, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.69140924550993876, -0.50000000000000000, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.86104678636125520, -0.50000000000000000, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0297439459053981, -0.50000000000000000, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1979214112912036, -0.50000000000000000, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.3659033858648930, -0.50000000000000000, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.5338490483665983, -0.50000000000000000, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.20000000000000001.
template <typename Tp>
void test043()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data043)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data043[i].k), Tp(data043[i].nu),
                   Tp(data043[i].phi));
      const Tp f0 = data043[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.29999999999999999.
testcase_ellint_3<double> data044[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17422708752228896, -0.50000000000000000, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34671739434855858, -0.50000000000000000, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51617616305641889, -0.50000000000000000, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.68200047612545178, -0.50000000000000000, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.84427217869498372, -0.50000000000000000, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0035637821389782, -0.50000000000000000, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1606800483933113, -0.50000000000000000, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.3164407134643459, -0.50000000000000000, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.4715681939859637, -0.50000000000000000, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.29999999999999999.
template <typename Tp>
void test044()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data044)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data044[i].k), Tp(data044[i].nu),
                   Tp(data044[i].phi));
      const Tp f0 = data044[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.40000000000000002.
testcase_ellint_3<double> data045[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17405275963859917, -0.50000000000000000, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34538761957029329, -0.50000000000000000, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.51200902646603919, -0.50000000000000000, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.67301522212868792, -0.50000000000000000, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.82853844466313320, -0.50000000000000000, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.97942097862681488, -0.50000000000000000, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1268429801220616, -0.50000000000000000, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2720406704533922, -0.50000000000000000, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.4161679518465340, -0.50000000000000000, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.40000000000000002.
template <typename Tp>
void test045()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data045)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data045[i].k), Tp(data045[i].nu),
                   Tp(data045[i].phi));
      const Tp f0 = data045[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.50000000000000000.
testcase_ellint_3<double> data046[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17387905570381157, -0.50000000000000000, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34407576010465207, -0.50000000000000000, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50795686560160835, -0.50000000000000000, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.66442115453330175, -0.50000000000000000, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.81373829119355345, -0.50000000000000000, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.95705743313235825, -0.50000000000000000, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0959131991362556, -0.50000000000000000, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.2318900529754597, -0.50000000000000000, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.3664739530045971, -0.50000000000000000, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.50000000000000000.
template <typename Tp>
void test046()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data046)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data046[i].k), Tp(data046[i].nu),
                   Tp(data046[i].phi));
      const Tp f0 = data046[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.59999999999999998.
testcase_ellint_3<double> data047[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17370597174637581, -0.50000000000000000, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34278139158591414, -0.50000000000000000, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50401419439302719, -0.50000000000000000, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.65618938076167221, -0.50000000000000000, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.79977959248855424, -0.50000000000000000, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.93625925190753545, -0.50000000000000000, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0674905658379710, -0.50000000000000000, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1953481298023048, -0.50000000000000000, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.3215740290190876, -0.50000000000000000, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.59999999999999998.
template <typename Tp>
void test047()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data047)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data047[i].k), Tp(data047[i].nu),
                   Tp(data047[i].phi));
      const Tp f0 = data047[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.69999999999999996.
testcase_ellint_3<double> data048[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17353350383131641, -0.50000000000000000, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34150410405436771, -0.50000000000000000, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50017589696443487, -0.50000000000000000, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.64829398188419962, -0.50000000000000000, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.78658270782402073, -0.50000000000000000, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.91684738336675053, -0.50000000000000000, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0412486789555937, -0.50000000000000000, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1619021847612001, -0.50000000000000000, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2807475181182502, -0.50000000000000000, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.69999999999999996.
template <typename Tp>
void test048()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data048)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data048[i].k), Tp(data048[i].nu),
                   Tp(data048[i].phi));
      const Tp f0 = data048[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.80000000000000004.
testcase_ellint_3<double> data049[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17336164805979126, -0.50000000000000000, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34024350132086773, -0.50000000000000000, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49643719555734084, -0.50000000000000000, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.64071162456976150, -0.50000000000000000, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.77407836177211908, -0.50000000000000000, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.89867058251905652, -0.50000000000000000, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0169181822134912, -0.50000000000000000, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.1311363312779448, -0.50000000000000000, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.2434165408189539, -0.50000000000000000, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.80000000000000004.
template <typename Tp>
void test049()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data049)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data049[i].k), Tp(data049[i].nu),
                   Tp(data049[i].phi));
      const Tp f0 = data049[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.50000000000000000, nu=0.90000000000000002.
testcase_ellint_3<double> data050[] = {
  { -0.0000000000000000, -0.50000000000000000, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17319040056865681, -0.50000000000000000, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33899920036578557, -0.50000000000000000, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49279362182695186, -0.50000000000000000, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.63342123379746151, -0.50000000000000000, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.76220595179550321, -0.50000000000000000, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.88160004743532294, -0.50000000000000000, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.99427448642310134, -0.50000000000000000, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.1027091512470093, -0.50000000000000000, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.2091116095504744, -0.50000000000000000, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.50000000000000000, nu=0.90000000000000002.
template <typename Tp>
void test050()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data050)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data050[i].k), Tp(data050[i].nu),
                   Tp(data050[i].phi));
      const Tp f0 = data050[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.0000000000000000.
testcase_ellint_3<double> data051[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17467414669441531, -0.40000000000000002, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35018222772483443, -0.40000000000000002, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52729015917508748, -0.40000000000000002, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.70662374407341255, -0.40000000000000002, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.88859210497602170, -0.40000000000000002, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0733136290471379, -0.40000000000000002, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2605612170157061, -0.40000000000000002, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4497513956433437, -0.40000000000000002, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.6399998658645112, -0.40000000000000002, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.0000000000000000.
template <typename Tp>
void test051()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data051)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data051[i].k), Tp(data051[i].nu),
                   Tp(data051[i].phi));
      const Tp f0 = data051[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.10000000000000001.
testcase_ellint_3<double> data052[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17449806706684673, -0.40000000000000002, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34880048623856075, -0.40000000000000002, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52277322065757403, -0.40000000000000002, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.69638072056918376, -0.40000000000000002, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.86968426619831540, -0.40000000000000002, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0428044206578095, -0.40000000000000002, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.2158651158274378, -0.40000000000000002, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3889447129893322, -0.40000000000000002, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5620566886683604, -0.40000000000000002, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.10000000000000001.
template <typename Tp>
void test052()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data052)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data052[i].k), Tp(data052[i].nu),
                   Tp(data052[i].phi));
      const Tp f0 = data052[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.20000000000000001.
testcase_ellint_3<double> data053[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17432262290723399, -0.40000000000000002, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34743795258968596, -0.40000000000000002, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51838919472805123, -0.40000000000000002, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.68663134739057918, -0.40000000000000002, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.85206432981833979, -0.40000000000000002, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0149595349004430, -0.40000000000000002, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1758349405464676, -0.40000000000000002, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.3353337673882635, -0.40000000000000002, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4941414344266770, -0.40000000000000002, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.20000000000000001.
template <typename Tp>
void test053()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data053)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data053[i].k), Tp(data053[i].nu),
                   Tp(data053[i].phi));
      const Tp f0 = data053[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.29999999999999999.
testcase_ellint_3<double> data054[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17414781013591543, -0.40000000000000002, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34609415696777285, -0.40000000000000002, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51413131295862546, -0.40000000000000002, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.67733527622935630, -0.40000000000000002, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.83558675182733266, -0.40000000000000002, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.98940140808865906, -0.40000000000000002, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1396968797728058, -0.40000000000000002, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2875920037865087, -0.40000000000000002, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.4342789859950078, -0.40000000000000002, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.29999999999999999.
template <typename Tp>
void test054()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data054)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data054[i].k), Tp(data054[i].nu),
                   Tp(data054[i].phi));
      const Tp f0 = data054[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.40000000000000002.
testcase_ellint_3<double> data055[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17397362471112710, -0.40000000000000002, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34476864603333196, -0.40000000000000002, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50999329415379357, -0.40000000000000002, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66845674551396017, -0.40000000000000002, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.82012848346231748, -0.40000000000000002, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.96582449258349057, -0.40000000000000002, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1068473749476286, -0.40000000000000002, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2447132729159986, -0.40000000000000002, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3809986210732901, -0.40000000000000002, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.40000000000000002.
template <typename Tp>
void test055()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data055)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data055[i].k), Tp(data055[i].nu),
                   Tp(data055[i].phi));
      const Tp f0 = data055[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.50000000000000000.
testcase_ellint_3<double> data056[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17380006262854139, -0.40000000000000002, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34346098216756610, -0.40000000000000002, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50596929935059420, -0.40000000000000002, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65996392089131262, -0.40000000000000002, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.80558463511364786, -0.40000000000000002, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.94397834522857704, -0.40000000000000002, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0768075114108115, -0.40000000000000002, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.2059184624251329, -0.40000000000000002, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.3331797176377398, -0.40000000000000002, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.50000000000000000.
template <typename Tp>
void test056()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data056)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data056[i].k), Tp(data056[i].nu),
                   Tp(data056[i].phi));
      const Tp f0 = data056[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.59999999999999998.
testcase_ellint_3<double> data057[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17362711992081248, -0.40000000000000002, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34217074276403953, -0.40000000000000002, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50205389185761617, -0.40000000000000002, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.65182834920372745, -0.40000000000000002, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.79186512820565136, -0.40000000000000002, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.92365535916287134, -0.40000000000000002, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0491915663957907, -0.40000000000000002, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1705934291745104, -0.40000000000000002, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2899514672527024, -0.40000000000000002, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.59999999999999998.
template <typename Tp>
void test057()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data057)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data057[i].k), Tp(data057[i].nu),
                   Tp(data057[i].phi));
      const Tp f0 = data057[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.69999999999999996.
testcase_ellint_3<double> data058[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17345479265712871, -0.40000000000000002, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34089751955950354, -0.40000000000000002, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49824200167361343, -0.40000000000000002, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.64402450341199413, -0.40000000000000002, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.77889207804122873, -0.40000000000000002, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.90468169720957992, -0.40000000000000002, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0236847823692916, -0.40000000000000002, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1382465247425164, -0.40000000000000002, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2506255923253344, -0.40000000000000002, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.69999999999999996.
template <typename Tp>
void test058()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data058)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data058[i].k), Tp(data058[i].nu),
                   Tp(data058[i].phi));
      const Tp f0 = data058[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.80000000000000004.
testcase_ellint_3<double> data059[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17328307694277156, -0.40000000000000002, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33964091800132007, -0.40000000000000002, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49452889372467451, -0.40000000000000002, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.63652940095937327, -0.40000000000000002, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.76659772511159097, -0.40000000000000002, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.88691047977338111, -0.40000000000000002, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0000273200611638, -0.40000000000000002, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.1084787902188007, -0.40000000000000002, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.2146499565727209, -0.40000000000000002, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.80000000000000004.
template <typename Tp>
void test059()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data059)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data059[i].k), Tp(data059[i].nu),
                   Tp(data059[i].phi));
      const Tp f0 = data059[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.40000000000000002, nu=0.90000000000000002.
testcase_ellint_3<double> data060[] = {
  { -0.0000000000000000, -0.40000000000000002, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17311196891868130, -0.40000000000000002, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33840055664911906, -0.40000000000000002, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49091013944075340, -0.40000000000000002, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62932228186809591, -0.40000000000000002, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.75492278323019801, -0.40000000000000002, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.87021659043854294, -0.40000000000000002, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.97800245228239246, -0.40000000000000002, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0809625773173694, -0.40000000000000002, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1815758115929846, -0.40000000000000002, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.40000000000000002, nu=0.90000000000000002.
template <typename Tp>
void test060()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data060)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data060[i].k), Tp(data060[i].nu),
                   Tp(data060[i].phi));
      const Tp f0 = data060[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.0000000000000000.
testcase_ellint_3<double> data061[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17461228653000102, -0.30000000000000004, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.34969146102798415, -0.30000000000000004, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52565822873726320, -0.30000000000000004, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.70284226512408532, -0.30000000000000004, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.88144139195111182, -0.30000000000000004, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0614897067260523, -0.30000000000000004, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2428416824174218, -0.30000000000000004, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4251795877015925, -0.30000000000000004, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.6080486199305126, -0.30000000000000004, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.0000000000000000.
template <typename Tp>
void test061()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data061)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data061[i].k), Tp(data061[i].nu),
                   Tp(data061[i].phi));
      const Tp f0 = data061[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.10000000000000001.
testcase_ellint_3<double> data062[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17443631884814378, -0.30000000000000004, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34831316835124926, -0.30000000000000004, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52116586276523857, -0.30000000000000004, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.69269385837910036, -0.30000000000000004, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.86279023163070856, -0.30000000000000004, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0315321461438265, -0.30000000000000004, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.1991449111869024, -0.30000000000000004, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3659561780923211, -0.30000000000000004, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5323534693557526, -0.30000000000000004, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.10000000000000001.
template <typename Tp>
void test062()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data062)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data062[i].k), Tp(data062[i].nu),
                   Tp(data062[i].phi));
      const Tp f0 = data062[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.20000000000000001.
testcase_ellint_3<double> data063[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17426098615372090, -0.30000000000000004, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34695402664689923, -0.30000000000000004, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51680555567038933, -0.30000000000000004, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.68303375225260210, -0.30000000000000004, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.84540662891295026, -0.30000000000000004, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0041834051646927, -0.30000000000000004, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1599952702345711, -0.30000000000000004, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.3137179520499163, -0.30000000000000004, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4663658145259875, -0.30000000000000004, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.20000000000000001.
template <typename Tp>
void test063()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data063)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data063[i].k), Tp(data063[i].nu),
                   Tp(data063[i].phi));
      const Tp f0 = data063[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.29999999999999999.
testcase_ellint_3<double> data064[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17408628437042845, -0.30000000000000004, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34561356761638401, -0.30000000000000004, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51257058617875850, -0.30000000000000004, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.67382207124602866, -0.30000000000000004, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.82914751587825131, -0.30000000000000004, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.97907434814374950, -0.30000000000000004, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1246399297351584, -0.30000000000000004, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2671793970398146, -0.30000000000000004, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.4081767433479089, -0.30000000000000004, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.29999999999999999.
template <typename Tp>
void test064()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data064)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data064[i].k), Tp(data064[i].nu),
                   Tp(data064[i].phi));
      const Tp f0 = data064[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.40000000000000002.
testcase_ellint_3<double> data065[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17391220945982730, -0.30000000000000004, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34429133937639689, -0.30000000000000004, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50845471668581632, -0.30000000000000004, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66502347027873854, -0.30000000000000004, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.81389191978012254, -0.30000000000000004, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.95590618002140593, -0.30000000000000004, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.0924915195213121, -0.30000000000000004, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2253651604038058, -0.30000000000000004, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3563643538969761, -0.30000000000000004, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.40000000000000002.
template <typename Tp>
void test065()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data065)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data065[i].k), Tp(data065[i].nu),
                   Tp(data065[i].phi));
      const Tp f0 = data065[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.50000000000000000.
testcase_ellint_3<double> data066[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17373875742088235, -0.30000000000000004, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34298690571124157, -0.30000000000000004, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50445214859646936, -0.30000000000000004, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65660648352418516, -0.30000000000000004, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.79953670639287289, -0.30000000000000004, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.93443393926588558, -0.30000000000000004, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0630838369016911, -0.30000000000000004, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.1875197325653026, -0.30000000000000004, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.3098448759814960, -0.30000000000000004, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.50000000000000000.
template <typename Tp>
void test066()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data066)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data066[i].k), Tp(data066[i].nu),
                   Tp(data066[i].phi));
      const Tp f0 = data066[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.59999999999999998.
testcase_ellint_3<double> data067[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17356592428950826, -0.30000000000000004, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34169984536697379, -0.30000000000000004, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50055748266498457, -0.30000000000000004, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.64854298527106768, -0.30000000000000004, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.78599329284207431, -0.30000000000000004, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.91445452089128221, -0.30000000000000004, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0360412952290587, -0.30000000000000004, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1530473919778639, -0.30000000000000004, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2677758800420666, -0.30000000000000004, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.59999999999999998.
template <typename Tp>
void test067()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data067)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data067[i].k), Tp(data067[i].nu),
                   Tp(data067[i].phi));
      const Tp f0 = data067[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.69999999999999996.
testcase_ellint_3<double> data068[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17339370613812227, -0.30000000000000004, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34042975138455933, -0.30000000000000004, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49676568368075985, -0.30000000000000004, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.64080774055753720, -0.30000000000000004, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.77318507779667278, -0.30000000000000004, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.89579782346548631, -0.30000000000000004, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0110573286052202, -0.30000000000000004, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1214710972949633, -0.30000000000000004, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2294913236274980, -0.30000000000000004, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.69999999999999996.
template <typename Tp>
void test068()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data068)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data068[i].k), Tp(data068[i].nu),
                   Tp(data068[i].phi));
      const Tp f0 = data068[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.80000000000000004.
testcase_ellint_3<double> data069[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17322209907520361, -0.30000000000000004, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33917623046949996, -0.30000000000000004, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49307204894329176, -0.30000000000000004, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.63337802830291723, -0.30000000000000004, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.76104540997689407, -0.30000000000000004, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.87832009635450736, -0.30000000000000004, 0.80000000000000004, 
          1.0471975511965976 },
  { 0.98787879723171790, -0.30000000000000004, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.0924036340069336, -0.30000000000000004, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.1944567571590046, -0.30000000000000004, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.80000000000000004.
template <typename Tp>
void test069()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data069)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data069[i].k), Tp(data069[i].nu),
                   Tp(data069[i].phi));
      const Tp f0 = data069[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.30000000000000004, nu=0.90000000000000002.
testcase_ellint_3<double> data070[] = {
  { -0.0000000000000000, -0.30000000000000004, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17305109924485948, -0.30000000000000004, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33793890239556984, -0.30000000000000004, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.48947218005089738, -0.30000000000000004, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62623332340775151, -0.30000000000000004, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.74951596581511148, -0.30000000000000004, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.86189886597756005, -0.30000000000000004, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.96629451153092005, -0.30000000000000004, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0655269133492680, -0.30000000000000004, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1622376896064912, -0.30000000000000004, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.30000000000000004, nu=0.90000000000000002.
template <typename Tp>
void test070()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data070)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data070[i].k), Tp(data070[i].nu),
                   Tp(data070[i].phi));
      const Tp f0 = data070[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.0000000000000000.
testcase_ellint_3<double> data071[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17456817290292811, -0.19999999999999996, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.34934315932086801, -0.19999999999999996, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52450880529443988, -0.19999999999999996, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.70020491009844910, -0.19999999999999996, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.87651006649967955, -0.19999999999999996, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0534305870298994, -0.19999999999999996, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2308975521670784, -0.19999999999999996, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4087733584990738, -0.19999999999999996, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.5868678474541664, -0.19999999999999996, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.0000000000000000.
template <typename Tp>
void test071()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data071)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data071[i].k), Tp(data071[i].nu),
                   Tp(data071[i].phi));
      const Tp f0 = data071[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.10000000000000001.
testcase_ellint_3<double> data072[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17439228502691750, -0.19999999999999996, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34796731137565740, -0.19999999999999996, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52003370294544848, -0.19999999999999996, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.69012222258631495, -0.19999999999999996, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.85803491465566772, -0.19999999999999996, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0238463961099364, -0.19999999999999996, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.1878691059202153, -0.19999999999999996, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3505985031831940, -0.19999999999999996, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5126513474261092, -0.19999999999999996, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.10000000000000001.
template <typename Tp>
void test072()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data072)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data072[i].k), Tp(data072[i].nu),
                   Tp(data072[i].phi));
      const Tp f0 = data072[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.20000000000000001.
testcase_ellint_3<double> data073[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17421703179583750, -0.19999999999999996, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34661057411998791, -0.19999999999999996, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51569006052647393, -0.19999999999999996, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.68052412821107278, -0.19999999999999996, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.84081341263313825, -0.19999999999999996, 0.20000000000000001, 
          0.87266462599716477 },
  { 0.99683359988842890, -0.19999999999999996, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1493086715118852, -0.19999999999999996, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.2992699693957541, -0.19999999999999996, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4479323932249568, -0.19999999999999996, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.20000000000000001.
template <typename Tp>
void test073()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data073)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data073[i].k), Tp(data073[i].nu),
                   Tp(data073[i].phi));
      const Tp f0 = data073[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.29999999999999999.
testcase_ellint_3<double> data074[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17404240913577707, -0.19999999999999996, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34527248032587193, -0.19999999999999996, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51147118981668416, -0.19999999999999996, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.67137107867777635, -0.19999999999999996, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.82470418188668893, -0.19999999999999996, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.97202873223594299, -0.19999999999999996, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1144773569375266, -0.19999999999999996, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2535292433701000, -0.19999999999999996, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.3908453514752481, -0.19999999999999996, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.29999999999999999.
template <typename Tp>
void test074()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data074)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data074[i].k), Tp(data074[i].nu),
                   Tp(data074[i].phi));
      const Tp f0 = data074[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.40000000000000002.
testcase_ellint_3<double> data075[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17386841301066677, -0.19999999999999996, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34395257914113253, -0.19999999999999996, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50737088376869466, -0.19999999999999996, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66262801717277664, -0.19999999999999996, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.80958766645079094, -0.19999999999999996, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.94913754236162040, -0.19999999999999996, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.0827985514223000, -0.19999999999999996, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2124212429050478, -0.19999999999999996, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3400002519661010, -0.19999999999999996, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.40000000000000002.
template <typename Tp>
void test075()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data075)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data075[i].k), Tp(data075[i].nu),
                   Tp(data075[i].phi));
      const Tp f0 = data075[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.50000000000000000.
testcase_ellint_3<double> data076[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17369503942181802, -0.19999999999999996, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34265043534362660, -0.19999999999999996, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50338337208655415, -0.19999999999999996, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65426373297163642, -0.19999999999999996, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.79536193036145808, -0.19999999999999996, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.92791875910061605, -0.19999999999999996, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0538145052725829, -0.19999999999999996, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.1752060022875899, -0.19999999999999996, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.2943374404397376, -0.19999999999999996, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.50000000000000000.
template <typename Tp>
void test076()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data076)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data076[i].k), Tp(data076[i].nu),
                   Tp(data076[i].phi));
      const Tp f0 = data076[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.59999999999999998.
testcase_ellint_3<double> data077[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17352228440746928, -0.19999999999999996, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34136562863713626, -0.19999999999999996, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.49950328177638481, -0.19999999999999996, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.64625032705690832, -0.19999999999999996, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.78193941198403094, -0.19999999999999996, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.90817230934317128, -0.19999999999999996, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0271563751276462, -0.19999999999999996, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1412999379040518, -0.19999999999999996, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2530330675914561, -0.19999999999999996, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.59999999999999998.
template <typename Tp>
void test077()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data077)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data077[i].k), Tp(data077[i].nu),
                   Tp(data077[i].phi));
      const Tp f0 = data077[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.69999999999999996.
testcase_ellint_3<double> data078[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17335014404233898, -0.19999999999999996, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34009775298617811, -0.19999999999999996, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49572560201923810, -0.19999999999999996, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.63856276669886525, -0.19999999999999996, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.76924438644867565, -0.19999999999999996, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.88973060843856466, -0.19999999999999996, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0025230471636377, -0.19999999999999996, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1102356376093103, -0.19999999999999996, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2154356555075867, -0.19999999999999996, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.69999999999999996.
template <typename Tp>
void test078()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data078)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data078[i].k), Tp(data078[i].nu),
                   Tp(data078[i].phi));
      const Tp f0 = data078[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.80000000000000004.
testcase_ellint_3<double> data079[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17317861443718541, -0.19999999999999996, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33884641598718701, -0.19999999999999996, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49204565281259494, -0.19999999999999996, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.63117851188220353, -0.19999999999999996, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.75721095949544170, -0.19999999999999996, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.87245201443919118, -0.19999999999999996, 0.80000000000000004, 
          1.0471975511965976 },
  { 0.97966584238831089, -0.19999999999999996, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.0816336325174360, -0.19999999999999996, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.1810223448909913, -0.19999999999999996, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.80000000000000004.
template <typename Tp>
void test079()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data079)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data079[i].k), Tp(data079[i].nu),
                   Tp(data079[i].phi));
      const Tp f0 = data079[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.19999999999999996, nu=0.90000000000000002.
testcase_ellint_3<double> data080[] = {
  { -0.0000000000000000, -0.19999999999999996, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17300769173837280, -0.19999999999999996, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33761123827372508, -0.19999999999999996, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.48845905690769426, -0.19999999999999996, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62407720017324986, -0.19999999999999996, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.74578146525124289, -0.19999999999999996, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.85621583540073076, -0.19999999999999996, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.95837725988001199, -0.19999999999999996, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0551821412633928, -0.19999999999999996, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1493679916141863, -0.19999999999999996, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.19999999999999996, nu=0.90000000000000002.
template <typename Tp>
void test080()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data080)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data080[i].k), Tp(data080[i].nu),
                   Tp(data080[i].phi));
      const Tp f0 = data080[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.0000000000000000.
testcase_ellint_3<double> data081[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17454173353063665, -0.099999999999999978, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.34913506721468085, -0.099999999999999978, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52382550016538953, -0.099999999999999978, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.69864700854177031, -0.099999999999999978, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.87361792586964870, -0.099999999999999978, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0487386319621685, -0.099999999999999978, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2239913752078757, -0.099999999999999978, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.3993423113684049, -0.099999999999999978, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.5747455615173562, -0.099999999999999978, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.0000000000000000.
template <typename Tp>
void test081()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data081)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data081[i].k), Tp(data081[i].nu),
                   Tp(data081[i].phi));
      const Tp f0 = data081[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.10000000000000001.
testcase_ellint_3<double> data082[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17436589347616618, -0.099999999999999978, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34776067871237354, -0.099999999999999978, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.51936064354727807, -0.099999999999999978, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.68860303749364360, -0.099999999999999978, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.85524561882332051, -0.099999999999999978, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0193708301908337, -0.099999999999999978, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.1813474067123044, -0.099999999999999978, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3417670770424983, -0.099999999999999978, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5013711111199950, -0.099999999999999978, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.10000000000000001.
template <typename Tp>
void test082()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data082)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data082[i].k), Tp(data082[i].nu),
                   Tp(data082[i].phi));
      const Tp f0 = data082[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.20000000000000001.
testcase_ellint_3<double> data083[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17419068786141345, -0.099999999999999978, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34640537686230127, -0.099999999999999978, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51502689171753957, -0.099999999999999978, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.67904147863672726, -0.099999999999999978, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.83811885126105179, -0.099999999999999978, 0.20000000000000001, 
          0.87266462599716477 },
  { 0.99255278555742810, -0.099999999999999978, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1431260546194930, -0.099999999999999978, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.2909589656532101, -0.099999999999999978, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4373749386463430, -0.099999999999999978, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.20000000000000001.
template <typename Tp>
void test083()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data083)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data083[i].k), Tp(data083[i].nu),
                   Tp(data083[i].phi));
      const Tp f0 = data083[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.29999999999999999.
testcase_ellint_3<double> data084[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17401611261390110, -0.099999999999999978, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34506869507511767, -0.099999999999999978, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51081757604259870, -0.099999999999999978, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.66992297597712303, -0.099999999999999978, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.82209722856174228, -0.099999999999999978, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.96792430487669612, -0.099999999999999978, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1085964108954092, -0.099999999999999978, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2456748370836999, -0.099999999999999978, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.3809159606704959, -0.099999999999999978, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.29999999999999999.
template <typename Tp>
void test084()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data084)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data084[i].k), Tp(data084[i].nu),
                   Tp(data084[i].phi));
      const Tp f0 = data084[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.40000000000000002.
testcase_ellint_3<double> data085[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17384216369897937, -0.099999999999999978, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34375018311376782, -0.099999999999999978, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50672650758380455, -0.099999999999999978, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66121264213337616, -0.099999999999999978, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.80706202005774441, -0.099999999999999978, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.94519376138245892, -0.099999999999999978, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.0771880300759584, -0.099999999999999978, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2049711557188272, -0.099999999999999978, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3306223265207477, -0.099999999999999978, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.40000000000000002.
template <typename Tp>
void test085()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data085)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data085[i].k), Tp(data085[i].nu),
                   Tp(data085[i].phi));
      const Tp f0 = data085[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.50000000000000000.
testcase_ellint_3<double> data086[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17366883711936554, -0.099999999999999978, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34244940634881876, -0.099999999999999978, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50274793281634378, -0.099999999999999978, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65287941633275093, -0.099999999999999978, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.79291198790315398, -0.099999999999999978, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.92412201537880345, -0.099999999999999978, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0484480076799370, -0.099999999999999978, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.1681168130475206, -0.099999999999999978, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.2854480708580160, -0.099999999999999978, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.50000000000000000.
template <typename Tp>
void test086()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data086)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data086[i].k), Tp(data086[i].nu),
                   Tp(data086[i].phi));
      const Tp f0 = data086[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.59999999999999998.
testcase_ellint_3<double> data087[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17349612891469018, -0.099999999999999978, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34116594505539438, -0.099999999999999978, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.49887649430466685, -0.099999999999999978, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.64489553282165157, -0.099999999999999978, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.77956016553782437, -0.099999999999999978, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.90451074530096309, -0.099999999999999978, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0220113666961632, -0.099999999999999978, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1345351441065563, -0.099999999999999978, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2445798942989255, -0.099999999999999978, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.59999999999999998.
template <typename Tp>
void test087()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data087)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data087[i].k), Tp(data087[i].nu),
                   Tp(data087[i].phi));
      const Tp f0 = data087[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.69999999999999996.
testcase_ellint_3<double> data088[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17332403516105052, -0.099999999999999978, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.33989939374896877, -0.099999999999999978, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49510719568614081, -0.099999999999999978, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.63723607776354974, -0.099999999999999978, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.76693133887935327, -0.099999999999999978, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.88619382078823827, -0.099999999999999978, 0.69999999999999996, 
          1.0471975511965976 },
  { 0.99758012018676490, -0.099999999999999978, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1037642270814410, -0.099999999999999978, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2073745911083187, -0.099999999999999978, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.69999999999999996.
template <typename Tp>
void test088()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data088)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data088[i].k), Tp(data088[i].nu),
                   Tp(data088[i].phi));
      const Tp f0 = data088[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.80000000000000004.
testcase_ellint_3<double> data089[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17315255197057020, -0.099999999999999978, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33864936055747985, -0.099999999999999978, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49143537041117619, -0.099999999999999978, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.62987861760047492, -0.099999999999999978, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.75496005490917517, -0.099999999999999978, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.86903081862701903, -0.099999999999999978, 0.80000000000000004, 
          1.0471975511965976 },
  { 0.97490814820725591, -0.099999999999999978, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.0754290107171083, -0.099999999999999978, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.1733158866987732, -0.099999999999999978, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.80000000000000004.
template <typename Tp>
void test089()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data089)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data089[i].k), Tp(data089[i].nu),
                   Tp(data089[i].phi));
      const Tp f0 = data089[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=-0.099999999999999978, nu=0.90000000000000002.
testcase_ellint_3<double> data090[] = {
  { -0.0000000000000000, -0.099999999999999978, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17298167549096569, -0.099999999999999978, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33741546662741584, -0.099999999999999978, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.48785665376856879, -0.099999999999999978, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62280288554518959, -0.099999999999999978, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.74358903115455199, -0.099999999999999978, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.85290207679298358, -0.099999999999999978, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.95379006645397379, -0.099999999999999978, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0492213119872327, -0.099999999999999978, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1419839485283374, -0.099999999999999978, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=-0.099999999999999978, nu=0.90000000000000002.
template <typename Tp>
void test090()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data090)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data090[i].k), Tp(data090[i].nu),
                   Tp(data090[i].phi));
      const Tp f0 = data090[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.0000000000000000.
testcase_ellint_3<double> data091[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17453292519943295, 0.0000000000000000, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.34906585039886584, 0.0000000000000000, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52359877559829882, 0.0000000000000000, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.69813170079773179, 0.0000000000000000, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.87266462599716477, 0.0000000000000000, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0471975511965976, 0.0000000000000000, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2217304763960304, 0.0000000000000000, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.3962634015954631, 0.0000000000000000, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.5707963267948966, 0.0000000000000000, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.0000000000000000.
template <typename Tp>
void test091()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data091)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data091[i].k), Tp(data091[i].nu),
                   Tp(data091[i].phi));
      const Tp f0 = data091[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.10000000000000001.
testcase_ellint_3<double> data092[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17435710107516608, 0.0000000000000000, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34769194715329604, 0.0000000000000000, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.51913731575866118, 0.0000000000000000, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.68810051897078461, 0.0000000000000000, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.85432615661706823, 0.0000000000000000, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0179006647340796, 0.0000000000000000, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.1792120640746322, 0.0000000000000000, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3388834245070498, 0.0000000000000000, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.4976955329233277, 0.0000000000000000, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.10000000000000001.
template <typename Tp>
void test092()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data092)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data092[i].k), Tp(data092[i].nu),
                   Tp(data092[i].phi));
      const Tp f0 = data092[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.20000000000000001.
testcase_ellint_3<double> data093[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17418191132226077, 0.0000000000000000, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34633712256943405, 0.0000000000000000, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51480684302043711, 0.0000000000000000, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.67855102942481949, 0.0000000000000000, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.83723056090326253, 0.0000000000000000, 0.20000000000000001, 
          0.87266462599716477 },
  { 0.99114645269578183, 0.0000000000000000, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1411014627915537, 0.0000000000000000, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.2882448138013969, 0.0000000000000000, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4339343023863691, 0.0000000000000000, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.20000000000000001.
template <typename Tp>
void test093()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data093)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data093[i].k), Tp(data093[i].nu),
                   Tp(data093[i].phi));
      const Tp f0 = data093[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.29999999999999999.
testcase_ellint_3<double> data094[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17400735186871727, 0.0000000000000000, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34500091027020219, 0.0000000000000000, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51060069523901541, 0.0000000000000000, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.66944393961375459, 0.0000000000000000, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.82123776744538157, 0.0000000000000000, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.96657579245516523, 0.0000000000000000, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1066703663542414, 0.0000000000000000, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2431094251944901, 0.0000000000000000, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.3776795151134889, 0.0000000000000000, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.29999999999999999.
template <typename Tp>
void test094()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data094)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data094[i].k), Tp(data094[i].nu),
                   Tp(data094[i].phi));
      const Tp f0 = data094[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.40000000000000002.
testcase_ellint_3<double> data095[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17383341868035865, 0.0000000000000000, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34368286022299821, 0.0000000000000000, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50651268947499406, 0.0000000000000000, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66074441806097550, 0.0000000000000000, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.80622931670113485, 0.0000000000000000, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.94389791565435233, 0.0000000000000000, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.0753503387899728, 0.0000000000000000, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2025374759127518, 0.0000000000000000, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3275651989026322, 0.0000000000000000, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.40000000000000002.
template <typename Tp>
void test095()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data095)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data095[i].k), Tp(data095[i].nu),
                   Tp(data095[i].phi));
      const Tp f0 = data095[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.50000000000000000.
testcase_ellint_3<double> data096[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17366010776037047, 0.0000000000000000, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34238253799539309, 0.0000000000000000, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50253707775976408, 0.0000000000000000, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65242145347295766, 0.0000000000000000, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.79210420018698058, 0.0000000000000000, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.92287437995632193, 0.0000000000000000, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0466900550798661, 0.0000000000000000, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.1658007366618623, 0.0000000000000000, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.2825498301618641, 0.0000000000000000, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.50000000000000000.
template <typename Tp>
void test096()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data096)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data096[i].k), Tp(data096[i].nu),
                   Tp(data096[i].phi));
      const Tp f0 = data096[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.59999999999999998.
testcase_ellint_3<double> data097[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17348741514884702, 0.0000000000000000, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34109952405241289, 0.0000000000000000, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.49866850781226296, 0.0000000000000000, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.64444732407062510, 0.0000000000000000, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.77877564686544720, 0.0000000000000000, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.90330743691883497, 0.0000000000000000, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0203257987604104, 0.0000000000000000, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1323247918768631, 0.0000000000000000, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2418235332245127, 0.0000000000000000, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.59999999999999998.
template <typename Tp>
void test097()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data097)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data097[i].k), Tp(data097[i].nu),
                   Tp(data097[i].phi));
      const Tp f0 = data097[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.69999999999999996.
testcase_ellint_3<double> data098[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17331533692234477, 0.0000000000000000, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.33983341309265941, 0.0000000000000000, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49490198805931990, 0.0000000000000000, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.63679715525145308, 0.0000000000000000, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.76616861049481944, 0.0000000000000000, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.88503143209004220, 0.0000000000000000, 0.69999999999999996, 
          1.0471975511965976 },
  { 0.99596060249112173, 0.0000000000000000, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1016495050260424, 0.0000000000000000, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2047457872617382, 0.0000000000000000, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.69999999999999996.
template <typename Tp>
void test098()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data098)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data098[i].k), Tp(data098[i].nu),
                   Tp(data098[i].phi));
      const Tp f0 = data098[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.80000000000000004.
testcase_ellint_3<double> data099[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17314386919344213, 0.0000000000000000, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33858381342073240, 0.0000000000000000, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49123285640844738, 0.0000000000000000, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.62944854858904520, 0.0000000000000000, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.75421778305499343, 0.0000000000000000, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.86790634112156639, 0.0000000000000000, 0.80000000000000004, 
          1.0471975511965976 },
  { 0.97334918087427558, 0.0000000000000000, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.0734012615283985, 0.0000000000000000, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.1708024551734544, 0.0000000000000000, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.80000000000000004.
template <typename Tp>
void test099()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data099)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data099[i].k), Tp(data099[i].nu),
                   Tp(data099[i].phi));
      const Tp f0 = data099[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.0000000000000000, nu=0.90000000000000002.
testcase_ellint_3<double> data100[] = {
  { -0.0000000000000000, 0.0000000000000000, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17297300811030600, 0.0000000000000000, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33735034635360817, 0.0000000000000000, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.48765675230233141, 0.0000000000000000, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62238126886123568, 0.0000000000000000, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.74286600807269243, 0.0000000000000000, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.85181283909264971, 0.0000000000000000, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.95228683995371122, 0.0000000000000000, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0472730487412552, 0.0000000000000000, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1395754288497419, 0.0000000000000000, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.0000000000000000, nu=0.90000000000000002.
template <typename Tp>
void test100()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data100)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data100[i].k), Tp(data100[i].nu),
                   Tp(data100[i].phi));
      const Tp f0 = data100[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.0000000000000000.
testcase_ellint_3<double> data101[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17454173353063665, 0.10000000000000009, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.34913506721468085, 0.10000000000000009, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52382550016538953, 0.10000000000000009, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.69864700854177031, 0.10000000000000009, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.87361792586964870, 0.10000000000000009, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0487386319621685, 0.10000000000000009, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2239913752078757, 0.10000000000000009, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.3993423113684049, 0.10000000000000009, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.5747455615173562, 0.10000000000000009, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.0000000000000000.
template <typename Tp>
void test101()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data101)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data101[i].k), Tp(data101[i].nu),
                   Tp(data101[i].phi));
      const Tp f0 = data101[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.10000000000000001.
testcase_ellint_3<double> data102[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17436589347616618, 0.10000000000000009, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34776067871237354, 0.10000000000000009, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.51936064354727807, 0.10000000000000009, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.68860303749364360, 0.10000000000000009, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.85524561882332051, 0.10000000000000009, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0193708301908337, 0.10000000000000009, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.1813474067123044, 0.10000000000000009, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3417670770424983, 0.10000000000000009, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5013711111199950, 0.10000000000000009, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.10000000000000001.
template <typename Tp>
void test102()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data102)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data102[i].k), Tp(data102[i].nu),
                   Tp(data102[i].phi));
      const Tp f0 = data102[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.20000000000000001.
testcase_ellint_3<double> data103[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17419068786141345, 0.10000000000000009, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34640537686230127, 0.10000000000000009, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51502689171753957, 0.10000000000000009, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.67904147863672726, 0.10000000000000009, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.83811885126105179, 0.10000000000000009, 0.20000000000000001, 
          0.87266462599716477 },
  { 0.99255278555742810, 0.10000000000000009, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1431260546194930, 0.10000000000000009, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.2909589656532101, 0.10000000000000009, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4373749386463430, 0.10000000000000009, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.20000000000000001.
template <typename Tp>
void test103()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data103)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data103[i].k), Tp(data103[i].nu),
                   Tp(data103[i].phi));
      const Tp f0 = data103[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.29999999999999999.
testcase_ellint_3<double> data104[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17401611261390110, 0.10000000000000009, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34506869507511767, 0.10000000000000009, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51081757604259870, 0.10000000000000009, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.66992297597712303, 0.10000000000000009, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.82209722856174228, 0.10000000000000009, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.96792430487669612, 0.10000000000000009, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1085964108954092, 0.10000000000000009, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2456748370836999, 0.10000000000000009, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.3809159606704959, 0.10000000000000009, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.29999999999999999.
template <typename Tp>
void test104()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data104)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data104[i].k), Tp(data104[i].nu),
                   Tp(data104[i].phi));
      const Tp f0 = data104[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.40000000000000002.
testcase_ellint_3<double> data105[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17384216369897937, 0.10000000000000009, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34375018311376782, 0.10000000000000009, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50672650758380455, 0.10000000000000009, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66121264213337616, 0.10000000000000009, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.80706202005774441, 0.10000000000000009, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.94519376138245892, 0.10000000000000009, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.0771880300759584, 0.10000000000000009, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2049711557188272, 0.10000000000000009, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3306223265207477, 0.10000000000000009, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.40000000000000002.
template <typename Tp>
void test105()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data105)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data105[i].k), Tp(data105[i].nu),
                   Tp(data105[i].phi));
      const Tp f0 = data105[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.50000000000000000.
testcase_ellint_3<double> data106[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17366883711936554, 0.10000000000000009, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34244940634881876, 0.10000000000000009, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50274793281634378, 0.10000000000000009, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65287941633275093, 0.10000000000000009, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.79291198790315398, 0.10000000000000009, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.92412201537880345, 0.10000000000000009, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0484480076799370, 0.10000000000000009, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.1681168130475206, 0.10000000000000009, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.2854480708580160, 0.10000000000000009, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.50000000000000000.
template <typename Tp>
void test106()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data106)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data106[i].k), Tp(data106[i].nu),
                   Tp(data106[i].phi));
      const Tp f0 = data106[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.59999999999999998.
testcase_ellint_3<double> data107[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17349612891469018, 0.10000000000000009, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34116594505539438, 0.10000000000000009, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.49887649430466685, 0.10000000000000009, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.64489553282165157, 0.10000000000000009, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.77956016553782437, 0.10000000000000009, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.90451074530096309, 0.10000000000000009, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0220113666961632, 0.10000000000000009, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1345351441065563, 0.10000000000000009, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2445798942989255, 0.10000000000000009, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.59999999999999998.
template <typename Tp>
void test107()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data107)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data107[i].k), Tp(data107[i].nu),
                   Tp(data107[i].phi));
      const Tp f0 = data107[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.69999999999999996.
testcase_ellint_3<double> data108[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17332403516105052, 0.10000000000000009, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.33989939374896877, 0.10000000000000009, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49510719568614081, 0.10000000000000009, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.63723607776354974, 0.10000000000000009, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.76693133887935327, 0.10000000000000009, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.88619382078823827, 0.10000000000000009, 0.69999999999999996, 
          1.0471975511965976 },
  { 0.99758012018676490, 0.10000000000000009, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1037642270814410, 0.10000000000000009, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2073745911083187, 0.10000000000000009, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.69999999999999996.
template <typename Tp>
void test108()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data108)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data108[i].k), Tp(data108[i].nu),
                   Tp(data108[i].phi));
      const Tp f0 = data108[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.80000000000000004.
testcase_ellint_3<double> data109[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17315255197057020, 0.10000000000000009, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33864936055747985, 0.10000000000000009, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49143537041117619, 0.10000000000000009, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.62987861760047492, 0.10000000000000009, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.75496005490917517, 0.10000000000000009, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.86903081862701903, 0.10000000000000009, 0.80000000000000004, 
          1.0471975511965976 },
  { 0.97490814820725591, 0.10000000000000009, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.0754290107171083, 0.10000000000000009, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.1733158866987732, 0.10000000000000009, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.80000000000000004.
template <typename Tp>
void test109()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data109)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data109[i].k), Tp(data109[i].nu),
                   Tp(data109[i].phi));
      const Tp f0 = data109[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.10000000000000009, nu=0.90000000000000002.
testcase_ellint_3<double> data110[] = {
  { -0.0000000000000000, 0.10000000000000009, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17298167549096569, 0.10000000000000009, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33741546662741584, 0.10000000000000009, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.48785665376856879, 0.10000000000000009, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62280288554518959, 0.10000000000000009, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.74358903115455199, 0.10000000000000009, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.85290207679298358, 0.10000000000000009, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.95379006645397379, 0.10000000000000009, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0492213119872327, 0.10000000000000009, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1419839485283374, 0.10000000000000009, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.10000000000000009, nu=0.90000000000000002.
template <typename Tp>
void test110()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data110)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data110[i].k), Tp(data110[i].nu),
                   Tp(data110[i].phi));
      const Tp f0 = data110[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.0000000000000000.
testcase_ellint_3<double> data111[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17456817290292811, 0.19999999999999996, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.34934315932086801, 0.19999999999999996, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52450880529443988, 0.19999999999999996, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.70020491009844910, 0.19999999999999996, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.87651006649967955, 0.19999999999999996, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0534305870298994, 0.19999999999999996, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2308975521670784, 0.19999999999999996, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4087733584990738, 0.19999999999999996, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.5868678474541664, 0.19999999999999996, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.0000000000000000.
template <typename Tp>
void test111()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data111)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data111[i].k), Tp(data111[i].nu),
                   Tp(data111[i].phi));
      const Tp f0 = data111[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.10000000000000001.
testcase_ellint_3<double> data112[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17439228502691750, 0.19999999999999996, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34796731137565740, 0.19999999999999996, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52003370294544848, 0.19999999999999996, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.69012222258631495, 0.19999999999999996, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.85803491465566772, 0.19999999999999996, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0238463961099364, 0.19999999999999996, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.1878691059202153, 0.19999999999999996, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3505985031831940, 0.19999999999999996, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5126513474261092, 0.19999999999999996, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.10000000000000001.
template <typename Tp>
void test112()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data112)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data112[i].k), Tp(data112[i].nu),
                   Tp(data112[i].phi));
      const Tp f0 = data112[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.20000000000000001.
testcase_ellint_3<double> data113[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17421703179583750, 0.19999999999999996, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34661057411998791, 0.19999999999999996, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51569006052647393, 0.19999999999999996, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.68052412821107278, 0.19999999999999996, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.84081341263313825, 0.19999999999999996, 0.20000000000000001, 
          0.87266462599716477 },
  { 0.99683359988842890, 0.19999999999999996, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1493086715118852, 0.19999999999999996, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.2992699693957541, 0.19999999999999996, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4479323932249568, 0.19999999999999996, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.20000000000000001.
template <typename Tp>
void test113()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data113)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data113[i].k), Tp(data113[i].nu),
                   Tp(data113[i].phi));
      const Tp f0 = data113[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.29999999999999999.
testcase_ellint_3<double> data114[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17404240913577707, 0.19999999999999996, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34527248032587193, 0.19999999999999996, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51147118981668416, 0.19999999999999996, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.67137107867777635, 0.19999999999999996, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.82470418188668893, 0.19999999999999996, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.97202873223594299, 0.19999999999999996, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1144773569375266, 0.19999999999999996, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2535292433701000, 0.19999999999999996, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.3908453514752481, 0.19999999999999996, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.29999999999999999.
template <typename Tp>
void test114()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data114)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data114[i].k), Tp(data114[i].nu),
                   Tp(data114[i].phi));
      const Tp f0 = data114[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.40000000000000002.
testcase_ellint_3<double> data115[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17386841301066677, 0.19999999999999996, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34395257914113253, 0.19999999999999996, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50737088376869466, 0.19999999999999996, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66262801717277664, 0.19999999999999996, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.80958766645079094, 0.19999999999999996, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.94913754236162040, 0.19999999999999996, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.0827985514223000, 0.19999999999999996, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2124212429050478, 0.19999999999999996, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3400002519661010, 0.19999999999999996, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.40000000000000002.
template <typename Tp>
void test115()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data115)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data115[i].k), Tp(data115[i].nu),
                   Tp(data115[i].phi));
      const Tp f0 = data115[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.50000000000000000.
testcase_ellint_3<double> data116[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17369503942181802, 0.19999999999999996, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34265043534362660, 0.19999999999999996, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50338337208655415, 0.19999999999999996, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65426373297163642, 0.19999999999999996, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.79536193036145808, 0.19999999999999996, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.92791875910061605, 0.19999999999999996, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0538145052725829, 0.19999999999999996, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.1752060022875899, 0.19999999999999996, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.2943374404397376, 0.19999999999999996, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.50000000000000000.
template <typename Tp>
void test116()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data116)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data116[i].k), Tp(data116[i].nu),
                   Tp(data116[i].phi));
      const Tp f0 = data116[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.59999999999999998.
testcase_ellint_3<double> data117[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17352228440746928, 0.19999999999999996, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34136562863713626, 0.19999999999999996, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.49950328177638481, 0.19999999999999996, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.64625032705690832, 0.19999999999999996, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.78193941198403094, 0.19999999999999996, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.90817230934317128, 0.19999999999999996, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0271563751276462, 0.19999999999999996, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1412999379040518, 0.19999999999999996, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2530330675914561, 0.19999999999999996, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.59999999999999998.
template <typename Tp>
void test117()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data117)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data117[i].k), Tp(data117[i].nu),
                   Tp(data117[i].phi));
      const Tp f0 = data117[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.69999999999999996.
testcase_ellint_3<double> data118[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17335014404233898, 0.19999999999999996, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34009775298617811, 0.19999999999999996, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49572560201923810, 0.19999999999999996, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.63856276669886525, 0.19999999999999996, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.76924438644867565, 0.19999999999999996, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.88973060843856466, 0.19999999999999996, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0025230471636377, 0.19999999999999996, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1102356376093103, 0.19999999999999996, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2154356555075867, 0.19999999999999996, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.69999999999999996.
template <typename Tp>
void test118()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data118)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data118[i].k), Tp(data118[i].nu),
                   Tp(data118[i].phi));
      const Tp f0 = data118[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.80000000000000004.
testcase_ellint_3<double> data119[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17317861443718541, 0.19999999999999996, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33884641598718701, 0.19999999999999996, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49204565281259494, 0.19999999999999996, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.63117851188220353, 0.19999999999999996, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.75721095949544170, 0.19999999999999996, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.87245201443919118, 0.19999999999999996, 0.80000000000000004, 
          1.0471975511965976 },
  { 0.97966584238831089, 0.19999999999999996, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.0816336325174360, 0.19999999999999996, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.1810223448909913, 0.19999999999999996, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.80000000000000004.
template <typename Tp>
void test119()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data119)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data119[i].k), Tp(data119[i].nu),
                   Tp(data119[i].phi));
      const Tp f0 = data119[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.19999999999999996, nu=0.90000000000000002.
testcase_ellint_3<double> data120[] = {
  { -0.0000000000000000, 0.19999999999999996, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17300769173837280, 0.19999999999999996, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33761123827372508, 0.19999999999999996, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.48845905690769426, 0.19999999999999996, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62407720017324986, 0.19999999999999996, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.74578146525124289, 0.19999999999999996, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.85621583540073076, 0.19999999999999996, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.95837725988001199, 0.19999999999999996, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0551821412633928, 0.19999999999999996, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1493679916141863, 0.19999999999999996, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.19999999999999996, nu=0.90000000000000002.
template <typename Tp>
void test120()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data120)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data120[i].k), Tp(data120[i].nu),
                   Tp(data120[i].phi));
      const Tp f0 = data120[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.0000000000000000.
testcase_ellint_3<double> data121[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17461228653000102, 0.30000000000000004, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.34969146102798415, 0.30000000000000004, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52565822873726320, 0.30000000000000004, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.70284226512408532, 0.30000000000000004, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.88144139195111182, 0.30000000000000004, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0614897067260523, 0.30000000000000004, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2428416824174218, 0.30000000000000004, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4251795877015925, 0.30000000000000004, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.6080486199305126, 0.30000000000000004, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.0000000000000000.
template <typename Tp>
void test121()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data121)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data121[i].k), Tp(data121[i].nu),
                   Tp(data121[i].phi));
      const Tp f0 = data121[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.10000000000000001.
testcase_ellint_3<double> data122[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17443631884814378, 0.30000000000000004, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34831316835124926, 0.30000000000000004, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52116586276523857, 0.30000000000000004, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.69269385837910036, 0.30000000000000004, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.86279023163070856, 0.30000000000000004, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0315321461438265, 0.30000000000000004, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.1991449111869024, 0.30000000000000004, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3659561780923211, 0.30000000000000004, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5323534693557526, 0.30000000000000004, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.10000000000000001.
template <typename Tp>
void test122()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data122)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data122[i].k), Tp(data122[i].nu),
                   Tp(data122[i].phi));
      const Tp f0 = data122[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.20000000000000001.
testcase_ellint_3<double> data123[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17426098615372090, 0.30000000000000004, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34695402664689923, 0.30000000000000004, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51680555567038933, 0.30000000000000004, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.68303375225260210, 0.30000000000000004, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.84540662891295026, 0.30000000000000004, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0041834051646927, 0.30000000000000004, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1599952702345711, 0.30000000000000004, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.3137179520499163, 0.30000000000000004, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4663658145259875, 0.30000000000000004, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.20000000000000001.
template <typename Tp>
void test123()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data123)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data123[i].k), Tp(data123[i].nu),
                   Tp(data123[i].phi));
      const Tp f0 = data123[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.29999999999999999.
testcase_ellint_3<double> data124[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17408628437042845, 0.30000000000000004, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34561356761638401, 0.30000000000000004, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51257058617875850, 0.30000000000000004, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.67382207124602866, 0.30000000000000004, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.82914751587825131, 0.30000000000000004, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.97907434814374950, 0.30000000000000004, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1246399297351584, 0.30000000000000004, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2671793970398146, 0.30000000000000004, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.4081767433479089, 0.30000000000000004, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.29999999999999999.
template <typename Tp>
void test124()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data124)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data124[i].k), Tp(data124[i].nu),
                   Tp(data124[i].phi));
      const Tp f0 = data124[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.40000000000000002.
testcase_ellint_3<double> data125[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17391220945982730, 0.30000000000000004, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34429133937639689, 0.30000000000000004, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50845471668581632, 0.30000000000000004, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66502347027873854, 0.30000000000000004, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.81389191978012254, 0.30000000000000004, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.95590618002140593, 0.30000000000000004, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.0924915195213121, 0.30000000000000004, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2253651604038058, 0.30000000000000004, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3563643538969761, 0.30000000000000004, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.40000000000000002.
template <typename Tp>
void test125()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data125)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data125[i].k), Tp(data125[i].nu),
                   Tp(data125[i].phi));
      const Tp f0 = data125[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.50000000000000000.
testcase_ellint_3<double> data126[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17373875742088235, 0.30000000000000004, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34298690571124157, 0.30000000000000004, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50445214859646936, 0.30000000000000004, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65660648352418516, 0.30000000000000004, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.79953670639287289, 0.30000000000000004, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.93443393926588558, 0.30000000000000004, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0630838369016911, 0.30000000000000004, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.1875197325653026, 0.30000000000000004, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.3098448759814960, 0.30000000000000004, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.50000000000000000.
template <typename Tp>
void test126()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data126)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data126[i].k), Tp(data126[i].nu),
                   Tp(data126[i].phi));
      const Tp f0 = data126[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.59999999999999998.
testcase_ellint_3<double> data127[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17356592428950826, 0.30000000000000004, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34169984536697379, 0.30000000000000004, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50055748266498457, 0.30000000000000004, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.64854298527106768, 0.30000000000000004, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.78599329284207431, 0.30000000000000004, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.91445452089128221, 0.30000000000000004, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0360412952290587, 0.30000000000000004, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1530473919778639, 0.30000000000000004, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2677758800420666, 0.30000000000000004, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.59999999999999998.
template <typename Tp>
void test127()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data127)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data127[i].k), Tp(data127[i].nu),
                   Tp(data127[i].phi));
      const Tp f0 = data127[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.69999999999999996.
testcase_ellint_3<double> data128[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17339370613812227, 0.30000000000000004, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34042975138455933, 0.30000000000000004, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49676568368075985, 0.30000000000000004, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.64080774055753720, 0.30000000000000004, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.77318507779667278, 0.30000000000000004, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.89579782346548631, 0.30000000000000004, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0110573286052202, 0.30000000000000004, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1214710972949633, 0.30000000000000004, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2294913236274980, 0.30000000000000004, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.69999999999999996.
template <typename Tp>
void test128()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data128)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data128[i].k), Tp(data128[i].nu),
                   Tp(data128[i].phi));
      const Tp f0 = data128[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.80000000000000004.
testcase_ellint_3<double> data129[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17322209907520361, 0.30000000000000004, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33917623046949996, 0.30000000000000004, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49307204894329176, 0.30000000000000004, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.63337802830291723, 0.30000000000000004, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.76104540997689407, 0.30000000000000004, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.87832009635450736, 0.30000000000000004, 0.80000000000000004, 
          1.0471975511965976 },
  { 0.98787879723171790, 0.30000000000000004, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.0924036340069336, 0.30000000000000004, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.1944567571590046, 0.30000000000000004, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.80000000000000004.
template <typename Tp>
void test129()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data129)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data129[i].k), Tp(data129[i].nu),
                   Tp(data129[i].phi));
      const Tp f0 = data129[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.30000000000000004, nu=0.90000000000000002.
testcase_ellint_3<double> data130[] = {
  { -0.0000000000000000, 0.30000000000000004, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17305109924485948, 0.30000000000000004, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33793890239556984, 0.30000000000000004, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.48947218005089738, 0.30000000000000004, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62623332340775151, 0.30000000000000004, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.74951596581511148, 0.30000000000000004, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.86189886597756005, 0.30000000000000004, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.96629451153092005, 0.30000000000000004, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0655269133492680, 0.30000000000000004, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1622376896064912, 0.30000000000000004, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.30000000000000004, nu=0.90000000000000002.
template <typename Tp>
void test130()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data130)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data130[i].k), Tp(data130[i].nu),
                   Tp(data130[i].phi));
      const Tp f0 = data130[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.0000000000000000.
testcase_ellint_3<double> data131[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17467414669441531, 0.39999999999999991, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35018222772483443, 0.39999999999999991, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52729015917508748, 0.39999999999999991, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.70662374407341255, 0.39999999999999991, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.88859210497602170, 0.39999999999999991, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0733136290471379, 0.39999999999999991, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2605612170157061, 0.39999999999999991, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4497513956433437, 0.39999999999999991, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.6399998658645112, 0.39999999999999991, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.0000000000000000.
template <typename Tp>
void test131()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data131)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data131[i].k), Tp(data131[i].nu),
                   Tp(data131[i].phi));
      const Tp f0 = data131[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.10000000000000001.
testcase_ellint_3<double> data132[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17449806706684673, 0.39999999999999991, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34880048623856075, 0.39999999999999991, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52277322065757403, 0.39999999999999991, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.69638072056918376, 0.39999999999999991, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.86968426619831540, 0.39999999999999991, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0428044206578095, 0.39999999999999991, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.2158651158274378, 0.39999999999999991, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.3889447129893322, 0.39999999999999991, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.5620566886683604, 0.39999999999999991, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.10000000000000001.
template <typename Tp>
void test132()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data132)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data132[i].k), Tp(data132[i].nu),
                   Tp(data132[i].phi));
      const Tp f0 = data132[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.20000000000000001.
testcase_ellint_3<double> data133[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17432262290723399, 0.39999999999999991, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34743795258968596, 0.39999999999999991, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.51838919472805123, 0.39999999999999991, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.68663134739057918, 0.39999999999999991, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.85206432981833979, 0.39999999999999991, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0149595349004430, 0.39999999999999991, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1758349405464676, 0.39999999999999991, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.3353337673882635, 0.39999999999999991, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.4941414344266770, 0.39999999999999991, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.20000000000000001.
template <typename Tp>
void test133()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data133)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data133[i].k), Tp(data133[i].nu),
                   Tp(data133[i].phi));
      const Tp f0 = data133[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.29999999999999999.
testcase_ellint_3<double> data134[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17414781013591543, 0.39999999999999991, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34609415696777285, 0.39999999999999991, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51413131295862546, 0.39999999999999991, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.67733527622935630, 0.39999999999999991, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.83558675182733266, 0.39999999999999991, 0.29999999999999999, 
          0.87266462599716477 },
  { 0.98940140808865906, 0.39999999999999991, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1396968797728058, 0.39999999999999991, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.2875920037865087, 0.39999999999999991, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.4342789859950078, 0.39999999999999991, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.29999999999999999.
template <typename Tp>
void test134()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data134)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data134[i].k), Tp(data134[i].nu),
                   Tp(data134[i].phi));
      const Tp f0 = data134[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.40000000000000002.
testcase_ellint_3<double> data135[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17397362471112710, 0.39999999999999991, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34476864603333196, 0.39999999999999991, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.50999329415379357, 0.39999999999999991, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.66845674551396017, 0.39999999999999991, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.82012848346231748, 0.39999999999999991, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.96582449258349057, 0.39999999999999991, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1068473749476286, 0.39999999999999991, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2447132729159986, 0.39999999999999991, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.3809986210732901, 0.39999999999999991, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.40000000000000002.
template <typename Tp>
void test135()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data135)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data135[i].k), Tp(data135[i].nu),
                   Tp(data135[i].phi));
      const Tp f0 = data135[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.50000000000000000.
testcase_ellint_3<double> data136[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17380006262854139, 0.39999999999999991, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34346098216756610, 0.39999999999999991, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50596929935059420, 0.39999999999999991, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.65996392089131262, 0.39999999999999991, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.80558463511364786, 0.39999999999999991, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.94397834522857704, 0.39999999999999991, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0768075114108115, 0.39999999999999991, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.2059184624251329, 0.39999999999999991, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.3331797176377398, 0.39999999999999991, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.50000000000000000.
template <typename Tp>
void test136()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data136)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data136[i].k), Tp(data136[i].nu),
                   Tp(data136[i].phi));
      const Tp f0 = data136[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.59999999999999998.
testcase_ellint_3<double> data137[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17362711992081248, 0.39999999999999991, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34217074276403953, 0.39999999999999991, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50205389185761617, 0.39999999999999991, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.65182834920372745, 0.39999999999999991, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.79186512820565136, 0.39999999999999991, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.92365535916287134, 0.39999999999999991, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0491915663957907, 0.39999999999999991, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1705934291745104, 0.39999999999999991, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.2899514672527024, 0.39999999999999991, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.59999999999999998.
template <typename Tp>
void test137()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data137)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data137[i].k), Tp(data137[i].nu),
                   Tp(data137[i].phi));
      const Tp f0 = data137[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.69999999999999996.
testcase_ellint_3<double> data138[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17345479265712871, 0.39999999999999991, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34089751955950354, 0.39999999999999991, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.49824200167361343, 0.39999999999999991, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.64402450341199413, 0.39999999999999991, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.77889207804122873, 0.39999999999999991, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.90468169720957992, 0.39999999999999991, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0236847823692916, 0.39999999999999991, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1382465247425164, 0.39999999999999991, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2506255923253344, 0.39999999999999991, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.69999999999999996.
template <typename Tp>
void test138()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data138)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data138[i].k), Tp(data138[i].nu),
                   Tp(data138[i].phi));
      const Tp f0 = data138[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.80000000000000004.
testcase_ellint_3<double> data139[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17328307694277156, 0.39999999999999991, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.33964091800132007, 0.39999999999999991, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49452889372467451, 0.39999999999999991, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.63652940095937327, 0.39999999999999991, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.76659772511159097, 0.39999999999999991, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.88691047977338111, 0.39999999999999991, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0000273200611640, 0.39999999999999991, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.1084787902188007, 0.39999999999999991, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.2146499565727209, 0.39999999999999991, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.80000000000000004.
template <typename Tp>
void test139()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data139)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data139[i].k), Tp(data139[i].nu),
                   Tp(data139[i].phi));
      const Tp f0 = data139[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.39999999999999991, nu=0.90000000000000002.
testcase_ellint_3<double> data140[] = {
  { -0.0000000000000000, 0.39999999999999991, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17311196891868130, 0.39999999999999991, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33840055664911906, 0.39999999999999991, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49091013944075340, 0.39999999999999991, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.62932228186809591, 0.39999999999999991, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.75492278323019801, 0.39999999999999991, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.87021659043854294, 0.39999999999999991, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.97800245228239246, 0.39999999999999991, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.0809625773173694, 0.39999999999999991, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.1815758115929846, 0.39999999999999991, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.39999999999999991, nu=0.90000000000000002.
template <typename Tp>
void test140()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data140)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data140[i].k), Tp(data140[i].nu),
                   Tp(data140[i].phi));
      const Tp f0 = data140[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.0000000000000000.
testcase_ellint_3<double> data141[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17475385514035785, 0.50000000000000000, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35081868470101585, 0.50000000000000000, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.52942862705190585, 0.50000000000000000, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.71164727562630326, 0.50000000000000000, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.89824523594227768, 0.50000000000000000, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.0895506700518851, 0.50000000000000000, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.2853005857432933, 0.50000000000000000, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.4845545520549484, 0.50000000000000000, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.6857503548125963, 0.50000000000000000, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.0000000000000000.
template <typename Tp>
void test141()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data141)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data141[i].k), Tp(data141[i].nu),
                   Tp(data141[i].phi));
      const Tp f0 = data141[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.10000000000000001.
testcase_ellint_3<double> data142[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17457763120814676, 0.50000000000000000, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.34943246340849154, 0.50000000000000000, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52487937869610801, 0.50000000000000000, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.70127785096388395, 0.50000000000000000, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.87898815988624479, 0.50000000000000000, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0582764576094172, 0.50000000000000000, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.2391936844060207, 0.50000000000000000, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.4214793542995841, 0.50000000000000000, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.6045524936084892, 0.50000000000000000, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.10000000000000001.
template <typename Tp>
void test142()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data142)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data142[i].k), Tp(data142[i].nu),
                   Tp(data142[i].phi));
      const Tp f0 = data142[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.20000000000000001.
testcase_ellint_3<double> data143[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17440204336345433, 0.50000000000000000, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34806552388338824, 0.50000000000000000, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.52046416757129821, 0.50000000000000000, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.69140924550993876, 0.50000000000000000, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.86104678636125520, 0.50000000000000000, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0297439459053981, 0.50000000000000000, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.1979214112912036, 0.50000000000000000, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.3659033858648930, 0.50000000000000000, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.5338490483665983, 0.50000000000000000, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.20000000000000001.
template <typename Tp>
void test143()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data143)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data143[i].k), Tp(data143[i].nu),
                   Tp(data143[i].phi));
      const Tp f0 = data143[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.29999999999999999.
testcase_ellint_3<double> data144[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17422708752228896, 0.50000000000000000, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34671739434855858, 0.50000000000000000, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51617616305641889, 0.50000000000000000, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.68200047612545178, 0.50000000000000000, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.84427217869498372, 0.50000000000000000, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0035637821389782, 0.50000000000000000, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1606800483933113, 0.50000000000000000, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.3164407134643459, 0.50000000000000000, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.4715681939859637, 0.50000000000000000, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.29999999999999999.
template <typename Tp>
void test144()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data144)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data144[i].k), Tp(data144[i].nu),
                   Tp(data144[i].phi));
      const Tp f0 = data144[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.40000000000000002.
testcase_ellint_3<double> data145[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17405275963859917, 0.50000000000000000, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34538761957029329, 0.50000000000000000, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.51200902646603919, 0.50000000000000000, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.67301522212868792, 0.50000000000000000, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.82853844466313320, 0.50000000000000000, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.97942097862681488, 0.50000000000000000, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1268429801220616, 0.50000000000000000, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.2720406704533922, 0.50000000000000000, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.4161679518465340, 0.50000000000000000, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.40000000000000002.
template <typename Tp>
void test145()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data145)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data145[i].k), Tp(data145[i].nu),
                   Tp(data145[i].phi));
      const Tp f0 = data145[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.50000000000000000.
testcase_ellint_3<double> data146[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17387905570381157, 0.50000000000000000, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34407576010465207, 0.50000000000000000, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.50795686560160835, 0.50000000000000000, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.66442115453330175, 0.50000000000000000, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.81373829119355345, 0.50000000000000000, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.95705743313235825, 0.50000000000000000, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.0959131991362556, 0.50000000000000000, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.2318900529754597, 0.50000000000000000, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.3664739530045971, 0.50000000000000000, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.50000000000000000.
template <typename Tp>
void test146()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data146)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data146[i].k), Tp(data146[i].nu),
                   Tp(data146[i].phi));
      const Tp f0 = data146[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.59999999999999998.
testcase_ellint_3<double> data147[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17370597174637581, 0.50000000000000000, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34278139158591414, 0.50000000000000000, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50401419439302719, 0.50000000000000000, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.65618938076167221, 0.50000000000000000, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.79977959248855424, 0.50000000000000000, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.93625925190753545, 0.50000000000000000, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0674905658379710, 0.50000000000000000, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.1953481298023048, 0.50000000000000000, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.3215740290190876, 0.50000000000000000, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.59999999999999998.
template <typename Tp>
void test147()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data147)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data147[i].k), Tp(data147[i].nu),
                   Tp(data147[i].phi));
      const Tp f0 = data147[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.69999999999999996.
testcase_ellint_3<double> data148[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17353350383131641, 0.50000000000000000, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34150410405436771, 0.50000000000000000, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50017589696443487, 0.50000000000000000, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.64829398188419962, 0.50000000000000000, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.78658270782402073, 0.50000000000000000, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.91684738336675053, 0.50000000000000000, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0412486789555937, 0.50000000000000000, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1619021847612001, 0.50000000000000000, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.2807475181182502, 0.50000000000000000, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.69999999999999996.
template <typename Tp>
void test148()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data148)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data148[i].k), Tp(data148[i].nu),
                   Tp(data148[i].phi));
      const Tp f0 = data148[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.80000000000000004.
testcase_ellint_3<double> data149[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17336164805979126, 0.50000000000000000, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34024350132086773, 0.50000000000000000, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49643719555734084, 0.50000000000000000, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.64071162456976150, 0.50000000000000000, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.77407836177211908, 0.50000000000000000, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.89867058251905652, 0.50000000000000000, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0169181822134912, 0.50000000000000000, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.1311363312779448, 0.50000000000000000, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.2434165408189539, 0.50000000000000000, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.80000000000000004.
template <typename Tp>
void test149()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data149)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data149[i].k), Tp(data149[i].nu),
                   Tp(data149[i].phi));
      const Tp f0 = data149[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.50000000000000000, nu=0.90000000000000002.
testcase_ellint_3<double> data150[] = {
  { -0.0000000000000000, 0.50000000000000000, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17319040056865681, 0.50000000000000000, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33899920036578557, 0.50000000000000000, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49279362182695186, 0.50000000000000000, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.63342123379746151, 0.50000000000000000, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.76220595179550321, 0.50000000000000000, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.88160004743532294, 0.50000000000000000, 0.90000000000000002, 
          1.0471975511965976 },
  { 0.99427448642310134, 0.50000000000000000, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.1027091512470093, 0.50000000000000000, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.2091116095504744, 0.50000000000000000, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.50000000000000000, nu=0.90000000000000002.
template <typename Tp>
void test150()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data150)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data150[i].k), Tp(data150[i].nu),
                   Tp(data150[i].phi));
      const Tp f0 = data150[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.0000000000000000.
testcase_ellint_3<double> data151[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17485154362988362, 0.60000000000000009, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35160509865544326, 0.60000000000000009, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.53210652578446160, 0.60000000000000009, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.71805304664485670, 0.60000000000000009, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.91082759030195970, 0.60000000000000009, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.1112333229323366, 0.60000000000000009, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.3191461190365270, 0.60000000000000009, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.5332022105084775, 0.60000000000000009, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.7507538029157526, 0.60000000000000009, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.0000000000000000.
template <typename Tp>
void test151()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data151)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data151[i].k), Tp(data151[i].nu),
                   Tp(data151[i].phi));
      const Tp f0 = data151[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.10000000000000001.
testcase_ellint_3<double> data152[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17467514275022014, 0.60000000000000009, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35021333086258255, 0.60000000000000009, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.52751664092962713, 0.60000000000000009, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.70752126971957885, 0.60000000000000009, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.89111058756112871, 0.60000000000000009, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.0789241202877773, 0.60000000000000009, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.2710800210399946, 0.60000000000000009, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.4669060574440278, 0.60000000000000009, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.6648615773343014, 0.60000000000000009, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.10000000000000001.
template <typename Tp>
void test152()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data152)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data152[i].k), Tp(data152[i].nu),
                   Tp(data152[i].phi));
      const Tp f0 = data152[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.20000000000000001.
testcase_ellint_3<double> data153[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17449937871800653, 0.60000000000000009, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34884093647346553, 0.60000000000000009, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.52306221119844110, 0.60000000000000009, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.69749955678982223, 0.60000000000000009, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.87274610682416853, 0.60000000000000009, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0494620540750796, 0.60000000000000009, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.2280847305507339, 0.60000000000000009, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.4085436279696888, 0.60000000000000009, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.5901418016279374, 0.60000000000000009, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.20000000000000001.
template <typename Tp>
void test153()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data153)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data153[i].k), Tp(data153[i].nu),
                   Tp(data153[i].phi));
      const Tp f0 = data153[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.29999999999999999.
testcase_ellint_3<double> data154[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17432424744393935, 0.60000000000000009, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34748744127146447, 0.60000000000000009, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.51873632743924847, 0.60000000000000009, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.68794610396313127, 0.60000000000000009, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.85558070175468726, 0.60000000000000009, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0224416343605658, 0.60000000000000009, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.1893144457936788, 0.60000000000000009, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.3566435377982575, 0.60000000000000009, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.5243814243493585, 0.60000000000000009, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.29999999999999999.
template <typename Tp>
void test154()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data154)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data154[i].k), Tp(data154[i].nu),
                   Tp(data154[i].phi));
      const Tp f0 = data154[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.40000000000000002.
testcase_ellint_3<double> data155[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17414974487670720, 0.60000000000000009, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34615238767335027, 0.60000000000000009, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.51453257838108579, 0.60000000000000009, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.67882386787534410, 0.60000000000000009, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.83948470233173578, 0.60000000000000009, 0.40000000000000002, 
          0.87266462599716477 },
  { 0.99753496200074021, 0.60000000000000009, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1541101404388487, 0.60000000000000009, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.3100911323398816, 0.60000000000000009, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.4659345278069984, 0.60000000000000009, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.40000000000000002.
template <typename Tp>
void test155()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data155)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data155[i].k), Tp(data155[i].nu),
                   Tp(data155[i].phi));
      const Tp f0 = data155[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.50000000000000000.
testcase_ellint_3<double> data156[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17397586700252810, 0.60000000000000009, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34483533397138516, 0.60000000000000009, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.51044500461706499, 0.60000000000000009, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.67009988034712675, 0.60000000000000009, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.82434762375735193, 0.60000000000000009, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.97447346702799043, 0.60000000000000009, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.1219494000522143, 0.60000000000000009, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.2680242605954488, 0.60000000000000009, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.4135484285693078, 0.60000000000000009, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.50000000000000000.
template <typename Tp>
void test156()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data156)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data156[i].k), Tp(data156[i].nu),
                   Tp(data156[i].phi));
      const Tp f0 = data156[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.59999999999999998.
testcase_ellint_3<double> data157[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17380260984469356, 0.60000000000000009, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34353585361777839, 0.60000000000000009, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50646805774321402, 0.60000000000000009, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.66174468108625517, 0.60000000000000009, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.81007462280278408, 0.60000000000000009, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.95303466945718773, 0.60000000000000009, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.0924118588677503, 0.60000000000000009, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.2297640574847937, 0.60000000000000009, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.3662507535812816, 0.60000000000000009, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.59999999999999998.
template <typename Tp>
void test157()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data157)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data157[i].k), Tp(data157[i].nu),
                   Tp(data157[i].phi));
      const Tp f0 = data157[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.69999999999999996.
testcase_ellint_3<double> data158[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17362996946312009, 0.60000000000000009, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34225353454870588, 0.60000000000000009, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50259656397799546, 0.60000000000000009, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.65373184496628944, 0.60000000000000009, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.79658372884056439, 0.60000000000000009, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.93303240100245466, 0.60000000000000009, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0651547944716557, 0.60000000000000009, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.1947676204853441, 0.60000000000000009, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.3232737468822811, 0.60000000000000009, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.69999999999999996.
template <typename Tp>
void test158()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data158)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data158[i].k), Tp(data158[i].nu),
                   Tp(data158[i].phi));
      const Tp f0 = data158[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.80000000000000004.
testcase_ellint_3<double> data159[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17345794195390687, 0.60000000000000009, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34098797854531027, 0.60000000000000009, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.49882569168826230, 0.60000000000000009, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.64603758566475511, 0.60000000000000009, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.78380365594769730, 0.60000000000000009, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.91430946255611223, 0.60000000000000009, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0398955217270607, 0.60000000000000009, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.1625948314277679, 0.60000000000000009, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.2840021261752192, 0.60000000000000009, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.80000000000000004.
template <typename Tp>
void test159()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data159)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data159[i].k), Tp(data159[i].nu),
                   Tp(data159[i].phi));
      const Tp f0 = data159[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.60000000000000009, nu=0.90000000000000002.
testcase_ellint_3<double> data160[] = {
  { -0.0000000000000000, 0.60000000000000009, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17328652344890033, 0.60000000000000009, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.33973880062929018, 0.60000000000000009, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49515092233122765, 0.60000000000000009, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.63864042139737043, 0.60000000000000009, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.77167205646538850, 0.60000000000000009, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.89673202848034428, 0.60000000000000009, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.0163984492661304, 0.60000000000000009, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.1328845785162431, 0.60000000000000009, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.2479362973851875, 0.60000000000000009, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.60000000000000009, nu=0.90000000000000002.
template <typename Tp>
void test160()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data160)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data160[i].k), Tp(data160[i].nu),
                   Tp(data160[i].phi));
      const Tp f0 = data160[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.0000000000000000.
testcase_ellint_3<double> data161[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17496737466916720, 0.69999999999999996, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35254687535677925, 0.69999999999999996, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.53536740275997130, 0.69999999999999996, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.72603797651684465, 0.69999999999999996, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.92698296348313458, 0.69999999999999996, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.1400447527693316, 0.69999999999999996, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.3657668117194071, 0.69999999999999996, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.6024686895959159, 0.69999999999999996, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.8456939983747236, 0.69999999999999996, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.0000000000000000.
template <typename Tp>
void test161()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data161)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data161[i].k), Tp(data161[i].nu),
                   Tp(data161[i].phi));
      const Tp f0 = data161[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.10000000000000001.
testcase_ellint_3<double> data162[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17479076384884681, 0.69999999999999996, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35114844900396364, 0.69999999999999996, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.53072776947527012, 0.69999999999999996, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.71530198262386246, 0.69999999999999996, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.90666760677828306, 0.69999999999999996, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.1063366517438080, 0.69999999999999996, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.3149477243092147, 0.69999999999999996, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.5314886725038925, 0.69999999999999996, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.7528050171757608, 0.69999999999999996, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.10000000000000001.
template <typename Tp>
void test162()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data162)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data162[i].k), Tp(data162[i].nu),
                   Tp(data162[i].phi));
      const Tp f0 = data162[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.20000000000000001.
testcase_ellint_3<double> data163[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17461479077791472, 0.69999999999999996, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.34976950621407538, 0.69999999999999996, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.52622533231350188, 0.69999999999999996, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.70508774017895226, 0.69999999999999996, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.88775302531730294, 0.69999999999999996, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.0756195476149006, 0.69999999999999996, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.2695349716654372, 0.69999999999999996, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.4690814617070540, 0.69999999999999996, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.6721098780092147, 0.69999999999999996, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.20000000000000001.
template <typename Tp>
void test163()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data163)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data163[i].k), Tp(data163[i].nu),
                   Tp(data163[i].phi));
      const Tp f0 = data163[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.29999999999999999.
testcase_ellint_3<double> data164[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17443945136076172, 0.69999999999999996, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34840956983535287, 0.69999999999999996, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.52185308551329179, 0.69999999999999996, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.69535240431168266, 0.69999999999999996, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.87007983473964923, 0.69999999999999996, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0474657975577066, 0.69999999999999996, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.2286225419931889, 0.69999999999999996, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.4136490671013271, 0.69999999999999996, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.6011813647733213, 0.69999999999999996, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.29999999999999999.
template <typename Tp>
void test164()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data164)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data164[i].k), Tp(data164[i].nu),
                   Tp(data164[i].phi));
      const Tp f0 = data164[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.40000000000000002.
testcase_ellint_3<double> data165[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17426474153983226, 0.69999999999999996, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34706817945773732, 0.69999999999999996, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.51760452851738159, 0.69999999999999996, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.68605801534722766, 0.69999999999999996, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.85351339387296532, 0.69999999999999996, 0.40000000000000002, 
          0.87266462599716477 },
  { 1.0215297967969537, 0.69999999999999996, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.1915051074460528, 0.69999999999999996, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.3639821911744707, 0.69999999999999996, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.5382162002954762, 0.69999999999999996, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.40000000000000002.
template <typename Tp>
void test165()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data165)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data165[i].k), Tp(data165[i].nu),
                   Tp(data165[i].phi));
      const Tp f0 = data165[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.50000000000000000.
testcase_ellint_3<double> data166[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17409065729516093, 0.69999999999999996, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34574489064986091, 0.69999999999999996, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.51347361925579793, 0.69999999999999996, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.67717079489579290, 0.69999999999999996, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.83793902055292280, 0.69999999999999996, 0.50000000000000000, 
          0.87266462599716477 },
  { 0.99752863545289705, 0.69999999999999996, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.1576240080401499, 0.69999999999999996, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.3191464023923762, 0.69999999999999996, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.4818433192178544, 0.69999999999999996, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.50000000000000000.
template <typename Tp>
void test166()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data166)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data166[i].k), Tp(data166[i].nu),
                   Tp(data166[i].phi));
      const Tp f0 = data166[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.59999999999999998.
testcase_ellint_3<double> data167[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17391719464391611, 0.69999999999999996, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34443927423869031, 0.69999999999999996, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.50945473266486074, 0.69999999999999996, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.66866056326513823, 0.69999999999999996, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.82325830002337352, 0.69999999999999996, 0.59999999999999998, 
          0.87266462599716477 },
  { 0.97522808245669357, 0.69999999999999996, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.1265300613705282, 0.69999999999999996, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.2784066076152003, 0.69999999999999996, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.4309994736080540, 0.69999999999999996, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.59999999999999998.
template <typename Tp>
void test167()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data167)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data167[i].k), Tp(data167[i].nu),
                   Tp(data167[i].phi));
      const Tp f0 = data167[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.69999999999999996.
testcase_ellint_3<double> data168[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17374434963995028, 0.69999999999999996, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34315091562900674, 0.69999999999999996, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50554262375653358, 0.69999999999999996, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.66050025406305812, 0.69999999999999996, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.80938620118847404, 0.69999999999999996, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.95443223855852144, 0.69999999999999996, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.0978573207128302, 0.69999999999999996, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.2411754575007123, 0.69999999999999996, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.3848459188329196, 0.69999999999999996, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.69999999999999996.
template <typename Tp>
void test168()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data168)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data168[i].k), Tp(data168[i].nu),
                   Tp(data168[i].phi));
      const Tp f0 = data168[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.80000000000000004.
testcase_ellint_3<double> data169[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17357211837335737, 0.69999999999999996, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34187941416012108, 0.69999999999999996, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.50173239465478270, 0.69999999999999996, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.65266550725988315, 0.69999999999999996, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.79624879865249298, 0.69999999999999996, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.93497577043296920, 0.69999999999999996, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.0713041566930748, 0.69999999999999996, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.2069772023255652, 0.69999999999999996, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.3427110650397533, 0.69999999999999996, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.80000000000000004.
template <typename Tp>
void test169()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data169)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data169[i].k), Tp(data169[i].nu),
                   Tp(data169[i].phi));
      const Tp f0 = data169[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.69999999999999996, nu=0.90000000000000002.
testcase_ellint_3<double> data170[] = {
  { -0.0000000000000000, 0.69999999999999996, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17340049697003634, 0.69999999999999996, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.34062438249741556, 0.69999999999999996, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.49801946510076878, 0.69999999999999996, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.64513432604750487, 0.69999999999999996, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.78378145487573758, 0.69999999999999996, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.91671799500854634, 0.69999999999999996, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.0466193579463123, 0.69999999999999996, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.1754218079199146, 0.69999999999999996, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.3040500499695911, 0.69999999999999996, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.69999999999999996, nu=0.90000000000000002.
template <typename Tp>
void test170()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data170)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data170[i].k), Tp(data170[i].nu),
                   Tp(data170[i].phi));
      const Tp f0 = data170[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.0000000000000000.
testcase_ellint_3<double> data171[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17510154241338902, 0.80000000000000004, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35365068839779390, 0.80000000000000004, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.53926804409084561, 0.80000000000000004, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.73587926028070383, 0.80000000000000004, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.94770942970071170, 0.80000000000000004, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.1789022995388239, 0.80000000000000004, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.4323027881876009, 0.80000000000000004, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.7069629739121674, 0.80000000000000004, 0.0000000000000000, 
          1.3962634015954636 },
  { 1.9953027776647296, 0.80000000000000004, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.0000000000000000.
template <typename Tp>
void test171()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data171)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data171[i].k), Tp(data171[i].nu),
                   Tp(data171[i].phi));
      const Tp f0 = data171[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.10000000000000001.
testcase_ellint_3<double> data172[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17492468824017166, 0.80000000000000004, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35224443521476911, 0.80000000000000004, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.53456851853226961, 0.80000000000000004, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.72488875602364944, 0.80000000000000004, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.92661354274638952, 0.80000000000000004, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.1432651144499077, 0.80000000000000004, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.3774479927211429, 0.80000000000000004, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.6287092337196041, 0.80000000000000004, 0.10000000000000001, 
          1.3962634015954636 },
  { 1.8910755418379521, 0.80000000000000004, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.10000000000000001.
template <typename Tp>
void test172()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data172)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data172[i].k), Tp(data172[i].nu),
                   Tp(data172[i].phi));
      const Tp f0 = data172[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.20000000000000001.
testcase_ellint_3<double> data173[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17474847286224943, 0.80000000000000004, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.35085779529084682, 0.80000000000000004, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.53000829263059157, 0.80000000000000004, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.71443466027453406, 0.80000000000000004, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.90698196872715420, 0.80000000000000004, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.1108198200558581, 0.80000000000000004, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.3284988909963957, 0.80000000000000004, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.5600369318140328, 0.80000000000000004, 0.20000000000000001, 
          1.3962634015954636 },
  { 1.8007226661734588, 0.80000000000000004, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.20000000000000001.
template <typename Tp>
void test173()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data173)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data173[i].k), Tp(data173[i].nu),
                   Tp(data173[i].phi));
      const Tp f0 = data173[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.29999999999999999.
testcase_ellint_3<double> data174[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17457289217669891, 0.80000000000000004, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.34949028801501258, 0.80000000000000004, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.52558024362769318, 0.80000000000000004, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.70447281740094914, 0.80000000000000004, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.88864745641528986, 0.80000000000000004, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.0811075819341465, 0.80000000000000004, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.2844589654082377, 0.80000000000000004, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.4991461361277849, 0.80000000000000004, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.7214611048717301, 0.80000000000000004, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.29999999999999999.
template <typename Tp>
void test174()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data174)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data174[i].k), Tp(data174[i].nu),
                   Tp(data174[i].phi));
      const Tp f0 = data174[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.40000000000000002.
testcase_ellint_3<double> data175[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17439794211872178, 0.80000000000000004, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34814144964568972, 0.80000000000000004, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.52127776285273075, 0.80000000000000004, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.69496411438966599, 0.80000000000000004, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.87146878427509589, 0.80000000000000004, 0.40000000000000002, 
          0.87266462599716477 },
  { 1.0537579024937762, 0.80000000000000004, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.2445534387922637, 0.80000000000000004, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.4446769766361993, 0.80000000000000004, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.6512267838651289, 0.80000000000000004, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.40000000000000002.
template <typename Tp>
void test175()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data175)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data175[i].k), Tp(data175[i].nu),
                   Tp(data175[i].phi));
      const Tp f0 = data175[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.50000000000000000.
testcase_ellint_3<double> data176[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17422361866118047, 0.80000000000000004, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34681083254170475, 0.80000000000000004, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.51709470815494440, 0.80000000000000004, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.68587375344080259, 0.80000000000000004, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.85532571852810624, 0.80000000000000004, 0.50000000000000000, 
          0.87266462599716477 },
  { 1.0284677391874906, 0.80000000000000004, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.2081693942686225, 0.80000000000000004, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.3955803006426311, 0.80000000000000004, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.5884528947755532, 0.80000000000000004, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.50000000000000000.
template <typename Tp>
void test176()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data176)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data176[i].k), Tp(data176[i].nu),
                   Tp(data176[i].phi));
      const Tp f0 = data176[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.59999999999999998.
testcase_ellint_3<double> data177[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17404991781414092, 0.80000000000000004, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34549800443625167, 0.80000000000000004, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.51302536167001556, 0.80000000000000004, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.67717065003912258, 0.80000000000000004, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.84011512421134416, 0.80000000000000004, 0.59999999999999998, 
          0.87266462599716477 },
  { 1.0049863847088742, 0.80000000000000004, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.1748145941898918, 0.80000000000000004, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.3510319699755071, 0.80000000000000004, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.5319262547427865, 0.80000000000000004, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.59999999999999998.
template <typename Tp>
void test177()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data177)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data177[i].k), Tp(data177[i].nu),
                   Tp(data177[i].phi));
      const Tp f0 = data177[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.69999999999999996.
testcase_ellint_3<double> data178[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17387683562442202, 0.80000000000000004, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34420254775101611, 0.80000000000000004, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.50906439222143685, 0.80000000000000004, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.66882693152688433, 0.80000000000000004, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.82574792844091316, 0.80000000000000004, 0.69999999999999996, 
          0.87266462599716477 },
  { 0.98310431309490953, 0.80000000000000004, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.1440884535113258, 0.80000000000000004, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.3103743938952537, 0.80000000000000004, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.4806912324625332, 0.80000000000000004, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.69999999999999996.
template <typename Tp>
void test178()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data178)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data178[i].k), Tp(data178[i].nu),
                   Tp(data178[i].phi));
      const Tp f0 = data178[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.80000000000000004.
testcase_ellint_3<double> data179[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17370436817515206, 0.80000000000000004, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34292405894783395, 0.80000000000000004, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.50520682176250087, 0.80000000000000004, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.66081751679736189, 0.80000000000000004, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.81214672249355102, 0.80000000000000004, 0.80000000000000004, 
          0.87266462599716477 },
  { 0.96264481387685574, 0.80000000000000004, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.1156611352656258, 0.80000000000000004, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.2730756225143889, 0.80000000000000004, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.4339837018309474, 0.80000000000000004, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.80000000000000004.
template <typename Tp>
void test179()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data179)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data179[i].k), Tp(data179[i].nu),
                   Tp(data179[i].phi));
      const Tp f0 = data179[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.80000000000000004, nu=0.90000000000000002.
testcase_ellint_3<double> data180[] = {
  { -0.0000000000000000, 0.80000000000000004, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17353251158533153, 0.80000000000000004, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.34166214791545768, 0.80000000000000004, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.50144799535130580, 0.80000000000000004, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.65311976193814447, 0.80000000000000004, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.79924384892320866, 0.80000000000000004, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.94345762353365625, 0.80000000000000004, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.0892582069219159, 0.80000000000000004, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.2387000876610268, 0.80000000000000004, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.3911845406776222, 0.80000000000000004, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.80000000000000004, nu=0.90000000000000002.
template <typename Tp>
void test180()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data180)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data180[i].k), Tp(data180[i].nu),
                   Tp(data180[i].phi));
      const Tp f0 = data180[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.0000000000000000.
testcase_ellint_3<double> data181[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.0000000000000000, 
          0.0000000000000000 },
  { 0.17525427376115027, 0.89999999999999991, 0.0000000000000000, 
          0.17453292519943295 },
  { 0.35492464591297446, 0.89999999999999991, 0.0000000000000000, 
          0.34906585039886590 },
  { 0.54388221416157123, 0.89999999999999991, 0.0000000000000000, 
          0.52359877559829882 },
  { 0.74797400423532501, 0.89999999999999991, 0.0000000000000000, 
          0.69813170079773179 },
  { 0.97463898451966458, 0.89999999999999991, 0.0000000000000000, 
          0.87266462599716477 },
  { 1.2334463254523438, 0.89999999999999991, 0.0000000000000000, 
          1.0471975511965976 },
  { 1.5355247765594910, 0.89999999999999991, 0.0000000000000000, 
          1.2217304763960306 },
  { 1.8882928567775117, 0.89999999999999991, 0.0000000000000000, 
          1.3962634015954636 },
  { 2.2805491384227699, 0.89999999999999991, 0.0000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.0000000000000000.
template <typename Tp>
void test181()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data181)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data181[i].k), Tp(data181[i].nu),
                   Tp(data181[i].phi));
      const Tp f0 = data181[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.10000000000000001.
testcase_ellint_3<double> data182[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.10000000000000001, 
          0.0000000000000000 },
  { 0.17507714233254659, 0.89999999999999991, 0.10000000000000001, 
          0.17453292519943295 },
  { 0.35350932904326521, 0.89999999999999991, 0.10000000000000001, 
          0.34906585039886590 },
  { 0.53911129989870987, 0.89999999999999991, 0.10000000000000001, 
          0.52359877559829882 },
  { 0.73666644254508407, 0.89999999999999991, 0.10000000000000001, 
          0.69813170079773179 },
  { 0.95250736612100184, 0.89999999999999991, 0.10000000000000001, 
          0.87266462599716477 },
  { 1.1950199550905591, 0.89999999999999991, 0.10000000000000001, 
          1.0471975511965976 },
  { 1.4741687286340848, 0.89999999999999991, 0.10000000000000001, 
          1.2217304763960306 },
  { 1.7968678183506053, 0.89999999999999991, 0.10000000000000001, 
          1.3962634015954636 },
  { 2.1537868513875282, 0.89999999999999991, 0.10000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.10000000000000001.
template <typename Tp>
void test182()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data182)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data182[i].k), Tp(data182[i].nu),
                   Tp(data182[i].phi));
      const Tp f0 = data182[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.20000000000000001.
testcase_ellint_3<double> data183[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.20000000000000001, 
          0.0000000000000000 },
  { 0.17490065089140930, 0.89999999999999991, 0.20000000000000001, 
          0.17453292519943295 },
  { 0.35211377590661436, 0.89999999999999991, 0.20000000000000001, 
          0.34906585039886590 },
  { 0.53448220334204111, 0.89999999999999991, 0.20000000000000001, 
          0.52359877559829882 },
  { 0.72591368943179591, 0.89999999999999991, 0.20000000000000001, 
          0.69813170079773179 },
  { 0.93192539780038763, 0.89999999999999991, 0.20000000000000001, 
          0.87266462599716477 },
  { 1.1600809679692681, 0.89999999999999991, 0.20000000000000001, 
          1.0471975511965976 },
  { 1.4195407225882508, 0.89999999999999991, 0.20000000000000001, 
          1.2217304763960306 },
  { 1.7168966476424521, 0.89999999999999991, 0.20000000000000001, 
          1.3962634015954636 },
  { 2.0443194576468890, 0.89999999999999991, 0.20000000000000001, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.20000000000000001.
template <typename Tp>
void test183()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data183)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data183[i].k), Tp(data183[i].nu),
                   Tp(data183[i].phi));
      const Tp f0 = data183[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.29999999999999999.
testcase_ellint_3<double> data184[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.29999999999999999, 
          0.0000000000000000 },
  { 0.17472479532647534, 0.89999999999999991, 0.29999999999999999, 
          0.17453292519943295 },
  { 0.35073750187374114, 0.89999999999999991, 0.29999999999999999, 
          0.34906585039886590 },
  { 0.52998766129466968, 0.89999999999999991, 0.29999999999999999, 
          0.52359877559829882 },
  { 0.71566993548699565, 0.89999999999999991, 0.29999999999999999, 
          0.69813170079773179 },
  { 0.91271517762560195, 0.89999999999999991, 0.29999999999999999, 
          0.87266462599716477 },
  { 1.1281241199843368, 0.89999999999999991, 0.29999999999999999, 
          1.0471975511965976 },
  { 1.3704929576917448, 0.89999999999999991, 0.29999999999999999, 
          1.2217304763960306 },
  { 1.6461981511487711, 0.89999999999999991, 0.29999999999999999, 
          1.3962634015954636 },
  { 1.9486280260314424, 0.89999999999999991, 0.29999999999999999, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.29999999999999999.
template <typename Tp>
void test184()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data184)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data184[i].k), Tp(data184[i].nu),
                   Tp(data184[i].phi));
      const Tp f0 = data184[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.40000000000000002.
testcase_ellint_3<double> data185[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.40000000000000002, 
          0.0000000000000000 },
  { 0.17454957156468839, 0.89999999999999991, 0.40000000000000002, 
          0.17453292519943295 },
  { 0.34938003933330430, 0.89999999999999991, 0.40000000000000002, 
          0.34906585039886590 },
  { 0.52562093533067444, 0.89999999999999991, 0.40000000000000002, 
          0.52359877559829882 },
  { 0.70589461324915681, 0.89999999999999991, 0.40000000000000002, 
          0.69813170079773179 },
  { 0.89472658511942849, 0.89999999999999991, 0.40000000000000002, 
          0.87266462599716477 },
  { 1.0987419542323438, 0.89999999999999991, 0.40000000000000002, 
          1.0471975511965976 },
  { 1.3261349565496301, 0.89999999999999991, 0.40000000000000002, 
          1.2217304763960306 },
  { 1.5831293909853761, 0.89999999999999991, 0.40000000000000002, 
          1.3962634015954636 },
  { 1.8641114227238347, 0.89999999999999991, 0.40000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.40000000000000002.
template <typename Tp>
void test185()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data185)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data185[i].k), Tp(data185[i].nu),
                   Tp(data185[i].phi));
      const Tp f0 = data185[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.50000000000000000.
testcase_ellint_3<double> data186[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.50000000000000000, 
          0.0000000000000000 },
  { 0.17437497557073336, 0.89999999999999991, 0.50000000000000000, 
          0.17453292519943295 },
  { 0.34804093691586013, 0.89999999999999991, 0.50000000000000000, 
          0.34906585039886590 },
  { 0.52137576320372903, 0.89999999999999991, 0.50000000000000000, 
          0.52359877559829882 },
  { 0.69655163996912262, 0.89999999999999991, 0.50000000000000000, 
          0.69813170079773179 },
  { 0.87783188683054236, 0.89999999999999991, 0.50000000000000000, 
          0.87266462599716477 },
  { 1.0716015959755183, 0.89999999999999991, 0.50000000000000000, 
          1.0471975511965976 },
  { 1.2857636916026749, 0.89999999999999991, 0.50000000000000000, 
          1.2217304763960306 },
  { 1.5264263913252358, 0.89999999999999991, 0.50000000000000000, 
          1.3962634015954636 },
  { 1.7888013241937859, 0.89999999999999991, 0.50000000000000000, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.50000000000000000.
template <typename Tp>
void test186()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data186)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data186[i].k), Tp(data186[i].nu),
                   Tp(data186[i].phi));
      const Tp f0 = data186[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.59999999999999998.
testcase_ellint_3<double> data187[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.59999999999999998, 
          0.0000000000000000 },
  { 0.17420100334657815, 0.89999999999999991, 0.59999999999999998, 
          0.17453292519943295 },
  { 0.34671975876122157, 0.89999999999999991, 0.59999999999999998, 
          0.34906585039886590 },
  { 0.51724631570707957, 0.89999999999999991, 0.59999999999999998, 
          0.52359877559829882 },
  { 0.68760879113743034, 0.89999999999999991, 0.59999999999999998, 
          0.69813170079773179 },
  { 0.86192157779698364, 0.89999999999999991, 0.59999999999999998, 
          0.87266462599716477 },
  { 1.0464279696166352, 0.89999999999999991, 0.59999999999999998, 
          1.0471975511965976 },
  { 1.2488156247094004, 0.89999999999999991, 0.59999999999999998, 
          1.2217304763960306 },
  { 1.4750988777188470, 0.89999999999999991, 0.59999999999999998, 
          1.3962634015954636 },
  { 1.7211781128919521, 0.89999999999999991, 0.59999999999999998, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.59999999999999998.
template <typename Tp>
void test187()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data187)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data187[i].k), Tp(data187[i].nu),
                   Tp(data187[i].phi));
      const Tp f0 = data187[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.69999999999999996.
testcase_ellint_3<double> data188[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.69999999999999996, 
          0.0000000000000000 },
  { 0.17402765093102210, 0.89999999999999991, 0.69999999999999996, 
          0.17453292519943295 },
  { 0.34541608382635131, 0.89999999999999991, 0.69999999999999996, 
          0.34906585039886590 },
  { 0.51322715827061693, 0.89999999999999991, 0.69999999999999996, 
          0.52359877559829882 },
  { 0.67903717872440283, 0.89999999999999991, 0.69999999999999996, 
          0.69813170079773179 },
  { 0.84690113601682671, 0.89999999999999991, 0.69999999999999996, 
          0.87266462599716477 },
  { 1.0229914311548416, 0.89999999999999991, 0.69999999999999996, 
          1.0471975511965976 },
  { 1.2148329639709381, 0.89999999999999991, 0.69999999999999996, 
          1.2217304763960306 },
  { 1.4283586501307799, 0.89999999999999991, 0.69999999999999996, 
          1.3962634015954636 },
  { 1.6600480747670936, 0.89999999999999991, 0.69999999999999996, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.69999999999999996.
template <typename Tp>
void test188()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data188)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data188[i].k), Tp(data188[i].nu),
                   Tp(data188[i].phi));
      const Tp f0 = data188[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.80000000000000004.
testcase_ellint_3<double> data189[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.80000000000000004, 
          0.0000000000000000 },
  { 0.17385491439925149, 0.89999999999999991, 0.80000000000000004, 
          0.17453292519943295 },
  { 0.34412950523113928, 0.89999999999999991, 0.80000000000000004, 
          0.34906585039886590 },
  { 0.50931321668729601, 0.89999999999999991, 0.80000000000000004, 
          0.52359877559829882 },
  { 0.67081081392296327, 0.89999999999999991, 0.80000000000000004, 
          0.69813170079773179 },
  { 0.83268846097293259, 0.89999999999999991, 0.80000000000000004, 
          0.87266462599716477 },
  { 1.0010985015814025, 0.89999999999999991, 0.80000000000000004, 
          1.0471975511965976 },
  { 1.1834394045489678, 0.89999999999999991, 0.80000000000000004, 
          1.2217304763960306 },
  { 1.3855695891683182, 0.89999999999999991, 0.80000000000000004, 
          1.3962634015954636 },
  { 1.6044591960982200, 0.89999999999999991, 0.80000000000000004, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.80000000000000004.
template <typename Tp>
void test189()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data189)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data189[i].k), Tp(data189[i].nu),
                   Tp(data189[i].phi));
      const Tp f0 = data189[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for k=0.89999999999999991, nu=0.90000000000000002.
testcase_ellint_3<double> data190[] = {
  { -0.0000000000000000, 0.89999999999999991, 0.90000000000000002, 
          0.0000000000000000 },
  { 0.17368278986240138, 0.89999999999999991, 0.90000000000000002, 
          0.17453292519943295 },
  { 0.34285962963961397, 0.89999999999999991, 0.90000000000000002, 
          0.34906585039886590 },
  { 0.50549974644993312, 0.89999999999999991, 0.90000000000000002, 
          0.52359877559829882 },
  { 0.66290623857720876, 0.89999999999999991, 0.90000000000000002, 
          0.69813170079773179 },
  { 0.81921183128847164, 0.89999999999999991, 0.90000000000000002, 
          0.87266462599716477 },
  { 0.98058481956066368, 0.89999999999999991, 0.90000000000000002, 
          1.0471975511965976 },
  { 1.1543223520473569, 0.89999999999999991, 0.90000000000000002, 
          1.2217304763960306 },
  { 1.3462119782292934, 0.89999999999999991, 0.90000000000000002, 
          1.3962634015954636 },
  { 1.5536420236310944, 0.89999999999999991, 0.90000000000000002, 
          1.5707963267948966 },
};

// Test function for k=0.89999999999999991, nu=0.90000000000000002.
template <typename Tp>
void test190()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data190)
                         / sizeof(testcase_ellint_3<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::ellint_3(Tp(data190[i].k), Tp(data190[i].nu),
                   Tp(data190[i].phi));
      const Tp f0 = data190[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

int main(int, char**)
{
  test001<double>();
  test002<double>();
  test003<double>();
  test004<double>();
  test005<double>();
  test006<double>();
  test007<double>();
  test008<double>();
  test009<double>();
  test010<double>();
  test011<double>();
  test012<double>();
  test013<double>();
  test014<double>();
  test015<double>();
  test016<double>();
  test017<double>();
  test018<double>();
  test019<double>();
  test020<double>();
  test021<double>();
  test022<double>();
  test023<double>();
  test024<double>();
  test025<double>();
  test026<double>();
  test027<double>();
  test028<double>();
  test029<double>();
  test030<double>();
  test031<double>();
  test032<double>();
  test033<double>();
  test034<double>();
  test035<double>();
  test036<double>();
  test037<double>();
  test038<double>();
  test039<double>();
  test040<double>();
  test041<double>();
  test042<double>();
  test043<double>();
  test044<double>();
  test045<double>();
  test046<double>();
  test047<double>();
  test048<double>();
  test049<double>();
  test050<double>();
  test051<double>();
  test052<double>();
  test053<double>();
  test054<double>();
  test055<double>();
  test056<double>();
  test057<double>();
  test058<double>();
  test059<double>();
  test060<double>();
  test061<double>();
  test062<double>();
  test063<double>();
  test064<double>();
  test065<double>();
  test066<double>();
  test067<double>();
  test068<double>();
  test069<double>();
  test070<double>();
  test071<double>();
  test072<double>();
  test073<double>();
  test074<double>();
  test075<double>();
  test076<double>();
  test077<double>();
  test078<double>();
  test079<double>();
  test080<double>();
  test081<double>();
  test082<double>();
  test083<double>();
  test084<double>();
  test085<double>();
  test086<double>();
  test087<double>();
  test088<double>();
  test089<double>();
  test090<double>();
  test091<double>();
  test092<double>();
  test093<double>();
  test094<double>();
  test095<double>();
  test096<double>();
  test097<double>();
  test098<double>();
  test099<double>();
  test100<double>();
  test101<double>();
  test102<double>();
  test103<double>();
  test104<double>();
  test105<double>();
  test106<double>();
  test107<double>();
  test108<double>();
  test109<double>();
  test110<double>();
  test111<double>();
  test112<double>();
  test113<double>();
  test114<double>();
  test115<double>();
  test116<double>();
  test117<double>();
  test118<double>();
  test119<double>();
  test120<double>();
  test121<double>();
  test122<double>();
  test123<double>();
  test124<double>();
  test125<double>();
  test126<double>();
  test127<double>();
  test128<double>();
  test129<double>();
  test130<double>();
  test131<double>();
  test132<double>();
  test133<double>();
  test134<double>();
  test135<double>();
  test136<double>();
  test137<double>();
  test138<double>();
  test139<double>();
  test140<double>();
  test141<double>();
  test142<double>();
  test143<double>();
  test144<double>();
  test145<double>();
  test146<double>();
  test147<double>();
  test148<double>();
  test149<double>();
  test150<double>();
  test151<double>();
  test152<double>();
  test153<double>();
  test154<double>();
  test155<double>();
  test156<double>();
  test157<double>();
  test158<double>();
  test159<double>();
  test160<double>();
  test161<double>();
  test162<double>();
  test163<double>();
  test164<double>();
  test165<double>();
  test166<double>();
  test167<double>();
  test168<double>();
  test169<double>();
  test170<double>();
  test171<double>();
  test172<double>();
  test173<double>();
  test174<double>();
  test175<double>();
  test176<double>();
  test177<double>();
  test178<double>();
  test179<double>();
  test180<double>();
  test181<double>();
  test182<double>();
  test183<double>();
  test184<double>();
  test185<double>();
  test186<double>();
  test187<double>();
  test188<double>();
  test189<double>();
  test190<double>();
  return 0;
}
