/****************************************************************************
  This file is part of the Freedom Remailer.  It is mostly borrowed from
  Matt Ghio's statistics functions as given in his remailer distribution,
  but significant changes have been made by John B. Fleming and Johannes
  Kroeger.  Changes are
  (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/file.h>
#include "freedom.h"

void
updatestats(enum stats_flag flag)
{
  char month[24][5];
  struct tm *curtime;
  time_t now;
  pid_t pid = getpid();
  FILE *file;
  int date[24], hour, currenthour, x, lockfd;
  int Tm, Tp;
  int Hm[24], Hp[24];

  now = time(NULL);
  curtime = localtime(&now);

  /* don't manipulate the stats file before locking stats.lock exclusively */
  lockfd = open("stats.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));

  if (!(file = fopen(STATS_FILE, "r"))) {
    /* STATS_FILE does not exist; make a new empty stats file */
    file = fopen(STATS_FILE, "w+");
    fprintf(file, "0\n");
    fprintf(file, "0 0\n");
    for (x = 0; x < 24; x++) /* tmp for month. First day, no messages */
      fprintf(file, "tmp 0 0 0\n");
    rewind(file);
  }
  fscanf(file, "%d", &hour);
  fscanf(file, "%d %d", &Tm, &Tp);
  for (x = 0; x < 24; x++)
    fscanf(file, "%s %d %d %d", month[x], &date[x], &Hm[x], &Hp[x]);
  fclose(file);

  currenthour = curtime->tm_hour;
  x = hour;
  while (x != currenthour) {
    if (x > 0) {
      strcpy(month[x], month[x - 1]);
      date[x] = date[x - 1];
    }
    else {
      strftime(month[0], 5, "%b", curtime);
      date[0] = curtime->tm_mday;
    }
    Hm[x] = Hp[x] = 0;
    x++;
    if (x > 23)
      x=0;
  }

  if (hour != currenthour) {
    Hm[hour] = Tm;
    Hp[hour] = Tp;
    Tm = Tp = 0;
  }

  if (flag == STATS_MESSAGE)
    Tm++;
  else if (flag == STATS_PGP)
    Tm++, Tp++;

  file = fopen(STATS_FILE, "w");
  fprintf(file, "%d\n", currenthour);
  fprintf(file, "%d %d\n", Tm, Tp);
  for (x = 0; x < 24; x++)
    fprintf(file, "%s %d %d %d\n", month[x], date[x], Hm[x], Hp[x]);
  fclose(file);
  flock(lockfd, LOCK_UN);
  close(lockfd);
}

void
mailstats(const char *to)
{
  char sendmail[BUFSIZ], month[24][5];
  struct tm *curtime;
  time_t now;
  pid_t pid = getpid();
  FILE *file;
  int date[24], hour, currenthour, x, lockfd;
  int Tm, Tp, x0, y, mmax = 0;
  int Hm[24], Hp[24];
  float mscale;

  now = time(NULL);
  curtime = localtime(&now);
  currenthour = curtime->tm_hour;


  /* don't manipulate the stats file before locking stats.lock exclusively */
  lockfd = open("stats.lock", O_WRONLY | O_CREAT | O_TRUNC, 0660);
  flock(lockfd, LOCK_EX);
  write(lockfd, &pid, sizeof(pid_t));

  if ((file = fopen(STATS_FILE, "r"))) {
    fscanf(file, "%d", &hour);
    fscanf(file, "%d %d", &Tm, &Tp);
    for (x = 0; x < 24; x++)
      fscanf(file, "%s %d %d %d", month[x], &date[x], &Hm[x], &Hp[x]);
    fclose(file);

    x = hour;
    while (x != currenthour) {
      if (x > 0) {
	strcpy(month[x], month[x - 1]);
	date[x] = date[x - 1];
      }
      else {
	strftime(month[0], 5, "%b", curtime);
	date[0] = curtime->tm_mday;
      }
      Hm[x] = Hp[x] = 0;
      x++;
      if (x > 23)
	x=0;
    }

    if (hour != currenthour) {
      Hm[hour] = Tm;
      Hp[hour] = Tp;
      Tm = Tp = 0;
    }

    sprintf(sendmail, "%s", SENDMAIL);
    file = popen(sendmail, "w");
    fprintf(file, "From: %s <%s>\n", REMAILER_NAME, REMAILER_ADDR);
    fprintf(file, "Subject: Cypherpunk Remailer Statistics\n");
    fprintf(file, "To: %s\n\n", to);
    fprintf(file, "Statistics for last 24 hours from anonymous remailer\n");
    fprintf(file, "%s\n\n", REMAILER_NAME);
    fprintf(file, "Number of messages per hour from "
		  "%s %2d %02d:00 to %s %2d %02d:59\n\n",
		  month[23], date[23], currenthour, month[0], date[0],
		  (currenthour + 23) % 24);
    for (x = 0; x < 24; x++)
      mmax = (Hm[x] > mmax) ? Hm[x] : mmax;
    mscale = (mmax > 67) ? 67.0 / mmax : 1.0;
    for (x0 = 0; x0 < 24; x0++) {
      x = (x0 + currenthour) % 24;
      fprintf(file, "%02d:00 (%3d) ", x, Hm[x]);
      if (Hm[x] > 0) {
	for (y = 0; y < Hp[x] * mscale; y++)
	  fprintf (file, "*");
	for (y = 0; y < (Hm[x] - Hp[x]) * mscale; y++)
	  fprintf (file, "+");
	Tm += Hm[x];
	Tp += Hp[x];
      }
      fprintf(file, "\n");
    }
    fprintf(file, "\n");
    fprintf(file, "Total messages received in last 24 hours: \t%5d\n", Tm);
    if (ALLOW_PGP)
      fprintf(file, "Number of PGP encrypted messages: \t\t%5d\n", Tp);
    fprintf(file, "Number of plaintext messages: \t\t\t%5d\n", Tm - Tp);
    pclose(file);
  }
  flock(lockfd, LOCK_UN);
  close(lockfd);
}

void
mailconf(const char *to)
{
  FILE *p;
  char sendmail[BUFSIZ];

  sprintf(sendmail, "%s", SENDMAIL);
  p = popen(sendmail, "w");
  fprintf(p, "From: %s <%s>\n", REMAILER_NAME, REMAILER_ADDR);
  fprintf(p, "Subject: Capabilities of the %s\n", REMAILER_NAME);
  fprintf(p, "To: %s\n\n", to);

  fprintf(p, "Remailer-Type: Freedom %s\n\n", VERSION);

  if (SIZE_LIMIT)
    fprintf(p, "Maximum message size: %d bytes\n", SIZE_LIMIT);
  else
    fprintf(p, "Maximum message size: unlimited\n");

  if (USE_MIX)
    fprintf(p, "Using Mixmaster reordering pool: supported\n");
  else
    fprintf(p, "Using Mixmaster reordering pool: not supported\n");

  if (ALLOW_LTIME)
    fprintf(p, "Queuing messages with Latent-Time: supported\n");
  else
    fprintf(p, "Queuing messages with Latent-Time: not supported\n");

  if (USE_STATS)
    fprintf(p, "Message statistics autoresponder: supported\n");
  else
    fprintf(p, "Message statistics autoresponder: not supported\n");

  if (ALLOW_PGP) {
    fprintf(p, "PGP encryption of incoming messages: supported\n");
    fprintf(p, "Message encryption with Encrypt-Key: supported\n");
  }
  else {
    fprintf(p, "PGP encryption of incoming messages: not supported\n");
    fprintf(p, "Message encryption with Encrypt-Key: not supported\n");
  }

  fprintf(p, "Subject encryption with Encrypt-Subject: supported\n");

  if (ALLOW_WWW) {
    fprintf(p, "Retrieving WWW files with Get-URL: supported\n");
#ifdef USE_RX
    fprintf(p, "Blocked URLs (regular expressions):\n");
#else /* !USE_RX */
    fprintf(p, "Blocked URLs:\n");
#endif /* !USE_RX */
    print_blocked(p, URL_BLOCK);
  }
  else
    fprintf(p, "Retrieving WWW files with Get-URL: not supported\n");

  if (ALLOW_POST) {
    if (ALLOW_POST == 1)
      fprintf(p, "News posting with Anon-Post-To: via local news server\n");
    else if (ALLOW_POST == 2)
      fprintf(p, "News posting with Anon-Post-To: via %s\n", MAIL2NEWS);
#ifdef USE_RX
    fprintf(p, "Blocked newsgroups (regular expressions):\n");
#else /* !USE_RX */
    fprintf(p, "Blocked newsgroups:\n");
#endif /* !USE_RX */
    print_blocked(p, GROUP_BLOCK);
  }
  else
    fprintf(p, "News posting with Anon-Post-To: not supported\n");

#ifdef USE_RX
    fprintf(p, "Blocked subjects (regular expressions):\n");
#else /* !USE_RX */
    fprintf(p, "Blocked subjects:\n");
#endif /* !USE_RX */
    print_blocked(p, SUBJ_BLOCK);

  pclose(p);
}
