/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997 Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/file.h>
#include "freedom.h"

int USE_SYSLOG = 0;
int USE_STATS = 1;
int USE_MIX = 0;
int ALLOW_PGP = 1;
int ALLOW_POST = 1;
int ALLOW_WWW = 1;
int ALLOW_LTIME = 1;
int SIZE_LIMIT = 0;
char remailer_dir[BUFSIZ] = "";
char cur_dir[BUFSIZ];
char MIX_DIR[BUFSIZ] = "";
char QUEUE_DIR[BUFSIZ] = "";
char PGP_DIR[BUFSIZ] = "";
char MAIL_SPOOL[BUFSIZ] = "/dev/null";
/* the -t option tells sendmail to expect a To: header in stdin */
char SENDMAIL[BUFSIZ] = "/usr/sbin/sendmail -t";
/* the -h option tells inews to expect message headers in stdin */
char INEWS[BUFSIZ] = "/usr/bin/inews -h";
char PGP[BUFSIZ] = "/usr/bin/pgp";
char LYNX[BUFSIZ] = "/usr/bin/lynx";
char HELP_FILE[BUFSIZ] = "freedom-help";
char KEY_FILE[BUFSIZ] = "freedom-key";
char STATS_FILE[BUFSIZ] = "freedom-stats";
char SOURCE_BLOCK[BUFSIZ] = "blocked.source";
char DEST_BLOCK[BUFSIZ] = "blocked.destination";
char SUBJ_BLOCK[BUFSIZ] = "blocked.subject";
char GROUP_BLOCK[BUFSIZ] = "blocked.newsgroup";
char URL_BLOCK[BUFSIZ] = "blocked.url";
char HDRFILTER[BUFSIZ] = "headers.del";
char REMAILER_NAME[BUFSIZ] = "Anonymous Remailer";
char ANON_NAME[BUFSIZ] = "Anonymous";
char ORGANIZATION[BUFSIZ] = "Anonymous Posting Service";
char MAIL2NEWS[BUFSIZ] = "mail2news_nospam@nym.alias.net";
char REMAILER_ADDR[BUFSIZ] = "";
char ANON_ADDR[BUFSIZ] = "";
char COMPLAINTS[BUFSIZ] = "";

/* read word from freedom.conf */
#define read_conf(t) \
if (!strncmp(line, #t, sizeof(#t) - 1)) \
  sscanf(line, "%s %s", junk, t)

/* read line from freedom.conf, with whitespace */
#define read_conf_w(t) \
if (!strncmp(line, #t, sizeof(#t) - 1)) {\
  foo = line;\
  while (*foo > ' ') foo++; /* scan to first space */ \
  while (*foo > '\0' && *foo <= ' ') foo++; /* scan to first non whitespace */ \
  strcpy(t, foo);\
  chop(t);\
}

/* read number from freedom.conf */
#define read_conf_i(t) \
if (!strncmp(line, #t, sizeof(#t) - 1)) \
  sscanf(line, "%s %d", junk, &t);

int init_remailer(void)
{
  FILE *infile;
  char line[BUFSIZ], junk[BUFSIZ], *foo;

  umask(007);
  remailer_dir[0] = '\0';
  strncat(remailer_dir, REMAILER_DIR, sizeof(remailer_dir) - 1);

  /* environment variable replaces defined value */
  if ((foo = getenv("FREEDOM"))) {
    remailer_dir[0] = '\0';
    strncat(remailer_dir, foo, sizeof(remailer_dir) - 1);
  }

  if (!getcwd(cur_dir, sizeof(cur_dir))) { /*could not get current dir */
    fprintf(stderr, "Could not get current directory!\n");
    strcpy(cur_dir, remailer_dir); /* So set cur_dir to remailer_dir */
  }
  if (cur_dir[strlen(cur_dir) - 1] != '/')
    strcat(cur_dir, "/");

  /* drop our privileges if remailer_dir != REMAILER_DIR */
  if (strcmp(REMAILER_DIR, remailer_dir))
    setuid(getuid()), setgid(getgid());

  if (remailer_dir[0] == '\0') {
    foo = getenv("HOME"); /* try ~/freedom */
    if (foo) {
      strncat(remailer_dir, foo, sizeof(remailer_dir) - 1);
      strncat(remailer_dir, "/freedom", sizeof(remailer_dir)
	      - strlen(remailer_dir) - 1);
    }
  }
  if (chdir(remailer_dir))
    fprintf(stderr, "Error changing to directory %s\n", remailer_dir);

  /* initialize the random number generator with time and pid */
  srand((time(NULL) ^ (getpid() | (getpid() << 16))) & RAND_MAX);

  if (!(infile = fopen("freedom.conf", "r"))) {
    fprintf (stderr, "Error: Could not open freedom.conf!\n");
    return -1;
  }
  else {
    while (fgets(line, sizeof(line), infile)) {
      if (line[0] != '#' && strlen(line) > 1) {
	if (!strncmp(line, "REMAILER_DIR", 12)) {
	  sscanf(line, "%s %s", junk, remailer_dir);
	  if (chdir(remailer_dir))
	    fprintf(stderr, "Error changing to REMAILER_DIR %s\n",
		    remailer_dir);
	}
	read_conf_i(USE_SYSLOG);
	read_conf_i(USE_STATS);
	read_conf_i(USE_MIX);
	read_conf_i(ALLOW_PGP);
	read_conf_i(ALLOW_POST);
	read_conf_i(ALLOW_WWW);
	read_conf_i(ALLOW_LTIME);
	read_conf_i(SIZE_LIMIT);
	read_conf(MIX_DIR);
	read_conf(QUEUE_DIR);
	read_conf(PGP_DIR);
	read_conf(MAIL_SPOOL);
	read_conf_w(SENDMAIL);
	read_conf_w(INEWS);
	read_conf(PGP);
	read_conf(LYNX);
	read_conf(HELP_FILE);
	read_conf(KEY_FILE);
	read_conf(STATS_FILE);
	read_conf(SOURCE_BLOCK);
	read_conf(DEST_BLOCK);
	read_conf(SUBJ_BLOCK);
	read_conf(GROUP_BLOCK);
	read_conf(URL_BLOCK);
	read_conf(HDRFILTER);
	read_conf_w(REMAILER_NAME);
	read_conf_w(ANON_NAME);
	read_conf_w(ORGANIZATION);
	read_conf(MAIL2NEWS);
	read_conf(REMAILER_ADDR);
	read_conf(ANON_ADDR);
	read_conf(COMPLAINTS);
      }
    }
    if (ALLOW_LTIME && (QUEUE_DIR[0] == '\0'))
      strcpy(QUEUE_DIR, REMAILER_DIR "/queue");
    if (ALLOW_PGP &&(PGP_DIR[0] == '\0'))
      strcpy(PGP_DIR, REMAILER_DIR "/.pgp");
    if (ANON_NAME[0] == '\0')
      strcpy(ANON_NAME, REMAILER_NAME);
    if (ANON_ADDR[0] == '\0')
      strcpy(ANON_ADDR, REMAILER_ADDR);
    if (COMPLAINTS[0] == '\0')
      strcpy(COMPLAINTS, REMAILER_ADDR);
    fclose(infile);
    return 0;
  }
}

int main(int argc, char *argv[])
{
  FILE *infile, *outfile, *p;
  char tempfilename[BUFSIZ], bFilename[BUFSIZ], dFilename[BUFSIZ],
       eFilename[BUFSIZ], cFilename[BUFSIZ], kFilename[BUFSIZ],
       uFilename[BUFSIZ], poolfilename[BUFSIZ],
       line[BUFSIZ], send_to[BUFSIZ], message_from[BUFSIZ];
  int recursing, trimmings = 0;
  int tempfd;
  enum message_type anon;
  struct stat tempfilestat;

  init_remailer();

  if (USE_SYSLOG)
    openlog(basename(argv[0]), LOG_PID, SYSLOG_FACILITY);

  if (argc > 1) {
    if (!strcmp(argv[1], "-q")) {
      if (USE_SYSLOG)
	syslog(LOG_DEBUG, "Running the latent message queue");
      runlatentqueue();
      exit(0);
    }
    else {
      printf("Freedom Remailer 2.1 "
	     "(C) John B. Fleming 1995, Johannes Kroeger 1997\n\n"
	     "Usage: %s\t\t# read mail on stdin and process\n"
	     "       %s -q\t# run queue of latent messages\n",
	     basename(argv[0]), basename(argv[0]));
      exit(1);
    }
  }

  do { /* make sure we don't overwrite other messages */
    sprintf(tempfilename, "mail%08x", rand());
  } while ((tempfd = open(tempfilename, O_WRONLY | O_CREAT | O_EXCL, 0660))
	   < 0);
  sprintf(bFilename, "%sB", tempfilename);
  sprintf(dFilename, "%sD", tempfilename);
  sprintf(eFilename, "%sE", tempfilename);
  sprintf(cFilename, "%sC", tempfilename);
  sprintf(kFilename, "%sK", tempfilename);
  sprintf(uFilename, "%sU", tempfilename);

  /* copy message coming in on stdin to a temporary file */
  if ((outfile = fdopen(tempfd, "w"))) {
    if (USE_SYSLOG)
      syslog(LOG_DEBUG, "Writing new message to %s", basename(tempfilename));
    while (fgets(line, sizeof(line), stdin))
      fputs(line, outfile);
    fclose(outfile);
  }
  else {
    if (USE_SYSLOG)
      syslog(LOG_ERR, "Can't write new message: %m");
    exit(1);
  }

  if (SIZE_LIMIT) {
    stat(tempfilename, &tempfilestat);
    if (tempfilestat.st_size > SIZE_LIMIT) {
      if (USE_SYSLOG) {
	syslog(LOG_WARNING, "Message too large: %lu bytes",
	       tempfilestat.st_size);
	syslog(LOG_DEBUG, "Deleting %s", tempfilename);
      }
      unlink(tempfilename);
      exit(0);
    }
  }

  recursing = 1;
  while (recursing) {
    recursing = 0;
    anon = scan_message(tempfilename, send_to, message_from);
    if (anon == NULL_MESSAGE) {
      if (USE_SYSLOG)
	syslog(LOG_DEBUG, "Discarding dummy message");
      unlink(tempfilename);
    }
    else if (anon == PGP_MESSAGE) {
      if (USE_STATS)
	updatestats(STATS_PGP);
      infile = fopen(tempfilename, "r");
      /* extract first PGP packet of incoming message for decryption */
      if ((outfile = fopen(eFilename, "w"))) {
	while (fgets(line, sizeof(line), infile) &&
	       strcmp(line, "-----BEGIN PGP MESSAGE-----\n"))
	  continue;
	do {
	  fputs(line, outfile);
	} while (fgets(line, sizeof(line), infile) &&
		 strcmp(line, "-----END PGP MESSAGE-----\n"));
	fputs(line, outfile);
	fclose(outfile);
      }
      fclose(infile);
      pgpdecrypt(eFilename, dFilename);
      rename(tempfilename, eFilename);
      outfile = fopen(tempfilename, "w");
      if ((infile = fopen(dFilename, "r"))) {
	fputs("PGP-Recursive: Yes\n\n", outfile);
	while (fgets(line, sizeof(line), infile))
	  fputs(line, outfile);
	fclose(infile);
      }
      /* Text after the first PGP packet is appended to decrypted message */
      if ((infile = fopen(eFilename, "r"))) {
	while (fgets(line, sizeof(line), infile)) {
	  if (!strcmp(line, "-----END PGP MESSAGE-----\n")) {
	    while (fgets(line, sizeof(line), infile))
	      fputs(line, outfile);
	  }
	}
	fclose(infile);
      }
      fclose(outfile);
      unlink(eFilename);
      unlink(dFilename);
      recursing = 1;
    }
    else if (anon == ANON_MESSAGE) {
      if (USE_SYSLOG)
	syslog(LOG_INFO, "Anonymous message detected");
      if (USE_STATS) /* do stats whether or not blocked */
        updatestats(STATS_MESSAGE);
      if (blocked(message_from, SOURCE_BLOCK)) {
	if (USE_SYSLOG) {
	  syslog(LOG_NOTICE, "Blocked From: %s", message_from);
	  syslog(LOG_NOTICE, "Discarding message");
	}
	unlink(tempfilename);
      }
      else {
	remove_headers(tempfilename, bFilename, send_to);
	unlink(tempfilename);
	if (scan_subject(bFilename) == -1) {
	  if (USE_SYSLOG)
	    syslog(LOG_NOTICE, "Discarding message");
	  unlink(bFilename);
	}
	else {
	  if (USE_SYSLOG)
	    syslog(LOG_INFO, "Remailing message");
	  if (USE_MIX == 1) {
	    do { /* make sure we don't overwrite other messages */
	      sprintf(poolfilename, "%s/mail%08x", MIX_DIR, rand());
	    } while (link(bFilename, poolfilename) < 0);
	  }
	  else {
	    if (!USE_MIX)
	      sprintf(line, "%s", SENDMAIL);
	    else if (USE_MIX == 2)
	      sprintf(line, "%s/%s", MIX_DIR, MIX_MAIL);
	    p = (FILE *) popen(line, "w");
	    if (!USE_MIX) {
	      fprintf(p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	      fprintf(p, DISCLAIMER);
	    }
	    if ((infile = fopen(bFilename, "r"))) {
	      while (fgets(line, sizeof(line), infile))
		fputs(line, p);
	      fclose(infile);
	    }
	    pclose(p);
	  }
	  unlink(bFilename);
	}
      }
    }
    else if (anon == ANON_POST) {
      if (USE_SYSLOG)
	syslog(LOG_INFO, "Anonymous post detected");
      if (USE_STATS)
	updatestats(STATS_POST);
      if (blocked(message_from, SOURCE_BLOCK)) {
	if (USE_SYSLOG) {
	  syslog(LOG_NOTICE, "Blocked From: %s", message_from);
	  syslog(LOG_NOTICE, "Discarding message");
	}
        unlink(tempfilename);
      }
      else {
	remove_headers(tempfilename, bFilename, send_to);
	unlink(tempfilename);
	if (scan_subject(bFilename) == -1) {
	  if (USE_SYSLOG)
	    syslog(LOG_NOTICE, "Discarding message");
	  unlink(bFilename);
	}
	else {
	  if (USE_SYSLOG)
	    syslog(LOG_INFO, "Posting message");
	  if (USE_MIX == 1) {
	    do { /* make sure we don't overwrite other messages */
	      sprintf(poolfilename, "%s/mail%08x", MIX_DIR, rand());
	    } while (link(bFilename, poolfilename) < 0);
	  }
	  else {
	    if (!USE_MIX) {
	      if (ALLOW_POST == 1)
		sprintf(line, "%s", INEWS);
	      else if (ALLOW_POST == 2)
		sprintf(line, "%s", SENDMAIL);
	    }
	    else if (USE_MIX == 2)
	      sprintf(line, "%s/%s", MIX_DIR, MIX_NEWS);
	    p = (FILE *) popen(line, "w");
	    if (!USE_MIX) {
	      if (ALLOW_POST == 1)
		fprintf(p, "Organization: %s\n", ORGANIZATION);
	      else if (ALLOW_POST == 2)
		fprintf(p, "To: %s\n", MAIL2NEWS);
	      fprintf(p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	      fprintf(p, DISCLAIMER);
	      if (!scan_subject(bFilename))
		fprintf(p, "Subject: none\n");
	    }
	    if ((infile = fopen(bFilename, "r"))) {
	      while (fgets(line, sizeof(line), infile))
		fputs(line, p);
	      fclose(infile);
	    }
	    pclose(p);
	  }
	  unlink(bFilename);
	}
      }
    }
    else if (anon == STATS_REQ) {
      if (USE_SYSLOG)
	syslog(LOG_INFO, "Statistics were requested");
      updatestats(STATS_STAT);
      if (blocked(message_from, DEST_BLOCK)) {
	if (USE_SYSLOG) {
	  syslog(LOG_NOTICE, "Blocked From: %s", message_from);
	  syslog(LOG_NOTICE, "Ignoring request");
	}
      }
      else {
	if (USE_SYSLOG)
	  syslog(LOG_INFO, "Mailing statistics");
	mailstats(message_from);
      }
      unlink(tempfilename);
    }
    else if (anon == HELP_REQ) {
      if (USE_SYSLOG)
	syslog(LOG_INFO, "Help was requested");
      if (USE_STATS)
	updatestats(STATS_HELP);
      mailfile(HELP_FILE, message_from);
      unlink(tempfilename);
    }
    else if (anon == KEY_REQ) {
      if (USE_SYSLOG)
	syslog(LOG_INFO, "PGP key was requested");
      if (USE_STATS)
	updatestats(STATS_KEY);
      mailfile(KEY_FILE, message_from);
      unlink(tempfilename);
    }
    else if (anon == CUTMARKS) {
      if (USE_SYSLOG)
	syslog(LOG_DEBUG, "Cutmarks: %s detected", send_to);
      cut(tempfilename, cFilename, send_to, "cutmarks:", 1);
      trimmings = 1;
    }
    else if (anon == ENCRYPT_SUB) {
      if (USE_SYSLOG)
	syslog(LOG_DEBUG, "Encrypt-Subject: %s detected", send_to);
      encrypt_sub(tempfilename, send_to);
      recursing = 1;
    }
    else if (anon == ENCRYPT_KEY) {
      if (USE_SYSLOG)
	syslog(LOG_DEBUG, "Encrypt-Key: %s detected", send_to);
      cut(tempfilename, kFilename, "**", "encrypt-key:", 0);
      encrypt_key(kFilename, tempfilename, send_to);
      recursing = 1;
    }
    else if (anon == LATENT_TIME) {
      if (USE_SYSLOG)
	syslog(LOG_DEBUG, "Latent-Time: %s detected", send_to);
      if (USE_STATS)
        updatestats(STATS_LATENT);
      queuelatent(tempfilename, send_to);
      unlink(tempfilename);
    }
    else if ((anon == URL_REQUEST) || (anon == HTML_REQUEST)) {
      if (anon == URL_REQUEST) {
	if (USE_SYSLOG)
	  syslog(LOG_INFO, "WWW page requested");
      }
      else if (anon == HTML_REQUEST) {
	if (USE_SYSLOG)
	  syslog(LOG_INFO, "HTML source requested");
      }
      if (USE_STATS)
	updatestats(STATS_WWW);
      if (strlen(send_to) == 0) {
	if (USE_SYSLOG)
	  syslog(LOG_NOTICE, "Ignoring request");
	unlink(bFilename);
      }
      else {
	remove_headers(tempfilename, bFilename, send_to);
	unlink(tempfilename);
	geturl(message_from, uFilename, anon);
	if (USE_SYSLOG)
	  syslog(LOG_INFO, "Remailing message");
	if (USE_MIX == 1) {
	  do { /* make sure we don't overwrite other messages */
	    sprintf(poolfilename, "%s/mail%08x", MIX_DIR, rand());
	  } while (link(bFilename, poolfilename) < 0);
	  outfile = fopen(poolfilename, "a");
	  if ((infile = fopen(uFilename, "r"))) {
	    fputs("\n", outfile);
	    while (fgets(line, sizeof(line), infile))
	      fputs(line, outfile);
	    fclose(infile);
          }
	  fclose(outfile);
	}
	else {
	  if (!USE_MIX)
	    sprintf(line, "%s", SENDMAIL);
	  else if (USE_MIX == 2)
	    sprintf(line, "%s/%s", MIX_DIR, MIX_MAIL);
	  p = (FILE *) popen(line, "w");
	  if (!USE_MIX) {
	    fprintf(p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	    fprintf(p, DISCLAIMER);
	  }
	  if ((infile = fopen(bFilename, "r"))) {
	    while (fgets(line, sizeof(line), infile))
	      fputs(line, p);
	    fclose(infile);
	  }
	  if ((infile = fopen(uFilename, "r"))) {
	    fputs("\n", p);
	    while (fgets(line, sizeof(line), infile))
	      fputs(line, p);
	    fclose(infile);
	  }
	  pclose(p);
	}
	unlink(bFilename);
	unlink(uFilename);
      }
    }
    else if (trimmings) {
      if (USE_SYSLOG)
	syslog(LOG_DEBUG, "Processing cutmarks leftovers");
      rename (cFilename, tempfilename);
      trimmings = 0;
      recursing = 1;
    }
    else {
      if (USE_SYSLOG)
	syslog(LOG_INFO, "Non-anonymous message detected");
      if ((infile = fopen(tempfilename, "r"))) {
	if (fgets(line, sizeof(line), infile)) {
	  if (!strcmp(line, "Cut-Recursive: Yes\n")) {
	    if (USE_SYSLOG) {
	      syslog(LOG_DEBUG, "It's cutmarks leftovers");
	      syslog(LOG_INFO, "Discarding undeliverable message");
	    }
	  }
	  else if (!strcmp(line, "PGP-Recursive: Yes\n")) {
	    if (USE_SYSLOG) {
	      syslog(LOG_DEBUG, "Could not decrypt reply block");
	      syslog(LOG_INFO, "Discarding undeliverable message");
	    }
	  }
	  else {
	    if (USE_SYSLOG)
	      syslog(LOG_INFO, "Appending message to spool");
	    if ((outfile = fopen(MAIL_SPOOL, "a"))) {
	      do
		fputs(line, outfile);
	      while (fgets(line, sizeof(line), infile));
	      fclose(outfile);
	    }
	  }
	}
	fclose(infile);
      }
      unlink(tempfilename);
    }
    if (trimmings)
      recursing = 1;
  }
  if (USE_SYSLOG)
    closelog();
  exit(0);
}
