/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997 Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <fcntl.h>
#include <dirent.h>
#include <syslog.h>
#include <sys/wait.h>
#include "freedom.h"

void
runlatentqueue(void)
{
  DIR *queuedir;
  FILE *infile;
  struct dirent *dir_entry;
  struct tm *curtime;
  time_t now;
  char filename[BUFSIZ], queuefilename[BUFSIZ], line[BUFSIZ];
  int ctime, stime, cday, sday, status;

  now = time(NULL);
  curtime = localtime(&now);
  ctime = (curtime->tm_hour * 60) + curtime->tm_min;
  cday = curtime->tm_wday;

  if ((queuedir = opendir(QUEUE_DIR))) {
    while ((dir_entry = readdir(queuedir))) {
      strcpy(filename, dir_entry->d_name);
      if (filename[0] == 'm') {
	/* control file found - process it */
	sprintf(queuefilename, "%s/%s", QUEUE_DIR, filename);
	if ((infile = fopen(queuefilename, "r"))) {
	  fgets(line, sizeof(line), infile);
	  sday = atoi(line);
	  fgets(line, sizeof(line), infile);
	  stime = atoi(line);
	  fclose(infile);
	}
	else {
	  sday = cday;
	  stime = ctime;
	}

	if ((cday > sday) || ((cday == 0) && (sday == 6)) ||
	    ((cday == sday) && (ctime >= stime))) {
	  /* time to send the message */
	  filename[0] = 'q';
	  sprintf(queuefilename, "%s/%s", QUEUE_DIR, filename);
	  sprintf(line, "%s/%s", remailer_dir, REMAILER_BIN);
	  if (USE_SYSLOG)
	    syslog(LOG_INFO, "Dequeuing message %s", filename);

	  /* pipe queued message to another remailer process */
	  if (!fork()) { /* child */
	    infile = fopen(queuefilename, "r");
	    dup2(fileno(infile), STDIN_FILENO);
	    fclose(infile);
	    execl(line, REMAILER_BIN, NULL);
	  }
	  else /* parent */
	    wait(&status);

	  if (WIFEXITED(status)) {
	    unlink(queuefilename);
	    filename[0] = 'm';
	    sprintf(queuefilename, "%s/%s", QUEUE_DIR, filename);
	    unlink(queuefilename);
	  }
	  else {
	    if (USE_SYSLOG)
	      syslog(LOG_WARNING, "Couldn't dequeue message %s", filename);
	  }
	}
      }
    }
    closedir(queuedir);
  }
}

void
queuelatent(const char *origfilename, const char *latentTime)
{
  char line[BUFSIZ], hexrand[9],
  qFilename[BUFSIZ], mFilename[BUFSIZ];
  FILE *infile, *outfile;
  int time1, time2, day, qfd, mfd;
  int headers = 1, linenum = 0;
  struct tm *curtime;
  time_t now;

  now = time(NULL);
  curtime = localtime(&now);
  time1 = (curtime->tm_hour * 60) + curtime->tm_min;
  day = curtime->tm_wday;
  time2 = parsetime(latentTime, time1);
  if (time2 < time1) {
    day++;
    if (day == 7)
      day = 0;
  }

  do { /* make sure we don't overwrite other messages */
    sprintf(hexrand, "%08x", rand());
    sprintf(qFilename, "%s/q%s", QUEUE_DIR, hexrand);
    sprintf(mFilename, "%s/m%s", QUEUE_DIR, hexrand);
  }
  while ((qfd = open(qFilename, O_WRONLY | O_CREAT | O_EXCL, 0660)) < 0 ||
	 (mfd = open(mFilename, O_WRONLY | O_CREAT | O_EXCL, 0660)) < 0);

  if ((outfile = fdopen(mfd, "w"))) {
    fprintf(outfile, "%d\n%d\n", day, time2);
    fclose(outfile);
  }
  if (!(infile = fopen(origfilename, "r"))) {
    if (USE_SYSLOG)
      syslog(LOG_WARNING, "Can't open %s: %m", basename(origfilename));
    close(qfd);
    return;
  }
  outfile = fdopen(qfd, "w");
  if (USE_SYSLOG)
    syslog(LOG_INFO, "Queuing message to %s", basename(qFilename));
  while (fgets(line, sizeof(line), infile)) {
    if (!headers) {
      if ((linenum > 1) || (strlen(line) > 1))
	linenum++;
      if ((linenum == 1) && strstr(line, "::"))
	headers = 1;
      fputs(line, outfile);
    }
    else {
      if (strlen(line) <= 1)
	headers = 0;
      if (!strileft(line, "latent-time:"))
	fputs(line, outfile);
    }
  }
  fclose(infile);
  fclose(outfile);
}

int
parsetime(const char *latent, const int time_now)
{
  char hour[3], minute[3];
  int tim, ranmax;
  hour[2] = '\0'; minute[2] = '\0';

  if (isdigit(latent[0])) { /* no '+': parse as time given */
    if (isdigit(latent[1]) && (latent[2] == ':') &&
	isdigit(latent[3]) && isdigit(latent[4])) { /* two-digit hour */
      hour[0] = latent[0]; hour[1] = latent[1];
      minute[0] = latent[3]; minute[1] = latent[4];
      ranmax = (atoi(hour) * 60) + atoi(minute) - time_now;
      if (ranmax <= 0)
	ranmax += 1440;
      if (tolower(latent[5]) == 'r') /* hh:mmR */
	tim = ((rand() % ranmax) + time_now) % 1440;
      else /* hh:mm */
	tim = ((atoi(hour) * 60) + atoi(minute)) % 1440;
    }
    else if ((latent[1] == ':') &&
	     isdigit(latent[2]) && isdigit(latent[3])) { /* one-digit hour */
      hour[0] = latent[0]; hour[1] = '\0';
      minute[0] = latent[2]; minute[1] = latent[3];
      ranmax = (atoi(hour) * 60) + atoi(minute) - time_now;
      if (ranmax <= 0)
	ranmax += 1440;
      if (tolower(latent[4]) == 'r') /* h:mmR */
	tim = ((rand() % ranmax) + time_now) % 1440;
      else /* h:mm */
	tim = ((atoi(hour) * 60) + atoi(minute)) % 1440;
    }
    else /* invalid time string: no latency */
      tim = time_now % 1440;
  }
  else if (latent[0] == '+') { /* '+': parse as delay */
    if (isdigit(latent[1]) && isdigit(latent[2]) && (latent[3] == ':') &&
	isdigit(latent[4]) && isdigit(latent[5])) { /* two-digit hour */
      hour[0] = latent[1]; hour[1] = latent[2];
      minute[0] = latent[4]; minute[1] = latent[5];
      ranmax = (atoi(hour) * 60) + atoi(minute);
      if (tolower(latent[6]) == 'r' && ranmax > 0) /* +hh:mmR */
	tim = ((rand() % ranmax) + time_now) % 1440;
      else /* +hh:mm */
	tim = (time_now + (atoi(hour) * 60) + atoi(minute)) % 1440;
    }
    else if (isdigit(latent[1]) && (latent[2] == ':') &&
	     isdigit(latent[3]) && isdigit(latent[4])) { /* one-digit hour */
      hour[0] = latent[1]; hour[1] = '\0';
      minute[0] = latent[3]; minute[1] = latent[4];
      ranmax = (atoi(hour) * 60) + atoi(minute);
      if (tolower(latent[5]) == 'r' && ranmax > 0) /* +h:mmR */
	tim = ((rand() % ranmax) + time_now) % 1440;
      else /* +h:mm */
	tim = (time_now + (atoi(hour) * 60) + atoi(minute)) % 1440;
    }
    else /* invalid time string: no latency */
      tim = time_now % 1440;
  }
  else /* invalid time string: no latency */
    tim = time_now % 1440;
  return tim;
}
