/* pgg.h - general include file for libpgg usage
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#ifndef PGG_INCLUDE_PGG_H_19990604145212
#define PGG_INCLUDE_PGG_H_19990604145212


/*
 * PGG object are opaque. The only way to manipulate them is 
 * to use the provided functions.
 */
typedef struct { int dummy; } * PggBuffer;
typedef struct { int dummy; } * PggInput;
typedef struct { int dummy; } * PggExe;
typedef struct { int dummy; } * PggOutopts;
typedef struct { int dummy; } * PggStdio;
typedef struct { int dummy; } * PggOutput;
typedef struct { int dummy; } * PggSymcrypt;
typedef struct { int dummy; } * PggAlgo;
typedef struct { int dummy; } * PggConfig;
typedef struct { int dummy; } * PggPasscache;
typedef struct { int dummy; } * PggSign;
typedef struct { int dummy; } * PggEncrypt;




/*
 * These are primary error numbers. They describe error classes.
 */
typedef enum {
    PGG_ERROR_SUCCESS	= 0,		/* Succes, no error at all */
    PGG_ERROR_UNKNOWN,			/* Generic, unknown error */
    PGG_ERROR_ARGUMENT,			/* An argument was invalid */
    PGG_ERROR_OBJECT,			/* An object was wrong */
    PGG_ERROR_GNUPG,			/* Execution of GnuPG failed */
    PGG_ERROR_RESOURCE,			/* No more resources */
    PGG_ERROR_REQUEST,			/* A request couldn't be fullfilled */
    PGG_ERROR_INTERNAL,			/* An internal error occured */
    PGG_ERROR_CRYPT			/* An encrypting/decrypting specific error occured */
} PggError;


/*
 * Sub error codes desribe an error more precisly
 */
typedef enum {
    PGG_SUBERR_NONE = 0,		/* No sub error */
    
    /*
     * Sub codes for encryption or decryption problems: PGG_ERROR_CRYPT
     */
    PGG_SUBERR_PASSPHRASE,		/* A passphrase was wrong or missed */
    PGG_SUBERR_NOPUBKEY,		/* A public key was missed */
    PGG_SUBERR_NOSECKEY,		/* A secret key was missed */
    
    /*
     * Sub codes for request which couldn't be fullfilled: PGG_ERROR_REQUEST
     */
    PGG_SUBERR_NOTFOUND,		/* Something you asked for could not be found (key, ...) */
    PGG_SUBERR_NOTSET,			/* A value the caller asked for was not set */
    PGG_SUBERR_STATE,			/* The state of the object was not suited for the request */
    
    /*
     * Sub error codes for invalid arguments: PGG_ERROR_ARGUMENT
     */
    PGG_SUBERR_NULLPTR,			/* An argument was a forbidden null pointer */
    PGG_SUBERR_RANGE,			/* An index for something was out of valid bounds */
    PGG_SUBERR_TOLARGE,			/* A string or something else was to long or to large */
    PGG_SUBERR_TOSMALL,			/* A string or somthingg else was to short or to small */
    PGG_SUBERR_VALUE,			/* A value was invalid. For example an invalid cipher id or something else */
    
    /*
     * Sub codes for wrong objects: PGG_ERROR_OBJECT
     */
    PGG_SUBERR_NULLOBJ,			/* An object was null */
    PGG_SUBERR_MAGIC,			/* The magic value of an object was wrong (false type casts) */
    PGG_SUBERR_REFCOUNT,		/* The reference counter of an object was 0. This could cause core dumps! */
    
    /*
     * Errors on excuting GnuPG: PGG_ERROR_GNUPG
     */
    PGG_SUBERR_DIED,			/* GnuPG died unexpected */
    PGG_SUBERR_EXECUTE,			/* Gpg binary not executable */
    PGG_SUBERR_UNEXPECTED,		/* An unexpected event/behaviour from GnuPG */
    
    /*
     * Errors for non available resources: PGG_ERROR_RESOURCE
     */
    PGG_SUBERR_MEMORY,			/* No more memory available */
    PGG_SUBERR_PIPE,			/* No pipe available */
    PGG_SUBERR_PROCESS			/* No process available to create (fork() for example)*/

} PggSuberr;



typedef struct {
    PggError		error;
    PggSuberr		suberr;
} PggErrenv[1];


extern void			pgg_errenv_reset(PggErrenv errenv);
extern PggError			pgg_errenv_get_error(PggErrenv errenv);
extern PggSuberr		pgg_errenv_get_suberr(PggErrenv errenv);
extern void			pgg_errenv_set(PggErrenv errenv, PggError error, PggSuberr suberr);
extern void			pgg_errenv_copy(PggErrenv dest, PggErrenv source);
extern int			pgg_errenv_is_set(PggErrenv errenv);




extern PggBuffer		pgg_buffer_new(PggErrenv errenv);
extern PggBuffer		pgg_buffer_clone(PggBuffer buf, PggErrenv errenv);
extern void			pgg_buffer_addref(PggBuffer buf, PggErrenv errenv);
extern void			pgg_buffer_release(PggBuffer buf, PggErrenv errenv);

extern void			pgg_buffer_reclaim(PggBuffer buf, PggErrenv errenv);

extern void			pgg_buffer_append(PggBuffer buf, const void * data, long amount, PggErrenv errenv);
extern void			pgg_buffer_insert(PggBuffer buf, long offset, const void * data, long amount, PggErrenv errenv);
extern void			pgg_buffer_delete(PggBuffer buf, long offset, long amount, PggErrenv errenv);
extern void			pgg_buffer_clear(PggBuffer buf, PggErrenv errenv);

extern void			pgg_buffer_set_size(PggBuffer buf, long size, PggErrenv errenv);
extern void			pgg_buffer_set_wipeout(PggBuffer buf, int yesno, PggErrenv errenv);
extern void			pgg_buffer_set_maxsize(PggBuffer buf, long maxsize, PggErrenv errenv);
extern void			pgg_buffer_ensure_capacity(PggBuffer buf, long capacity, PggErrenv errenv);

extern long			pgg_buffer_get_maxsize(PggBuffer buf, PggErrenv errenv);
extern long			pgg_buffer_get_capacity(PggBuffer buf, PggErrenv errenv);
extern long			pgg_buffer_get_size(PggBuffer buf, PggErrenv errenv);
extern void *			pgg_buffer_get_data(PggBuffer buf, PggErrenv errenv);

extern void			pgg_buffer_detach_data(PggBuffer buf, PggErrenv errenv);
extern void			pgg_buffer_atach_data(PggBuffer buf, void * data, long size, PggErrenv errenv);




extern PggInput			pgg_input_new(PggErrenv errenv);
extern PggInput			pgg_input_clone(PggInput in, PggErrenv errenv);
extern void			pgg_input_addref(PggInput in, PggErrenv errenv);
extern void			pgg_input_release(PggInput in, PggErrenv errenv);

extern void			pgg_input_set_filename(PggInput in, char *filename, PggErrenv errenv);
extern void			pgg_input_set_buffer(PggInput in, PggBuffer buf, PggErrenv errenv);
extern void			pgg_input_set_fd(PggInput in, int fd, PggErrenv errenv);
extern void			pgg_input_set_rawdata(PggInput in, void *data, long amount, PggErrenv errenv);

extern void			pgg_input_open(PggInput in, PggErrenv errenv);
extern long			pgg_input_read(PggInput in, void *data, long amount, PggErrenv errenv);
extern void			pgg_input_close(PggInput in, PggErrenv errenv);




typedef enum {
    PGG_EXE_EVENT_ERROR = -1,			/* An error occured */
    PGG_EXE_EVENT_NONE = 0,			/* No event at all */
    PGG_EXE_EVENT_FINISHED,			/* Execution finished */
    PGG_EXE_EVENT_STATUS			/* New status code available */
} PggExe_Event;


typedef enum {
    PGG_EXE_STATUS_ERROR = -1,			/* An error occured */
    PGG_EXE_STATUS_NONE = 0,			/* No status message at all */
    PGG_EXE_STATUS_GOODSIG,
    PGG_EXE_STATUS_BADSIG,
    PGG_EXE_STATUS_ERRSIG,
    PGG_EXE_STATUS_VALIDSIG,
    PGG_EXE_STATUS_SIG_ID,
    PGG_EXE_STATUS_ENC_TO,
    PGG_EXE_STATUS_NODATA,
    PGG_EXE_STATUS_TRUST_UNDEFINED,
    PGG_EXE_STATUS_TRUST_NEVER,
    PGG_EXE_STATUS_TRUST_MARGINAL,
    PGG_EXE_STATUS_TRUST_FULLY,
    PGG_EXE_STATUS_TRUST_ULTIMATE,
    PGG_EXE_STATUS_SIGEXPIRED,
    PGG_EXE_STATUS_KEYREVOKED,
    PGG_EXE_STATUS_BADARMOR,
    PGG_EXE_STATUS_RSA_OR_IDEA,
    PGG_EXE_STATUS_SHM_INFO,
    PGG_EXE_STATUS_SHM_GET,
    PGG_EXE_STATUS_SHM_GET_BOOL,
    PGG_EXE_STATUS_SHM_GET_HIDDEN,
    PGG_EXE_STATUS_NEED_PASSPHRASE,
    PGG_EXE_STATUS_NEED_PASSPHRASE_SYM,
    PGG_EXE_STATUS_MISSING_PASSPHRASE,
    PGG_EXE_STATUS_BAD_PASSPHRASE,
    PGG_EXE_STATUS_GOOD_PASSPHRASE,
    PGG_EXE_STATUS_DECRYPTION_FAILED,
    PGG_EXE_STATUS_DECRYPTION_OKAY,
    PGG_EXE_STATUS_NO_PUBKEY,
    PGG_EXE_STATUS_NO_SECKEY,
    PGG_EXE_STATUS_UNKNOWN,			/* unknown status message */
} PggExe_Status;


typedef long (*PggExe_ReadCB)(void *opaque, void *data, long amount);
typedef long (*PggExe_WriteCB)(void *opaque, void *data, long amount);


extern PggExe			pgg_exe_new(PggErrenv errenv);
extern PggExe			pgg_exe_clone(PggExe exe, PggErrenv errenv);
extern void			pgg_exe_addref(PggExe exe, PggErrenv errenv);
extern void			pgg_exe_release(PggExe exe, PggErrenv errenv);

extern void 			pgg_exe_set_gpgpath(PggExe exe, const char *gpgpath, PggErrenv errenv);
extern const char *		pgg_exe_get_gpgpath(PggExe exe, PggErrenv errenv);

extern int			pgg_exe_get_returnCode(PggExe exe, PggErrenv errenv);
extern void			pgg_exe_start(PggExe exe, PggErrenv errenv);
extern void			pgg_exe_add_arg(PggExe exe, char *arg, PggErrenv errenv);
extern void			pgg_exe_abort(PggExe exe, PggErrenv errenv);
extern void			pgg_exe_request_status(PggExe exe, PggErrenv errenv);
extern PggExe_Status		pgg_exe_get_status_code(PggExe exe, PggErrenv errenv);
extern char *			pgg_exe_get_status_arg(PggExe exe, PggErrenv errenv);
extern PggExe_Event		pgg_exe_wait_event(PggExe exe, PggErrenv errenv);
extern void			pgg_exe_request_shm(PggExe exe, PggErrenv errenv);
extern void			pgg_exe_reply_str(PggExe exe, const char * reply, PggErrenv errenv);
extern void			pgg_exe_reply_bool(PggExe exe, int yesno, PggErrenv errenv);

extern void			pgg_exe_set_read_cb(PggExe exe, PggExe_ReadCB cb, void *opaque, PggErrenv errenv);
extern void			pgg_exe_set_write_cb(PggExe exe, PggExe_WriteCB cb, void *opaque, PggErrenv errenv);




#define PGG_OUTOPTS_COMMENT_NONE	((char *)0)
#define PGG_OUTOPTS_COMMENT_DEFAULT	((char *)1)

extern PggOutopts		pgg_outopts_new(PggErrenv errenv);
extern PggOutopts		pgg_outopts_clone(PggOutopts oo, PggErrenv errenv);
extern void			pgg_outopts_addref(PggOutopts oo, PggErrenv errenv);
extern void			pgg_outopts_release(PggOutopts oo, PggErrenv errenv);

extern void			pgg_outopts_set_comment(PggOutopts oo, char *text, PggErrenv errenv);
extern void			pgg_outopts_set_version(PggOutopts oo, int yesno, PggErrenv errenv);
extern void			pgg_outopts_set_armor(PggOutopts oo, int yesno, PggErrenv errenv);

extern char *			pgg_outopts_get_comment(PggOutopts oo, PggErrenv errenv);
extern int			pgg_outopts_get_version(PggOutopts oo, PggErrenv errenv);
extern int			pgg_outopts_get_armor(PggOutopts oo, PggErrenv errenv);

extern void			pgg_outopts_setup_exe(PggOutopts oo, PggExe exe, PggErrenv errenv);




extern PggStdio			pgg_stdio_new(PggErrenv errenv);
extern PggStdio			pgg_stdio_clone(PggStdio io, PggErrenv errenv);
extern void			pgg_stdio_addref(PggStdio io, PggErrenv errenv);
extern void			pgg_stdio_release(PggStdio io, PggErrenv errenv);

extern void			pgg_stdio_set_input(PggStdio io, PggInput input, PggErrenv errenv);
extern void			pgg_stdio_set_output(PggStdio io, PggOutput output, PggErrenv errenv);
extern void			pgg_stdio_set_outopts(PggStdio io, PggOutopts opts, PggErrenv errenv);
extern void			pgg_stdio_set_textmode(PggStdio io, int yesno, PggErrenv errenv);

extern PggInput			pgg_stdio_get_input(PggStdio io, PggErrenv errenv);
extern PggOutput		pgg_stdio_get_output(PggStdio io, PggErrenv errenv);
extern PggOutopts		pgg_stdio_get_outopts(PggStdio io, PggErrenv errenv);
extern int			pgg_stdio_get_textmode(PggStdio io, PggErrenv errenv);

extern void			pgg_stdio_setup_exe(PggStdio io, PggExe exe, PggErrenv errenv);




extern PggOutput		pgg_output_new(PggErrenv errenv);
extern PggOutput		pgg_output_clone(PggOutput out, PggErrenv errenv);
extern void			pgg_output_addref(PggOutput out, PggErrenv errenv);
extern void			pgg_output_release(PggOutput out, PggErrenv errenv);

extern void			pgg_output_set_filename(PggOutput out, char *filename, PggErrenv errenv);
extern void			pgg_output_set_buffer(PggOutput out, PggBuffer buf, PggErrenv errenv);
extern void			pgg_output_set_fd(PggOutput out, int fd, PggErrenv errenv);

extern void			pgg_output_open(PggOutput out, PggErrenv errenv);
extern long			pgg_output_write(PggOutput out, void *data, long amount, PggErrenv errenv);
extern void			pgg_output_close(PggOutput out, PggErrenv errenv);




extern PggSymcrypt		pgg_symcrypt_new(PggErrenv errenv);
extern PggSymcrypt		pgg_symcrypt_clone(PggSymcrypt sc, PggErrenv errenv);
extern void			pgg_symcrypt_addref(PggSymcrypt sc, PggErrenv errenv);
extern void			pgg_symcrypt_release(PggSymcrypt sc, PggErrenv errenv);

extern void			pgg_symcrypt_set_passphrase(PggSymcrypt sc, char *passphrase, PggErrenv errenv);
extern void			pgg_symcrypt_set_stdio(PggSymcrypt sc, PggStdio stdio, PggErrenv errenv);
extern void			pgg_symcrypt_set_outopts(PggSymcrypt sc, PggOutopts opts, PggErrenv errenv);
extern void			pgg_symcrypt_set_config(PggSymcrypt sc, PggConfig cfg, PggErrenv errenv);
extern void			pgg_symcrypt_set_algo(PggSymcrypt sc, PggAlgo algo, PggErrenv errenv);

extern void			pgg_symcrypt_encrypt(PggSymcrypt sc, PggErrenv errenv);
extern void			pgg_symcrypt_decrypt(PggSymcrypt sc, PggErrenv errenv);




typedef enum {
    PGG_ALGO_COMPRESS_ERROR = -1,
    PGG_ALGO_COMPRESS_USEPREFS,
    PGG_ALGO_COMPRESS_NONE,
    PGG_ALGO_COMPRESS_PGPZLIB,
    PGG_ALGO_COMPRESS_RFCZLIB
} PggAlgo_Compress;


typedef enum {
    PGG_ALGO_CIPHER_ERROR = -1,
    PGG_ALGO_CIPHER_USEPREFS,
    PGG_ALGO_CIPHER_IDEA,
    PGG_ALGO_CIPHER_3DES,
    PGG_ALGO_CIPHER_CAST5,
    PGG_ALGO_CIPHER_BLOWFISH,
    PGG_ALGO_CIPHER_TWOFISH
} PggAlgo_Cipher;


typedef enum {
    PGG_ALGO_DIGEST_ERROR = -1,
    PGG_ALGO_DIGEST_USEPREFS,
    PGG_ALGO_DIGEST_MD5,
    PGG_ALGO_DIGEST_SHA1,
    PGG_ALGO_DIGEST_RIPEMD160
} PggAlgo_Digest;


typedef enum {
    PGG_ALGO_PACKET_ERROR = -1,
    PGG_ALGO_PACKET_OPENPGP = 1,
    PGG_ALGO_PACKET_V3SIGS,
    PGG_ALGO_PACKET_RFC1991
} PggAlgo_Packet;


typedef enum {
    PGG_ALGO_S2K_MODE_ERROR = -1,
    PGG_ALGO_S2K_MODE_SIMPLE = 1,
    PGG_ALGO_S2K_MODE_SALTED,
    PGG_ALGO_S2K_MODE_ITERATEDSALTED
} PggAlgo_S2kMode;


extern PggAlgo			pgg_algo_new(PggErrenv errenv);
extern PggAlgo			pgg_algo_clone(PggAlgo algo, PggErrenv errenv);
extern void			pgg_algo_addref(PggAlgo algo, PggErrenv errenv);
extern void			pgg_algo_release(PggAlgo algo, PggErrenv errenv);

extern void			pgg_algo_set_cipher(PggAlgo algo, PggAlgo_Cipher cipher, PggErrenv errenv);
extern void			pgg_algo_set_digest(PggAlgo algo, PggAlgo_Digest digest, PggErrenv errenv);
extern void			pgg_algo_set_compress(PggAlgo algo, PggAlgo_Compress compress, PggErrenv errenv);
extern void			pgg_algo_set_packet(PggAlgo algo, PggAlgo_Packet packet, PggErrenv errenv);
extern void			pgg_algo_set_s2k_mode(PggAlgo algo, PggAlgo_S2kMode mode, PggErrenv errenv);
extern void			pgg_algo_set_s2k_cipher(PggAlgo algo, PggAlgo_Cipher cipher, PggErrenv errenv);
extern void			pgg_algo_set_s2k_digest(PggAlgo algo, PggAlgo_Digest digest, PggErrenv errenv);

extern PggAlgo_Cipher		pgg_algo_get_cipher(PggAlgo algo, PggErrenv errenv);
extern PggAlgo_Digest		pgg_algo_get_digest(PggAlgo algo, PggErrenv errenv);
extern PggAlgo_Compress		pgg_algo_get_compress(PggAlgo algo, PggErrenv errenv);
extern PggAlgo_Packet		pgg_algo_get_packet(PggAlgo algo, PggErrenv errenv);
extern PggAlgo_S2kMode		pgg_algo_get_s2k_mode(PggAlgo algo, PggErrenv errenv);
extern PggAlgo_Cipher		pgg_algo_get_s2k_cipher(PggAlgo algo, PggErrenv errenv);
extern PggAlgo_Digest		pgg_algo_get_s2k_digest(PggAlgo algo, PggErrenv errenv);

extern void			pgg_algo_setup_exe(PggAlgo algo, PggExe exe, PggErrenv errenv);




extern PggConfig		pgg_config_new(PggErrenv errenv);
extern PggConfig		pgg_config_clone(PggConfig cfg, PggErrenv errenv);
extern void			pgg_config_release(PggConfig cfg, PggErrenv errenv);
extern void			pgg_config_addref(PggConfig cfg, PggErrenv errenv);

extern const char *		pgg_config_get_gpgpath(PggConfig cfg, PggErrenv errenv);
extern void			pgg_config_set_gpgpath(PggConfig cfg, const char *gpgpath, PggErrenv errenv);
extern int			pgg_config_search_gpg(PggConfig cfg, const char * path, PggErrenv errenv);

extern void			pgg_config_set_configfile(PggConfig cfg, const char * configfile, PggErrenv errenv);
extern const char *		pgg_config_get_configfile(PggConfig cfg, PggErrenv errenv);

extern void			pgg_config_set_public_keyring(PggConfig cfg, const char * public_keyring, PggErrenv errenv);
extern const char *		pgg_config_get_public_keyring(PggConfig cfg, PggErrenv errenv);
extern void			pgg_config_set_secret_keyring(PggConfig cfg, const char * secret_keyring, PggErrenv errenv);
extern const char *		pgg_config_get_secret_keyring(PggConfig cfg, PggErrenv errenv);
extern void			pgg_config_set_trustdb(PggConfig cfg, const char * trustdb, PggErrenv errenv);
extern const char *		pgg_config_get_trustdb(PggConfig cfg, PggErrenv errenv);

extern void			pgg_config_setup_exe(PggConfig cfg, PggExe exe, PggErrenv errenv);




typedef enum
{
    PGG_PASSCACHE_POLICY_ERROR = -1,		/* error indicator */
    PGG_PASSCACHE_POLICY_REFRESH,		/* using a passphrase refreshes its timeout */
    PGG_PASSCACHE_POLICY_TIMEOUT		/* fixed timeout, no refreshing of passphrase timeouts */
} PggPasscache_Policy;


typedef enum
{
    PGG_PASSCACHE_CBCMD_INIT,			/* Called when the callback is attached to a PggPasscache object */
    PGG_PASSCACHE_CBCMD_ASK,			/* Request to fill the passcache with the passphrase of the given keyid in `arg' */
    PGG_PASSCACHE_CBCMD_RELEASE			/* The callback is detached from a PggPasscache object */
} PggPasscache_CbCmd;


typedef int (*PggPasscache_Cb)(PggPasscache pc, PggPasscache_CbCmd cmd, const char *arg);


extern PggPasscache		pgg_passcache_new(PggErrenv errenv);
extern PggPasscache		pgg_passcache_clone(PggPasscache pc, PggErrenv errenv);
extern void			pgg_passcache_addref(PggPasscache pc, PggErrenv errenv);
extern void			pgg_passcache_release(PggPasscache pc, PggErrenv errenv);

extern void			pgg_passcache_set_timeout(PggPasscache pc, long seconds, PggErrenv errenv);
extern long			pgg_passcache_get_timeout(PggPasscache pc, PggErrenv errenv);

extern void			pgg_passcache_set_policy(PggPasscache pc, PggPasscache_Policy policy, PggErrenv errenv);
extern PggPasscache_Policy	pgg_passcache_get_policy(PggPasscache pc, PggErrenv errenv);

extern void			pgg_passcache_clear(PggPasscache pc, PggErrenv errenv);
extern void			pgg_passcache_expire(PggPasscache pc, PggErrenv errenv);

extern void			pgg_passcache_set_callback(PggPasscache pc, PggPasscache_Cb callback, PggErrenv errenv);

extern void			pgg_passcache_set_passphrase(PggPasscache pc, const char * keyid, const char *passphrase, PggErrenv errenv);
extern void			pgg_passcache_del_passphrase(PggPasscache pc, const char * keyid, PggErrenv errenv);
extern const char *		pgg_passcache_get_passphrase(PggPasscache pc, const char * keyid, PggErrenv errenv);




typedef enum
{
    PGG_SIGN_MODE_ERROR = -1,		/* Error indicator */
    PGG_SIGN_MODE_NORMAL = 0,		/* Normal mode. Result is a packet including message and signature (default) */
    PGG_SIGN_MODE_DETACH,		/* Detached signature. Output is the signature without the message */
    PGG_SIGN_MODE_CLEARSIGN,		/* Clearsign a message. (Used for emails and so on) */
} PggSign_Mode;


typedef enum
{
    PGG_SIGN_ESCAPE_ERROR = -1,		/* Error indicator */
    PGG_SIGN_ESCAPE_NONE = 0,		/* Don't alter the input in any way (Default) */
    PGG_SIGN_ESCAPE_FROM,		/* Escape line beginning with "From " */
    PGG_SIGN_ESCAPE_NOTDASH		/* Don't escape dashes. (Special mode for patch files) */
} PggSign_Escape;


extern PggSign			pgg_sign_new(PggErrenv errenv);
extern PggSign			pgg_sign_clone(PggSign sign, PggErrenv errenv);
extern void			pgg_sign_addref(PggSign sign, PggErrenv errenv);
extern void			pgg_sign_release(PggSign sign, PggErrenv errenv);

extern void			pgg_sign_set_signer(PggSign sign, const char *signer, PggErrenv errenv);
extern const char *		pgg_sign_get_signer(PggSign sign, PggErrenv errenv);

extern void			pgg_sign_set_escape(PggSign sign, PggSign_Escape escape, PggErrenv errenv);
extern PggSign_Escape		pgg_sign_get_escape(PggSign sign, PggErrenv);

extern void			pgg_sign_set_mode(PggSign sign, PggSign_Mode mode, PggErrenv errenv);
extern PggSign_Mode		pgg_sign_get_mode(PggSign sign, PggErrenv);

extern void			pgg_sign_set_stdio(PggSign sign, PggStdio stdio, PggErrenv errenv);
extern PggStdio			pgg_sign_get_stdio(PggSign sign, PggErrenv errenv);

extern void			pgg_sign_set_passcache(PggSign sign, PggPasscache passcache, PggErrenv errenv);
extern PggPasscache		pgg_sign_get_passcache(PggSign sign, PggErrenv errenv);

extern void			pgg_sign_set_algo(PggSign sign, PggAlgo algo, PggErrenv errenv);
extern PggAlgo			pgg_sign_get_algo(PggSign sign, PggErrenv);

extern void			pgg_sign_set_config(PggSign sign, PggConfig config, PggErrenv errenv);
extern PggConfig		pgg_sign_get_config(PggSign sign, PggErrenv errenv);

extern void			pgg_sign_execute(PggSign sign, PggErrenv errenv);




#endif


