/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 *
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "mount_locl.h"

RCSID("$Id: mount_xfs.c,v 1.12 1998/10/31 15:16:56 lha Exp $");

static const struct mntopt mopts[] = {
    MOPT_STDOPTS,
    MOPT_ASYNC,
    MOPT_SYNC,
    MOPT_UPDATE,
    MOPT_RELOAD,
    {NULL}
};

static void
usage(void)
{
    fprintf(stderr, "Usage: %s [-o options] [-F flags] device path\n", __progname);
    exit(1);
}

#if defined(HAVE_GETVFSBYNAME) && defined(HAVE_VFSISLOADABLE) && defined(HAVE_VFSLOAD)

/*
 * Try to load the kernel module if we have getvfsbyname (currently
 * only FreeBSD)
 */

#ifdef HAVE_GETVFSBYNAME_TWO_ARGS

static int
get_vfc_index (const char *vfc_name)
{
    int ret;
    struct vfsconf vfc;
    
    ret = getvfsbyname (vfc_name, &vfc);
    if (ret != 0 && errno == ENOENT && vfsisloadable (vfc_name)) {
	if (vfsload (vfc_name))
	    err (1, "vfsload(%s)", vfc_name);
	endvfsent ();
	ret = getvfsbyname (vfc_name, &vfc);
    }
    if (ret != 0)
	err (1, "%s filesystem not available", vfc_name);
    return vfc.vfc_typenum;
}

#else

static int
get_vfc_index (const char *vfc_name)
{
    struct vfsconf *vfc;

    vfc = getvfsbyname (vfc_name);
    if (vfc == NULL && vfsisloadable (vfc_name)) {
	if (vfsload (vfc_name))
	    err (1, "vfsload(%s)", vfc_name);
	endvfsent ();
	vfc = getvfsbyname (vfc_name);
    }
    if (vfc == NULL)
	err (1, "%s filesystem not available", vfc_name);
    return vfc->vfc_index;
}

#endif

#else /* HAVE_GETVFSBYNAME */

#define get_vfc_index(x) MOUNT_XFS 

#endif

int
main(int argc, char **argv)
{
    int error;
    int ch, mntflags = 0;

    optind = 1;
#ifdef HAVE_OPTRESET
    optreset = 1;
#endif
    set_progname(argv[0]);
    while ((ch = getopt(argc, argv, "o:F:")) != -1)
	switch (ch) {
	case 'o':
	    getmntopts(optarg, mopts, &mntflags);
	    break;
	case 'F':
	    mntflags = atoi(optarg);
	    break;
	case '?':
	default:
	    usage();
	}

    argc -= optind;
    argv += optind;

    if (argc != 2)
	usage();

    error = mount(get_vfc_index ("xfs"), argv[1], mntflags, argv[0]);

    if (error != 0)
	err(1, "mount");

    return 0;
}
