/*
 * Copyright (c) 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "milko_locl.h"

RCSID("$Id: bufcache.c,v 1.2 1998/06/23 17:11:41 lha Exp $");

static Hashtab *bufhash = NULL;
static List *lrulist = NULL;
static int nbufentries = 0;

static AFSFid dummyfid = {17,42,4711};


static unsigned int 
bufcachehash(void *a)
{
    BufCacheEntry *bc = (BufCacheEntry *) a;
    return bc->fid.Volume ^ bc->fid.Vnode ^ bc->fid.Unique;
}

static int 
bufcachecmp(void *a, void *b)
{
    BufCacheEntry *bc1 = (BufCacheEntry *) a;
    BufCacheEntry *bc2 = (BufCacheEntry *) b;

    return bc1->fid.Volume == bc2->fid.Volume &&
	bc1->fid.Vnode == bc2->fid.Vnode &&
	bc1->fid.Unique == bc2->fid.Unique;
}


int
bufcache_init(int size)
{
    BufCacheEntry *bc;
    int i;

    bufhash = hashtabnew(size, bufcachecmp, bufcachehash);
    if (bufhash == NULL) 
	errx(1, "bufcache_init: hashtabnew");

    bc = (BufCacheEntry *) calloc(size, sizeof(BufCacheEntry));
    if (bc == NULL) 
	errx(1, "bufcache_init: calloc");

    lrulist = listnew();
    if (lrulist == NULL)
	errx(1, "bufcache_init: listnew");

    for (i = 0 ; i < size ; i++)
    {
	bc[i].hits = 0;
	bc[i].buf = NULL;
	bc[i].fid = dummyfid;
	listaddtail(lrulist, &bc[i]);
    }

    nbufentries = size;
    return 0;
}

/*
 * Find a free entry to recycle
 */

static BufCacheEntry *
find_free_entry(void)
{
    Listitem *li = listtail(lrulist);
    BufCacheEntry *bc;
    
    bc = listdata(li);
    
    if (bc->buf)
	mbuf_end(bc->buf);
    bc->buf = NULL;
    bc->fid = dummyfid;

    listaddhead(lrulist, bc);

    return bc;
}

int 
bufcache_add(AFSFid fid, mbuf *buf)
{
    BufCacheEntry *bc, key;
    key.fid = fid;

    if (hashtabsearch(bufhash, &key))
	return EEXIST;
    
    bc = find_free_entry() ;
    if (bc == NULL)
	return ENOMEM;
    
    bc->fid = fid;
    mbuf_use(buf);
    bc->buf = buf;
    bc->hits = 0;
    
    return 0;
}

int 
bufcache_remove(AFSFid fid)
{
    BufCacheEntry *bc, key;

    key.fid = fid;
    if ((bc = hashtabsearch(bufhash, &key)) == NULL)
	return ENOENT;

    assert(bc->buf);

    mbuf_end(bc->buf);
    bc->buf = NULL;
    return 0;
}

mbuf *
bufcache_get(AFSFid fid)
{
    BufCacheEntry *bc, key;

    key.fid = fid;
    if ((bc = hashtabsearch(bufhash, &key)) == NULL)
	return NULL;

    assert(bc->buf);
    mbuf_use(bc->buf);
    
    return bc->buf;
}

