/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * XFS Cache functions
 */

#define __NO_VERSION__

#include <xfs/xfs_locl.h>
#include <xfs/xfs_common.h>
#include <xfs/xfs_cache.h>

RCSID("$Id: xfs_cache.c,v 1.8 1998/12/22 13:16:30 lha Exp $");

static Cacheentry xfs_cache[XFS_CACHE_ENTRIES];

void
xfs_cache_init()
{
    int i;

    for (i = 0; i < XFS_CACHE_ENTRIES; ++i) {
	xfs_cache[i].dir = NULL;
	xfs_cache[i].xnode = NULL;
	xfs_cache[i].name = NULL;
	xfs_cache[i].name_len = 0;
    }
}

#define TWELVE (sizeof(unsigned))
#define SEVENTYFIVE (6*sizeof(unsigned))
#define HIGH_BITS (~((unsigned)(~0) >> TWELVE))

static unsigned int
xfs_cache_hashfunc(struct xfs_node *dir, unsigned const char *name, unsigned char len)
{
    unsigned const char *s;
    unsigned g;
    unsigned int h = 0;
    
    for (s = name; s - name < len; ++s) {
	h = (h << TWELVE) + *s;
	if ((g = h & HIGH_BITS))
	    h = (h ^ (g >> SEVENTYFIVE)) & ~HIGH_BITS;
    }
    h += (unsigned int) dir;
    
    return h;
}

/* Get element from cache */

struct xfs_node *
xfs_cache_lookup(struct xfs_node *dir, const char *name, unsigned char len)
{
    Cacheentry *tmp;
    XFSDEB(XDEBCACHE,("xfs_cache_lookup dir: %p\n",dir));

    tmp = (&xfs_cache[xfs_cache_hashfunc(dir,name,len) % XFS_CACHE_ENTRIES]);

    if (tmp == NULL)
	return NULL;
    if (tmp->xnode == NULL) {
	XFSDEB(XDEBCACHE,("xfs_cache_lookup: empty entry\n"));
	return NULL;
    }
    if (tmp->dir != dir) {
	XFSDEB(XDEBCACHE,("xfs_cache_lookup: tmp->dir(%p) != dir(%p)\n",tmp->dir,dir));
	return NULL;
    }
    if (tmp->name_len != len) {
	XFSDEB(XDEBCACHE,("xfs_cache_lookup: tmp->name_len(%d) != len(%d)\n",tmp->name_len,len));
	return NULL;
    }
    if (memcmp(tmp->name,name,len) != 0) {
	XFSDEB(XDEBCACHE,("xfs_cache_lookup: tmp->name(%s) != name(%s)",tmp->name,name));
	return NULL;
    }
    XFSDEB(XDEBCACHE,("xfs_cache_lookup returning %p\n",tmp->xnode));
    return tmp->xnode;
}

/* add element to cache */
/* if an entry is already there, throw it */

void
xfs_cache_add(struct xfs_node *dir, char *name, unsigned char len, struct xfs_node *xnode)
{
    Cacheentry *h = &xfs_cache[xfs_cache_hashfunc(dir,name,len) % XFS_CACHE_ENTRIES];
    XFSDEB(XDEBCACHE,("xfs_cache_add dir: %p xnode: %p\n",dir,xnode));

    if (h->xnode) {
	XFSDEB(XDEBCACHE,("xfs_cache_add: collision\n"));
	XFSDEB(XDEBCACHE,("xfs_cache_add h->xnode: %p\n",h->xnode));
	XFSDEB(XDEBCACHE,("xfs_cache_add XNODE_TO_VNODE(h->xnode): %p\n",XNODE_TO_VNODE(h->xnode)));
	iput(XNODE_TO_VNODE(h->xnode));
	xfs_free(h->name, XFS_MEM_XFSCACHE);
	XFSDEB(XDEBCACHE,("xfs_cache_add: free finished\n"));
    }
    h->name = (char *) xfs_alloc(len, XFS_MEM_XFSCACHE);
    if (h->name == NULL) {
	panic("malloc failed in xfs_cache_add");
    }
    memcpy(h->name,name,len);
    h->xnode = xnode;
    h->dir = dir;
    h->name_len = len;
    XFSDEB(XDEBCACHE,("xfs_cache_add return\n"));
}

/* delete cache entry with specified xnode */
/* does not do iput on the inode */
/* XXX: this is O(n), n is number of entries in hash table */

void
xfs_cache_delete(struct xfs_node *xnode)
{
    unsigned int i;

    XFSDEB(XDEBCACHE,("xfs_cache_delete xnode:%p\n",xnode));
    for (i = 0; i < XFS_CACHE_ENTRIES; i++)
	if (xnode && xfs_cache[i].xnode == xnode) {
	    XFSDEB(XDEBCACHE, ("xfs_cache_delete deleting entry: xnode = %p\n",
			       xfs_cache[i].xnode));
	    xfs_cache[i].dir = NULL;
	    xfs_cache[i].xnode = NULL;
	    xfs_free(xfs_cache[i].name, XFS_MEM_XFSCACHE);
	    xfs_cache[i].name = NULL;
	    xfs_cache[i].name_len = 0;
	}
}

/* delete cache entry with specified dir */
/* does not do iput on the inode */
/* XXX: this is O(n), n is number of entries in hash table */

void
xfs_cache_delete_dir(struct xfs_node *dir)
{
    unsigned int i;

    XFSDEB(XDEBCACHE,("xfs_cache_delete_dir dir:%p\n",dir));
    for (i = 0; i < XFS_CACHE_ENTRIES; i++)
	if (dir && xfs_cache[i].dir == dir) {
	    XFSDEB(XDEBCACHE, ("xfs_cache_delete deleting entry: dir = %p\n",
			       xfs_cache[i].dir));
	    xfs_cache[i].dir = NULL;
	    xfs_cache[i].xnode = NULL;
	    xfs_free(xfs_cache[i].name, XFS_MEM_XFSCACHE);
	    xfs_cache[i].name = NULL;
	    xfs_cache[i].name_len = 0;
	}
}

/* Purge the cache */
/* Do iput on all cache entries and delete the entry */

void
xfs_cache_purge()
{
    unsigned int i;

    XFSDEB(XDEBCACHE,("xfs_cache_purge\n"));
    for (i = 0; i < XFS_CACHE_ENTRIES; i++)
	if (xfs_cache[i].xnode != NULL) {
	    XFSDEB(XDEBCACHE,("xfs_cache_purge xnode: %p\n",xfs_cache[i].xnode));
	    iput(XNODE_TO_VNODE(xfs_cache[i].xnode));
	    xfs_cache[i].dir = NULL;
	    xfs_cache[i].xnode = NULL;
	    xfs_free(xfs_cache[i].name, XFS_MEM_XFSCACHE);
	    xfs_cache[i].name = NULL;
	    xfs_cache[i].name_len = 0;
	}
}
