/*
 * Copyright (c) 1995, 1996, 1997, 1998, 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 *
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * XFS operations.
 */

#include <xfs/xfs_locl.h>
#include <vm/vm_mmap.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_common.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_syscalls.h>
#include <xfs/xfs_vnodeops.h>

RCSID("$Id: xfs_vnodeops-osf.c,v 1.12 1999/02/06 03:49:44 assar Exp $");

/*
 * vnode functions
 */

static int
xfs_open(struct vnode **vpp, int mode, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_open\n"));

    if (mode & FWRITE)
	return xfs_open_valid(*vpp, cred, XFS_OPEN_NW);
    else
	return xfs_open_valid(*vpp, cred, XFS_OPEN_NR);
}

static int
xfs_fsync(struct vnode *vp, int fflags, struct ucred *cred, int waitfor)
{
    return xfs_fsync_common(vp, cred, waitfor, xfs_curproc());
}

static int
xfs_close(struct vnode *vp, int fflags, struct ucred *cred)
{
    return xfs_close_common(vp, fflags, xfs_curproc(), cred);
}

static int
xfs_read(struct vnode *vp, struct uio *uio, int ioflag, struct ucred *cred)
{
    return xfs_read_common(vp, uio, ioflag, cred);
}

static int
xfs_write(struct vnode *vp, struct uio *uiop, int ioflag, struct ucred *cred)
{
    return xfs_write_common(vp, uiop, ioflag, cred);
}

static int
xfs_ioctl(struct vnode *vp, 
	  int com, 
	  caddr_t data, 
	  int fflag, 
	  struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_ioctl\n"));
    return EOPNOTSUPP;
}

static int
xfs_getattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return xfs_getattr_common(vp, vap, cred);
}

static int
xfs_setattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return xfs_setattr_common (vp, vap, cred);
}

static int
xfs_access(struct vnode *vp, int mode, struct ucred *cred)
{
    return xfs_access_common(vp, mode, cred);
}

static int
xfs_lookup(struct vnode *dvp, struct nameidata *ndp)
{
    return xfs_lookup_common(dvp, ndp, &ndp->ni_vp);
}

static int
xfs_create(struct nameidata *ndp, struct vattr *vap)
{
    struct vnode *dvp  = ndp->ni_dvp;
    const char *name   = ndp->ni_dent.d_name;
    struct ucred *cred = ndp->ni_cred;
    int error;

    XFSDEB(XDEBVNOPS, ("xfs_create: vap->va_mode = %o\n",
		       vap->va_mode));

    error = xfs_create_common(dvp,
			      name,
			      vap,
			      cred);
    if(error == 0)
	error = xfs_lookup_name (dvp, name, NULL, cred,
				 &ndp->ni_vp);

    XFSDEB(XDEBVNOPS, ("xfs_create: error = %d\n", error));
    
    return error;
}

static int
xfs_remove(struct nameidata *ndp)
{
    return xfs_remove_common(ndp->ni_dvp, 
			     ndp->ni_vp, 
			     ndp->ni_dent.d_name, 
			     ndp->ni_cred);
}

static int
xfs_rename(struct nameidata *from, struct nameidata *to)
{
    int error =  xfs_rename_common(from->ni_dvp, 
			     from->ni_vp, 
			     from->ni_dent.d_name, 
			     to->ni_dvp, 
			     to->ni_vp, 
			     to->ni_dent.d_name, 
			     from->ni_cred);
    vrele(to->ni_dvp);
    if (to->ni_vp)
	vrele(to->ni_vp);
    vrele(from->ni_dvp);
    vrele(from->ni_vp);
    
    return error;
}

static int
xfs_mkdir(struct nameidata *ndp,
	  struct vattr *vap)
{
    struct vnode *dvp  = ndp->ni_dvp;
    const char *name   = ndp->ni_dent.d_name;
    struct ucred *cred = ndp->ni_cred;
    int error;

    error = xfs_mkdir_common(dvp,
			     name,
			     vap,
			     cred);

    if(error == 0)
	error = xfs_lookup_name (dvp, name, NULL, cred,
				 &ndp->ni_vp);

    XFSDEB(XDEBVNOPS, ("xfs_mkdir: error = %d\n", error));
    
    return error;
}

static int
xfs_rmdir(struct nameidata *ndp)
{
    return xfs_rmdir_common(ndp->ni_dvp, 
			    ndp->ni_vp, 
			    ndp->ni_dent.d_name, 
			    ndp->ni_cred);
}

static int
xfs_readdir(struct vnode *vp, 
	    struct uio *uio, 
	    struct ucred *cred, 
	    int *eofflagp)
{
    return xfs_readdir_common(vp, uio, cred, eofflagp);
}

static int
xfs_link(struct vnode* vp, struct nameidata *ndp)
{
    return xfs_link_common(ndp->ni_dvp, vp, ndp->ni_dent.d_name, ndp->ni_cred);
}

static int
xfs_symlink(struct nameidata *ndp, struct vattr *vap, char *target)
{
    return xfs_symlink_common(ndp->ni_dvp, &ndp->ni_vp,
			      ndp->ni_dent.d_name, NULL,
			      ndp->ni_cred, vap, target);
}

static int
xfs_readlink(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
    return xfs_readlink_common (vp, uiop, cred);
}

static int
xfs_inactive(struct vnode *vp)
{
    return xfs_inactive_common(vp, NULL);
}

static int
xfs_reclaim(struct vnode *vp)
{
    return xfs_reclaim_common (vp);
}

static int
xfs_lock(struct vnode *vp, int flag, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_lock\n"));
    return EOPNOTSUPP;
}

static int
xfs_unlock(struct vnode *vp, int flag, struct ucred *cred)
{
    XFSDEB(XDEBVNOPS, ("xfs_unlock\n"));
    return EOPNOTSUPP;
}

static int
xfs_abortop(struct componentname *cnp)
{
    XFSDEB(XDEBVNOPS, ("xfs_abortop\n"));

#ifndef __osf__
    if ((cnp->cn_flags & (HASBUF | SAVESTART)) == HASBUF)
	FREE(cnp->cn_pnbuf, M_NAMEI);
#endif
    return 0;
}

static int
xfs_mmap(struct vnode *vp,
	 vm_offset_t offset,
	 vm_map_t map,
	 vm_offset_t *addrp,
	 vm_size_t len,
	 vm_prot_t prot,
	 vm_prot_t maxprot,
	 int flags,
	 struct ucred *cred)
{
#if 0
    struct vp_mmap_args args;
    extern kern_return_t u_vp_create(vm_map_t, 
				     struct vm_ubc_object *, 
				     vm_offset_t);
    
    XFSDEB(XDEBVNOPS, ("xfs_mmap\n"));

    args.a_offset = offset;
    args.a_vaddr = addrp;
    args.a_size = len;
    args.a_prot = prot;
    args.a_maxprot = maxprot;
    args.a_flags = flags;
    return u_vp_create(map, vp->v_object, (vm_offset_t) &args);
#else
    XFSDEB(XDEBVNOPS, ("xfs_mmap\n"));

    return EOPNOTSUPP;
#endif
}

static int
xfs_bmap(struct vnode *vp, 
	 daddr_t bn, 
	 struct vnode **vpp, 
	 daddr_t *bnp)
{
    XFSDEB(XDEBVNOPS, ("xfs_bmap\n"));
    return EOPNOTSUPP;
}

/*
 *      common routine for unsupported vnode ops 
 */
int
xfs_unsupp()
{
    return ENOSYS;
}

/*
 *      common routine for unsupported property-list ops 
 */
int
xfs_notsupp()
{
    return EOPNOTSUPP;
}

int
xfs_ok()
{
    return 0;
}

struct vnodeops xfs_vnodeops = {
    xfs_lookup,		/* lookup */
    xfs_create,		/* create */
    xfs_unsupp,		/* mknod */
    xfs_open,		/* open */
    xfs_close,		/* close */
    xfs_access,		/* access */
    xfs_getattr,		/* getattr */
    xfs_setattr,		/* setattr */
    xfs_read,		/* read */
    xfs_write,		/* write */
    xfs_ioctl,		/* ioctl */
    xfs_unsupp,		/* select */
    xfs_mmap,		/* mmap */
    xfs_fsync,		/* fsync */
    xfs_ok,		/* seek */
    xfs_remove,		/* remove */
    xfs_link,		/* link */
    xfs_rename,		/* rename */
    xfs_mkdir,		/* mkdir */
    xfs_rmdir,		/* rmdir */
    xfs_symlink,		/* symlink */
    xfs_readdir,		/* readdir */
    xfs_readlink,		/* readlink */
    xfs_ok,			/* abortop */
    xfs_inactive,		/* inactive */
    xfs_reclaim,		/* reclaim */
    xfs_unsupp,		/* bmap */
    xfs_unsupp,		/* strategy */
    xfs_unsupp,		/* print */
    xfs_unsupp,		/* page_read */
    xfs_unsupp,		/* page_write */
    xfs_unsupp,		/* swap handler */
    xfs_unsupp,		/* buffer read */
    xfs_unsupp,		/* buffer release */
    xfs_unsupp,		/* file locking */
    xfs_unsupp,		/* fsync byte range */
    xfs_lock,		/* Lock a node */
    xfs_unlock,		/* Unlock a node */
    xfs_notsupp,	/* Get extended attributes */
    xfs_notsupp,	/* Set extended attributes */
    xfs_notsupp,	/* Delete extended attributes */
    vn_pathconf,		/* pathconf */		
};

int
xfs_refer(vm_ubc_object_t vop)
{
    VREF(vop->vu_vp);
    return 0;
}

int
xfs_release(vm_ubc_object_t vop)
{
    vrele(vop->vu_vp);
    return 0;
}

int
xfs_getpage(vm_ubc_object_t vop,
	    vm_offset_t offset,
	    vm_size_t len,
	    vm_prot_t *protp,
	    vm_page_t *pl,
	    int plsz,
	    vm_map_entry_t ep,
	    vm_offset_t addr,
	    int rwflg,
	    struct ucred *cred)
{
#if 0
    vm_page_t page;
    page = ubc_kluster(vop, NULL, offset + fsbsize & ~(fsbsize - 1), fsbsize,
		       B_CACHE | B_WANTFREE | B_BUSY, UBC_HNONE, &kplcnt);
    if(page) {
	error = nfs_rwblk(vp, page, kplcnt, B_READ | B_ASYNC, 0, 
			  offset + fsbsize, cred);
	if(error) {
	    *pl = VM_PAGE_NULL;
	    return error;
	}
	
    }
    if(ubc_incore(vop, offset, len))
	nfs_validate_caches(vp, cred);
    uflags = 0;
    error = ubc_lookup(vop, trunc_page(offset), fsbsize, len + offset, 
		       &pp, &uflags);
    if(error) {
	*pl = VM_PAGE_NULL;
	return error;
    }
    if(uflags & B_DIRTY);
    if(uflags & B_NOCACHE);
    pl[0] = pp;
    pl[1] = VM_PAGE_NULL;
    return 0;
    
#else
    printf("xfs_getpage: vop = %p, offset = %lx, len = %ld\n", 
	   vop, offset, len);
    printf("xfs_getpage: protp = %p, ep = %p, addr = %p, rwflg = %d\n", 
	   protp, ep, addr, rwflg);
    *pl = VM_PAGE_NULL;
    return EINVAL;
#endif
}

int
xfs_putpage(vm_ubc_object_t vop,
	    vm_page_t *pl,
	    int pcnt,
	    int flags,
	    struct ucred *cred)
{
    printf("xfs_putpage\n");
    return EINVAL;
}

int
xfs_write_check(vm_ubc_object_t vpp, vm_page_t pp)
{
    return TRUE;
}

struct vfs_ubcops xfs_ubcops = {
    xfs_refer,              /* refer vnode */
    xfs_release,            /* release vnode */
    xfs_getpage,            /* get page */
    xfs_putpage,            /* put page */
    xfs_write_check         /* check writability */
};
