/*
Copyright (C) 1992,1993,1994,1995 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/MOSS source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/


/* This locking package has one limitation, that being the file to be locked
   MUST exist.  This is where the "filchk" utility comes in handy.

   The file must exist because the device/inode pair for the file is used to
   create the lock file name.  This file name is guaranteed to be exactly 14
   characters long, and thus ports nicely to SYS5.

   The lock file is created in the directory where the file exists.  This has
   the good side effect of making the lock effective across remote file
   systems.  This has the unfortunate side effect of suggesting the file
   should not be locked open when you are opening it just to read.

   The reason is the directory the file is in may not be writeable to you and
   the open will fail if you try to lock it open.  This means code opening
   files for reading must be able to deal with files whose contents may change
   while being read, ie if the file has a specific syntax, you had better be
   able to deal with syntactically incorrect files.

   Finally, a process locking open files for writing must have write access to
   the directory the file is in.

   THE ROUTINES IN THIS FILE MAY NOT LOG, NOR MAY THEY USE ANYTHING
   THAT DOES LOG, SINCE THE LOGGING PACKAGE USES THEM.
*/

#include "config.h"

#include <sys/stat.h>
#include <stdio.h>

#include "general.h"
#include "util.h"

static int lockit(), lockname(), timerON(), timerOFF();


lk_open(file, access)
char *file;
int access;
{
    int i, j;
    long curtime;
    char *curlock = NULLCP, *tmplock = NULLCP;
    struct stat st;

    if (file == NULLCP || stat(file, &st) == NOTOK) 
	return(NOTOK);

    (void) lockname(&curlock, &tmplock, file,
		    (int)(st.st_dev), (int)(st.st_ino));

    for (i = 0;;)
	switch (lockit(tmplock, curlock)) {
	case OK: 
	    if ((i = open(file, access)) == NOTOK) {
		j = errno;
		(void) unlink(curlock);
		errno = j;
		return(NOTOK);
	    }
	    timerON(curlock, i);
	    return i;

	case EACCES:
	    errno = EACCES;
	    return -1;

	default:
	    if (stat(curlock, &st) == NOTOK) {
		if (i++ > 5)
		    return NOTOK;
		sleep(4);
		break;
	    }

	    i = 0;
	    curtime = TIME();
	    if (curtime < (long)st.st_ctime + 60L)
		sleep(4);
	    else
		(void) unlink(curlock);
	    break;
	}
}


static lockit(tmp, file)
char   *tmp,
       *file;
{
#ifdef PC
    return OK;
#else
    int    fd;

    if ((fd = creat (tmp, 0400)) == NOTOK) 
	return NOTOK;

    (void) close (fd);

    fd = OK;
    if (link(tmp, file)) 
	fd = errno;

    (void) unlink(tmp);

    return(fd);
#endif
}

static lockname (curlock, tmplock, file, dev, ino)
char **curlock, **tmplock, *file;
int dev, ino;
{
    char *cp;
    int i;

    /* error checking minimized since we are static and called correctly */

    FREE(*curlock);
    FREE(*tmplock);

    if ((cp = RINDEX(file, PATH_SEP)) == NULLCP)
	cp = file;
    else
	++cp;

    if ((*curlock = calloc(1, (unsigned)((cp - file) + 16))) == NULLCP) {
	return;
    }

    (void) strncpy(*curlock, file, cp - file);
    i = (char *)cp - (char *)file;
#ifndef PC
    (void) sprintf(*curlock + i, "LCK%05d.%05d", dev % 100000, ino % 100000);
#else
    (void) sprintf(*curlock + i, "%05d.LCK", ino % 100000);
#endif

    /* this gives us a bit more space than we need, but it is easier than
       trying to figure out exactly what we need */
    if ((*tmplock = calloc(1, (unsigned)(strlen(*curlock) + 12))) == NULLCP)
	return;

    if ((cp = RINDEX(*curlock, PATH_SEP)) == NULLCP || *++cp == NULLC)
#ifndef PC
	(void) strcpy(*tmplock, ",LCK.XXXXXX");
#else
        (void) sprintf(*tmplock, "%.*sXXXXXX.LCK", cp - *curlock, *curlock);
#endif
    else
#ifndef PC
	(void) sprintf(*tmplock, "%.*s,LCK.XXXXXX", cp - *curlock, *curlock);
#else   
        (void) strcpy(*tmplock, ",LCK.XXXXXX");
#endif
    (void) unlink(mktemp(*tmplock));
}

lk_close(fd)
int     fd;
{
    if (fd == NOTOK)
	return OK;

    timerOFF(fd);

    return(close(fd));
}

FILE	*lk_fopen (file, mode)
char   *file, *mode;
{
    int    fd;
    FILE  *fp;

    if ((fd = lk_open(file, strcmp(mode, "r") ? 2 : 0)) == NOTOK)
	return((FILE *)0);

    if ((fp = fdopen (fd, mode)) == (FILE *)0) {
	(void) lk_close(fd);
    }

    return fp;
}

int	lk_fclose (fp)
FILE	*fp;
{
    int ret, fd;

    if (fp == (FILE *)0)
	return OK;

    fd = fileno(fp);
    ret = fclose(fp);
    (void) lk_close(fd);

    return(ret);
}

#include <signal.h>

#define	NSECS	((unsigned)30)

struct lock {
    int		 l_fd;
    char	*l_lock;
    struct lock *l_next;
};
#define	NULLP	((struct lock *) 0)

static struct lock *l_top = NULLP;

/* ARGSUSED */

static void alrmser (sig)
int	sig;
{
    int    j;
    char  *cp;
    struct lock   *tp;

#ifndef	BSD42
#ifndef PC
    (void) signal (SIGALRM, alrmser);
#endif /* PC */
#endif	/* BSD42 */

    for (tp = l_top; tp != NULLP; tp = tp -> l_next)
	if (*(cp = tp -> l_lock) != NULLC && (j = creat(cp, 0400)) != NOTOK)
	    (void) close(j);

    (void) alarm(NSECS);
}

static timerON (lock, fd)
char   *lock;
int	fd;
{
    struct lock   *tp;

    if ((tp = (struct lock *) malloc ((unsigned) (sizeof *tp))) == NULLP)
	return;			/* XXX */

    tp -> l_fd = fd;
    if ((tp -> l_lock = malloc ((unsigned) (strlen (lock) + 1))) == NULLCP) {
	free ((char *) tp);
	return;			/* XXX */
    }
    (void) strcpy (tp -> l_lock, lock);
    tp -> l_next = NULLP;

    if (l_top)
	tp -> l_next = l_top -> l_next;
    else {
#ifndef PC
	(void) signal (SIGALRM, alrmser);/* perhaps SIGT{STP,TIN,TOU} */
#endif
#ifndef MAC
	(void) alarm (NSECS);
#endif
    }
    l_top = tp;
}


#define	FREE_LOCK(x)	if ((x) != NULLP) { \
			    FREE((x) -> l_lock); \
			    free((char *) (x)); \
			    (x) = NULLP; \
			} \
			else

static timerOFF(fd)
int	fd;
{
    struct lock   *pp, *tp;

    (void) alarm (0);

    if (l_top != NULLP) {
	for (pp = tp = l_top; tp != NULLP; pp = tp, tp = tp -> l_next)
	    if (tp -> l_fd == fd)
		break;
	if (tp != NULLP) {
	    if (tp == l_top)
		l_top = tp -> l_next;
	    else
		pp -> l_next = tp -> l_next;

	    if (tp -> l_lock != NULLCP)
		(void) unlink(tp -> l_lock);
	    FREE_LOCK(tp);
	}
    }

    if (l_top != NULLP)
	(void) alarm(NSECS);
}

