/*
 * Copyright (c) 1992 by Sun Microsystems, Inc.
 */

#include <sys/param.h>
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <sys/cmn_err.h>
#include <sys/debug.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/kmem.h>
#include <sys/types.h>

#include <fist.h>
#include <cryptfs.h>

extern vnodeops_t fist_crypt_vnodeops;

pid_t getsid(pid_t);
extern pid_t getpid();
pid_t setsid(void);

#ifdef FIST_CRYPTDEBUG
int fist_cryptdebug = 4;
#endif

unsigned char cbc_iv[8] = {0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10};


void
fist_crypt_encodefilename(
			  vfs_t *vfsp,
			  char *name,
			  char **uuencoded_name,
			  int *uuencoded_length,
			  int skip_dots,
			  cred_t *cr)
{
  char *crypted_name,*ptr;
  int length, rounded_length, n, i, j;
  unsigned char iv[8];
  short csum;

  ASSERT(fist_get_userpass(vfsp, cr) != NULL);

  if (skip_dots && (name[0] == '.' &&
		    (name[1] == '\0' ||
		     (name[1] == '.' && name[2] == '\0')))) {
    *uuencoded_length = strlen(name) + 1;
    *uuencoded_name = kmem_zalloc(*uuencoded_length, KM_SLEEP);
    strcpy(*uuencoded_name, name);
    return;
  }
  for (csum = 0, length = 0, ptr = name; *ptr; ptr++, length++)
    csum += *ptr;
  /*
   * rounded_length is an multiple of 3 rounded-up length
   * the uuencode algorithm processes 3 source bytes at a time
   * so we have to make sure we don't read past the memory
   * we have allocated
   *
   * it uses length + 3 to provide 2 bytes for the checksum
   * and one byte for the length
   */
  rounded_length = (((length + 3) + 2) / 3) * 3;
  crypted_name = kmem_zalloc(rounded_length, KM_SLEEP);

  bcopy(cbc_iv, iv, 8);
  n = 0;
  *(short *)crypted_name = csum;
  crypted_name[2] = length;
  BF_cfb64_encrypt(name, crypted_name + 3,
		   length, fist_get_userpass(vfsp, cr), iv, &n,
		   BF_ENCRYPT);

  *uuencoded_length = (((length + 3) + 2) / 3) * 4 + 1;
  *uuencoded_name = kmem_alloc(*uuencoded_length, KM_SLEEP);

  for (i = 0, j = 0; i < rounded_length; i += 3, j += 4) {
    (*uuencoded_name)[j] = 48 + ((crypted_name[i] >> 2) & 63);
    (*uuencoded_name)[j + 1] = 48 + (((crypted_name[i] << 4) & 48) | ((crypted_name[i + 1] >> 4) & 15));
    (*uuencoded_name)[j + 2] = 48 + (((crypted_name[i + 1] << 2) & 60) | ((crypted_name[i + 2] >> 6) & 3));
    (*uuencoded_name)[j + 3] = 48 + (crypted_name[i + 2] & 63);
  }
  (*uuencoded_name)[j] = '\0';

  kmem_free(crypted_name, rounded_length);
}


int
fist_crypt_decodefilename(vfs_t *vfsp,
			  char *name,
			  int length,
			  char *decrypted_name,
			  int *decrypted_length, /* w/ terminating null */
			  int skip_dots,
			  cred_t *cr)
{
  int n, i, j, saved_length, saved_csum, csum;
  int uudecoded_length, error = 0;
  unsigned char iv[8];
  char *uudecoded_name;

  ASSERT(fist_get_userpass(vfsp, cr) != NULL);

  if (skip_dots && (name[0] == '.' &&
		    (name[1] == '\0' ||
		     (name[1] == '.' && name[2] == '\0')))) {
    /*used to add 1 to following*/
    *decrypted_length = length;
//    *decrypted_name = kmem_alloc(*decrypted_length, KM_SLEEP);
    for (i = 0; i < length; i++)
      decrypted_name[i] = name[i];
    decrypted_name[i] = 0;
    return 0;
  }

  uudecoded_length = ((length + 3) / 4) * 3;
  uudecoded_name = kmem_alloc(uudecoded_length, KM_SLEEP);

  for (i = 0, j = 0; i < length; i += 4, j += 3) {
    uudecoded_name[j] = ((name[i] - 48) <<2) | ((name[i + 1] - 48) >>4);
    uudecoded_name[j + 1] = (((name[i + 1] - 48) <<4) & 240) | ((name[i + 2] - 48) >>2);
    uudecoded_name[j + 2] = (((name[i + 2] - 48) <<6) & 192) | ((name[i + 3] - 48) &63);
  }
  saved_csum = *(short *)uudecoded_name;
  saved_length = uudecoded_name[2];
  if (saved_length > uudecoded_length) {
    printk("Problems with the length - too big: %d", saved_length);
    error = -1;
    goto out;
  }
//  *decrypted_name = (char *)kmem_alloc(saved_length + 1, KM_SLEEP);
  bcopy(cbc_iv, iv, 8);
  n = 0;
  BF_cfb64_encrypt(uudecoded_name + 3, decrypted_name,
		   saved_length, fist_get_userpass(vfsp, cr), iv, &n,
		   BF_DECRYPT);
  for (csum = 0, i = 0; i < saved_length; i++)
    csum += decrypted_name[i];
  if (csum != saved_csum) {
    fist_dprint(6,"Checksum error\n");
//    kmem_free(*decrypted_name, saved_length + 1);
    error = -1;
    goto out;
  }
  decrypted_name[saved_length] = 0;
  *decrypted_length = saved_length + 1;
out:
  kmem_free(uudecoded_name, uudecoded_length);
  return error;
}


int
crypt_encode_block(char *func, int line, caddr_t base, int len, const vnode_t *vp,cred_t *cr)
{
  /* blowfish variables */
  unsigned char iv[8];
  int n;

  ASSERT(fist_get_userpass(vp->v_vfsp,cr) != NULL);

  if (len > PAGESIZE || len < 0)
    printk("CEB: %s:%d: base=%x, len=%d, vp=%x\n",
	   func, line, (int) base, len, (int) vp);

  bcopy(cbc_iv, iv, 8);
  n = 0;		/* opaque variable for blowfish's internal stat */
  BF_cfb64_encrypt(base,
		   base,
		   /* len, fist_get_bf_key(vp->v_vfsp, cr), iv, &n, */
		   len, fist_get_userpass(vp->v_vfsp, cr), iv, &n,
		   BF_ENCRYPT);
  return len;
}


int
crypt_decode_block(char *func, int line, caddr_t base, int len, const vnode_t *vp, cred_t *cr)
{
  /* blowfish variables */
  unsigned char iv[8];
  int n;

  ASSERT(fist_get_userpass(vp->v_vfsp,cr) != NULL);

  if (len > PAGESIZE || len < 0)
    printk("CDB: %s:%d: base=%x, len=%d, vp=%x\n",
	   func, line, (int) base, len, (int) vp);
  bcopy(cbc_iv, iv, 8);
  n = 0;		/* opaque variable for blowfish's internal stat */
  BF_cfb64_encrypt(base,
		   base,
		   len, fist_get_userpass(vp->v_vfsp, cr), iv, &n,
		   BF_DECRYPT);

  return len;
}


BF_KEY *
fist_get_userpass(const vfs_t * this_vfsp, cred_t * cr)
{
  fist_key_t *ht_entry;
  int pos = (cr->cr_uid & FIST_HASH_MASK);
  struct fist_cryptinfo *fwip = vfstofwi(this_vfsp);

  rw_enter(&fwip->fwi_user_rwlock, RW_READER);

  for (ht_entry = fwip->fwi_user[pos];
       ht_entry;
       ht_entry = ht_entry->next) {
    if ((ht_entry->uid == cr->cr_uid)
	/*	&& ((ht_entry->sid)== (getsid(getpid()))) */
	) {
      fist_dprint(5, "userhash found entry");
      rw_exit(&fwip->fwi_user_rwlock);
      return &ht_entry->key;
    }
  }
  fist_dprint(5, "User key HT did not find entry with uid %d\n", cr->cr_uid);
  printk("User key HT did not find entry with uid %d\n", cr->cr_uid);
  printk("uid %d, ruid %d, saved uid %d\n",cr->cr_uid, cr->cr_ruid,cr->cr_suid);
  rw_exit(&fwip->fwi_user_rwlock);
  return NULL;
}


void
fist_set_userpass(const vfs_t * this_vfsp, unsigned char *key, cred_t * cr)
{
  fist_key_t *ht_entry;
  int pos = (cr->cr_uid & FIST_HASH_MASK);
  struct fist_cryptinfo *fwip = vfstofwi(this_vfsp);

  printk("fist_set_userpass: vfsp:%x\n", this_vfsp);

  rw_enter(&fwip->fwi_user_rwlock, RW_WRITER);

  for (ht_entry = fwip->fwi_user[pos];
       ht_entry;
       ht_entry = ht_entry->next) {
    if (ht_entry->uid == cr->cr_uid) {
      /* key present just have to change it */

      /*       ht_entry->sid = setsid(); */
      BF_set_key(&ht_entry->key, 16, key);
      goto out;
    }
  }
  /* key not present */
  ht_entry = kmem_alloc(sizeof(fist_key_t), KM_SLEEP);
  if (!ht_entry) {
    printk("fist_set_userpass NO MORE MEMORY!\n");
    goto out;
  }
  ht_entry->uid = cr->cr_uid;
  /*   ht_entry->sid = (curproc->p_pidp)->pid_id; */
  BF_set_key(&ht_entry->key, 16, key);
  ht_entry->next = fwip->fwi_user[pos];
  fwip->fwi_user[pos] = ht_entry;

out:
  rw_exit(&fwip->fwi_user_rwlock);
}


void
fist_free_userpass(const vfs_t * this_vfsp)
{
  fist_key_t *ht_entry, *next_ht;
  int bucket;
  struct fist_cryptinfo *fwip = vfstofwi(this_vfsp);

  printk("fist_free_userpass: vfsp:%x\n", this_vfsp);

  rw_enter(&fwip->fwi_user_rwlock, RW_WRITER);
  for (bucket=0; bucket < FIST_HASH_SIZE; ++bucket) {
    for (ht_entry = fwip->fwi_user[bucket];
	 ht_entry;
	 ht_entry = next_ht) {
      next_ht = ht_entry->next;
      kmem_free(ht_entry, sizeof(fist_key_t));
    }
    fwip->fwi_user[bucket] = NULL;
  }
  rw_exit(&fwip->fwi_user_rwlock);
}
