/*
 * This is version 1.1 of CryptoLib
 *
 * The authors of this software are Jack Lacy, Don Mitchell and Matt Blaze
 *              Copyright (c) 1991, 1992, 1993, 1994, 1995 by AT&T.
 * Permission to use, copy, and modify this software without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies of any software which is or includes a copy or
 * modification of this software and in all copies of the supporting
 * documentation for such software.
 *
 * NOTE:
 * Some of the algorithms in cryptolib may be covered by patents.
 * It is the responsibility of the user to ensure that any required
 * licenses are obtained.
 *
 *
 * SOME PARTS OF CRYPTOLIB MAY BE RESTRICTED UNDER UNITED STATES EXPORT
 * REGULATIONS.
 *
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR AT&T MAKE ANY
 * REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY
 * OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/*
 *        Data Encryption Standard
 *        D.P.Mitchell  83/06/08.
 *
 *        Copyright 1994 AT&T Bell Laboratories
 *
 *        block_cipher(unsigned char expanded_key[128], unsigned char block[8], int decrypting)
 */
#include "libcrypt.h"

static void fp P((unsigned long, unsigned long, unsigned char []));
static unsigned long ip_low P((register unsigned char []));
static unsigned long ip_high P((register unsigned char []));

/*
 *        Tables for Combined S and P Boxes
 */

static unsigned long  s0p[] = {
	0x00410100,0x00010000,0x40400000,0x40410100,0x00400000,0x40010100,0x40010000,0x40400000,
	0x40010100,0x00410100,0x00410000,0x40000100,0x40400100,0x00400000,0x00000000,0x40010000,
	0x00010000,0x40000000,0x00400100,0x00010100,0x40410100,0x00410000,0x40000100,0x00400100,
	0x40000000,0x00000100,0x00010100,0x40410000,0x00000100,0x40400100,0x40410000,0x00000000,
	0x00000000,0x40410100,0x00400100,0x40010000,0x00410100,0x00010000,0x40000100,0x00400100,
	0x40410000,0x00000100,0x00010100,0x40400000,0x40010100,0x40000000,0x40400000,0x00410000,
	0x40410100,0x00010100,0x00410000,0x40400100,0x00400000,0x40000100,0x40010000,0x00000000,
	0x00010000,0x00400000,0x40400100,0x00410100,0x40000000,0x40410000,0x00000100,0x40010100,
};

static unsigned long  s1p[] = {
	0x08021002,0x00000000,0x00021000,0x08020000,0x08000002,0x00001002,0x08001000,0x00021000,
	0x00001000,0x08020002,0x00000002,0x08001000,0x00020002,0x08021000,0x08020000,0x00000002,
	0x00020000,0x08001002,0x08020002,0x00001000,0x00021002,0x08000000,0x00000000,0x00020002,
	0x08001002,0x00021002,0x08021000,0x08000002,0x08000000,0x00020000,0x00001002,0x08021002,
	0x00020002,0x08021000,0x08001000,0x00021002,0x08021002,0x00020002,0x08000002,0x00000000,
	0x08000000,0x00001002,0x00020000,0x08020002,0x00001000,0x08000000,0x00021002,0x08001002,
	0x08021000,0x00001000,0x00000000,0x08000002,0x00000002,0x08021002,0x00021000,0x08020000,
	0x08020002,0x00020000,0x00001002,0x08001000,0x08001002,0x00000002,0x08020000,0x00021000,
};

static unsigned long  s2p[] = {
	0x20800000,0x00808020,0x00000020,0x20800020,0x20008000,0x00800000,0x20800020,0x00008020,
	0x00800020,0x00008000,0x00808000,0x20000000,0x20808020,0x20000020,0x20000000,0x20808000,
	0x00000000,0x20008000,0x00808020,0x00000020,0x20000020,0x20808020,0x00008000,0x20800000,
	0x20808000,0x00800020,0x20008020,0x00808000,0x00008020,0x00000000,0x00800000,0x20008020,
	0x00808020,0x00000020,0x20000000,0x00008000,0x20000020,0x20008000,0x00808000,0x20800020,
	0x00000000,0x00808020,0x00008020,0x20808000,0x20008000,0x00800000,0x20808020,0x20000000,
	0x20008020,0x20800000,0x00800000,0x20808020,0x00008000,0x00800020,0x20800020,0x00008020,
	0x00800020,0x00000000,0x20808000,0x20000020,0x20800000,0x20008020,0x00000020,0x00808000,
};

static unsigned long  s3p[] = {
	0x00080201,0x02000200,0x00000001,0x02080201,0x00000000,0x02080000,0x02000201,0x00080001,
	0x02080200,0x02000001,0x02000000,0x00000201,0x02000001,0x00080201,0x00080000,0x02000000,
	0x02080001,0x00080200,0x00000200,0x00000001,0x00080200,0x02000201,0x02080000,0x00000200,
	0x00000201,0x00000000,0x00080001,0x02080200,0x02000200,0x02080001,0x02080201,0x00080000,
	0x02080001,0x00000201,0x00080000,0x02000001,0x00080200,0x02000200,0x00000001,0x02080000,
	0x02000201,0x00000000,0x00000200,0x00080001,0x00000000,0x02080001,0x02080200,0x00000200,
	0x02000000,0x02080201,0x00080201,0x00080000,0x02080201,0x00000001,0x02000200,0x00080201,
	0x00080001,0x00080200,0x02080000,0x02000201,0x00000201,0x02000000,0x02000001,0x02080200,
};

static unsigned long  s4p[] = {
	0x01000000,0x00002000,0x00000080,0x01002084,0x01002004,0x01000080,0x00002084,0x01002000,
	0x00002000,0x00000004,0x01000004,0x00002080,0x01000084,0x01002004,0x01002080,0x00000000,
	0x00002080,0x01000000,0x00002004,0x00000084,0x01000080,0x00002084,0x00000000,0x01000004,
	0x00000004,0x01000084,0x01002084,0x00002004,0x01002000,0x00000080,0x00000084,0x01002080,
	0x01002080,0x01000084,0x00002004,0x01002000,0x00002000,0x00000004,0x01000004,0x01000080,
	0x01000000,0x00002080,0x01002084,0x00000000,0x00002084,0x01000000,0x00000080,0x00002004,
	0x01000084,0x00000080,0x00000000,0x01002084,0x01002004,0x01002080,0x00000084,0x00002000,
	0x00002080,0x01002004,0x01000080,0x00000084,0x00000004,0x00002084,0x01002000,0x01000004,
};

static unsigned long  s5p[] = {
	0x10000008,0x00040008,0x00000000,0x10040400,0x00040008,0x00000400,0x10000408,0x00040000,
	0x00000408,0x10040408,0x00040400,0x10000000,0x10000400,0x10000008,0x10040000,0x00040408,
	0x00040000,0x10000408,0x10040008,0x00000000,0x00000400,0x00000008,0x10040400,0x10040008,
	0x10040408,0x10040000,0x10000000,0x00000408,0x00000008,0x00040400,0x00040408,0x10000400,
	0x00000408,0x10000000,0x10000400,0x00040408,0x10040400,0x00040008,0x00000000,0x10000400,
	0x10000000,0x00000400,0x10040008,0x00040000,0x00040008,0x10040408,0x00040400,0x00000008,
	0x10040408,0x00040400,0x00040000,0x10000408,0x10000008,0x10040000,0x00040408,0x00000000,
	0x00000400,0x10000008,0x10000408,0x10040400,0x10040000,0x00000408,0x00000008,0x10040008,
};

static unsigned long  s6p[] = {
	0x00000800,0x00000040,0x00200040,0x80200000,0x80200840,0x80000800,0x00000840,0x00000000,
	0x00200000,0x80200040,0x80000040,0x00200800,0x80000000,0x00200840,0x00200800,0x80000040,
	0x80200040,0x00000800,0x80000800,0x80200840,0x00000000,0x00200040,0x80200000,0x00000840,
	0x80200800,0x80000840,0x00200840,0x80000000,0x80000840,0x80200800,0x00000040,0x00200000,
	0x80000840,0x00200800,0x80200800,0x80000040,0x00000800,0x00000040,0x00200000,0x80200800,
	0x80200040,0x80000840,0x00000840,0x00000000,0x00000040,0x80200000,0x80000000,0x00200040,
	0x00000000,0x80200040,0x00200040,0x00000840,0x80000040,0x00000800,0x80200840,0x00200000,
	0x00200840,0x80000000,0x80000800,0x80200840,0x80200000,0x00200840,0x00200800,0x80000800,
};

static unsigned long  s7p[] = {
	0x04100010,0x04104000,0x00004010,0x00000000,0x04004000,0x00100010,0x04100000,0x04104010,
	0x00000010,0x04000000,0x00104000,0x00004010,0x00104010,0x04004010,0x04000010,0x04100000,
	0x00004000,0x00104010,0x00100010,0x04004000,0x04104010,0x04000010,0x00000000,0x00104000,
	0x04000000,0x00100000,0x04004010,0x04100010,0x00100000,0x00004000,0x04104000,0x00000010,
	0x00100000,0x00004000,0x04000010,0x04104010,0x00004010,0x04000000,0x00000000,0x00104000,
	0x04100010,0x04004010,0x04004000,0x00100010,0x04104000,0x00000010,0x00100010,0x04004000,
	0x04104010,0x00100000,0x04100000,0x04000010,0x00104000,0x00004010,0x04004010,0x04100000,
	0x00000010,0x04104000,0x00104010,0x00000000,0x04000000,0x04100010,0x00004000,0x00104010,
};

/*
 *        DES electronic codebook encryption of one block
 */
#ifdef K_AND_R
_TYPE( void )
block_cipher(expanded_key, text, decrypting)
  unsigned char expanded_key[128];
  unsigned char text[8];
  int decrypting;
#else
_TYPE( void ) block_cipher(unsigned char expanded_key[128],
			   unsigned char text[8],
			   int decrypting)
#endif
{
	register unsigned char *key;
	register unsigned long temp;
	register unsigned long left, right;
	register int i;
	register int key_offset;
	
	key = expanded_key;
	left = ip_low(text);
	right = ip_high(text);
	if (decrypting) {
		key_offset = 16 - 8;
		key = key + 128 - 8;
	} else
		key_offset = 0 - 8;
	for (i = 0; i < 16; i++) {
		temp = (right << 1) | ((right >> 31) & 1);
		left ^= s0p[(int)(((temp      ) & 0x3f) ^ key[0])]
		      ^ s1p[(int)(((temp >>  4) & 0x3f) ^ key[1])]
		      ^ s2p[(int)(((temp >>  8) & 0x3f) ^ key[2])]
		      ^ s3p[(int)(((temp >> 12) & 0x3f) ^ key[3])]
		      ^ s4p[(int)(((temp >> 16) & 0x3f) ^ key[4])]
		      ^ s5p[(int)(((temp >> 20) & 0x3f) ^ key[5])]
		      ^ s6p[(int)(((temp >> 24) & 0x3f) ^ key[6])];
		temp = ((right & 1) << 5) | ((right >> 27) & 0x1f);
		left ^= s7p[(int)(temp ^ key[7])];
		temp = left;
		left = right;
		right = temp;
		key -= key_offset;
	}
	temp = left;
	left = right;
	right = temp;
	/*
	 *        standard final permutation (IPI)
	 */
	fp(left, right, text);
}

#ifdef K_AND_R
_TYPE( void )
triple_block_cipher(expanded_key, text, decrypting)
  unsigned char expanded_key[3][128];
  unsigned char text[8];
  int decrypting;
#else
_TYPE( void ) triple_block_cipher(unsigned char expanded_key[3][128],
				  unsigned char text[8],
				  int decrypting)
#endif
{
	register unsigned char *key;
	register unsigned long temp;
	register unsigned long left, right;
	register int i, j;
	register int key_offset;
	
	left = ip_low(text);
	right = ip_high(text);
	
	for (j=0; j<3; j++) {
		key = expanded_key[j];
		if (decrypting) {
			key_offset = 16 - 8;
			key = key + 128 - 8;
		} else
			key_offset = 0 - 8;
		for (i = 0; i < 16; i++) {
			temp = (right << 1) | ((right >> 31) & 1);
			left ^= s0p[(int)(((temp      ) & 0x3f) ^ key[0])]
				^ s1p[(int)(((temp >>  4) & 0x3f) ^ key[1])]
					^ s2p[(int)(((temp >>  8) & 0x3f) ^ key[2])]
						^ s3p[(int)(((temp >> 12) & 0x3f) ^ key[3])]
							^ s4p[(int)(((temp >> 16) & 0x3f) ^ key[4])]
								^ s5p[(int)(((temp >> 20) & 0x3f) ^ key[5])]
									^ s6p[(int)(((temp >> 24) & 0x3f) ^ key[6])];
			temp = ((right & 1) << 5) | ((right >> 27) & 0x1f);
			left ^= s7p[(int)(temp ^ key[7])];
			temp = left;
			left = right;
			right = temp;
			key -= key_offset;
		}
		temp = left;
		left = right;
		right = temp;
		
	}
	/*
	 *        standard final permutation (IPI)
	 */
	fp(left, right, text);
}

/*
 *        Final Permutation
 */
static unsigned long        fph0[16] = {
	0x00000000,0x40000000,0x00400000,0x40400000,0x00004000,0x40004000,0x00404000,0x40404000,
	0x00000040,0x40000040,0x00400040,0x40400040,0x00004040,0x40004040,0x00404040,0x40404040,
};
static unsigned long        fpl1[16] = {
	0x00000000,0x40000000,0x00400000,0x40400000,0x00004000,0x40004000,0x00404000,0x40404000,
	0x00000040,0x40000040,0x00400040,0x40400040,0x00004040,0x40004040,0x00404040,0x40404040,
};
static unsigned long        fph2[16] = {
	0x00000000,0x10000000,0x00100000,0x10100000,0x00001000,0x10001000,0x00101000,0x10101000,
	0x00000010,0x10000010,0x00100010,0x10100010,0x00001010,0x10001010,0x00101010,0x10101010,
};
static unsigned long        fpl3[16] = {
	0x00000000,0x10000000,0x00100000,0x10100000,0x00001000,0x10001000,0x00101000,0x10101000,
	0x00000010,0x10000010,0x00100010,0x10100010,0x00001010,0x10001010,0x00101010,0x10101010,
};
static unsigned long        fph4[16] = {
	0x00000000,0x04000000,0x00040000,0x04040000,0x00000400,0x04000400,0x00040400,0x04040400,
	0x00000004,0x04000004,0x00040004,0x04040004,0x00000404,0x04000404,0x00040404,0x04040404,
};
static unsigned long        fpl5[16] = {
	0x00000000,0x04000000,0x00040000,0x04040000,0x00000400,0x04000400,0x00040400,0x04040400,
	0x00000004,0x04000004,0x00040004,0x04040004,0x00000404,0x04000404,0x00040404,0x04040404,
};
static unsigned long        fph6[16] = {
	0x00000000,0x01000000,0x00010000,0x01010000,0x00000100,0x01000100,0x00010100,0x01010100,
	0x00000001,0x01000001,0x00010001,0x01010001,0x00000101,0x01000101,0x00010101,0x01010101,
};
static unsigned long        fpl7[16] = {
	0x00000000,0x01000000,0x00010000,0x01010000,0x00000100,0x01000100,0x00010100,0x01010100,
	0x00000001,0x01000001,0x00010001,0x01010001,0x00000101,0x01000101,0x00010101,0x01010101,
};
static unsigned long        fph8[16] = {
	0x00000000,0x80000000,0x00800000,0x80800000,0x00008000,0x80008000,0x00808000,0x80808000,
	0x00000080,0x80000080,0x00800080,0x80800080,0x00008080,0x80008080,0x00808080,0x80808080,
};
static unsigned long        fpl9[16] = {
	0x00000000,0x80000000,0x00800000,0x80800000,0x00008000,0x80008000,0x00808000,0x80808000,
	0x00000080,0x80000080,0x00800080,0x80800080,0x00008080,0x80008080,0x00808080,0x80808080,
};
static unsigned long        fpha[16] = {
	0x00000000,0x20000000,0x00200000,0x20200000,0x00002000,0x20002000,0x00202000,0x20202000,
	0x00000020,0x20000020,0x00200020,0x20200020,0x00002020,0x20002020,0x00202020,0x20202020,
};
static unsigned long        fplb[16] = {
	0x00000000,0x20000000,0x00200000,0x20200000,0x00002000,0x20002000,0x00202000,0x20202000,
	0x00000020,0x20000020,0x00200020,0x20200020,0x00002020,0x20002020,0x00202020,0x20202020,
};
static unsigned long        fphc[16] = {
	0x00000000,0x08000000,0x00080000,0x08080000,0x00000800,0x08000800,0x00080800,0x08080800,
	0x00000008,0x08000008,0x00080008,0x08080008,0x00000808,0x08000808,0x00080808,0x08080808,
};
static unsigned long        fpld[16] = {
	0x00000000,0x08000000,0x00080000,0x08080000,0x00000800,0x08000800,0x00080800,0x08080800,
	0x00000008,0x08000008,0x00080008,0x08080008,0x00000808,0x08000808,0x00080808,0x08080808,
};
static unsigned long        fphe[16] = {
	0x00000000,0x02000000,0x00020000,0x02020000,0x00000200,0x02000200,0x00020200,0x02020200,
	0x00000002,0x02000002,0x00020002,0x02020002,0x00000202,0x02000202,0x00020202,0x02020202,
};
static unsigned long        fplf[16] = {
	0x00000000,0x02000000,0x00020000,0x02020000,0x00000200,0x02000200,0x00020200,0x02020200,
	0x00000002,0x02000002,0x00020002,0x02020002,0x00000202,0x02000202,0x00020202,0x02020202,
};

#ifdef K_AND_R
static void
fp(left, right, text)
  unsigned long left, right;
  unsigned char text[8];
#else
  static void fp(unsigned long left,
		 unsigned long right,
		 unsigned char text[8])
#endif
{
	register unsigned long low, high;
	register unsigned long temp;
	
	temp = left;
	high = fph0[(int)(temp & 0xf)];
	temp >>= 4;
	low  = fpl1[(int)(temp & 0xf)];
	temp >>= 4;
	high |= fph2[(int)(temp & 0xf)];
	temp >>= 4;
	low  |= fpl3[(int)(temp & 0xf)];
	temp >>= 4;
	high |= fph4[(int)(temp & 0xf)];
	temp >>= 4;
	low  |= fpl5[(int)(temp & 0xf)];
	temp >>= 4;
	high |= fph6[(int)(temp & 0xf)];
	temp >>= 4;
	low  |= fpl7[(int)(temp & 0xf)];
	temp = right;
	high |= fph8[(int)(temp & 0xf)];
	temp >>= 4;
	low  |= fpl9[(int)(temp & 0xf)];
	temp >>= 4;
	high |= fpha[(int)(temp & 0xf)];
	temp >>= 4;
	low  |= fplb[(int)(temp & 0xf)];
	temp >>= 4;
	high |= fphc[(int)(temp & 0xf)];
	temp >>= 4;
	low  |= fpld[(int)(temp & 0xf)];
	temp >>= 4;
	high |= fphe[(int)(temp & 0xf)];
	temp >>= 4;
	low  |= fplf[(int)(temp & 0xf)];
	
	text[0] = (unsigned char)low;
	text[1] = (unsigned char)(low >> 8);
	text[2] = (unsigned char)(low >> 16);
	text[3] = (unsigned char)(low >> 24);
	text[4] = (unsigned char)high;
	text[5] = (unsigned char)(high >> 8);
	text[6] = (unsigned char)(high >> 16);
	text[7] = (unsigned char)(high >> 24);
}

/*
 *        Initial Permutation
 */
static unsigned long        ipl0[16] = {
	0x00000000,
	0x00008000,0x00000000,0x00008000,0x00000080,0x00008080,0x00000080,0x00008080,0x00000000,
	0x00008000,0x00000000,0x00008000,0x00000080,0x00008080,0x00000080,0x00008080,};
static unsigned long        iph0[16] = {
	0x00000000,
	0x00000000,0x00008000,0x00008000,0x00000000,0x00000000,0x00008000,0x00008000,0x00000080,
	0x00000080,0x00008080,0x00008080,0x00000080,0x00000080,0x00008080,0x00008080,};
static unsigned long        ipl1[16] = {
	0x00000000,
	0x80000000,0x00000000,0x80000000,0x00800000,0x80800000,0x00800000,0x80800000,0x00000000,
	0x80000000,0x00000000,0x80000000,0x00800000,0x80800000,0x00800000,0x80800000,};
static unsigned long        iph1[16] = {
	0x00000000,
	0x00000000,0x80000000,0x80000000,0x00000000,0x00000000,0x80000000,0x80000000,0x00800000,
	0x00800000,0x80800000,0x80800000,0x00800000,0x00800000,0x80800000,0x80800000,};
static unsigned long        ipl2[16] = {
	0x00000000,
	0x00004000,0x00000000,0x00004000,0x00000040,0x00004040,0x00000040,0x00004040,0x00000000,
	0x00004000,0x00000000,0x00004000,0x00000040,0x00004040,0x00000040,0x00004040,};
static unsigned long        iph2[16] = {
	0x00000000,
	0x00000000,0x00004000,0x00004000,0x00000000,0x00000000,0x00004000,0x00004000,0x00000040,
	0x00000040,0x00004040,0x00004040,0x00000040,0x00000040,0x00004040,0x00004040,};
static unsigned long        ipl3[16] = {
	0x00000000,
	0x40000000,0x00000000,0x40000000,0x00400000,0x40400000,0x00400000,0x40400000,0x00000000,
	0x40000000,0x00000000,0x40000000,0x00400000,0x40400000,0x00400000,0x40400000,};
static unsigned long        iph3[16] = {
	0x00000000,
	0x00000000,0x40000000,0x40000000,0x00000000,0x00000000,0x40000000,0x40000000,0x00400000,
	0x00400000,0x40400000,0x40400000,0x00400000,0x00400000,0x40400000,0x40400000,};
static unsigned long        ipl4[16] = {
	0x00000000,
	0x00002000,0x00000000,0x00002000,0x00000020,0x00002020,0x00000020,0x00002020,0x00000000,
	0x00002000,0x00000000,0x00002000,0x00000020,0x00002020,0x00000020,0x00002020,};
static unsigned long        iph4[16] = {
	0x00000000,
	0x00000000,0x00002000,0x00002000,0x00000000,0x00000000,0x00002000,0x00002000,0x00000020,
	0x00000020,0x00002020,0x00002020,0x00000020,0x00000020,0x00002020,0x00002020,};
static unsigned long        ipl5[16] = {
	0x00000000,
	0x20000000,0x00000000,0x20000000,0x00200000,0x20200000,0x00200000,0x20200000,0x00000000,
	0x20000000,0x00000000,0x20000000,0x00200000,0x20200000,0x00200000,0x20200000,};
static unsigned long        iph5[16] = {
	0x00000000,
	0x00000000,0x20000000,0x20000000,0x00000000,0x00000000,0x20000000,0x20000000,0x00200000,
	0x00200000,0x20200000,0x20200000,0x00200000,0x00200000,0x20200000,0x20200000,};
static unsigned long        ipl6[16] = {
	0x00000000,
	0x00001000,0x00000000,0x00001000,0x00000010,0x00001010,0x00000010,0x00001010,0x00000000,
	0x00001000,0x00000000,0x00001000,0x00000010,0x00001010,0x00000010,0x00001010,};
static unsigned long        iph6[16] = {
	0x00000000,
	0x00000000,0x00001000,0x00001000,0x00000000,0x00000000,0x00001000,0x00001000,0x00000010,
	0x00000010,0x00001010,0x00001010,0x00000010,0x00000010,0x00001010,0x00001010,};
static unsigned long        ipl7[16] = {
	0x00000000,
	0x10000000,0x00000000,0x10000000,0x00100000,0x10100000,0x00100000,0x10100000,0x00000000,
	0x10000000,0x00000000,0x10000000,0x00100000,0x10100000,0x00100000,0x10100000,};
static unsigned long        iph7[16] = {
	0x00000000,
	0x00000000,0x10000000,0x10000000,0x00000000,0x00000000,0x10000000,0x10000000,0x00100000,
	0x00100000,0x10100000,0x10100000,0x00100000,0x00100000,0x10100000,0x10100000,};
static unsigned long        ipl8[16] = {
	0x00000000,
	0x00000800,0x00000000,0x00000800,0x00000008,0x00000808,0x00000008,0x00000808,0x00000000,
	0x00000800,0x00000000,0x00000800,0x00000008,0x00000808,0x00000008,0x00000808,};
static unsigned long        iph8[16] = {
	0x00000000,
	0x00000000,0x00000800,0x00000800,0x00000000,0x00000000,0x00000800,0x00000800,0x00000008,
	0x00000008,0x00000808,0x00000808,0x00000008,0x00000008,0x00000808,0x00000808,};
static unsigned long        ipl9[16] = {
	0x00000000,
	0x08000000,0x00000000,0x08000000,0x00080000,0x08080000,0x00080000,0x08080000,0x00000000,
	0x08000000,0x00000000,0x08000000,0x00080000,0x08080000,0x00080000,0x08080000,};
static unsigned long        iph9[16] = {
	0x00000000,
	0x00000000,0x08000000,0x08000000,0x00000000,0x00000000,0x08000000,0x08000000,0x00080000,
	0x00080000,0x08080000,0x08080000,0x00080000,0x00080000,0x08080000,0x08080000,};
static unsigned long        ipla[16] = {
	0x00000000,
	0x00000400,0x00000000,0x00000400,0x00000004,0x00000404,0x00000004,0x00000404,0x00000000,
	0x00000400,0x00000000,0x00000400,0x00000004,0x00000404,0x00000004,0x00000404,};
static unsigned long        ipha[16] = {
	0x00000000,
	0x00000000,0x00000400,0x00000400,0x00000000,0x00000000,0x00000400,0x00000400,0x00000004,
	0x00000004,0x00000404,0x00000404,0x00000004,0x00000004,0x00000404,0x00000404,};
static unsigned long        iplb[16] = {
	0x00000000,
	0x04000000,0x00000000,0x04000000,0x00040000,0x04040000,0x00040000,0x04040000,0x00000000,
	0x04000000,0x00000000,0x04000000,0x00040000,0x04040000,0x00040000,0x04040000,};
static unsigned long        iphb[16] = {
	0x00000000,
	0x00000000,0x04000000,0x04000000,0x00000000,0x00000000,0x04000000,0x04000000,0x00040000,
	0x00040000,0x04040000,0x04040000,0x00040000,0x00040000,0x04040000,0x04040000,};
static unsigned long        iplc[16] = {
	0x00000000,
	0x00000200,0x00000000,0x00000200,0x00000002,0x00000202,0x00000002,0x00000202,0x00000000,
	0x00000200,0x00000000,0x00000200,0x00000002,0x00000202,0x00000002,0x00000202,};
static unsigned long        iphc[16] = {
	0x00000000,
	0x00000000,0x00000200,0x00000200,0x00000000,0x00000000,0x00000200,0x00000200,0x00000002,
	0x00000002,0x00000202,0x00000202,0x00000002,0x00000002,0x00000202,0x00000202,};
static unsigned long        ipld[16] = {
	0x00000000,
	0x02000000,0x00000000,0x02000000,0x00020000,0x02020000,0x00020000,0x02020000,0x00000000,
	0x02000000,0x00000000,0x02000000,0x00020000,0x02020000,0x00020000,0x02020000,};
static unsigned long        iphd[16] = {
	0x00000000,
	0x00000000,0x02000000,0x02000000,0x00000000,0x00000000,0x02000000,0x02000000,0x00020000,
	0x00020000,0x02020000,0x02020000,0x00020000,0x00020000,0x02020000,0x02020000,};
static unsigned long        iple[16] = {
	0x00000000,
	0x00000100,0x00000000,0x00000100,0x00000001,0x00000101,0x00000001,0x00000101,0x00000000,
	0x00000100,0x00000000,0x00000100,0x00000001,0x00000101,0x00000001,0x00000101,};
static unsigned long        iphe[16] = {
	0x00000000,
	0x00000000,0x00000100,0x00000100,0x00000000,0x00000000,0x00000100,0x00000100,0x00000001,
	0x00000001,0x00000101,0x00000101,0x00000001,0x00000001,0x00000101,0x00000101,};
static unsigned long        iplf[16] = {
	0x00000000,
	0x01000000,0x00000000,0x01000000,0x00010000,0x01010000,0x00010000,0x01010000,0x00000000,
	0x01000000,0x00000000,0x01000000,0x00010000,0x01010000,0x00010000,0x01010000,};
static unsigned long        iphf[16] = {
	0x00000000,
	0x00000000,0x01000000,0x01000000,0x00000000,0x00000000,0x01000000,0x01000000,0x00010000,
	0x00010000,0x01010000,0x01010000,0x00010000,0x00010000,0x01010000,0x01010000,};

#ifdef K_AND_R
static unsigned long
ip_low(block)
  register unsigned char block[8];
#else
  static unsigned long ip_low(register unsigned char block[8])
#endif
{
	register unsigned long l;
	
	l  = ipl1[block[0] & 0xf];
	l |= ipl0[(block[0] >> 4) & 0xf];
	l |= ipl3[block[1] & 0xf];
	l |= ipl2[(block[1] >> 4) & 0xf];
	l |= ipl5[block[2] & 0xf];
	l |= ipl4[(block[2] >> 4) & 0xf];
	l |= ipl7[block[3] & 0xf];
	l |= ipl6[(block[3] >> 4) & 0xf];
	l |= ipl9[block[4] & 0xf];
	l |= ipl8[(block[4] >> 4) & 0xf];
	l |= iplb[block[5] & 0xf];
	l |= ipla[(block[5] >> 4) & 0xf];
	l |= ipld[block[6] & 0xf];
	l |= iplc[(block[6] >> 4) & 0xf];
	l |= iplf[block[7] & 0xf];
	l |= iple[(block[7] >> 4) & 0xf];
	return l;
}

#ifdef K_AND_R
static unsigned long
ip_high(block)
  register unsigned char block[8];
#else
  static unsigned long ip_high(register unsigned char block[8])
#endif
{
	register unsigned long l;
	
	l  = iph1[block[0] & 0xf];
	l |= iph0[(block[0] >> 4) & 0xf];
	l |= iph3[block[1] & 0xf];
	l |= iph2[(block[1] >> 4) & 0xf];
	l |= iph5[block[2] & 0xf];
	l |= iph4[(block[2] >> 4) & 0xf];
	l |= iph7[block[3] & 0xf];
	l |= iph6[(block[3] >> 4) & 0xf];
	l |= iph9[block[4] & 0xf];
	l |= iph8[(block[4] >> 4) & 0xf];
	l |= iphb[block[5] & 0xf];
	l |= ipha[(block[5] >> 4) & 0xf];
	l |= iphd[block[6] & 0xf];
	l |= iphc[(block[6] >> 4) & 0xf];
	l |= iphf[block[7] & 0xf];
	l |= iphe[(block[7] >> 4) & 0xf];
	return l;
}

/*
 *        Key set-up
 */

#ifdef K_AND_R
_TYPE( void )
key_setup(key, subkeys)
  unsigned char key[8], *subkeys;
#else
_TYPE( void ) key_setup(unsigned char key[8],
			unsigned char *subkeys)
#endif
{
	register int octet;
	register unsigned char *kp;
	
	kp = subkeys;
	octet = 16;
	do {
		*kp++ = 0;        *kp++ = 0;
		*kp++ = 0;        *kp++ = 0;
		*kp++ = 0;        *kp++ = 0;
		*kp++ = 0;        *kp++ = 0;
	} while (--octet);
	kp = subkeys;
	octet = key[0];
	if (octet & 0x80) {
		kp[  3] |=  2; kp[  9] |=  8; kp[ 18] |=  8; 
		kp[ 27] |= 32; kp[ 33] |=  2; kp[ 42] |= 16; 
		kp[ 48] |=  8; kp[ 65] |= 16; kp[ 74] |=  2; 
		kp[ 80] |=  2; kp[ 89] |=  4; kp[ 99] |= 16; 
		kp[104] |=  4; kp[122] |= 32; 
	}
	if (octet & 0x40) {
		kp[  1] |=  4; kp[  8] |=  1; kp[ 18] |=  4; 
		kp[ 25] |= 32; kp[ 34] |= 32; kp[ 41] |=  8; 
		kp[ 50] |=  8; kp[ 59] |= 32; kp[ 64] |= 16; 
		kp[ 75] |=  4; kp[ 90] |=  1; kp[ 97] |= 16; 
		kp[106] |=  2; kp[112] |=  2; kp[123] |=  1; 
	}
	if (octet & 0x20) {
		kp[  2] |=  1; kp[ 19] |=  8; kp[ 35] |=  1; 
		kp[ 40] |=  1; kp[ 50] |=  4; kp[ 57] |= 32; 
		kp[ 75] |=  2; kp[ 80] |= 32; kp[ 89] |=  1; 
		kp[ 96] |= 16; kp[107] |=  4; kp[120] |=  8; 
	}
	if (octet & 0x10) {
		kp[  4] |= 32; kp[ 20] |=  2; kp[ 31] |=  4; 
		kp[ 37] |= 32; kp[ 47] |=  1; kp[ 54] |=  1; 
		kp[ 63] |=  2; kp[ 68] |=  1; kp[ 78] |=  4; 
		kp[ 84] |=  8; kp[101] |= 16; kp[108] |=  4; 
		kp[119] |= 16; kp[126] |=  8; 
	}
	if (octet & 0x8) {
		kp[  5] |=  4; kp[ 15] |=  4; kp[ 21] |= 32; 
		kp[ 31] |=  1; kp[ 38] |=  1; kp[ 47] |=  2; 
		kp[ 53] |=  2; kp[ 68] |=  8; kp[ 85] |= 16; 
		kp[ 92] |=  4; kp[103] |= 16; kp[108] |= 32; 
		kp[118] |= 32; kp[124] |=  2; 
	}
	if (octet & 0x4) {
		kp[ 15] |=  2; kp[ 21] |=  2; kp[ 39] |=  8; 
		kp[ 46] |= 16; kp[ 55] |= 32; kp[ 61] |=  1; 
		kp[ 71] |= 16; kp[ 76] |= 32; kp[ 86] |= 32; 
		kp[ 93] |=  4; kp[102] |=  2; kp[108] |= 16; 
		kp[117] |=  8; kp[126] |=  1; 
	}
	if (octet & 0x2) {
		kp[ 14] |= 16; kp[ 23] |= 32; kp[ 29] |=  1; 
		kp[ 38] |=  8; kp[ 52] |=  2; kp[ 63] |=  4; 
		kp[ 70] |=  2; kp[ 76] |= 16; kp[ 85] |=  8; 
		kp[100] |=  1; kp[110] |=  4; kp[116] |=  8; 
		kp[127] |=  8; 
	}
	octet = key[1];
	if (octet & 0x80) {
		kp[  1] |=  8; kp[  8] |= 32; kp[ 17] |=  1; 
		kp[ 24] |= 16; kp[ 35] |=  4; kp[ 50] |=  1; 
		kp[ 57] |= 16; kp[ 67] |=  8; kp[ 83] |=  1; 
		kp[ 88] |=  1; kp[ 98] |=  4; kp[105] |= 32; 
		kp[114] |= 32; kp[123] |=  2; 
	}
	if (octet & 0x40) {
		kp[  0] |=  1; kp[ 11] |= 16; kp[ 16] |=  4; 
		kp[ 35] |=  2; kp[ 40] |= 32; kp[ 49] |=  1; 
		kp[ 56] |= 16; kp[ 65] |=  2; kp[ 74] |= 16; 
		kp[ 80] |=  8; kp[ 99] |=  8; kp[115] |=  1; 
		kp[121] |=  4; 
	}
	if (octet & 0x20) {
		kp[  9] |= 16; kp[ 18] |=  2; kp[ 24] |=  2; 
		kp[ 33] |=  4; kp[ 43] |= 16; kp[ 48] |=  4; 
		kp[ 66] |= 32; kp[ 73] |=  8; kp[ 82] |=  8; 
		kp[ 91] |= 32; kp[ 97] |=  2; kp[106] |= 16; 
		kp[112] |=  8; kp[122] |=  1; 
	}
	if (octet & 0x10) {
		kp[ 14] |= 32; kp[ 21] |=  4; kp[ 30] |=  2; 
		kp[ 36] |= 16; kp[ 45] |=  8; kp[ 60] |=  1; 
		kp[ 69] |=  2; kp[ 87] |=  8; kp[ 94] |= 16; 
		kp[103] |= 32; kp[109] |=  1; kp[118] |=  8; 
		kp[124] |= 32; 
	}
	if (octet & 0x8) {
		kp[  7] |=  4; kp[ 14] |=  2; kp[ 20] |= 16; 
		kp[ 29] |=  8; kp[ 44] |=  1; kp[ 54] |=  4; 
		kp[ 60] |=  8; kp[ 71] |=  8; kp[ 78] |= 16; 
		kp[ 87] |= 32; kp[ 93] |=  1; kp[102] |=  8; 
		kp[116] |=  2; kp[125] |=  4; 
	}
	if (octet & 0x4) {
		kp[  7] |=  2; kp[ 12] |=  1; kp[ 22] |=  4; 
		kp[ 28] |=  8; kp[ 45] |= 16; kp[ 52] |=  4; 
		kp[ 63] |= 16; kp[ 70] |=  8; kp[ 84] |=  2; 
		kp[ 95] |=  4; kp[101] |= 32; kp[111] |=  1; 
		kp[118] |=  1; 
	}
	if (octet & 0x2) {
		kp[  6] |= 16; kp[ 13] |= 16; kp[ 20] |=  4; 
		kp[ 31] |= 16; kp[ 36] |= 32; kp[ 46] |= 32; 
		kp[ 53] |=  4; kp[ 62] |=  2; kp[ 69] |= 32; 
		kp[ 79] |=  1; kp[ 86] |=  1; kp[ 95] |=  2; 
		kp[101] |=  2; kp[119] |=  8; 
	}
	octet = key[2];
	if (octet & 0x80) {
		kp[  0] |= 32; kp[ 10] |=  8; kp[ 19] |= 32; 
		kp[ 25] |=  2; kp[ 34] |= 16; kp[ 40] |=  8; 
		kp[ 59] |=  8; kp[ 66] |=  2; kp[ 72] |=  2; 
		kp[ 81] |=  4; kp[ 91] |= 16; kp[ 96] |=  4; 
		kp[115] |=  2; kp[121] |=  8; 
	}
	if (octet & 0x40) {
		kp[  3] |= 16; kp[ 10] |=  4; kp[ 17] |= 32; 
		kp[ 26] |= 32; kp[ 33] |=  8; kp[ 42] |=  8; 
		kp[ 51] |= 32; kp[ 57] |=  2; kp[ 67] |=  4; 
		kp[ 82] |=  1; kp[ 89] |= 16; kp[ 98] |=  2; 
		kp[104] |=  2; kp[113] |=  4; kp[120] |=  1; 
	}
	if (octet & 0x20) {
		kp[  1] |= 16; kp[ 11] |=  8; kp[ 27] |=  1; 
		kp[ 32] |=  1; kp[ 42] |=  4; kp[ 49] |= 32; 
		kp[ 58] |= 32; kp[ 67] |=  2; kp[ 72] |= 32; 
		kp[ 81] |=  1; kp[ 88] |= 16; kp[ 99] |=  4; 
		kp[114] |=  1; 
	}
	if (octet & 0x10) {
		kp[  6] |= 32; kp[ 12] |=  2; kp[ 23] |=  4; 
		kp[ 29] |= 32; kp[ 39] |=  1; kp[ 46] |=  1; 
		kp[ 55] |=  2; kp[ 61] |=  2; kp[ 70] |=  4; 
		kp[ 76] |=  8; kp[ 93] |= 16; kp[100] |=  4; 
		kp[111] |= 16; kp[116] |= 32; 
	}
	if (octet & 0x8) {
		kp[  6] |=  2; kp[ 13] |= 32; kp[ 23] |=  1; 
		kp[ 30] |=  1; kp[ 39] |=  2; kp[ 45] |=  2; 
		kp[ 63] |=  8; kp[ 77] |= 16; kp[ 84] |=  4; 
		kp[ 95] |= 16; kp[100] |= 32; kp[110] |= 32; 
		kp[117] |=  4; kp[127] |=  4; 
	}
	if (octet & 0x4) {
		kp[  4] |=  1; kp[ 13] |=  2; kp[ 31] |=  8; 
		kp[ 38] |= 16; kp[ 47] |= 32; kp[ 53] |=  1; 
		kp[ 62] |=  8; kp[ 68] |= 32; kp[ 78] |= 32; 
		kp[ 85] |=  4; kp[ 94] |=  2; kp[100] |= 16; 
		kp[109] |=  8; kp[127] |=  2; 
	}
	if (octet & 0x2) {
		kp[  5] |= 16; kp[ 15] |= 32; kp[ 21] |=  1; 
		kp[ 30] |=  8; kp[ 44] |=  2; kp[ 55] |=  4; 
		kp[ 61] |= 32; kp[ 68] |= 16; kp[ 77] |=  8; 
		kp[ 92] |=  1; kp[102] |=  4; kp[108] |=  8; 
		kp[126] |= 16; 
	}
	octet = key[3];
	if (octet & 0x80) {
		kp[  2] |=  8; kp[  9] |=  1; kp[ 16] |= 16; 
		kp[ 27] |=  4; kp[ 42] |=  1; kp[ 49] |= 16; 
		kp[ 58] |=  2; kp[ 75] |=  1; kp[ 80] |=  1; 
		kp[ 90] |=  4; kp[ 97] |= 32; kp[106] |= 32; 
		kp[113] |=  8; kp[120] |= 32; 
	}
	if (octet & 0x40) {
		kp[  2] |=  4; kp[  8] |=  4; kp[ 27] |=  2; 
		kp[ 32] |= 32; kp[ 41] |=  1; kp[ 48] |= 16; 
		kp[ 59] |=  4; kp[ 66] |= 16; kp[ 72] |=  8; 
		kp[ 91] |=  8; kp[107] |=  1; kp[112] |=  1; 
		kp[123] |= 16; 
	}
	if (octet & 0x20) {
		kp[  3] |=  8; kp[ 10] |=  2; kp[ 16] |=  2; 
		kp[ 25] |=  4; kp[ 35] |= 16; kp[ 40] |=  4; 
		kp[ 59] |=  2; kp[ 65] |=  8; kp[ 74] |=  8; 
		kp[ 83] |= 32; kp[ 89] |=  2; kp[ 98] |= 16; 
		kp[104] |=  8; kp[121] |= 16; 
	}
	if (octet & 0x10) {
		kp[  4] |=  2; kp[ 13] |=  4; kp[ 22] |=  2; 
		kp[ 28] |= 16; kp[ 37] |=  8; kp[ 52] |=  1; 
		kp[ 62] |=  4; kp[ 79] |=  8; kp[ 86] |= 16; 
		kp[ 95] |= 32; kp[101] |=  1; kp[110] |=  8; 
		kp[126] |= 32; 
	}
	if (octet & 0x8) {
		kp[  5] |= 32; kp[ 12] |= 16; kp[ 21] |=  8; 
		kp[ 36] |=  1; kp[ 46] |=  4; kp[ 52] |=  8; 
		kp[ 70] |= 16; kp[ 79] |= 32; kp[ 85] |=  1; 
		kp[ 94] |=  8; kp[108] |=  2; kp[119] |=  4; 
		kp[126] |=  2; 
	}
	if (octet & 0x4) {
		kp[  5] |=  2; kp[ 14] |=  4; kp[ 20] |=  8; 
		kp[ 37] |= 16; kp[ 44] |=  4; kp[ 55] |= 16; 
		kp[ 60] |= 32; kp[ 76] |=  2; kp[ 87] |=  4; 
		kp[ 93] |= 32; kp[103] |=  1; kp[110] |=  1; 
		kp[119] |=  2; kp[124] |=  1; 
	}
	if (octet & 0x2) {
		kp[  7] |= 32; kp[ 12] |=  4; kp[ 23] |= 16; 
		kp[ 28] |= 32; kp[ 38] |= 32; kp[ 45] |=  4; 
		kp[ 54] |=  2; kp[ 60] |= 16; kp[ 71] |=  1; 
		kp[ 78] |=  1; kp[ 87] |=  2; kp[ 93] |=  2; 
		kp[111] |=  8; kp[118] |= 16; kp[125] |= 16; 
	}
	octet = key[4];
	if (octet & 0x80) {
		kp[  1] |=  1; kp[ 11] |= 32; kp[ 17] |=  2; 
		kp[ 26] |= 16; kp[ 32] |=  8; kp[ 51] |=  8; 
		kp[ 64] |=  2; kp[ 73] |=  4; kp[ 83] |= 16; 
		kp[ 88] |=  4; kp[107] |=  2; kp[112] |= 32; 
		kp[122] |=  8; 
	}
	if (octet & 0x40) {
		kp[  0] |=  4; kp[  9] |= 32; kp[ 18] |= 32; 
		kp[ 25] |=  8; kp[ 34] |=  8; kp[ 43] |= 32; 
		kp[ 49] |=  2; kp[ 58] |= 16; kp[ 74] |=  1; 
		kp[ 81] |= 16; kp[ 90] |=  2; kp[ 96] |=  2; 
		kp[105] |=  4; kp[115] |= 16; kp[122] |=  4; 
	}
	if (octet & 0x20) {
		kp[  2] |=  2; kp[ 19] |=  1; kp[ 24] |=  1; 
		kp[ 34] |=  4; kp[ 41] |= 32; kp[ 50] |= 32; 
		kp[ 57] |=  8; kp[ 64] |= 32; kp[ 73] |=  1; 
		kp[ 80] |= 16; kp[ 91] |=  4; kp[106] |=  1; 
		kp[113] |= 16; kp[123] |=  8; 
	}
	if (octet & 0x10) {
		kp[  3] |=  4; kp[ 10] |= 16; kp[ 16] |=  8; 
		kp[ 35] |=  8; kp[ 51] |=  1; kp[ 56] |=  1; 
		kp[ 67] |= 16; kp[ 72] |=  4; kp[ 91] |=  2; 
		kp[ 96] |= 32; kp[105] |=  1; kp[112] |= 16; 
		kp[121] |=  2; 
	}
	if (octet & 0x8) {
		kp[  4] |= 16; kp[ 15] |=  1; kp[ 22] |=  1; 
		kp[ 31] |=  2; kp[ 37] |=  2; kp[ 55] |=  8; 
		kp[ 62] |= 16; kp[ 69] |= 16; kp[ 76] |=  4; 
		kp[ 87] |= 16; kp[ 92] |= 32; kp[102] |= 32; 
		kp[109] |=  4; kp[118] |=  2; kp[125] |= 32; 
	}
	if (octet & 0x4) {
		kp[  6] |=  4; kp[ 23] |=  8; kp[ 30] |= 16; 
		kp[ 39] |= 32; kp[ 45] |=  1; kp[ 54] |=  8; 
		kp[ 70] |= 32; kp[ 77] |=  4; kp[ 86] |=  2; 
		kp[ 92] |= 16; kp[101] |=  8; kp[116] |=  1; 
		kp[125] |=  2; 
	}
	if (octet & 0x2) {
		kp[  4] |=  4; kp[ 13] |=  1; kp[ 22] |=  8; 
		kp[ 36] |=  2; kp[ 47] |=  4; kp[ 53] |= 32; 
		kp[ 63] |=  1; kp[ 69] |=  8; kp[ 84] |=  1; 
		kp[ 94] |=  4; kp[100] |=  8; kp[117] |= 16; 
		kp[127] |= 32; 
	}
	octet = key[5];
	if (octet & 0x80) {
		kp[  3] |= 32; kp[  8] |= 16; kp[ 19] |=  4; 
		kp[ 34] |=  1; kp[ 41] |= 16; kp[ 50] |=  2; 
		kp[ 56] |=  2; kp[ 67] |=  1; kp[ 72] |=  1; 
		kp[ 82] |=  4; kp[ 89] |= 32; kp[ 98] |= 32; 
		kp[105] |=  8; kp[114] |=  8; kp[121] |=  1; 
	}
	if (octet & 0x40) {
		kp[  1] |= 32; kp[ 19] |=  2; kp[ 24] |= 32; 
		kp[ 33] |=  1; kp[ 40] |= 16; kp[ 51] |=  4; 
		kp[ 64] |=  8; kp[ 83] |=  8; kp[ 99] |=  1; 
		kp[104] |=  1; kp[114] |=  4; kp[120] |=  4; 
	}
	if (octet & 0x20) {
		kp[  8] |=  2; kp[ 17] |=  4; kp[ 27] |= 16; 
		kp[ 32] |=  4; kp[ 51] |=  2; kp[ 56] |= 32; 
		kp[ 66] |=  8; kp[ 75] |= 32; kp[ 81] |=  2; 
		kp[ 90] |= 16; kp[ 96] |=  8; kp[115] |=  8; 
		kp[122] |=  2; 
	}
	if (octet & 0x10) {
		kp[  2] |= 16; kp[ 18] |=  1; kp[ 25] |= 16; 
		kp[ 34] |=  2; kp[ 40] |=  2; kp[ 49] |=  4; 
		kp[ 59] |= 16; kp[ 66] |=  4; kp[ 73] |= 32; 
		kp[ 82] |= 32; kp[ 89] |=  8; kp[ 98] |=  8; 
		kp[107] |= 32; kp[113] |=  2; kp[123] |=  4; 
	}
	if (octet & 0x8) {
		kp[  7] |=  1; kp[ 13] |=  8; kp[ 28] |=  1; 
		kp[ 38] |=  4; kp[ 44] |=  8; kp[ 61] |= 16; 
		kp[ 71] |= 32; kp[ 77] |=  1; kp[ 86] |=  8; 
		kp[100] |=  2; kp[111] |=  4; kp[117] |= 32; 
		kp[124] |= 16; 
	}
	if (octet & 0x4) {
		kp[ 12] |=  8; kp[ 29] |= 16; kp[ 36] |=  4; 
		kp[ 47] |= 16; kp[ 52] |= 32; kp[ 62] |= 32; 
		kp[ 68] |=  2; kp[ 79] |=  4; kp[ 85] |= 32; 
		kp[ 95] |=  1; kp[102] |=  1; kp[111] |=  2; 
		kp[117] |=  2; kp[126] |=  4; 
	}
	if (octet & 0x2) {
		kp[  5] |=  1; kp[ 15] |= 16; kp[ 20] |= 32; 
		kp[ 30] |= 32; kp[ 37] |=  4; kp[ 46] |=  2; 
		kp[ 52] |= 16; kp[ 61] |=  8; kp[ 70] |=  1; 
		kp[ 79] |=  2; kp[ 85] |=  2; kp[103] |=  8; 
		kp[110] |= 16; kp[119] |= 32; kp[124] |=  4; 
	}
	octet = key[6];
	if (octet & 0x80) {
		kp[  0] |= 16; kp[  9] |=  2; kp[ 18] |= 16; 
		kp[ 24] |=  8; kp[ 43] |=  8; kp[ 59] |=  1; 
		kp[ 65] |=  4; kp[ 75] |= 16; kp[ 80] |=  4; 
		kp[ 99] |=  2; kp[104] |= 32; kp[113] |=  1; 
		kp[123] |= 32; 
	}
	if (octet & 0x40) {
		kp[ 10] |= 32; kp[ 17] |=  8; kp[ 26] |=  8; 
		kp[ 35] |= 32; kp[ 41] |=  2; kp[ 50] |= 16; 
		kp[ 56] |=  8; kp[ 66] |=  1; kp[ 73] |= 16; 
		kp[ 82] |=  2; kp[ 88] |=  2; kp[ 97] |=  4; 
		kp[107] |= 16; kp[112] |=  4; kp[121] |= 32; 
	}
	if (octet & 0x20) {
		kp[  0] |=  2; kp[ 11] |=  1; kp[ 16] |=  1; 
		kp[ 26] |=  4; kp[ 33] |= 32; kp[ 42] |= 32; 
		kp[ 49] |=  8; kp[ 58] |=  8; kp[ 65] |=  1; 
		kp[ 72] |= 16; kp[ 83] |=  4; kp[ 98] |=  1; 
		kp[105] |= 16; kp[114] |=  2; 
	}
	if (octet & 0x10) {
		kp[  8] |=  8; kp[ 27] |=  8; kp[ 43] |=  1; 
		kp[ 48] |=  1; kp[ 58] |=  4; kp[ 64] |=  4; 
		kp[ 83] |=  2; kp[ 88] |= 32; kp[ 97] |=  1; 
		kp[104] |= 16; kp[115] |=  4; kp[122] |= 16; 
	}
	if (octet & 0x8) {
		kp[  5] |=  8; kp[ 14] |=  1; kp[ 23] |=  2; 
		kp[ 29] |=  2; kp[ 47] |=  8; kp[ 54] |= 16; 
		kp[ 63] |= 32; kp[ 68] |=  4; kp[ 79] |= 16; 
		kp[ 84] |= 32; kp[ 94] |= 32; kp[101] |=  4; 
		kp[110] |=  2; kp[116] |= 16; kp[127] |=  1; 
	}
	if (octet & 0x4) {
		kp[  4] |=  8; kp[ 15] |=  8; kp[ 22] |= 16; 
		kp[ 31] |= 32; kp[ 37] |=  1; kp[ 46] |=  8; 
		kp[ 60] |=  2; kp[ 69] |=  4; kp[ 78] |=  2; 
		kp[ 84] |= 16; kp[ 93] |=  8; kp[108] |=  1; 
		kp[118] |=  4; 
	}
	if (octet & 0x2) {
		kp[  7] |= 16; kp[ 14] |=  8; kp[ 28] |=  2; 
		kp[ 39] |=  4; kp[ 45] |= 32; kp[ 55] |=  1; 
		kp[ 62] |=  1; kp[ 76] |=  1; kp[ 86] |=  4; 
		kp[ 92] |=  8; kp[109] |= 16; kp[116] |=  4; 
		kp[125] |=  1; 
	}
	octet = key[7];
	if (octet & 0x80) {
		kp[  1] |=  2; kp[ 11] |=  4; kp[ 26] |=  1; 
		kp[ 33] |= 16; kp[ 42] |=  2; kp[ 48] |=  2; 
		kp[ 57] |=  4; kp[ 64] |=  1; kp[ 74] |=  4; 
		kp[ 81] |= 32; kp[ 90] |= 32; kp[ 97] |=  8; 
		kp[106] |=  8; kp[115] |= 32; kp[120] |= 16; 
	}
	if (octet & 0x40) {
		kp[  2] |= 32; kp[ 11] |=  2; kp[ 16] |= 32; 
		kp[ 25] |=  1; kp[ 32] |= 16; kp[ 43] |=  4; 
		kp[ 58] |=  1; kp[ 75] |=  8; kp[ 91] |=  1; 
		kp[ 96] |=  1; kp[106] |=  4; kp[113] |= 32; 
	}
	if (octet & 0x20) {
		kp[  3] |=  1; kp[  9] |=  4; kp[ 19] |= 16; 
		kp[ 24] |=  4; kp[ 43] |=  2; kp[ 48] |= 32; 
		kp[ 57] |=  1; kp[ 67] |= 32; kp[ 73] |=  2; 
		kp[ 82] |= 16; kp[ 88] |=  8; kp[107] |=  8; 
		kp[120] |=  2; 
	}
	if (octet & 0x10) {
		kp[  0] |=  8; kp[ 10] |=  1; kp[ 17] |= 16; 
		kp[ 26] |=  2; kp[ 32] |=  2; kp[ 41] |=  4; 
		kp[ 51] |= 16; kp[ 56] |=  4; kp[ 65] |= 32; 
		kp[ 74] |= 32; kp[ 81] |=  8; kp[ 90] |=  8; 
		kp[ 99] |= 32; kp[105] |=  2; kp[114] |= 16; 
	}
	if (octet & 0x8) {
		kp[  6] |=  1; kp[ 20] |=  1; kp[ 30] |=  4; 
		kp[ 36] |=  8; kp[ 53] |= 16; kp[ 60] |=  4; 
		kp[ 69] |=  1; kp[ 78] |=  8; kp[ 92] |=  2; 
		kp[103] |=  4; kp[109] |= 32; kp[119] |=  1; 
		kp[125] |=  8; 
	}
	if (octet & 0x4) {
		kp[  7] |=  8; kp[ 21] |= 16; kp[ 28] |=  4; 
		kp[ 39] |= 16; kp[ 44] |= 32; kp[ 54] |= 32; 
		kp[ 61] |=  4; kp[ 71] |=  4; kp[ 77] |= 32; 
		kp[ 87] |=  1; kp[ 94] |=  1; kp[103] |=  2; 
		kp[109] |=  2; kp[124] |=  8; 
	}
	if (octet & 0x2) {
		kp[  6] |=  8; kp[ 12] |= 32; kp[ 22] |= 32; 
		kp[ 29] |=  4; kp[ 38] |=  2; kp[ 44] |= 16; 
		kp[ 53] |=  8; kp[ 71] |=  2; kp[ 77] |=  2; 
		kp[ 95] |=  8; kp[102] |= 16; kp[111] |= 32; 
		kp[117] |=  1; kp[127] |= 16; 
	}
}



