/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

/*
 *  This driver was modeled on the PCB 1.4.0 PS driver by Thomas Nau.  It was
 *  modified to produce Gerber/RS-274D photoplotter command files suitable
 *  for use in the fabrication of printed circuit boards by Albert John
 *  FitzPatrick III on April 6, 1996.
 *
 *  Contact address for Email:
 *  Albert John FitzPatrick III <ajf_nylorac@acm.org>
 *
 */

/*
 * Monte Bateman contributed the support for the recognition and the filling of
 * rectangular polygons.
 */

static	char	*rcsid = "$Id: dev_gerber.c,v 144.1 1996/11/29 07:07:52 nau Exp $";

/*
 * Gerber/RS-274D device driver
 */

/*
 * - RS-274D as used within this driver:
 *    + A RS-274D file consists of a sequence of blocks.
 *    + Each block consists of:
 *        = An optional setup sequence. (G)
 *        = An optional aperture selection. (D)
 *        = An optional arc parameter specification sequence. (I,J)
 *        = An optional motion parameter specification sequence. (X,Y)
 *        = An optional control command sequence (D,M)
 *        = A mandatory "*" termination character. ("*")
 *        = An optional ASCII CR-LF sequence. ("\015\012")
 *    + Setup Codes:
 *        = G01 - Enter vector mode.
 *        = G02 - Enter clockwise arc mode.
 *        = G03 - Enter counter clockwise arc mode.
 *        = G04 - Enter comment mode until the block ends.
 *        = G54 - Prepare to change aperture.
 *        = G75 - Select full 360 degree arc mode.
 *        = G90 - Enter absolute coordinate mode.
 *    + Aperture Selection Codes:
 *        = Dn  - Select aperture n, where n is:
 *                   # 10,11,...,19,70,71,20,21,...,29,72,73 for traditional
 *                     (usually vector) 24 aperture photoplotters.
 *                   # 10,11,... for modern (usually raster) photoplotters.
 *    + Arc Parameter Specification Codes:
 *        = In  - Specifies n as an arc center's X coordinate relative to
 *                the arc's origin.
 *        = Jn  - Specifies n as an arc center's Y coordinate relative to
 *                the arc's origin.
 *    + Motion Parameter Specfication Codes:
 *        = Xn  - Specifies the new X coordinate.
 *        = Yn  - Specifies the new Y coordinate.
 *    + Control Command Codes:
 *        = D01 - Move with aperture light on.
 *        = D02 - Move with aperture light off.
 *        = D03 - Move with aperture light off, flash it on, turn it off.
 *        = M02 - End photoplotting.
 *    + Note:
 *        = Optional codes are modal.  Previous values will be used in future
 *          blocks unless new values are provided within those blocks.
 *        = This driver makes little use of modality because it:
 *             # Only saves output file size, which was only critical in the
 *               age of paper tape files.
 *             # Leads to many additional failure modes based upon the
 *               unpredictability of layout content and the fact that I am
 *               too lazy to write a lower-level output monitor that could
 *               reliably elide redundant codes.
 *        = This driver attempts to make the output files editable by humans
 *          using standard POSIX text processing facilities.  Since the RS-274D
 *          standard allows for the optional termination of blocks with an
 *          ASCII CR-LF sequence, and the ASCII LF character has the same
 *          value as the POSIX newline character, and the POSIX text processing
 *          facilities coexist well with ASCII CR characters, and output file
 *          size is not a critical factor, this driver makes extensive use of
 *          the optional CR-LF sequence to punctuate the output codes.
 *
 *          Typically, each and every PCB drawing entity is placed within it's
 *          own block.  Large entities are placed in several consecutive
 *          blocks.  On occasion, comments are inserted to aid in the
 *          recognition of the entities (I.E. text).
 */

/* 
 * FIXME: Fill polygons (including rectangles).
 * FIXME: Handle production of physical layer (PCB group) files based upon the
 *	combination of logical layer (PCB layer) data.
 * FIXME: Beware of RS-274D character set limitations.  Filter non-RS-274D
 *	characters from output ('#' is not valid).
 * FIXME: Enhance media selection (allow offset 0, etc.) in printdialog.c.
 * FIXME: Handled arcs where height != width.
 * FIXME: Analyze computational accuracy especially with respect to sin()/cos().
 */

#include <math.h>
#include <time.h>
#include <pwd.h>
#include <sys/types.h>
#include <errno.h>
#include <varargs.h>

#include "global.h"

#include "data.h"
#include "dev_gerber.h"
#include "error.h"
#include "misc.h"
#include "rotate.h"


/* ---------------------------------------------------------------------------
 * some additional defines
 */

/* FIXME: Shouldn't this be in macro.h? */
#define TO_RADIANS(degrees) (M180 * (degrees))


/*----------------------------------------------------------------------------*/
/* Private constants                                                          */
/*----------------------------------------------------------------------------*/

/* FIXME: Port these constants to PCB; set based upon X resource and/or
	PCB constants. */

/* Maximum length of line with a PCB file. */
#define GB_MAXLINELEN 1024

/* Maximum length of token within a PCB file. */
#define GB_MAXTOKENLEN 64

/* Maximum number of aperture variations in use within a PCB file. */
#define GB_MAXAPERTURECOUNT 256


/*----------------------------------------------------------------------------*/
/* Private data structures                                                    */
/*----------------------------------------------------------------------------*/

enum ApertureShape {
	ROUND,					/* Shaped like a circle */
	SQUARE					/* Shaped like a square */
};
typedef enum ApertureShape ApertureShape;

enum ApertureFunction {
	FLASH,					/* Suitable for flashing */
	DRAW,					/* Suitable for drawing */
	DRILL					/* Suitable for drilling */
};
typedef enum ApertureFunction ApertureFunction;

typedef struct Aperture {
	int dCode;				/* The RS-274D D code */
	int apertureSize;			/* Size in mils */
	ApertureShape apertureShape;		/* ROUND/SQUARE */
	ApertureFunction apertureFunction;	/* FLASH/DRAW/DRILL */
} Aperture; 

typedef struct Apertures {
	int nextAvailable;			/* Number of apertures */
	Aperture aperture[GB_MAXAPERTURECOUNT];
} Apertures;


/*----------------------------------------------------------------------------*/
/* Utility routines                                                           */
/*----------------------------------------------------------------------------*/

#define gerberX(pcb, x) ((long) (x))
#define gerberY(pcb, y) ((long) ((pcb)->MaxHeight - (y)))
#define gerberXOffset(pcb, x) ((long) (x))
#define gerberYOffset(pcb, y) ((long) -(y))


/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	char	*GB_Init(PrintInitTypePtr, char *);
static	void	GB_Exit(void);
static	void	GB_SetColor(XColor);
static	void	GB_Drill(PinTypePtr);
static	void	GB_PrintLayer(LayerTypePtr, int, Boolean);
static	void	GB_PrintElementPackage(ElementTypePtr);
static	void	GB_PrintSomeText(TextTypePtr);
static	void	GB_PrintPad(PadTypePtr);
static	void	GB_PrintPinOrVia(PinTypePtr);
static	void	GB_PrintPadMask(PadTypePtr);
static	void	GB_PrintPinOrViaMask(PinTypePtr);
static	void	GB_PrintClearPinOrViaOnGroundplane(PinTypePtr);
static	void	GB_PrintMaskOrGroundplaneRectangle(Position, Position, Position, Position);
static  void    GB_PrintOutline(Position, Position, Position, Position);
static  void    GB_PrintAlignment(Position, Position, Position, Position);
static	void	GB_PrintDrillingHelper(PinTypePtr);

static	void	GB_PrintPolygon(FILE *, PolygonTypePtr);
static	void	GB_FPrintFilledRectangle(FILE *, Position, Position, Position, Position);
static	void	GB_FPrintPinOrVia(FILE *, PinTypePtr);
static	void	GB_PrintText(FILE *, TextTypePtr);
static  void    GB_PrintLine(FILE *, LineTypePtr);


/* ----------------------------------------------------------------------
 * some local identifiers
 *
 */
static	PrintDeviceType	GB_QueryConstants = {
			"Gerber/RS-274D",					/* name of driver */
			"gbr",								/* filename suffix */

			GB_Init,							/* initializes driver */
			GB_Exit,							/* exit code */

			GB_SetColor,						/* set color */
			GB_Drill,							/* drilling information */

			GB_PrintLayer,						/* print layer */
			GB_PrintElementPackage,				/* print element package */
			GB_PrintSomeText,					/* print some (silkscreen, etc.) text */
			GB_PrintPad,						/* print pad */
			GB_PrintPinOrVia,					/* print pin or via */
			GB_PrintPadMask,					/* print pad mask */
			GB_PrintPinOrViaMask,				/* print pin or via mask */
			GB_PrintClearPinOrViaOnGroundplane,	/* print clear pin or via on groundplane */
			GB_PrintMaskOrGroundplaneRectangle,	/* print filled rectangle
													for ground planes and/or
													for solder masks */
			GB_PrintOutline,					/* print board outline */
			GB_PrintAlignment,					/* print alignment marks */
			GB_PrintDrillingHelper,				/* print drilling helper marks */

			False,								/* handles colored output */
			False,								/* handles inverted output */
			False,								/* handles ground planes */
			True,								/* handles drill info */
			True,								/* handles masks */
			False,								/* handles media */
			};

static	PrintInitType	GB_Flags;

static	char	*GB_Functions[] = { "",
	};

static int GB_debug = 0;			/* Stimulates tracing */

static Apertures GB_Apertures = { 0 };		/* Holds aperture definitions */

static Boolean GB_ErrorOccurred;


/*----------------------------------------------------------------------------*/
/* Error Logging Routines                                                     */
/*----------------------------------------------------------------------------*/
static void logError(fp, format, va_alist)
	FILE *fp;
	char *format;
	va_dcl
{
	va_list args;
	char    s[1024];

		/* FIXME: Is it legitimate to use Message() from within a
			driver? */

	va_start(args);
	vsprintf(s, format, args);
	fputs(s, fp);
/*
	FIXME: to much error output...
	Message("%s", s);
*/
	va_end(args);
}


/*----------------------------------------------------------------------------*/
/* Aperture Routines                                                          */
/*----------------------------------------------------------------------------*/

char *apertureShape(ApertureShape shape)
{
	char *cp;

	if (shape == ROUND)
		cp = "round";
	else if (shape == SQUARE)
		cp = "square";
	else
		cp = "*invalid-shape*";

	return (cp);
}

char *apertureFunction(ApertureFunction function)
{
	char *cp;

	if (function == FLASH)
		cp = "flash";
	else if (function == DRAW)
		cp = "draw";
	else if (function == DRILL)
		cp = "drill";
	else
		cp = "*invalid-function*";

	return (cp);
}

static int findApertureCode(
	Apertures *apertures,
	int width,
	ApertureShape shape,
	ApertureFunction function)
{
	int i;

	/* Search for an appropriate aperture. */

	for (i = 0; i < apertures->nextAvailable; i++) {
		Aperture *ap = &apertures->aperture[i];

		if (ap->apertureSize == width
			&& ap->apertureShape == shape
			&& ap->apertureFunction == function) {
			return (ap->dCode);
		}
	}

	/* Report the the exception and repair the error with a default aperture. */

	GB_ErrorOccurred = True;
	logError(GB_Flags.FP,
		"G04 Missing aperture with width=%d shape=%s function=%s *\015\012",
		width,
		apertureShape(shape),
		apertureFunction(function)
		);
	return (10);
}


static void loadApertures(Apertures *apertures)
{
	FILE 	*apertureFile;
	char 	line[GB_MAXLINELEN];
	char	*command;

	apertures->nextAvailable = 0;

		/* evaluate command and read data */
	command = EvaluateFilename(Settings.ApertureCommand, Settings.AperturePath,
		Settings.ApertureFile, NULL);
	if (*command == '\0' || (apertureFile = popen(command, "r")) == NULL)
	{
		PopenErrorMessage(command);
		return;
	}

	while (fgets(line, sizeof line, apertureFile) != (char *) NULL &&
		apertures->nextAvailable < GB_MAXAPERTURECOUNT) {
		if (*line == '#') {
			/* Skip comment lines. */
			continue;
		} else if (strcmp(line, "\n") == 0) {
			/* Skip blank lines. */
			continue;
		} else {
			int dCode;
			int apertureSize;
			char shape[GB_MAXTOKENLEN];
			char function[GB_MAXTOKENLEN];

			if (sscanf(line, "aperture D%d %d %s %s",
				&dCode, &apertureSize, shape, function) == 4) {
				int i;
				Aperture *ap;

				i = apertures->nextAvailable++;
				ap = &apertures->aperture[i];

				ap->dCode = dCode;
				ap->apertureSize = apertureSize;

				if (strcmp(shape, "round") == 0) {
					ap->apertureShape = ROUND;
				} else if (strcmp(shape, "square") == 0) {
					ap->apertureShape = SQUARE;
				}

				if (strcmp(function, "flash") == 0) {
					ap->apertureFunction = FLASH;
				} else if (strcmp(function, "draw") == 0) {
					ap->apertureFunction = DRAW;
				} else if (strcmp(function, "drill") == 0) {
					ap->apertureFunction = DRILL;
				}

				if (GB_debug >= 5) {
					fprintf(GB_Flags.FP,
						"G04 Aperture: %d %d %d %d *\015\012",
						ap->dCode,
						ap->apertureSize,
						(int) ap->apertureShape,
						(int) ap->apertureFunction
						);
				}
			} else {
				char *cp;

				GB_ErrorOccurred = True;

					/* Strip the trailing newline. */
				cp = line;
				while (*cp != '\0') {
					if (*cp == '\n')
						*cp = ' ';
					cp++;
				}

					/* Log the error for the user. */
				logError(GB_Flags.FP,
					"G04 Invalid aperture definition: %s *\015\012",
					line
					);
			}
		}
	}

	pclose(apertureFile);
}

/* ---------------------------------------------------------------------------
 * returns information about this driver
 */
PrintDeviceTypePtr GB_Query(void)
{
	return(&GB_QueryConstants);
}

/* ----------------------------------------------------------------------
 * prints custom Gerber/RS-274D compatible header with function definition
 * info struct is are passed in
 */
static char *GB_Init(PrintInitTypePtr Flags, char *Description)
{
	int		i;
	int		j;
	time_t		currenttime;
        char		utcTime[64];
        struct  passwd  *pwentry;

		/* save passed-in data */
	GB_Flags = *Flags;
	currenttime = time(NULL);

		/* Create a portable timestamp. */
        strftime(utcTime, sizeof utcTime, "%c UTC", gmtime(&currenttime));

		/* No errors have occurred so far. */
	GB_ErrorOccurred = False;

		/* ID the user. */
        pwentry = getpwuid(getuid());

	/* Print a cute file header at the beginning of each file. */
	fprintf(GB_Flags.FP, "G04 Title: %s, %s *\015\012", UNKNOWN(PCB->Name),
		UNKNOWN(Description));
	fprintf(GB_Flags.FP, "G04 Creator: %s "RELEASE" *\015\012", Progname);
	fprintf(GB_Flags.FP, "G04 CreationDate: %s *\015\012", utcTime);
	fprintf(GB_Flags.FP, "G04 For: %s *\015\012", pwentry->pw_name);
	fprintf(GB_Flags.FP, "G04 Format: Gerber/RS-274D 2,3 leading inch *\015\012");
	fprintf(GB_Flags.FP, "G04 PCB-Dimensions: %ld %ld *\015\012",
		(long) PCB->MaxWidth,
		(long) PCB->MaxHeight
		);
	fprintf(GB_Flags.FP, "G04 PCB-Coordinate-Origin: lower left *\015\012");

		/* black is default drawing color */
	fprintf(GB_Flags.FP,
		"G04 Color: R%ld G%ld B%ld *\015\012",
		0L,
		0L,
		0L);

		/* print own procedures */
	for (j = 0; j < ENTRIES(GB_Functions); j++)
		fprintf(GB_Flags.FP, "%s*\015\012", GB_Functions[j]);

        	/* We assume that the initial state has the light off. */

       		/* Signal Absolute Data. */
	fprintf(GB_Flags.FP,
        	"G90*\015\012");

       		/* Signal Straight Line Data. */
	fprintf(GB_Flags.FP,
       		"G01*\015\012");

		/* Load databases. */
	loadApertures(&GB_Apertures);

	return(NULL);
}

/* ---------------------------------------------------------------------------
 * exit code for this driver is empty
 */
static void GB_Exit(void)
{
	int i;

		/* print trailing commands */

        	/* Turn off the light */
	fprintf(GB_Flags.FP,
       		"D02*\015\012");

		/* Signal End-of-Plot. */
	fprintf(GB_Flags.FP,
		"M02*\015\012");

	if (GB_ErrorOccurred != False)
		logError(stderr, "An error occurred.  See the output file(s) for details.\n");
}

/* ----------------------------------------------------------------------
 * handles drilling information
 */
static void GB_Drill(PinTypePtr PinOrVia)
{
	/* FIXME: Accumulate and sort by drill size. */
	fprintf(GB_Flags.FP,
		"G54D%d*X%ldY%ldD03*\015\012",
		findApertureCode(&GB_Apertures, PinOrVia->DrillingHole, ROUND, DRILL),
		gerberX(PCB, PinOrVia->X),
		gerberY(PCB, PinOrVia->Y)
		);
}

/* ----------------------------------------------------------------------
 * prints layer data
 */
static void GB_PrintLayer(LayerTypePtr Layer, int GroupNumber,
	Boolean SilkscreenTextFlag)
{
	FILE *FP;

	/* FIXME: Check layer bounds. */
	FP = GB_Flags.FP;
	fprintf(FP,
		"G04 LayerGroup: %i *\015\012",
		GroupNumber + 1);
	fprintf(FP,
		"G04 Layer: \"%s\" (%i) *\015\012",
		UNKNOWN(Layer->Name),
		GetLayerNumber(PCB->Data, Layer) + 1);
	LINE_LOOP(Layer, GB_PrintLine(FP, line));
	if (! SilkscreenTextFlag)
		TEXT_LOOP(Layer, GB_PrintText(FP, text));
	POLYGON_LOOP(Layer, GB_PrintPolygon(FP, polygon));
}

/* ----------------------------------------------------------------------
 * prints a line
 */
static void GB_PrintLine(FILE *FP, LineTypePtr Line)
{
	fprintf(FP,
		"G54D%dX%ldY%ldD02*X%ldY%ldD01*\015\012",
		findApertureCode(&GB_Apertures, Line->Thickness, ROUND, DRAW),
		gerberX(PCB, Line->Point1.X),
		gerberY(PCB, Line->Point1.Y),
		gerberX(PCB, Line->Point2.X),
		gerberY(PCB, Line->Point2.Y)
		);
}

/* ----------------------------------------------------------------------
 * Checks a four-point polygon to see if it's a rectangle.
 * Tick off pairs of X & Y coords; if we get four matches,
 * we have a rectangle.
 */
static int isRectangle(PolygonTypePtr Ptr)
{
	Cardinal i, j;
	int matches = 0;

        if (Ptr->PointN != 4)
	   return(0);

        for (i=0; i<4; i++)
           for (j = i+1; j<4; j++) {
	      if (Ptr->Points[i].X == Ptr->Points[j].X)
	         matches++;

	      if (Ptr->Points[i].Y == Ptr->Points[j].Y)
	         matches++;
	   }

        if (matches == 4)
	   return(1);
	else
	   return(0);
 }

/* ---------------------------------------------------------------------------
 * prints a filled polygon
 */
static void GB_PrintPolygon(FILE *FP, PolygonTypePtr Ptr)
{
	static int isRectangle(PolygonTypePtr Ptr);
	int firstTime = 1;
	Position startX, startY;

	/* FIXME: Steal code from somewhere to complete this function. */
	/* FIXME: Beware of overexposure; handle it, if possible
		(probably too hard and only needed for a true photoplotter). */
	/* FIXME: Don't use 5 mil line width: Fill the polygon. */
	/* FIXME: The polygon wire-frames use 5 mil lines on center, therefore
		2.5 mil overhangs each ideal polygon edge. */
	/* FIXME: Consider the following notes by Monte Bateman:
		Here is how I would implement the more general polygon fill:

		1.  Use the built-in variables for minx, maxx, miny, and maxy for the
    		specified polygon.  Allocate a block of memory (2-D array) that is
    		(maxx - minx) X (maxy - miny) large.  Init the array to all zeroes.

		2.  Run through the list of points and "draw" lines between them into this
    		array.  Set to one each array element where a line crossess.
    		Use Bresenham's algorithm to draw the lines.

		3.  When finished, scan each row of the array.  When you find a '1',
    		start a horizontal line there.  When you find the next '1', end the
    		horizontal line there.  This will take a bit of experimenting to
    		find the proper spacing; I suggest using a 2mil round aperture and
    		scan every other horizontal row in the array.

		4.  This should give (close to) the desired results.  There is probably
    		a much better way to allocate the memory and "draw" the polygon
    		lines --- we really only need one bit per pixel, but that makes the
    		memory mapping tricky.  In reality, the large areas will be filled
    		with rectangles, and that is already hadled with my earlier code.
    		Non-rectangular polygons will be used to fill in weird-shaped little
    		corners, etc.  So it won't need too huge a block of memory, and you 
    		can get rid of it when finished.

    		I also wouldn't worry too much about the bizarre, folded polygons
    		(like the one in your test file).  If a user gets something that
    		doesn't fill correctly, they can go back a re-draw with "regular"
    		polygons and rectangles.  I would bet that 90% of the filling will be
    		simple rectangles, and the next largest percentage will be triangles.
		*/

	if (isRectangle(Ptr)) {
		GB_FPrintFilledRectangle(
			FP,
			Ptr->BoundingBox.X1,
			Ptr->BoundingBox.Y1,
			Ptr->BoundingBox.X2,
			Ptr->BoundingBox.Y2
			);
	} else {
		fprintf(FP,
			"G54D%d",
			findApertureCode(&GB_Apertures, 5, ROUND, DRAW));
		POLYGONPOINT_LOOP(Ptr,
			{
				fprintf(FP,
					"X%ldY%ld%s*\015\012",
					gerberX(PCB, point->X),
					gerberY(PCB, point->Y),
					(firstTime ? "D02" : "D01")
					);
				if (firstTime) {
					firstTime = 0;
					startX = point->X;
					startY = point->Y;
				}
			}
 		);
		fprintf(FP,
			"X%ldY%ldD01*\015\012",
			gerberX(PCB, startX),
			gerberY(PCB, startY)
			);
	}
}

/* ----------------------------------------------------------------------
 * prints a text
 * the routine is identical to DrawText() in module draw.c except
 * that DrawLine() and DrawRectangle() are replaced by their corresponding
 * printing routines
 */
static void GB_PrintText(FILE *FP, TextTypePtr Text)
{
	Position		x = 0,
					width;
	unsigned char	*string = (unsigned char *) Text->TextString;
	Cardinal		n;
	FontTypePtr		font = &PCB->Font;

		/* Add the text to the gerber file as a comment. */
	if (string && *string)
		fprintf(FP,
			"G04 Text: %s *\015\012",
			string);

		/* get the center of the text for mirroring */
	width = Text->Direction & 0x01 ? 
				Text->BoundingBox.Y2 -Text->BoundingBox.Y1 :
				Text->BoundingBox.X2 -Text->BoundingBox.X1;
	while (string && *string)
	{
			/* draw lines if symbol is valid and data is present */
		if (*string <= MAX_FONTPOSITION && font->Symbol[*string].Valid)
		{
			LineTypePtr	line = font->Symbol[*string].Line;
			LineType	newline;

			for (n = font->Symbol[*string].LineN; n; n--, line++)
			{
					/* create one line, scale, move, rotate and swap it */
				newline = *line;
				newline.Point1.X = (newline.Point1.X +x) *Text->Scale /100;
				newline.Point1.Y = newline.Point1.Y      *Text->Scale /100;
				newline.Point2.X = (newline.Point2.X +x) *Text->Scale /100;
				newline.Point2.Y = newline.Point2.Y      *Text->Scale /100;
				newline.Thickness = newline.Thickness *Text->Scale /100;

					/* do some mirroring, swaping and rotating */
				if (TEST_FLAG(MIRRORFLAG, Text))
				{
					newline.Point1.X = width -newline.Point1.X;
					newline.Point2.X = width -newline.Point2.X;
				}
				RotateLineLowLevel(&newline, 0, 0, Text->Direction);

					/* the labels of SMD objects on the bottom
					 * side haven't been swapped yet, only their offset
					 */
/* FIXME:
				if (TEST_FLAG(ONBOTTOMFLAG, Text))
				{
					newline.Point1.X = SWAP_SIGN_X(newline.Point1.X);
					newline.Point1.Y = SWAP_SIGN_Y(newline.Point1.Y);
					newline.Point2.X = SWAP_SIGN_X(newline.Point2.X);
					newline.Point2.Y = SWAP_SIGN_Y(newline.Point2.Y);
				}
*/
					/* add offset and draw line */
				newline.Point1.X += Text->X;
				newline.Point1.Y += Text->Y;
				newline.Point2.X += Text->X;
				newline.Point2.Y += Text->Y;
				GB_PrintLine(FP, &newline);
			}

				/* move on to next cursor position */
			x += (font->Symbol[*string].Width +font->Symbol[*string].Delta);
		}
		else
		{
				/* the default symbol is a filled box */
			BoxType		defaultsymbol = PCB->Font.DefaultSymbol;
			Position	size = (defaultsymbol.X2 -defaultsymbol.X1) *6/5;

			defaultsymbol.X1 = (defaultsymbol.X1 +x) *Text->Scale /100;
			defaultsymbol.Y1 = defaultsymbol.Y1      *Text->Scale /100;
			defaultsymbol.X2 = (defaultsymbol.X2 +x) *Text->Scale /100;
			defaultsymbol.Y2 = defaultsymbol.Y2      *Text->Scale /100;

				/* do some mirroring ... */
			if (TEST_FLAG(MIRRORFLAG, Text))
			{
				defaultsymbol.X1 = width -defaultsymbol.X1;
				defaultsymbol.X2 = width -defaultsymbol.X2;
			}
			RotateBoxLowLevel(&defaultsymbol, 0, 0, Text->Direction);

				/* add offset and draw filled box */
			defaultsymbol.X1 += Text->X;
			defaultsymbol.Y1 += Text->Y;
			defaultsymbol.X2 += Text->X;
			defaultsymbol.Y2 += Text->Y;
			GB_FPrintFilledRectangle(FP,
				defaultsymbol.X1,
				defaultsymbol.Y1,
				defaultsymbol.X2,
				defaultsymbol.Y2
				);

				/* move on to next cursor position */
			x += size;
		}
		string++;
	}
}

/* ----------------------------------------------------------------------
 * prints text
 */

static void GB_PrintSomeText(TextTypePtr Text)
{
	GB_PrintText(GB_Flags.FP, Text);
}

/* ----------------------------------------------------------------------
 * prints package outline and selected text (Canonical, Instance or Value)
 */
static void GB_PrintElementPackage(ElementTypePtr Element)
{
	FILE *FP;
	long arcStartX, arcStartY;
	long arcStopX, arcStopY;
	int useCWArc;

	FP = GB_Flags.FP;

	ELEMENTLINE_LOOP(Element, GB_PrintLine(FP, line););
	ARC_LOOP(Element, 
		/* FIXME: Handle impure arcs where the height is not equal to
			the width. */
		/* FIXME: Determine whether or not negative I/J parameters are
			valid.  If they are not, then this code must be
			reconsidered, possible with quadarant arcs in mind. */
		/* FIXME: Handle arcs which start at other than 0/90/180/270. */

		if (arc->StartAngle == 0 || arc->StartAngle == 180)
			useCWArc = 1;
		else if (arc->StartAngle == 90 || arc->StartAngle == 270)
			useCWArc = 0;
		else {
			useCWArc = -1;
		}

		if (useCWArc == -1) {
			GB_ErrorOccurred = True;
			logError(GB_Flags.FP,
				"G04 %d %d %d %d %d %d %d Arc *\015\012",
				(int) arc->X, (int) arc->Y,
				(int) arc->Width, (int) arc->Height,
				(int) arc->Thickness,
				arc->StartAngle,
				arc->Delta);
		} else {
			arcStartX = arc->X
				+ arc->Width * cos(TO_RADIANS(arc->StartAngle));
			arcStartY = arc->Y
				+ arc->Width * sin(TO_RADIANS(arc->StartAngle));
			arcStopX= arc->X
				+ arc->Width * cos(TO_RADIANS(arc->StartAngle + arc->Delta));
			arcStopY = arc->Y
				+ arc->Width * sin(TO_RADIANS(arc->StartAngle + arc->Delta));

			fprintf(FP,
				"G54D%d",
				findApertureCode(&GB_Apertures, arc->Thickness, ROUND, DRAW)
				);
			fprintf(FP,
				"X%ldY%ldD02*G75%sI%ldJ%ldX%ldY%ldD01*G01*\015\012",
				gerberX(PCB, arcStartX),
				gerberY(PCB, arcStartY),
				(useCWArc ? "G02" : "G03"),
				gerberXOffset(PCB, arc->X - arcStartX),
				gerberYOffset(PCB, arc->Y - arcStartY),
				gerberX(PCB, arcStopX),
				gerberY(PCB, arcStopY)
				);
		}
	);
	GB_PrintText(FP, &ELEMENT_TEXT(PCB, Element));
}

/* ----------------------------------------------------------------------
 * prints a pad
 */
static void GB_PrintPad(PadTypePtr Pad)
{
	fprintf(GB_Flags.FP,
		"G54D%dX%ldY%ldD02*X%ldY%ldD01*\015\012",
		findApertureCode(&GB_Apertures, Pad->Thickness, ROUND, DRAW),
		gerberX(PCB, Pad->Point1.X),
		gerberY(PCB, Pad->Point1.Y),
		gerberX(PCB, Pad->Point2.X),
		gerberY(PCB, Pad->Point2.Y)
		);
}

/* ----------------------------------------------------------------------
 * prints a via or pin to a specified file
 */
static void GB_FPrintPinOrVia(FILE *FP, PinTypePtr Ptr)
{
	fprintf(FP,
		"G54D%d*X%ldY%ldD03*\015\012",
		findApertureCode(&GB_Apertures, Ptr->Thickness, ROUND, FLASH),
		gerberX(PCB, Ptr->X),
		gerberY(PCB, Ptr->Y)
		);

}

/* ----------------------------------------------------------------------
 * prints a via or pin
 */
static void GB_PrintPinOrVia(PinTypePtr Ptr)
{
	int i;

		/* Add the pin/via to the layer file. */
	GB_FPrintPinOrVia(GB_Flags.FP, Ptr);
}

/* ----------------------------------------------------------------------
 * clears the area around a via or pin on the groundplane
 */
static void GB_PrintClearPinOrViaOnGroundplane(PinTypePtr Pin)
{
	/* FIXME: Implement this function? */

	GB_ErrorOccurred = True;
	logError(GB_Flags.FP, "G04 %d %d ClearPinOrViaOnGroundplane *\015\012",
		(int) Pin->X, (int) Pin->Y);
}

/* ----------------------------------------------------------------------
 * prints a pad mask
 */
static void GB_PrintPadMask(PadTypePtr Pad)
{
		/* FIXME: The thickness need to be oversized slightly to make
			soldermask registration less critical. */

		/* Add pad to the solder mask. */
	fprintf(GB_Flags.FP,
		"G54D%dX%ldY%ldD02*X%ldY%ldD01*\015\012",
		findApertureCode(&GB_Apertures, Pad->Thickness, ROUND, DRAW),
		gerberX(PCB, Pad->Point1.X),
		gerberY(PCB, Pad->Point1.Y),
		gerberX(PCB, Pad->Point2.X),
		gerberY(PCB, Pad->Point2.Y)
		);
}

/* ----------------------------------------------------------------------
 * prints a via or pin mask
 */
static void GB_PrintPinOrViaMask(PinTypePtr Pin)
{
		/* FIXME: The thickness need to be oversized slightly to make
			soldermask registration less critical. */

		/* Add pin/via to the solder mask. */
	GB_FPrintPinOrVia(GB_Flags.FP, Pin);
}

/* ---------------------------------------------------------------------------
 * draws a filled rectangle to the specified file
 */
static void GB_FPrintFilledRectangle(FILE *FP, Position X1, Position Y1,
	Position X2, Position Y2)
{
	int i, j;

	/* FIXME: Compensate for the overhang of the 5 mil aperture. */

	fprintf(FP,
		"G54D%d",
		findApertureCode(&GB_Apertures, 5, ROUND, DRAW));

	for (j = Y1; j < Y2; j += 4) {
 		fprintf(FP,
			"X%ldY%ldD02*\015\012",
			gerberX(PCB, X1 + 1),
			gerberY(PCB, j)
 			);
		fprintf(FP,
			"X%ldY%ldD01*\015\012",
			gerberX(PCB, X2 - 1),
			gerberY(PCB, j)
			);
 	}
}

/* ---------------------------------------------------------------------------
 * draws a filled rectangle for the ground plane and/or solder mask.
 */
static void GB_PrintMaskOrGroundplaneRectangle(Position X1, Position Y1,
	Position X2, Position Y2)
{
		/* FIXME: Since we cannot easily implement the PCB paradigm, we
			will ignore it.  PCB expects the driver to be able
			to fill an area an also be able to un-fill selected
			portions of that area.  PCB currently supports only
			one ground plane, therefore, we can simply treat
			this call as an opportunity to make a notation in
			the Gerber file that this file will be a negative
			Gerber file.  That is, all areas that are not drawn
			are to be considered as filled.  The support for
			"clear" pads and pins will be simply to draw opaque
			pads and pins. */
	fprintf(GB_Flags.FP, "G04 Image-Polarity: This is a negative Gerber file *\015\012");
		/* FIXME: Should we force additional, properly sized, apertures
			to be defined.  Perhaps an extension to the aperture
			definition file is required to allow a mask/groundplane
			aperture to be associated with a flash/draw aperture. */
	fprintf(GB_Flags.FP, "G04 Note: All apertures give are undersized; adjust for clearances *\015\012");
}

/* ---------------------------------------------------------------------------
 * draw the outlines of a layout;
 * the upper/left and lower/right corner are passed
 */
static void GB_PrintOutline(Position X1, Position Y1, Position X2, Position Y2)
{
	/* FIXME: Implement this function? */

	GB_ErrorOccurred = True;
	logError(GB_Flags.FP, "G04 %d %d %d %d Outline *\015\012",
		(int) X1, (int) Y1, (int) X2, (int) Y2);
}

/* ---------------------------------------------------------------------------
 * draw the alignment targets;
 * the upper/left and lower/right corner are passed
 */
static void GB_PrintAlignment(Position X1, Position Y1, Position X2, Position Y2)
{
	/* FIXME: Implement this function? */

	GB_ErrorOccurred = True;
	logError(GB_Flags.FP, "G04 %d %d %d %d %d Alignment *\015\012",
		(int) X1, (int) Y1, (int) X2, (int) Y2,
		(int) Settings.AlignmentDistance);
}

/* ---------------------------------------------------------------------------
 * prints a via or pin
 */
static void GB_PrintDrillingHelper(PinTypePtr Pin)
{
	/* FIXME: Implement this function? */

	GB_ErrorOccurred = True;
	logError(GB_Flags.FP, "G04 %d %d DrillingHelper *\015\012",
		(int) Pin->X, (int) Pin->Y);
}

/* ----------------------------------------------------------------------
 * queries color from X11 database and generates Gerber/RS-274D compatible
 * comment.
 */
static void GB_SetColor(XColor RGB)
{
	int i;

	fprintf(GB_Flags.FP,
		"G04 Color: R%ld G%ld B%ld *\015\012",
		(long) ((float) RGB.red / 65535.0 * 1000),
		(long) ((float) RGB.green / 65535.0 * 1000),
		(long) ((float) RGB.blue / 65535.0) * 1000);
}
