/* Copyright (C) 1991, 1992, 1994 Free Software Foundation, Inc.
This file is part of the GNU C Library.

The GNU C Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the GNU C Library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.  */

/* Modified by Guido Flohr <gufl0000@stud.uni-sb.de>, 25 Nov 96.
The first part was taken from the GNU-libc version 1.09,  I've tried it
with the system.c in the sysdeps/posix subdirectory.  Hope it 
works. The system_hack() function was the original system() from MiNTLIB
PL 46.  There's a separate Copyright for this one:  */
/*
 * system(): execute a command, passed as a string
 *
 * Written by Eric R. Smith and placed in the public domain.
 *
 * Modified by Allan Pratt to call _unx2dos on redirect file names
 * and to call spawnvp() without calling fork() -- why bother?
 *
 * Modified by Frank Ridderbusch in _parseargs() to handle the case
 * >'file'. Without the modification, the quotes would end up in the
 * filename for redirection
 *
 */
/* And I have mixed the two versions together.  No, don't want a copyright,  
   it's yours... --gfl */

#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <sys/types.h>
#include <errno.h>

/* These are the includes from the MiNTLib version.  */
#include <limits.h>
#include <string.h>
#include <ctype.h>
/* #include <stdlib.h> */
#include <process.h>
/* #include <errno.h> */
#include <file.h>
#include <osbind.h>
/* #include <unistd.h> */
/* #include "lib.h"  */


/* Make it work with the MiNTLib naming conventions --gfl.  */
extern char** environ;
#define	__environ	environ

#define __sigemptyset sigemptyset
#define __sigaction sigaction
#define __sigaddset sigaddset
#define __sigprocmask sigprocmask
#define __vfork vfork
#define __execve execve
#define __waitpid waitpid

/* No #ifndef!  Want to get an error.  */
#define CONST const
/* End of MiNT specific stuff.  */

#define	SHELL_PATH	"/bin/sh"	/* Path of the shell.  */
#define	SHELL_NAME	"sh"		/* Name to give it.  */

/* We should remember what system() is supported.  --gfl */
static int posix_system;
static int initialized = 0;  /* If non-zero the value in posix_system is valid.  */
static char* shell_path;
static char* shell_name;
static int system_hack(CONST char* s);
/*********************************************************/

/* Execute LINE as a shell command, returning its status.  */
int system (line)
register CONST char* line;
{
  int status = 0;  /* Better initialize it.  -- gfl*/
  int save;
  pid_t pid;
  struct sigaction sa, intr, quit;
#ifndef WAITPID_CANNOT_BLOCK_SIGCHLD
  sigset_t block, omask;
#endif

  if (line == NULL)
    return 1;

/* Check what kind of function to use.  --gfl */
  if (!initialized) {
    initialized = 1;
    if (access (SHELL_PATH, X_OK) == 0) {
      /* We can use the Posix version.  */
      shell_path = SHELL_PATH;
      shell_name = SHELL_NAME;
      posix_system = 1;
    } else {
      /* /bin/sh is not executable.  We give it another try.  */
      char* cur;
      
      shell_path = getenv ("SHELL");
      if (shell_path) {
        cur = shell_path;
        while (*cur) cur++;
        cur--;
        /* FIXME: Should check UNIXMODE if slash is a valid path separator.  */
        while (*cur && *cur != '/' && *cur != '\\') cur--;        
        if (*cur == '\\' || *cur == '/') cur++;
        shell_name = cur;
        posix_system = 1;
      } else {
        posix_system = 0;
      }
    }
  }

  if (!posix_system)
    return (system_hack (line));
/************************************************/

  sa.sa_handler = SIG_IGN;
  sa.sa_flags = 0;
  __sigemptyset (&sa.sa_mask);

  if (__sigaction (SIGINT, &sa, &intr) < 0)
    return -1;
  if (__sigaction (SIGQUIT, &sa, &quit) < 0)
    {
      save = errno;
      (void) __sigaction (SIGINT, &intr, (struct sigaction *) NULL);
      errno = save;
      return -1;
    }

#ifndef WAITPID_CANNOT_BLOCK_SIGCHLD

/* SCO 3.2v4 has a bug where `waitpid' will never return if SIGCHLD is
   blocked.  This makes it impossible for `system' to be implemented in
   compliance with POSIX.2-1992.  They have acknowledged that this is a bug
   but I have not seen nor heard of any forthcoming fix.  */

  __sigemptyset (&block);
  __sigaddset (&block, SIGCHLD);
  save = errno;
  if (__sigprocmask (SIG_BLOCK, &block, &omask) < 0)
    {
      if (errno == ENOSYS)
	errno = save;
      else
	{
	  save = errno;
	  (void) __sigaction (SIGINT, &intr, (struct sigaction *) NULL);
	  (void) __sigaction (SIGQUIT, &quit, (struct sigaction *) NULL);
	  errno = save;
	  return -1;
	}
    }
#define UNBLOCK	__sigprocmask (SIG_SETMASK, &omask, (sigset_t *) NULL)
#else
#define UNBLOCK 0
#endif

  pid = __vfork ();
  if (pid == (pid_t) 0)
    {
      /* Child side.  */
      CONST char *new_argv[4];
      new_argv[0] = shell_name;
      new_argv[1] = "-c";
      new_argv[2] = line;
      new_argv[3] = NULL;

      /* Restore the signals.  */
      (void) __sigaction (SIGINT, &intr, (struct sigaction *) NULL);
      (void) __sigaction (SIGQUIT, &quit, (struct sigaction *) NULL);
      (void) UNBLOCK;

      /* Exec the shell.  */
      (void) __execve (shell_path, (char *CONST *) new_argv, __environ);
      _exit (127);
    }
  else if (pid < (pid_t) 0)
    /* The fork failed.  */
    status = -1;
  else
    /* Parent side.  */
#ifdef	NO_WAITPID
    {
      pid_t child;
      do
	{
	  child = __wait (&status);
	  if (child <= -1)
	    {
	      status = -1;
	      break;
	    }
	} while (child != pid);
    }
#else
    if (__waitpid (pid, &status, 0) != pid)
      status = -1;
#endif

  save = errno;
  if ((__sigaction (SIGINT, &intr, (struct sigaction *) NULL) |
       __sigaction (SIGQUIT, &quit, (struct sigaction *) NULL) |
       UNBLOCK) != 0)
    {
      if (errno == ENOSYS)
	errno = save;
      else
	return -1;
    }

  return status;
}

/* Now the original MiNTLib function renamed to system_hack().  --gfl  */
#define isquote(c) ((c) == '\"' || (c) == '\'' || (c) == '`')
#define ARG_ERR	   ( (Argentry *) -1L )

/* struct. used to build a list of arguments for the command */

typedef struct argentry {
	struct argentry *next;
	char	string[1];
} Argentry;

static Argentry *_argalloc __PROTO((const char *s));
static void _argfree __PROTO((Argentry *p));
static Argentry *_parseargs __PROTO((const char *s));

/* allocate an Argentry that will hold the string "s" */

static Argentry *_argalloc(s)
	const char *s;
{
	Argentry *x;

	x = (Argentry *) malloc((size_t)(sizeof(Argentry) + strlen(s) + 1));
	if (!x)
		return ARG_ERR;
	x->next = (Argentry *) 0;
	strcpy(x->string, s);
	return x;
}

/* free a list of Argentries */

static void _argfree(p)
	Argentry *p;
{
	Argentry *oldp;

	while (p) {
		oldp = p;
		p = p->next;
		free(oldp);
	}
}

/* parse a string into a list of Argentries. Words are defined to be
 * (1) any sequence of non-blank characters
 * (2) any sequence of characters starting with a ', ", or ` and ending
 *     with the same character. These quotes are stripped off.
 * (3) any spaces after an unquoted > or < are skipped, so
 *     "ls > junk" is parsed as 'ls' '>junk'.
 */

static Argentry *_parseargs(s)
	const char *s;
{
	Argentry *cur, *res;
	char buf[1024];
	char *t, quote;

	res = cur = _argalloc("");

	for(;;) {
		t = buf;
again:
		while (isspace(*s)) s++;
		if (!*s) break;
		if (isquote(*s)) {
			quote = *s++;
			while (*s && *s != quote)
				*t++ = *s++;
			if (*s) s++;	/* skip final quote */
		}
		else {
			while (*s && !isspace(*s)) {
				*t++ = *s++;
				if (isquote(*s))
					goto again;
			}
			if (*s && ( *(s-1) == '>' || *(s-1) == '<' ))
				goto again;
		}
		*t = 0;
		cur->next = _argalloc(buf);
		if ((cur = cur->next) == NULL)	  /* couldn't alloc() */
			return ARG_ERR;
	}
	cur->next = (Argentry *) 0;
	cur = res; res = res->next; free(cur);
	return res;
}


/* Here is system() itself.
 * FIXME: we probably should do wildcard expansion.
 * also, should errno get set here??
 *
 */

static int system_hack(s)
	const char *s;
{
	Argentry *al, *cur;
	char **argv, *p;
	int  argc, i;
	char const *infile, *outfile;
	int  infd = 0, outfd = 1, append = 0;
	int oldin = 0, oldout = 1;	/* hold the Fdup'd in, out */
	char path[PATH_MAX];
	int retval;

	if (!s)		/* check for system() supported ?? */
		return 1;
	al = _parseargs(s);		/* get a list of args */
	if (al == ARG_ERR) {		/* not enough memory */
		errno = ENOMEM;
		return -1;
	}

	infile = outfile = "";

/* convert the list returned by _parseargs to the normal char *argv[] */
	argc = i = 0;
	for (cur = al; cur; cur = cur->next)
		argc++;
	if ((argv = (char **) malloc((size_t)(argc * sizeof(char *))))
		== NULL)
	{
		errno = ENOMEM; return -1;
	}
	for (cur = al; cur; cur = cur->next) {
		p = cur->string;
		if (*p == '>') {
			outfile = p+1;
			if (*outfile == '>') {
				outfile++;
				append = 1;
			}
			else
			append = 0;
		}
		else if (*p == '<') {
			infile = p+1;
		}
		else
			argv[i++] = p;
	}
 	argv[i] = (char *)0;
 
/* now actually run the program */

	if (*infile) {
		(void)_unx2dos(infile,path);
		infd = (int)Fopen(path, 0);
		if (infd < __SMALLEST_VALID_HANDLE) {
			perror(infile);
			return(2);
		}
		oldin = (int)Fdup(0);
		(void)Fforce(0, infd);
	}
	if (*outfile) {
		(void)_unx2dos(outfile,path);
		if (append) {
			outfd = (int)Fopen(path, 2);
			if (outfd < __SMALLEST_VALID_HANDLE)
				outfd = (int)Fcreate(path, 0);
			else
				(void)Fseek(0L, outfd, 2);
		}
		else
			outfd = (int)Fcreate(path, 0);
		if (outfd < __SMALLEST_VALID_HANDLE) {
			perror(outfile);
			return(2);
		}
		oldout = (int)Fdup(1);
		(void)Fforce(1, outfd);
	}

	retval = spawnvp(P_WAIT, argv[0], argv);

	if (*infile) (void)(Fforce(0,oldin), Fclose(oldin), Fclose(infd));
	if (*outfile) (void)(Fforce(1,oldout), Fclose(oldout), Fclose(outfd));
	free(argv);
	_argfree(al);
	return retval;
}
