/*
 * $Header: /home/orchestra5/davy/stuff/misc/xsat/RCS/satellites.c,v 1.1 92/04/10 14:08:14 davy Exp $
 *
 * Copyright 1992 by David A. Curry
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The
 * author makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * Routines for dealing with the "satellites" menu, reading the satellite
 * information, etc.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * 1285 Electrical Engineering Building
 * West Lafayette, IN 47907
 * davy@ecn.purdue.edu
 *
 * $Log:	satellites.c,v $
 * Revision 1.1  92/04/10  14:08:14  davy
 * Initial revision
 *
 * Revision 1.2  94/05/24  11:22:33  trf
 * 
 */
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Dialog.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "xsat.h"

static SatInfo	*satInfo;

static void	setSat();
static void	addSat();
static int	satCompare();

/*
 * Show current satellite information.
 */
void
satellitesShowSatellite(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	Arg argList[2];
	char buf[BUFSIZ];

	if (currentSat == NULL) {
		prompt("No satellite selected.", NULL, promptOkay);
		return;
	}

	rewind(tmpFP);
	ftruncate(fileno(tmpFP), 0);

	fprintf(tmpFP, "Satellite:       %s\n", currentSat->s_name);
	fprintf(tmpFP, "Catalog number:  %lu\n", currentSat->s_satnum);
	fprintf(tmpFP, "Epoch time:      %.8lf\n", currentSat->s_epochday);
	fprintf(tmpFP, "Element set:     %s\n", currentSat->s_elementset);
	fprintf(tmpFP, "Inclination:     %lf deg\n",
		currentSat->s_inclination);
	fprintf(tmpFP, "RA of node:      %lf deg\n", currentSat->s_raan);
	fprintf(tmpFP, "Eccentricity:    %.7lf\n", currentSat->s_eccentricity);
	fprintf(tmpFP, "Arg of perigee:  %lf deg\n", currentSat->s_argperigee);
	fprintf(tmpFP, "Mean anomaly:    %lf deg\n",
		currentSat->s_meananomaly);
	fprintf(tmpFP, "Mean motion:     %.8lf rev/day\n",
		currentSat->s_meanmotion);
	fprintf(tmpFP, "Decay rate:      %le rev/day^2\n",
		currentSat->s_decayrate);
	fprintf(tmpFP, "Epoch rev:       %lu\n", currentSat->s_epochrev);

	fflush(tmpFP);

	/* reset the text source to get the new file contents read in */
	(void)changeTextSource(orbitText,tmpFile);

	sprintf(buf, "Information for Current Satellite");
	XtSetArg(argList[0], XtNlabel, buf);
	XtSetValues(botLabel, argList, ONE);
}

/*
 * Prompt the user for a satellite name.
 */
void
satellitesSelectSatellite(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char buf[BUFSIZ];

	bzero(buf, sizeof(buf));

	if (currentSat != NULL)
		sprintf(buf, "%s", currentSat->s_name);

	prompt("Enter satellite name:", buf, setSat);
}

/*
 * Prompt the user for a new satellite.
 */
void
satellitesAddSatellite(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char buf[BUFSIZ];

	bzero(buf, sizeof(buf));

	prompt("Enter satellite name and elements:", buf, addSat);
}

/*
 * List all satellites.
 */
void
satellitesListSatellites(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	int i, n;
	Arg argList[2];
	char buf[BUFSIZ];
	SatInfo *si, **sip;

	rewind(tmpFP);
	ftruncate(fileno(tmpFP), 0);

	/*
	 * Sort them into alphabetical order first.
	 */
	n = 0;
	for (si = satInfo; si != NULL; si = si->s_next)
		n++;

	sip = (SatInfo **) safeAlloc(n * sizeof(SatInfo *));

	n = 0;
	for (si = satInfo; si != NULL; si = si->s_next)
		sip[n++] = si;

	qsort(sip, n, sizeof(SatInfo *), satCompare);

	for (i=0; i < n; i++)
		fprintf(tmpFP, "%-15s%lu\n", sip[i]->s_name, sip[i]->s_satnum);

	fflush(tmpFP);

	/* reset the text source to get the new file contents read in */
	(void)changeTextSource(orbitText,tmpFile);

	sprintf(buf, "List of Satellites and Catalog Numbers");
	XtSetArg(argList[0], XtNlabel, buf);
	XtSetValues(botLabel, argList, ONE);
}

/*
 * Read in the orbital elements.
 */
void
readSatList()
{
	FILE *fp;
	char line[BUFSIZ];

	sprintf(line, "%s/%s", XSATLIBDIR, SATFILE);

	if ((fp = fopen(line, "r")) == NULL) {
		perror(line);
		exit(1);
	}

	while (fgets(line, MAX_TLE_LINE, fp) != NULL) {
		/* skip the initial ID line, if present */
		if (line[0] == '-')
			continue;

		/* add in the next two lines of the element set */
		if (fgets(&line[strlen(line)], MAX_TLE_LINE, fp) == NULL)
			break;
		if (fgets(&line[strlen(line)], MAX_TLE_LINE, fp) == NULL)
			break;

		/* fill in a new satellite node */
		(void)handle_twoline_format(&satInfo,line);
	}

	fclose(fp);
}

/*
 * Find the named satellite in the SatInfo structures.
 */
SatInfo *
findSatByName(sname)
char *sname;
{
	SatInfo *si;

	for (si = satInfo; si != NULL; si = si->s_next) {
		if (strcasecmp(si->s_name, sname) == 0)
			return(si);
	}

	return(NULL);
}

/*
 * Find the numbered satellite in the SatInfo structures.
 */
SatInfo *
findSatByNumber(snum)
int snum;
{
	SatInfo *si;

	for (si = satInfo; si != NULL; si = si->s_next) {
		if (si->s_satnum == snum)
			return(si);
	}

	return(NULL);
}

/*
 * Callback for setting the satellite.
 */
static void
setSat(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
	char *new;
	SatInfo *si;
	char buf[BUFSIZ];

	new = XawDialogGetValueString(XtParent(w));
	stripSpace(new);

	if ((si = findSatByName(new)) == NULL) {
		sprintf(buf, "Unknown satellite: %s", new);
		prompt(buf, NULL, promptOkay);
	}
	else {
		currentSat = si;
	}

	XtDestroyWidget(clientData);
}

/*
 * Callback for adding a satellite.
 */
static void
addSat(w, clientData, callData)
XtPointer clientData, callData;
Widget w;
{
  char *new;
  SatInfo *si;

  new = XawDialogGetValueString(XtParent(w));

/* skip possible leading white space */
  while (isspace(*new))
    new++;

/* check for AMSAT vs. two-line format */
  if (strncmp(new,"Satellite:",strlen("Satellite:")) == 0)
    si = handle_amsat_format(&satInfo,new);
  else
    si = handle_twoline_format(&satInfo,new);

/* if we got a valid satellite info structure, make it the current satellite */
  if (si == (SatInfo *)NULL) {
    /* invalid; complain and invalidate currentSat */
    XBell(XtDisplay(w),0);
    currentSat = (SatInfo *)NULL;
  }
  else
    currentSat = si;

  XtDestroyWidget(clientData);
}


/*
 * Comparison for qsort.
 */
static int
satCompare(a, b)
SatInfo **a, **b;
{
	return(strcmp((*a)->s_name, (*b)->s_name));
}
