/*
 * $Header: /home/orchestra5/davy/stuff/misc/xsat/RCS/orbit.c,v 1.1 92/04/10 14:08:12 davy Exp $
 *
 * Copyright 1992 by David A. Curry
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The
 * author makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * Orbit calculation routines.  The main routine here is based in part on the
 * main routine from N3EMO-ORBIT; the rest of the code is takend directly
 * from N3EMO-ORBIT with a few minor modifications for use with xsat.
 *
 * The N3EMO-ORBIT code is:
 *
 *	Copyright (c) 1986,1987,1988,1989,1990 Robert W. Berger N3EMO
 *	May be freely distributed, provided this notice remains intact.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * 1285 Electrical Engineering Building
 * West Lafayette, IN 47907
 * davy@ecn.purdue.edu
 *
 * $Log:	orbit.c,v $
 * Revision 1.1  92/04/10  14:08:12  davy
 * Initial revision
 *
 * Revision 1.2  94/05/24  11:22:33  trf
 * 
 */
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <stdio.h>
#include <math.h>
#include <time.h>

#include "xsat.h"

#ifdef CurrentTime
#undef CurrentTime
#endif

#ifndef PI
#define PI			3.1415926535
#endif

#ifdef PI2
#undef PI2
#endif

#ifdef E
#undef E
#endif

#define SSPELLIPSE 0		/* If non zero, use ellipsoidal earth model
				   when calculating longitude, latitude, and
				   height */

#define PI2			(PI*2)
#define ABS(x)			((x) < 0 ? (-(x)) : (x))
#define SQR(x)			((x) * (x))

#define MinutesPerDay		(24*60.0)
#define SecondsPerDay		(24*60.0*60.0)
#define HalfSecond		(0.5/SecondsPerDay)
#define DegreesPerRadian	(180/PI)
#define RadiansPerDegree	(PI/180)

#define Epsilon			(RadiansPerDegree/3600)     /* 1 arc second */
#define SunRadius		695000		
#define SunSemiMajorAxis	149598845.0  	/* Kilometers		*/
#define EarthRadius		6378.16		/* kilometers		*/
#define C			2.997925e5	/* kilometers/second	*/
#define TropicalYear		365.24199	/* mean solar days	*/
#define EarthEccentricity	0.016713
#define EarthFlat		(1/298.25)	/* Earth Flattening Coeff. */
#define SiderealSolar		1.0027379093
#define SidRate			(PI2*SiderealSolar/SecondsPerDay)	/* radians/second */
#define GM			398600		/* Kilometers^3/seconds^2 */

typedef double	mat3x3[3][3];

static double	EndTime;
static double	StepTime;
static double	StartTime;

static double	SiteLat;
static double	SiteLong;
static double	SiteAltitude;
static double	SiteMinElev;

static double	OldSiteLat;
static double	OldSiteElevation;

static double	SidDay;
static double	SunRAAN;
static double	MaxPhase;
static double	EpochDay;
static double	EpochRAAN;
static double	Inclination;
static double	SinPenumbra;
static double	CosPenumbra;
static double	Eccentricity;
static double	OrbitalDecay;
static double	SidReference;
static double	SunMeanMotion;
static double	SunMeanAnomaly;
static double	SunEpochTime;
static double	SunArgPerigee;
static double	SunInclination;
static double	SunEccentricity;
static double	epochMeanMotion;
static double	EpochArgPerigee;
static double	EpochMeanAnomaly;

static int	EpochYear;
static char    *ElementSet;
static long	EpochOrbitNum;

static void	GetDate();
static void	GetRange();
static void	dumpstats();
static void	PrintTime();
static void	PrintDate();
static void	SPrintDate();
static void	SPrintTime();
static void	GetBearings();
static void	GetPrecession();
static void	PrintDayOfWeek();
static void	GetSubSatPoint();
static void	GetSatPosition();
static void	GetSitPosition();
static void	GetTopocentric();
static void	InitOrbitRoutines();

static int	Eclipsed();
static int	GetDayOfWeek();

static long	GetDayNum();

static double	Kepler();

void
orbitPredict(lineFunc, textFunc)
void (*lineFunc)(), (*textFunc)();
{
    char *tzn;
    struct tm *tp;
    double Radius;
    double Height;
    double PrevTime;
    int PrevVisible;
    double RangeRate;
    char timestr[128];
    double Dayf;
    long Day, PrevDay;
    double CurrentTime;
    int nsteps, tsteps;
    double CurrentOrbit;
    double SemiMajorAxis;
    double ReferenceOrbit;
    double SiteVX, SiteVY;
    double SSPLat, SSPLong;
    double SiteMatrix[3][3];
    double SatX, SatY, SatZ;
    double LastLat, LastLong;
    double SatVX, SatVY, SatVZ;
    double SiteX, SiteY, SiteZ;
    long OrbitNum, PrevOrbitNum;
    double MeanAnomaly, TrueAnomaly;
    double Azimuth, Elevation, Range;
    double AverageMotion, CurrentMotion;
    double RAANPrecession, PerigeePrecession;

    EpochArgPerigee = currentSat->s_argperigee * RadiansPerDegree;
    EpochMeanAnomaly = currentSat->s_meananomaly * RadiansPerDegree;
    Inclination = currentSat->s_inclination * RadiansPerDegree;
    EpochRAAN = currentSat->s_raan * RadiansPerDegree;
    epochMeanMotion = currentSat->s_meanmotion;
    Eccentricity = currentSat->s_eccentricity;
    OrbitalDecay = currentSat->s_decayrate;
    EpochOrbitNum = currentSat->s_epochrev;
    ElementSet = currentSat->s_elementset;
    EpochDay = currentSat->s_epochday;

    EpochYear = EpochDay / 1000.0;
    EpochDay -= EpochYear * 1000.0;
    EpochDay += GetDayNum(EpochYear, 1, 0);

    SiteLat = currentCity->c_latitude * RadiansPerDegree;
    SiteLong = currentCity->c_longitude * RadiansPerDegree;

    /*
     * Convert from "normal" longitude (180W to 180E, W negative)
     * to "internal" longitude (0 to 360W).
     */
    if (SiteLong < 0.0)
	SiteLong = -SiteLong;
    else
        SiteLong = PI2 - SiteLong;

    if (currentCity->c_elevation >= 0)
	SiteAltitude = currentCity->c_elevation / 1000.0;
    else
	SiteAltitude = 0.0;

    SiteMinElev = minElevation * RadiansPerDegree;

    tp = gmtime(&startTime);
    StartTime = GetDayNum(tp->tm_year, tp->tm_mon+1, tp->tm_mday);
    StartTime += tp->tm_hour/24.0;

    tp = gmtime(&stopTime);
    EndTime = GetDayNum(tp->tm_year, tp->tm_mon+1, tp->tm_mday);
    EndTime += tp->tm_hour/24.0;

    tp = localtime(&startTime);

    if (appResources.utc)
	tzn = "UTC";
    else
#ifdef STRUCT_TM_HAS_ZONE
	tzn = tp->tm_zone;
#else
	if (tp->tm_isdst)
	    tzn = tzname[1];
	else
	    tzn = tzname[0];
#endif

    StepTime = (stepTime / 60.0) / MinutesPerDay;

    nsteps = 0;
    tsteps = timeLabelFreq / (stepTime / 60.0);

    InitOrbitRoutines((StartTime + EndTime) / 2);

    fprintf(tmpFP, "%s Element Set %s\n", currentSat->s_name, ElementSet);

    SemiMajorAxis = 331.25 * exp(2*log(MinutesPerDay/epochMeanMotion)/3);
    GetPrecession(SemiMajorAxis, Eccentricity, Inclination,
		  &RAANPrecession, &PerigeePrecession);

    ReferenceOrbit = EpochMeanAnomaly/PI2 + EpochOrbitNum;

    PrevTime = StartTime - 2 * StepTime;
    PrevOrbitNum = -10000;
    PrevDay = -10000;

    LastLat = 1000.0;
    LastLong = 1000.0;

    for (CurrentTime = StartTime; CurrentTime <= EndTime;
	 CurrentTime += StepTime) {
	AverageMotion = epochMeanMotion +
		(CurrentTime - EpochDay) * OrbitalDecay / 2;
	CurrentMotion = epochMeanMotion +
		(CurrentTime - EpochDay) * OrbitalDecay;

	SemiMajorAxis = 331.25 * exp(2*log(MinutesPerDay/CurrentMotion)/3);

	CurrentOrbit = ReferenceOrbit + (CurrentTime - EpochDay) *
		AverageMotion;

	OrbitNum = CurrentOrbit;

	MeanAnomaly = (CurrentOrbit - OrbitNum) * PI2;

	TrueAnomaly = Kepler(MeanAnomaly, Eccentricity);

	GetSatPosition(EpochDay, EpochRAAN, EpochArgPerigee, SemiMajorAxis,
		       Inclination, Eccentricity, RAANPrecession,
		       PerigeePrecession, CurrentTime, TrueAnomaly,
		       &SatX, &SatY, &SatZ, &Radius, &SatVX, &SatVY, &SatVZ);

	GetSitPosition(SiteLat, SiteLong, SiteAltitude, CurrentTime,
		       &SiteX, &SiteY, &SiteZ, &SiteVX, &SiteVY, SiteMatrix);

	GetBearings(SatX, SatY, SatZ, SiteX, SiteY, SiteZ, SiteMatrix,
		    &Azimuth, &Elevation);

	if (CurrentTime >= StartTime) {
	    GetRange(SiteX, SiteY, SiteZ, SiteVX, SiteVY, SatX, SatY, SatZ,
		     SatVX, SatVY, SatVZ, &Range, &RangeRate);

	    GetSubSatPoint(SatX, SatY, SatZ, CurrentTime, &SSPLat, &SSPLong,
			   &Height);

	    SSPLat *= DegreesPerRadian;
	    SSPLong *= DegreesPerRadian;

	    /*
	     * Convert from "internal" longitude (0 to 360W)
	     * to "normal" longitude (180E to 180W, W negative).
	     */
	    if (SSPLong <= 180.0)
		SSPLong = -SSPLong;
	    else
		SSPLong = 360.0 - SSPLong;

	    Dayf = CurrentTime + HalfSecond;
	    if (appResources.utc == 0)
#ifdef STRUCT_TM_HAS_GMTOFF
		Dayf += tp->tm_gmtoff / SecondsPerDay;
#else
		Dayf -= (timezone - (tp->tm_isdst ? (60 * 60) : 0))
		                                              / SecondsPerDay;
#endif

	    if (Elevation >= SiteMinElev) {
		Day = Dayf;
		if (((double) Day) > Dayf)
		    Day -= 1;

		if (OrbitNum == PrevOrbitNum && Day == PrevDay && !PrevVisible)
		    fprintf(tmpFP, "\n");

		if (OrbitNum != PrevOrbitNum || Day != PrevDay) {
		    fprintf(tmpFP, "\n");
		    PrintDayOfWeek(tmpFP, Day);
		    fprintf(tmpFP, " ");
		    PrintDate(tmpFP, Day);

		    fprintf(tmpFP, " ---- Orbit #%d ----\n", OrbitNum);

		    if (tzn != NULL)
		        fprintf(tmpFP, "Time (%s)  Azim (deg)", tzn);
		    else
		        fprintf(tmpFP, "   Time     Azim (deg)");
		    fprintf(tmpFP, "  Elev (deg)  Range (km)");
		    fprintf(tmpFP, "  Hgt (km)  Lat (deg)");
		    fprintf(tmpFP, "  Long (deg)\n");
	        }

		fprintf(tmpFP, " ");

		PrintTime(tmpFP, Dayf);

		fprintf(tmpFP, "%11.1lf %11.1lf", Azimuth * DegreesPerRadian,
			Elevation * DegreesPerRadian);

		fprintf(tmpFP, "%13.1lf %9.1lf %8.2lf %c %9.2lf %c", Range,
			Height, abs(SSPLat), SSPLat < 0.0 ? 'S' : 'N',
			abs(SSPLong), SSPLong < 0.0 ? 'W' : 'E');

		fprintf(tmpFP, "\n");

		PrevOrbitNum = OrbitNum;
	        PrevVisible = 1;
		PrevDay = Day;
	    }
	    else {
		PrevVisible = 0;
	    }

	    if (currentMap != NULL) {
		if ((SSPLat >= currentMap->m_miny) &&
		    (SSPLat <= currentMap->m_maxy) &&
		    (SSPLong >= currentMap->m_minx) &&
		    (SSPLong <= currentMap->m_maxx)) {
		    if ((LastLat > 360.0) || (LastLong > 360.0) ||
		      ((sgn(LastLong) != sgn(SSPLong)) && abs(SSPLong) > 90)) {
			LastLong = SSPLong;
			LastLat = SSPLat;
		    }
		    else {
			(*lineFunc)(LastLong, LastLat, SSPLong, SSPLat);

			if (((nsteps == 0) || (nsteps >= tsteps)) &&
			    (tsteps > 0)) {
			    SPrintTime(timestr, Dayf);
			    (*textFunc)(SSPLong, SSPLat, timestr);
			    nsteps = 0;
			}
		
			LastLong = SSPLong;
			LastLat = SSPLat;
		    }
		}
		else {
		    LastLong = 1000.0;
		    LastLat = 1000.0;
		}
	    }

	    nsteps++;
	}
	else {
	    PrevVisible = 0;
	}
    }

    fflush(tmpFP);
}

/*
 * All code below this point is the contents of "orbitr.c" from the
 * N3EMO Orbit Simulator, v3.7.  This code is:
 *
 * Copyright (c) 1986,1987,1988,1989,1990 Robert W. Berger N3EMO
 * May be freely distributed, provided this notice remains intact.
 *
 * Slight reformatting has been done for consistency.
 */

/*
 * Solve Kepler's equation
 *
 * Inputs:
 *      MeanAnomaly     Time Since last perigee, in radians.
 *                      PI2 = one complete orbit.
 *      Eccentricity    Eccentricity of orbit's ellipse.
 * Output:
 *      TrueAnomaly     Angle between perigee, geocenter, and
 *                      current position.
 */
 
static long calls = 0;
static long iters = 0;

static void
dumpstats()
{
printf("Average iterations = %lf\n",((double) iters)/calls);
}

static double
Kepler(MeanAnomaly,Eccentricity)
register double MeanAnomaly,Eccentricity;
{
register double E;              /* Eccentric Anomaly                    */
register double Error;
register double TrueAnomaly;
 
calls++;

    E = MeanAnomaly ;/*+ Eccentricity*sin(MeanAnomaly);   /* Initial guess */
    do
        {
        Error = (E - Eccentricity*sin(E) - MeanAnomaly)
                / (1 - Eccentricity*cos(E));
        E -= Error;
iters++;
        }
   while (ABS(Error) >= Epsilon);

    if (ABS(E-PI) < Epsilon)
        TrueAnomaly = PI;
      else
        TrueAnomaly = 2*atan(sqrt((1+Eccentricity)/(1-Eccentricity))
                                *tan(E/2));
    if (TrueAnomaly < 0)
        TrueAnomaly += PI2;
 
    return TrueAnomaly;
}
 
static void
GetSubSatPoint(SatX,SatY,SatZ,TTime,Latitude,Longitude,Height)
double SatX,SatY,SatZ,TTime;
double *Latitude,*Longitude,*Height;
{
    double r;
    long i;

    r = sqrt(SQR(SatX) + SQR(SatY) + SQR(SatZ));

    *Longitude = PI2*((TTime-SidDay)*SiderealSolar + SidReference)
		    - atan2(SatY,SatX);

    /* i = floor(Longitude/2*pi)        */
    i = *Longitude/PI2;
    if(i < 0)
        i--;
 
    *Longitude -= i*PI2;

    *Latitude = atan(SatZ/sqrt(SQR(SatX) + SQR(SatY)));

#if SSPELLIPSE
#else
    *Height = r - EarthRadius;
#endif
}
 
static void 
GetPrecession(SemiMajorAxis,Eccentricity,Inclination,
        RAANPrecession,PerigeePrecession)
double SemiMajorAxis,Eccentricity,Inclination;
double *RAANPrecession,*PerigeePrecession;
{
  *RAANPrecession = 9.95*pow(EarthRadius/SemiMajorAxis,3.5) * cos(Inclination)
                 / SQR(1-SQR(Eccentricity)) * RadiansPerDegree;
 
  *PerigeePrecession = 4.97*pow(EarthRadius/SemiMajorAxis,3.5)
         * (5*SQR(cos(Inclination))-1)
                 / SQR(1-SQR(Eccentricity)) * RadiansPerDegree;
}
 
/*
 * Compute the satellite postion and velocity in the RA based coordinate
 * system
 */
static void
GetSatPosition(EpochTime,EpochRAAN,EpochArgPerigee,SemiMajorAxis,
	Inclination,Eccentricity,RAANPrecession,PerigeePrecession,
	TTime,TrueAnomaly,X,Y,Z,Radius,VX,VY,VZ)
 
double EpochTime,EpochRAAN,EpochArgPerigee;
double SemiMajorAxis,Inclination,Eccentricity;
double RAANPrecession,PerigeePrecession,TTime,TrueAnomaly;
double *X,*Y,*Z,*Radius,*VX,*VY,*VZ;

{
    double RAAN,ArgPerigee;
 

    double Xw,Yw,VXw,VYw;	/* In orbital plane */
    double Tmp;
    double Px,Qx,Py,Qy,Pz,Qz;	/* Escobal transformation 31 */
    double CosArgPerigee,SinArgPerigee;
    double CosRAAN,SinRAAN,CoSinclination,SinInclination;

        *Radius = SemiMajorAxis*(1-SQR(Eccentricity))
                        / (1+Eccentricity*cos(TrueAnomaly));


    Xw = *Radius * cos(TrueAnomaly);
    Yw = *Radius * sin(TrueAnomaly);
    
    Tmp = sqrt(GM/(SemiMajorAxis*(1-SQR(Eccentricity))));

    VXw = -Tmp*sin(TrueAnomaly);
    VYw = Tmp*(cos(TrueAnomaly) + Eccentricity);

    ArgPerigee = EpochArgPerigee + (TTime-EpochTime)*PerigeePrecession;
    RAAN = EpochRAAN - (TTime-EpochTime)*RAANPrecession;

    CosRAAN = cos(RAAN); SinRAAN = sin(RAAN);
    CosArgPerigee = cos(ArgPerigee); SinArgPerigee = sin(ArgPerigee);
    CoSinclination = cos(Inclination); SinInclination = sin(Inclination);
    
    Px = CosArgPerigee*CosRAAN - SinArgPerigee*SinRAAN*CoSinclination;
    Py = CosArgPerigee*SinRAAN + SinArgPerigee*CosRAAN*CoSinclination;
    Pz = SinArgPerigee*SinInclination;
    Qx = -SinArgPerigee*CosRAAN - CosArgPerigee*SinRAAN*CoSinclination;
    Qy = -SinArgPerigee*SinRAAN + CosArgPerigee*CosRAAN*CoSinclination;
    Qz = CosArgPerigee*SinInclination;

    *X = Px*Xw + Qx*Yw;		/* Escobal, transformation #31 */
    *Y = Py*Xw + Qy*Yw;
    *Z = Pz*Xw + Qz*Yw;

    *VX = Px*VXw + Qx*VYw;
    *VY = Py*VXw + Qy*VYw;
    *VZ = Pz*VXw + Qz*VYw;
}

/*
 * Compute the site postion and velocity in the RA based coordinate
 * system. SiteMatrix is set to a matrix which is used by GetTopoCentric
 * to convert geocentric coordinates to topocentric (observer-centered)
 * coordinates.
 */
static void
GetSitPosition(SiteLat,SiteLong,SiteElevation,CurrentTime,
             SiteX,SiteY,SiteZ,SiteVX,SiteVY,SiteMatrix)

double SiteLat,SiteLong,SiteElevation,CurrentTime;
double *SiteX,*SiteY,*SiteZ,*SiteVX,*SiteVY;
mat3x3 SiteMatrix;

{
    static double G1,G2; /* Used to correct for flattening of the Earth */
    static double CosLat,SinLat;
    double Lat;
    double SiteRA;	/* Right Ascension of site			*/
    double CosRA,SinRA;

    if ((SiteLat != OldSiteLat) || (SiteElevation != OldSiteElevation))
	{
	OldSiteLat = SiteLat;
	OldSiteElevation = SiteElevation;
	Lat = atan(1/(1-SQR(EarthFlat))*tan(SiteLat));

	CosLat = cos(Lat);
	SinLat = sin(Lat);

	G1 = EarthRadius/(sqrt(1-(2*EarthFlat-SQR(EarthFlat))*SQR(SinLat)));
	G2 = G1*SQR(1-EarthFlat);
	G1 += SiteElevation;
	G2 += SiteElevation;
	}


    SiteRA = PI2*((CurrentTime-SidDay)*SiderealSolar + SidReference)
	         - SiteLong;
    CosRA = cos(SiteRA);
    SinRA = sin(SiteRA);
    

    *SiteX = G1*CosLat*CosRA;
    *SiteY = G1*CosLat*SinRA;
    *SiteZ = G2*SinLat;
    *SiteVX = -SidRate * *SiteY;
    *SiteVY = SidRate * *SiteX;

    SiteMatrix[0][0] = SinLat*CosRA;
    SiteMatrix[0][1] = SinLat*SinRA;
    SiteMatrix[0][2] = -CosLat;
    SiteMatrix[1][0] = -SinRA;
    SiteMatrix[1][1] = CosRA;
    SiteMatrix[1][2] = 0.0;
    SiteMatrix[2][0] = CosRA*CosLat;
    SiteMatrix[2][1] = SinRA*CosLat;
    SiteMatrix[2][2] = SinLat;
}

static void
GetRange(SiteX,SiteY,SiteZ,SiteVX,SiteVY,
	SatX,SatY,SatZ,SatVX,SatVY,SatVZ,Range,RangeRate)

double SiteX,SiteY,SiteZ,SiteVX,SiteVY;
double SatX,SatY,SatZ,SatVX,SatVY,SatVZ;
double *Range,*RangeRate;
{
    double DX,DY,DZ;

    DX = SatX - SiteX; DY = SatY - SiteY; DZ = SatZ - SiteZ;

    *Range = sqrt(SQR(DX)+SQR(DY)+SQR(DZ));    

    *RangeRate = ((SatVX-SiteVX)*DX + (SatVY-SiteVY)*DY + SatVZ*DZ)
			/ *Range;
}

/*
 * Convert from geocentric RA based coordinates to topocentric
 * (observer centered) coordinates
 */
static void
GetTopocentric(SatX,SatY,SatZ,SiteX,SiteY,SiteZ,SiteMatrix,X,Y,Z)
double SatX,SatY,SatZ,SiteX,SiteY,SiteZ;
double *X,*Y,*Z;
mat3x3 SiteMatrix;
{
    SatX -= SiteX;
    SatY -= SiteY;
    SatZ -= SiteZ;

    *X = SiteMatrix[0][0]*SatX + SiteMatrix[0][1]*SatY
	+ SiteMatrix[0][2]*SatZ; 
    *Y = SiteMatrix[1][0]*SatX + SiteMatrix[1][1]*SatY
	+ SiteMatrix[1][2]*SatZ; 
    *Z = SiteMatrix[2][0]*SatX + SiteMatrix[2][1]*SatY
	+ SiteMatrix[2][2]*SatZ; 
}

static void
GetBearings(SatX,SatY,SatZ,SiteX,SiteY,SiteZ,SiteMatrix,Azimuth,Elevation)
double SatX,SatY,SatZ,SiteX,SiteY,SiteZ;
mat3x3 SiteMatrix;
double *Azimuth,*Elevation;
{
    double x,y,z;

    GetTopocentric(SatX,SatY,SatZ,SiteX,SiteY,SiteZ,SiteMatrix,&x,&y,&z);

    *Elevation = atan(z/sqrt(SQR(x) + SQR(y)));

    *Azimuth = PI - atan2(y,x);

    if (*Azimuth < 0)
	*Azimuth += PI;
}

static int
Eclipsed(SatX,SatY,SatZ,SatRadius,CurrentTime)
double SatX,SatY,SatZ,SatRadius,CurrentTime;
{
    double MeanAnomaly,TrueAnomaly;
    double SunX,SunY,SunZ,SunRad;
    double vx,vy,vz;
    double CosTheta;

    MeanAnomaly = SunMeanAnomaly+ (CurrentTime-SunEpochTime)*SunMeanMotion*PI2;
    TrueAnomaly = Kepler(MeanAnomaly,SunEccentricity);

    GetSatPosition(SunEpochTime,SunRAAN,SunArgPerigee,SunSemiMajorAxis,
		SunInclination,SunEccentricity,0.0,0.0,CurrentTime,
		TrueAnomaly,&SunX,&SunY,&SunZ,&SunRad,&vx,&vy,&vz);

    CosTheta = (SunX*SatX + SunY*SatY + SunZ*SatZ)/(SunRad*SatRadius)
		 *CosPenumbra + (SatRadius/EarthRadius)*SinPenumbra;

    if (CosTheta < 0)
        if (CosTheta < -sqrt(SQR(SatRadius)-SQR(EarthRadius))/SatRadius
	    		*CosPenumbra + (SatRadius/EarthRadius)*SinPenumbra)
	  
	    return 1;
    return 0;
}

/*
 * Initialize the Sun's keplerian elements for a given epoch.
 * Formulas are from "Explanatory Supplement to the Astronomical Ephemeris".
 * Also init the sidereal reference
 */
static void
InitOrbitRoutines(EpochDay)
double EpochDay;
{
    double T,T2,T3,Omega;
    int n;
    double SunTrueAnomaly,SunDistance;

    OldSiteLat = -100000;
    OldSiteElevation = -100000;

    T = (floor(EpochDay)-0.5)/36525;
    T2 = T*T;
    T3 = T2*T;

    SidDay = floor(EpochDay);

    SidReference = (6.6460656 + 2400.051262*T + 0.00002581*T2)/24;
    SidReference -= floor(SidReference);

    /* Omega is used to correct for the nutation and the abberation */
    Omega = (259.18 - 1934.142*T) * RadiansPerDegree;
    n = Omega / PI2;
    Omega -= n*PI2;

    SunEpochTime = EpochDay;
    SunRAAN = 0;

    SunInclination = (23.452294 - 0.0130125*T - 0.00000164*T2
		    + 0.000000503*T3 +0.00256*cos(Omega)) * RadiansPerDegree;
    SunEccentricity = (0.01675104 - 0.00004180*T - 0.000000126*T2);
    SunArgPerigee = (281.220833 + 1.719175*T + 0.0004527*T2
			+ 0.0000033*T3) * RadiansPerDegree;
    SunMeanAnomaly = (358.475845 + 35999.04975*T - 0.00015*T2
			- 0.00000333333*T3) * RadiansPerDegree;
    n = SunMeanAnomaly / PI2;
    SunMeanAnomaly -= n*PI2;

    SunMeanMotion = 1/(365.24219879 - 0.00000614*T);

    SunTrueAnomaly = Kepler(SunMeanAnomaly,SunEccentricity);
    SunDistance = SunSemiMajorAxis*(1-SQR(SunEccentricity))
			/ (1+SunEccentricity*cos(SunTrueAnomaly));

    SinPenumbra = (SunRadius-EarthRadius)/SunDistance;
    CosPenumbra = sqrt(1-SQR(SinPenumbra));
}

static void 
SPrintTime(Str,TTime)
char *Str;
double TTime;
{
    int day,hours,minutes,seconds;

    day = TTime;
    TTime -= day;
    if (TTime < 0)
        TTime += 1.0;   /* Correct for truncation problems with negatives */

    hours = TTime*24;
    TTime -=  hours/24.0;
 
    minutes = TTime*MinutesPerDay;
    TTime -= minutes/MinutesPerDay;
 
    seconds = TTime*SecondsPerDay;
    seconds -= seconds/SecondsPerDay;
 
    sprintf(Str,"%02d:%02d:%02d",hours,minutes,seconds);
}
 
static void
PrintTime(OutFile,TTime)
FILE *OutFile;
double TTime;
{
    char str[100];

    SPrintTime(str,TTime);
    fprintf(OutFile,"%s",str);
}

/*
 * Get the Day Number for a given date. January 1 of the reference year
 * is day 0. Note that the Day Number may be negative, if the sidereal
 * reference is in the future.
 */
 
/*
 * Date calculation routines
 * Robert Berger @ Carnegie Mellon
 *
 * January 1, 1900 is day 0
 * valid from 1900 through 2099
 */
static char *MonthNames[] = { "Jan","Feb","Mar","Apr","May","Jun","Jul",
			      "Aug","Sep","Oct","Nov","Dec" };
 
static int MonthDays[] = {0,31,59,90,120,151,181,212,243,273,304,334};
		
 
static char *DayNames[] = { "Sunday","Monday","Tuesday","Wednesday","Thursday",
			    "Friday","Saturday"};

static long
GetDayNum(Year,Month,Day)
{
    long Result;
    
    /* Heuristic to allow 4 or 2 digit year specifications */
    if (Year < 50)
	Year += 2000;
      else if (Year < 100)
	Year += 1900;
	
    Result = ((((long) Year-1901)*1461)>>2) + MonthDays[Month-1] + Day + 365;
    if (Year%4 == 0 && Month > 2)
        Result++;

    return Result;
}

static void
GetDate(DayNum,Year,Month,Day)
long DayNum;
int *Year,*Month,*Day;    
{
    int M,L;
    long Y;
	
    Y = 4*DayNum;
    Y /= 1461;

    DayNum =  DayNum -365 - (((Y-1)*1461)>>2);
    
    L = 0;
    if (Y%4 == 0 && DayNum > MonthDays[2])
        L = 1;
	
    M = 1;
	     
    while (DayNum > MonthDays[M]+L)
	M++;
	
    DayNum -= (MonthDays[M-1]);
    if (M > 2)
        DayNum -= L;
   
    *Year = Y+1900;
    *Month = M;
    *Day = DayNum;
}    

/* Sunday = 0 */
static int
GetDayOfWeek(DayNum)
long DayNum;
{
    return DayNum % 7;
}    

static void
SPrintDate(Str,DayNum)
char *Str;
long DayNum;
{
    int Month,Day,Year;
    
    GetDate(DayNum,&Year,&Month,&Day);
    sprintf(Str,"%d %s %d",Day,
                MonthNames[Month-1],Year);
} 

static void
SPrintDayOfWeek(Str,DayNum)
char *Str;
long DayNum;
{
    strcpy(Str,DayNames[DayNum%7]);
}

static void
PrintDate(OutFile,DayNum)
FILE *OutFile;
long DayNum;
{
    char str[100];

    SPrintDate(str,DayNum);
    fprintf(OutFile,"%s",str);
}

static void
PrintDayOfWeek(OutFile,DayNum)
FILE *OutFile;
long DayNum;
{
    fprintf(OutFile,"%s",DayNames[DayNum%7]);
}    
