/*
 * Routines for dealing with AMSAT-format satellite elements
 *
 * Terry R. Friedrichsen
 * Sunquest Information Systems, Inc.
 * terry@venus.sunquest.com
 *
 * Revision 1.0  95/07/07  05:23:36  trf
 *
 */

#include <X11/Intrinsic.h>

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "xsat.h"

#define until(x)   while (! (x))

static void handle_amsat_line();

static void save_amsat_string();
static void save_amsat_double();
static void save_amsat_u_long();

static SatInfo ssi;

/* set up a table for conversion of AMSAT-format elements */
struct amsat_key {
  char *key;
  char *delims;
  Boolean present_flag;
  void (*fcn)();
  void *value_addr;
};

static struct amsat_key amsat_keys[] = {
  { "Satellite",      "\n",    False, save_amsat_string, &ssi.s_name },
  { "Catalog number", " \t\n", False, save_amsat_u_long, &ssi.s_satnum },
  { "Epoch time",     " \t\n", False, save_amsat_double, &ssi.s_epochday },
  { "Element set",    "\n",    False, save_amsat_string, &ssi.s_elementset },
  { "Inclination",    " \t\n", False, save_amsat_double, &ssi.s_inclination },
  { "RA of node",     " \t\n", False, save_amsat_double, &ssi.s_raan },
  { "Eccentricity",   " \t\n", False, save_amsat_double, &ssi.s_eccentricity },
  { "Arg of perigee", " \t\n", False, save_amsat_double, &ssi.s_argperigee },
  { "Mean anomaly",   " \t\n", False, save_amsat_double, &ssi.s_meananomaly },
  { "Mean motion",    " \t\n", False, save_amsat_double, &ssi.s_meanmotion },
  { "Decay rate",     " \t\n", False, save_amsat_double, &ssi.s_decayrate },
  { "Epoch rev",      " \t\n", False, save_amsat_u_long, &ssi.s_epochrev },
  { (char *)NULL,(char *)NULL, False, (void (*)())NULL,  (void *)NULL }
};



/* handle satellite elements in two-line format */

SatInfo *handle_twoline_format(si_head,buff)

SatInfo **si_head;
char *buff;

{
  char *tok;
  
  char *line1, *line2;
  SatInfo *si;

/* save the satellite name */
  if ((tok = strtok(buff,"\n")) == (char *)NULL) {
    prompt("No data in two-line element set.", NULL, promptOkay);
    return (SatInfo *)NULL;
  }
  stripSpace(tok);
  ssi.s_name = saveString(tok);

/* get the two data lines */
  line1 = strtok((char *)NULL,"\n");
  if (line1 == (char *)NULL) {
    prompt("Missing data in two-line element set.", NULL, promptOkay);
    (void)free(ssi.s_name);
    return (SatInfo *)NULL;
  }
  line2 = strtok((char *)NULL,"\n");
  if (line2 == (char *)NULL) {
    prompt("Missing data in two-line element set.", NULL, promptOkay);
    (void)free(ssi.s_name);
    return (SatInfo *)NULL;
  }

/* crack the two-line data */
  ssi.s_elementset = (char *)NULL;
  if ((handle_line_1(&ssi,line1)) && (handle_line_2(&ssi,line2)))
    ;
  else {
    (void)free(ssi.s_name);
    return (SatInfo *)NULL;
  }

/* do we have old elements for this satellite? */
  if ((si = findSatByNumber(ssi.s_satnum)) == (SatInfo *)NULL) {
    /* no; get a new satellite node */
    si = (SatInfo *) safeAlloc(sizeof(SatInfo));
    si->s_next = *si_head;
    *si_head = si;
  }
  else {
    /* it does; free any old name strings in the satellite node */
    if (si->s_name != (char *)NULL)
      (void)free(si->s_name);
    if (si->s_elementset != (char *)NULL)
      (void)free(si->s_elementset);
  }

/* save the elements in the new node, preserving the link */
  ssi.s_next = si->s_next;
  *si = ssi;

  return si;
}



/* handle satellite element set line 1 */

Boolean handle_line_1(si,line1)

SatInfo *si;
char *line1;

{
  unsigned long elementset;
  char elementbuff[5];

/* verify that this has some possibility of being the correct line */
  if (line1[0] != '1') {
    fprintf(stderr, "Line 1 missing for satellite %s.\n", si->s_name);
    return False;
  }

/* trim trailing spaces and verify the checksum */
  stripSpace(line1);
  if (! verify_tle_checksum(line1)) {
    fprintf(stderr, "Line 1 bad checksum for satellite %s.\n", si->s_name);
    return False;
  }

/* scan in the values */
  sscanf(line1, "%*2c%5lu%*10c%2lf%12lf%10lf%*21c%5lu",
		       &si->s_satnum, &si->s_epochyear, &si->s_epochday,
		       &si->s_decayrate, &elementset);

/* apply necessary scaling to certain fields */
  si->s_epochday += si->s_epochyear * 1000;
  elementset /= 10;

/* convert the element set from numeric to string for AMSAT compatibility */
  sprintf(elementbuff,"%lu",elementset);
  si->s_elementset = saveString(elementbuff);

  return True;
}



/* handle satellite element set line 2 */

Boolean handle_line_2(si,line2)

SatInfo *si;
char *line2;

{
/* verify that this has some possibility of being the correct line */
  if (line2[0] != '2') {
    fprintf(stderr, "Line 2 missing for satellite %s.\n", si->s_name);
    return False;
  }
  
/* trim trailing spaces and verify the checksum */
  stripSpace(line2);
  if (! verify_tle_checksum(line2)) {
    fprintf(stderr, "Line 2 bad checksum for satellite %s.\n", si->s_name);
    return False;
  }

/* scan in the values */
  sscanf(line2, "%*8c%8lf%8lf%7lf%8lf%8lf%11lf%6lu",
		       &si->s_inclination, &si->s_raan, &si->s_eccentricity,
		       &si->s_argperigee, &si->s_meananomaly,
		       &si->s_meanmotion, &si->s_epochrev);

/* apply necessary scaling to certain fields */
  si->s_eccentricity *= 1E-7;
  si->s_epochrev /= 10;

  return True;
}



/* handle satellite elements in AMSAT format */

SatInfo *handle_amsat_format(si_head,buff)

SatInfo **si_head;
char *buff;

{
  int amsat_key_idx;
  char *line;
  char *nextline;
  char *ptr;
  SatInfo *si;

/* clear the string pointers from the local structure */
  ssi.s_name = ssi.s_elementset = (char *)NULL;

/* flag that we've seen none of the keywords yet */
  amsat_key_idx = 0;
  while (amsat_keys[amsat_key_idx].key != (char *)NULL)
    amsat_keys[amsat_key_idx++].present_flag = False;

/* split the input into lines */
  line = buff;
  do {
    while (isspace(*line))
      line++;
    nextline = strpbrk(line,"\n");
    if (nextline != (char *)NULL)
      nextline++;
    handle_amsat_line(line);
    line = nextline;
  } until (line == (char *)NULL);

/* verify that we've seen all of the keywords */
  amsat_key_idx = 0;
  while (amsat_keys[amsat_key_idx].key != (char *)NULL)
    if (! amsat_keys[amsat_key_idx++].present_flag) {
      /* we didn't get all the data; don't bother with this element set */
      if (ssi.s_name != (char *)NULL)
	(void)free(ssi.s_name);
      if (ssi.s_elementset != (char *)NULL)
	(void)free(ssi.s_elementset);

      prompt("Missing data in AMSAT element set.", NULL, promptOkay);
      return (SatInfo *)NULL;
    }      

/* extract the epoch year from the day */
  ssi.s_epochyear = (int)ssi.s_epochday / 1000;

/* see if this satellite already exists */
  if ((si = findSatByNumber(ssi.s_satnum)) == (SatInfo *)NULL) {
    /* it doesn't; get a new node and link it into the list */
    si = (SatInfo *) safeAlloc(sizeof(SatInfo));
    si->s_next = *si_head;
    *si_head = si;
  }
  else {
    /* it does; free any old name strings in the satellite node */
    if (si->s_name != (char *)NULL)
      (void)free(si->s_name);
    if (si->s_elementset != (char *)NULL)
      (void)free(si->s_elementset);
  }

/* save the elements in the new node, preserving the link */
  ssi.s_next = si->s_next;
  *si = ssi;

  return si;
}



/* handle a line of AMSAT format data */

static void handle_amsat_line(line)

char *line;

{
  char *token;
  int amsat_key_idx;

/* get the keyword from the line */
  token = strtok(line,":");
  if (token == (char *)NULL)
    return;

/* find the keyword in the list */
  amsat_key_idx = 0;
  while (amsat_keys[amsat_key_idx].key != (char *)NULL)
    if (strcmp(amsat_keys[amsat_key_idx].key,token) == 0)
      break;
    else
      amsat_key_idx++;

/* if we found a match, handle the value and flag that we've seen it */
  if (amsat_keys[amsat_key_idx].key != (char *)NULL) {
    token = strtok((char *)NULL,amsat_keys[amsat_key_idx].delims);
    (*amsat_keys[amsat_key_idx].fcn)(token,
				        amsat_keys[amsat_key_idx].value_addr);
    amsat_keys[amsat_key_idx].present_flag = True;
  }

  return;
}



/* save a string from AMSAT format */

static void save_amsat_string(src,dst)

char *src;
void *dst;

{
  char *ptr;
  char **dest = (char **)dst;

/* skip leading spaces */
  ptr = src;
  while (isspace(*ptr))
    ptr++;

/* trim trailing spaces */
  stripSpace(ptr);

/* allocate space for the string and save it */
  *dest = saveString(ptr);

  return;
}



/* save an unsigned long from AMSAT format */

static void save_amsat_u_long(src,dst)

char *src;
void *dst;

{
  unsigned long *dest = (unsigned long *)dst;

/* scan in the unsigned long */
  sscanf(src,"%lu",dest);

  return;
}



/* save a double from AMSAT format */

static void save_amsat_double(src,dst)

char *src;
void *dst;

{
  double *dest = (double *)dst;

/* scan in the double */
  sscanf(src,"%lf",dest);

  return;
}
