/* lock.c: The opielock() library function.

Portions of this software are Copyright 1996 by Craig Metz, All Rights
Reserved. The Inner Net Copyright Notice and License Agreement applies to
these portions of the software.

Portions of this software are Copyright 1995 by Randall Atkinson and Dan
McDonald, All Rights Reserved. All Rights under this copyright are assigned
to the U.S. Naval Research Laboratory (NRL). The NRL Copyright Notice and
License Agreement applies to this software.

        History:

	Modified by cmetz for OPIE 2.2. Use FUNCTION declaration et al.
             Use "principal" instead of "name" to make it clearer.
             Ifdef around some headers, be more careful about allowed
             error return values. Check open() return value properly.
             Avoid NULL.
        Created at NRL for OPIE 2.2 from opiesubr2.c
*/
#include "opie_cfg.h"
#if HAVE_STRING_H
#include <string.h>
#endif /* HAVE_STRING_H */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <fcntl.h>
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#include "opie.h"

#if USER_LOCKING
char *opielockfilename = (char *)0;

/* atexit() handler for opielock() */
static VOIDRET opieunlockaeh FUNCTION_NOARGS
{
  if (opielockfilename)
    opieunlock();
}
#endif /* USER_LOCKING */

/* 
   Serialize (we hope) authentication of user to prevent race conditions.
   Creates a lock file with a name of OPIE_LOCK_PREFIX with the user name
   appended. This file contains the pid of the lock's owner and a time()
   stamp. We use the former to check for dead owners and the latter to
   provide an upper bound on the lock duration. If there are any problems,
   we assume the lock is bogus.

   The value of this locking and its security implications are still not
   completely clear and require further study.

   One could conceivably hack this facility to provide locking of user
   accounts after several authentication failures.
 
   Return -1 on low-level error, 0 if ok, 1 on locking failure.
*/
int opielock FUNCTION((principal), char *principal)
{
#ifdef USER_LOCKING
  int fh, waits = 0, rval = -1, pid, t, i;
  char buffer[128], buffer2[128], *c, *c2;

  if (!(opielockfilename = (char *)malloc(sizeof(OPIE_LOCK_PREFIX) + strlen(principal))))
    return -1;

  strcpy(opielockfilename, OPIE_LOCK_PREFIX);
  strcat(opielockfilename, principal);

  fh = 0;
  while (!fh)
    if ((fh = open(opielockfilename, O_WRONLY | O_CREAT | O_EXCL, 0600)) < 0) {
      if ((fh = open(opielockfilename, O_RDWR, 0600)) < 0)
        goto lockret;
      if ((i = read(fh, buffer, sizeof(buffer))) <= 0)
        goto lockret;

      buffer[sizeof(buffer) - 1] = 0;
      buffer[i - 1] = 0;

      if (!(c = strchr(buffer, '\n')))
        break;

      *(c++) = 0;

      if (!(c2 = strchr(c, '\n')))
        break;

      *(c2++) = 0;

      if (!(pid = atoi(buffer)))
        break;

      if (!(t = atoi(c)))
        break;

      if ((time(0) + OPIE_LOCK_TIMEOUT) < t)
        break;

      if (kill(pid, 0))
        break;

      close(fh);
      fh = 0;
      sleep(1);
      if (waits++ > 3) {
        rval = 1; 
        goto lockret;
      };
    };

  sprintf(buffer, "%d\n%d\n", getpid(), time(0));
  i = strlen(buffer) + 1;
  if (lseek(fh, 0, SEEK_SET)) { 
    close(fh);
    unlink(opielockfilename);
    fh = 0;
    goto lockret;
  };
  if (write(fh, buffer, i) != i) {
    close(fh);
    unlink(opielockfilename);
    fh = 0;
    goto lockret;
  };
  close(fh);
  if ((fh = open(opielockfilename, O_RDWR, 0600)) < 0) {
    unlink(opielockfilename);
    goto lockret;
  };
  if (read(fh, buffer2, i) != i) {
    close(fh);
    unlink(opielockfilename);
    fh = 0;
    goto lockret;
  };
  close(fh);
  if (memcmp(buffer, buffer2, i)) {
    unlink(opielockfilename);
    goto lockret;
  };
    
  rval = 0;
  atexit(opieunlockaeh);

lockret:
  if (fh)
    close(fh);
  return rval;
#else /* USER_LOCKING */
  return 0;
#endif /* USER_LOCKING */
}
