--
-- This file is part of TALER
-- Copyright (C) 2014--2023 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_prewire(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'prewire';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(prewire_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY'
      ',wire_method TEXT NOT NULL'
      ',finished BOOLEAN NOT NULL DEFAULT FALSE'
      ',failed BOOLEAN NOT NULL DEFAULT FALSE'
      ',buf BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (prewire_uuid)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'pre-commit data for wire transfers we are about to execute'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE if the bank responded with a non-transient failure to our transfer request'
    ,'failed'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE once bank confirmed receiving the wire transfer request'
    ,'finished'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'serialized data to send to the bank to execute the wire transfer'
    ,'buf'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_prewire(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'prewire';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_finished_index '
    'ON ' || table_name || ' '
    '(finished)'
    ' WHERE finished;'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_finished_index '
    'IS ' || quote_literal('for do_gc') || ';'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_failed_finished_index '
    'ON ' || table_name || ' '
    '(prewire_uuid)'
    ' WHERE finished=FALSE'
    '   AND failed=FALSE;'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_failed_finished_index '
    'IS ' || quote_literal('for wire_prepare_data_get') || ';'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('prewire'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('prewire'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
