/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/* comm.c
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/12/08  B. Dorsey      Module created by breakup of nautilus.c
 */

#include <stdio.h>
#include <stdlib.h>		/* plm */
#include <string.h>		/* plm */
#include <ctype.h>		/* plm */
#include <malloc.h>		/* plm */
#if defined(unix)
#include <unistd.h>
#endif

#include "nautilus.h"

#define XCHG_TIMEOUT    15
#define RP_TIMEOUT      3

/* external variables */
extern struct param_t params;
extern struct coder_t coders[NCODERS];
extern struct negotiate_t negotiate;



/*
 * Dial phone number specified by argument and wait for connection.
 * Return SUCCESS or FAIL.
 *
 * This function will behave unpredictably if the modem initialization
 * string modem_init, or the phone number in phone, overflow the fixed
 * size internal string buf.
 */

int
Connect(char *phone, char *modem_init)
{
    char            buf[512];

    WritePort("AT\r", 3);
    if (WaitFor("OK", 2, 0) < 0) {
	WritePort("AT\r", 3);
	if (WaitFor("OK", 2, 0) < 0) {
	    return FAIL;
	}
    }
    PAUSE(100);				/* wait 100ms */
    if (strlen(modem_init) != 0) {	/* config string defined? */
        strcpy(buf, "AT");
        strcat(buf, modem_init);
        strcat(buf, "\r");
        WritePort(buf, strlen(buf));
        if (WaitFor("OK", 2, 0) < 0) {
            return FAIL;
        }
    }
    PAUSE(100);			/* wait 100ms */
    strcpy(buf, "AT");
    if (strlen(phone) != 0) {
        sprintf(buf + strlen(buf), "DT%s\r", phone);
    } else {
        strcat(buf, "D\r");
    }
    WritePort(buf, strlen(buf));

    return WaitFor("CONNECT", 60, 1);
}



/*
 * If 'flag' is true, put the modem in answer mode (ATS0=1) and wait
 * indefinitely for an incoming call.  Otherwise, take the modem
 * offhook immediately in answer mode.  Return SUCCESS or FAIL.
 */

int
AnswerMode(int flag)
{
    WritePort("AT\r", 3);
    if (WaitFor("OK", 2, 0) < 0) {
	WritePort("AT\r", 3);
	if (WaitFor("OK", 2, 0) < 0) {
	    return FAIL;
	}
    }
    PAUSE(100);			/* wait 100ms */
    if (flag) {
	WritePort("ATS0=1\r", 7);
	if (WaitFor("OK", 2, 0) < 0) {
	    return FAIL;
	}
	return WaitFor("CONNECT", 0, 1);
    }
    else {
	WritePort("ATA\r", 4);
	return WaitFor("CONNECT", 45, 1);
    }
}



/*
 * Monitor serial rx stream for 'string'.  Return SUCCESS if found.  If not
 * found after 'wait' seconds, return FAIL for failure.  If 'show' is
 * nonzero, seconds remaining and each string received are displayed on
 * console.  If 'wait' is zero, wait forever for desired string.
 */
int
WaitFor(char *string, int wait, int show)
{
    int		c, i=0, timeout, old=-1;
    unsigned	speed;
    char	ibuf[80], tbuf[16];
    UINT32	start = Clock();

    if (show) {
	sprintf(ibuf, "Waiting for \"%s\"", string);
	error(MSG_WARNING, ibuf);
    }

    for (;;) {
	/*
	 * Construct a line of input from the serial port.  Control
	 * characters are ignored with the exception of the linefeed
	 * which marks the end of the line.
	 */
	if (IncomingPort()) {
	    c = ReadPort(1);
	    if (i && (c == '\r')) {	/* end of line detected */
		ibuf[i] = '\0';
		if (show) {	/* print the buffer if show is true */
		    debug_puts(ibuf);
		    old = -1;
		}
		/*
		 * Check line against string to be matched and return
		 * a status or continue searching.
		 */
		if (strstr(ibuf, string)) {
		    /*
		     * User's string to be matched has been found.  If
		     * it's a "CONNECT" string, see if there is a connect
		     * speed (hopefully the DCE speed) present and store
		     * any such value in the params structure.
		     */
		    if (!strcmp(string, "CONNECT") && strlen(ibuf) > 7) {
			speed = atoi(&ibuf[8]);
			if (speed >= 300)
			    params.modem.speed = speed;
		    }
		    return SUCCESS;
		}
		/*
		 * If any of the following string(s) are detected,
		 * throw them out and continue waiting for the user's
		 * string.
		 */
		else if (strstr(ibuf, "AT") ||
		         strstr(ibuf, "RING"))
		    i = 0;
		/*
		 * We've reached the end of a line without finding the
		 * search string.  Return an error status.
		 */
		else
		    break;
	    }
	    else if (!iscntrl(c)) {
		ibuf[i++] = c;
		if (i == 80)	/* overflow detection */
		    break;
	    }
	}
	/*
	 * Check for timeout
	 */
	if (wait) {
	    timeout = Clock() - start;
	    if (timeout != old) {
		if (show) {
		    sprintf(tbuf, "\r%3d: ", wait-timeout);
		    debug_puts_nr(tbuf);
		}
		if (timeout >= wait)
		    break;
		old = timeout;
	    }
	}
	/*
	 * keypress will abort if waiting "forever".
	 */
	else
	    if (ptt())
	    	break;
    }

    if (show)
        debug_putc('\n');

    return FAIL;
}



int
XChange(enum modes mode)
{
    int                i, done, speed, i_am_newer, compatible;
    long               start, start2;
    UINT8              *p1, *p2;
    struct negotiate_t *orig, *ans, *older_version, *newer_version;
    struct keyexch_t   keyexch1, keyexch2;
    struct packet_t    packet;

    /*
     * Immediately after the modems connect, there is occasionally a
     * bit of garbage on the line.  If there are any characters
     * present before we expect them to be, discard them.
     */
    while (IncomingPort()) {
	ReadPort(1);
    }
    sleep(1);

    /* Set modem speed in negotiation parameters (if known) */
    if (params.modem.speed > 0) {
        negotiate.modem_speed[0] = params.modem.speed % 256;	/* LSB */
        negotiate.modem_speed[1] = params.modem.speed / 256;	/* MSB */
    }

    /* Exchange negotiation parameters */
    switch (mode) {
    case ORIGINATE:
	/*
	 * The call originator sends its negotiation parameters once
	 * every two seconds until either a packet is received, or
	 * until XCHG_TIMEOUT seconds have passed.
	 */
	done = 0;
	start = Clock();
	do {
	    SendPkt(DATA, (UINT8 *) &negotiate, sizeof(struct negotiate_t));
	    start2 = Clock();
	    do {
		if (IncomingPort()) {
		    if (ReadPkt(&packet) == SUCCESS) {
			done = 1;
			break;
		    }
		}
	    } while (Clock() - start2 < 2);
	    if (Clock() - start > XCHG_TIMEOUT)
		return FAIL;
	} while (done == 0);
	orig = &negotiate;
	ans = (struct negotiate_t *) packet.data;
	break;

    case ANSWER:
    case AUTO_ANSWER:
	/*
	 * The call answerer repeatedly attempts to receive packets
	 * until XCHG_TIMEOUT seconds have passed.  If it receives
	 * one, it sends a reply packet containing its negotiation
	 * parameters.
	 */
	start = Clock();
	for (;;) {
	    if (IncomingPort()) {
		if (ReadPkt(&packet) == SUCCESS)
		    break;
	    }
	    if (Clock() - start > XCHG_TIMEOUT)
		return FAIL;
	}
	SendPkt(DATA, (UINT8 *) &negotiate, sizeof(struct negotiate_t));
	orig = (struct negotiate_t *) packet.data;
	ans = &negotiate;
	break;
    }

    /* Verify that packet is the right type & size */
    if ((packet.type != DATA) ||
	(packet.length != sizeof(struct negotiate_t))) {
	/*
	 * If it is not what was expected, the most likely explanation
	 * is that the other endpoint is using a version of Nautilus
	 * older than 0.9.2, which had a negotiation structure size of
	 * 10 bytes.  Give this explanation to the user of the newer
	 * version.
	 */
	if (packet.length == 10) {
	    CantNotifyBetaVersion();
	}
        return FAIL;
    }

    /* Verify version compatibility */
    if ((orig->major == ans->major) && (orig->minor == ans->minor)) {
        /*
	 * The major and minor version numbers match exactly.  Version
	 * numbers 1.0 and later will be assigned such that if this
	 * condition is true, then the two versions are compatible.
	 * For example, it will never be the case that a version 1.0.1
	 * and 1.0.2 will be incompatible.  Version 1.0.2 might be
	 * compatible with 1.1.0, or it might not.
	 * 
	 * Do nothing here, and continue with the rest of the program.
	 */
    }
    else {
	/*
	 * There is a version mismatch.  The older version has no way
	 * to know whether it is compatible with the newer version.
	 * The newer of the two participants must tell the older one
	 * whether they are compatible.  If not, it also sends clear
	 * text to the older version explaining the problem and how to
	 * upgrade.  The older version displays this information on
	 * the screen and saves it to a file UPGRADE_FILE.
	 */
	if ((orig->major > ans->major) ||
	    ((orig->major == ans->major) && (orig->minor > ans->minor))) {
	    /*
	     * Originator is newer.
	     */
	    if (mode == ORIGINATE) {
		i_am_newer = TRUE;
	    }
	    else {
		i_am_newer = FALSE;
	    }
	    older_version = ans;
	    newer_version = orig;
	}
	else {
	    /*
	     * Answerer is newer.
	     */
	    if (mode == ORIGINATE) {
		i_am_newer = FALSE;
	    }
	    else {
		i_am_newer = TRUE;
	    }
	    older_version = orig;
	    newer_version = ans;
	}

	if (i_am_newer) {
	    if (older_version->major < 1) {
		/*
		 * Versions before 1.0 do not expect to receive a
		 * packet indicating whether it is compatible or not,
		 * because this mechanism was not implemented then.
		 * The pre-1.0 version will detect that the versions
		 * are not the same, and quit with no useful messages.
		 * The user with version >= 1.0 should see a message
		 * explaining the reason for the disconnection, and
		 * they can choose to notify the other party about how
		 * to upgrade.
		 *
		 * HOWEVER, the released version called 1.0 actually
		 * contains VERSION_MAJOR 0 and VERSION_MINOR 9
		 * compiled into it, because this made it backwards
		 * compatible with version 0.9.2.  Therefore, if the
		 * older version is reported as 0.9, try notifying it.
		 * If it hangs up, it was version 0.9.2.  If it
		 * doesn't hang up immediately, it was the version
		 * called 1.0, but with 0.9 compiled into it, and it
		 * will listen to this upgrade information sent to it.
		 */
		SendCompatible(FALSE);
		SendUpgradeInfo(older_version->major, older_version->minor);
		if (WaitForAck() == FAIL) {
		    /*
		     * The most likely explanation that the final
		     * acknowledgement never came is that the other
		     * version is 0.9.2, and it hung up, ignoring the
		     * upgrade info we just tried to send.  Notify the
		     * local user of this.
		     */
		    CantNotifyBetaVersion();
		}
	        return FAIL;
	    }
	    else if (VersionsCompatible(older_version->major,
					older_version->minor)) {
		/*
		 * Tell the older version that all is well, and go on.
		 *
		 * There is a possibility that this packet will be
		 * garbled and discarded, and the rest of the protocol
		 * would likely fail.
		 *
		 * A similar problem can occur in other parts of the
		 * protocol as well.  For example, in the negotiation
		 * parameter exchange above, the answerer could
		 * successfully receive the originator's parameters,
		 * and reply with its own, but the answerer's packet
		 * could then be lost.  The answerer goes on with the
		 * protocol, but the originator keeps sending
		 *
		 * I'm not certain, but the protocol used here may
		 * contain pieces which are similar to the well known
		 * Coordinated Attack problem in the distributed
		 * algorithms area of research.  If so, then there is
		 * no solution that is guaranteed to work within any
		 * fixed time limit, but there could easily be
		 * solutions which don't fail due to the loss of a
		 * single packet.  I'll think about it.  It is more
		 * likely similar to problems in committing operations
		 * in a distributed database.  Then again, I could be
		 * imagining the problem to be more difficult than it
		 * really is.
		 *
		 * See what happens to you when you take a course in
		 * distributed algorithms.  You turn weird :-)
		 *
		 * To protect the innocent, the comment above was
		 * written by Andy Fingerhut.
		 */
		SendCompatible(TRUE);
		/*
		 * Go on with the rest of the program in this case.
		 */
	    }
	    else {
		/* 
		 * Tell the older version about incompatibility, and
		 * how to upgrade.
		 */
		SendCompatible(FALSE);
		SendUpgradeInfo(older_version->major, older_version->minor);
		return FAIL;
	    }
	}
	else {
	    /*
	     * This is the code executed by the older version.  Wait
	     * for the packet containing compatibility info, and if
	     * the verdict is 'incompatible', then receive, display,
	     * and save the updgrade info.  Acknowledge the receipt
	     * of this message.
	     */
	    if (ReceiveCompatible(&compatible) == FAIL) {
		error(MSG_ERROR, "never received compatibility information");
		return FAIL;
	    }
	    if (!compatible) {
		ReceiveUpgradeInfo(newer_version->major, newer_version->minor);
		SendPkt(ACK, NULL, 0);
		return FAIL;
	    }
	    /*
	     * If compatible, go on with the rest of the program.
	     */
	}
    }

    /* Get modem speed or assume 14400 bps if unknown */
    speed = orig->modem_speed[0] + (orig->modem_speed[1] << 8);
    if (speed == 0)
        speed = ans->modem_speed[0] + (ans->modem_speed[1] << 8);
    if (speed == 0)
        speed = DEFAULT_DCE_SPEED;

    /* Find best coder to use */
    for (i=NCODERS-1; i>=0; i--) {
        if ((coders[i].bandwidth <= speed)
	    && CoderUsable(orig, i) && CoderUsable(ans, i)) {
	    break;
	}
    }
    if (i >= 0) {
        params.coder.index = i;
    } else {
        return FAIL;
    }

    /* Did answerer specify a coder? */
    if (ans->coder < NCODERS) {
        if ((coders[ans->coder].bandwidth <= speed)
	    && CoderUsable(orig, ans->coder) && CoderUsable(ans, ans->coder)) {
	    params.coder.index = ans->coder;
        }
    }

    /* Did originator specify a coder? */
    if (orig->coder < NCODERS) {
        if ((coders[orig->coder].bandwidth <= speed)
	    && CoderUsable(orig, orig->coder) && CoderUsable(ans, orig->coder)) {
	    params.coder.index = orig->coder;
        }
    }

    /*
     * For now, make sure that both ends have specified the same type
     * of encryption (or none).  If they haven't, treat it as an
     * error.
     *
     * NOTE: If the behavior of quitting like this remains in future
     * versions, it would be nicer if the program told the user which
     * coders were selected by the caller and callee, to help them
     * determine what is going wrong.
     */
    if (orig->encrypt_type != ans->encrypt_type) {
	error(MSG_ERROR, "selected encryption must match");
	return FAIL;
    }

    if (params.crypto.key1.type != NONE) {
        /*
         * Generate keyexch structure.
         */
        params.crypto.skey_len = MAX_SKEY_LEN;
        random_bytes((UINT8 *) &keyexch1.sess_iv, 8);
        random_bytes((UINT8 *) &keyexch1.xmit_iv, 8);
        random_bytes((UINT8 *) &keyexch1.recv_iv, 8);
        random_bytes((UINT8 *) &keyexch1.skey, params.crypto.skey_len);

	/*
	 * Copy the keyexch structure just generated and encrypt it
	 * in CFB mode using the sess_iv element (first 8 bytes) as
	 * the IV.  Obviously, the first 8 bytes are not encrypted.
	 */
	keyexch2 = keyexch1;
	cfb_init(&params.crypto.sess_ctx, &params.crypto.key1, &keyexch2.sess_iv);
	cfb_encrypt(&params.crypto.sess_ctx, (UINT8 *) &keyexch2.xmit_iv,
	            sizeof(struct keyexch_t) - sizeof(BLOCK));
	keydestroy(&params.crypto.key1);

    	/*
    	 * Exchange keyexch structures.  Originator goes first.
    	 */
	if (mode == ORIGINATE) {
	    SendPkt(DATA, (UINT8 *) &keyexch2, sizeof(struct keyexch_t));
	    if (ReadPkt(&packet) < 0) {
		error(MSG_ERROR, "timeout during crypto parameter negotiation");
		return FAIL;
	    }
	    if ((packet.type != DATA) || (packet.length != sizeof(struct keyexch_t))) {
		error(MSG_ERROR, "negotiation of crypto parameters failed");
		return FAIL;
	    }
	}
	else {
	    if (ReadPkt(&packet) < 0) {
		error(MSG_ERROR, "timeout during crypto parameter negotiation");
		return FAIL;
	    }
	    if ((packet.type != DATA) || (packet.length != sizeof(struct keyexch_t))) {
	        error(MSG_ERROR, "negotiation of crypto parameters failed");
	        return FAIL;
	    }
	    SendPkt(DATA, (UINT8 *) &keyexch2, sizeof(struct keyexch_t));
	}

	/*
	 * Copy the received packet data into the second keyexch structure.
	 * Decrypt the second keyexch structure using the sess_iv element
	 * (first 8 bytes) as the iv.  Since we're done using the key
	 * schedule generated from the pass phrase, burn it.
	 */
	memcpy((char *) &keyexch2, (char *) packet.data, sizeof(struct keyexch_t));
	cfb_init(&params.crypto.sess_ctx, &params.crypto.key2, &keyexch2.sess_iv);
	cfb_decrypt(&params.crypto.sess_ctx, (UINT8 *) &keyexch2.xmit_iv,
		    sizeof(struct keyexch_t) - sizeof(BLOCK));
	keydestroy(&params.crypto.key2);

	/*
	 * Check the CRC on the received keyexch structure to make sure
	 * the other user typed the same passphrase we did (optional)
	 */
	/* XXX */

	/*
	 * XOR the two decrypted keyexch structures together and copy
	 * the result to the params structure.
	 */
	p1 = (UINT8 *) &keyexch1.xmit_iv;
	p2 = (UINT8 *) &keyexch2.xmit_iv;
	for (i=0; i<sizeof(struct keyexch_t) - sizeof(BLOCK); i++) {
	    *p1++ ^= *p2++;
	}
	memcpy((char *) params.crypto.skey, (char *) keyexch1.skey, params.crypto.skey_len);

	/*
	 * Call keyinit with the session key.
	 */
	if (keyinit(&params.crypto.key1, params.crypto.skey, params.crypto.skey_len, 0) < 0) {
	    error(MSG_ERROR, "key initialization failed");
	    return FAIL;
	}
	memset(params.crypto.skey, '\0', MAX_SKEY_LEN);

	/*
	 * Initialize crypto contexts for transmitting/receiving.
	 */
	if (mode == ORIGINATE) {
	    cfb_init(&params.crypto.xmit_ctx, &params.crypto.key1, &keyexch1.xmit_iv);
	    cfb_init(&params.crypto.recv_ctx, &params.crypto.key1, &keyexch1.recv_iv);
	}
	else {
	    cfb_init(&params.crypto.xmit_ctx, &params.crypto.key1, &keyexch1.recv_iv);
	    cfb_init(&params.crypto.recv_ctx, &params.crypto.key1, &keyexch1.xmit_iv);
	}

	/*
	 * Burn sensitive data
	 */
	memset((char *) &keyexch1, '\0', sizeof(struct keyexch_t));
	memset((char *) &keyexch2, '\0', sizeof(struct keyexch_t));
    }

    return SUCCESS;
}



int
ReadPkt(struct packet_t *pkt)
{
    int             i, c;
    UINT16          recv_crc, crc;

    UINT8           buf[2 * MAX_PKT_DATA];

    /* Initialize packet structure */
    pkt->type = (UINT8) - 1;
    pkt->length = (UINT16) - 1;
    pkt->crc = 0;

    /*
     * Expect next incoming character to be a FRAME character. If it isn't,
     * there's a framing problem so advance to the beginning of the next
     * frame and return an error (FAIL).
     */
    if ((c = ReadPort(RP_TIMEOUT)) != FRAME) {
	SyncPkt();
	return FAIL;
    }

    /* Allow up to two FRAME characters to separate frames. */
    if ((c = ReadPort(RP_TIMEOUT)) == FRAME)
	c = ReadPort(RP_TIMEOUT);

    /* Check packet type for validity */
    if ((c < 0) || (c > MAX_PACKET_TYPE)) {
	SyncPkt();
	return FAIL;
    }

    pkt->type = c;

#if DEBUG > 0
    switch (pkt->type) {
    case DATA:
	break;
    case EOT:
	debug_puts("Reading packet type EOT");
	break;
    case XMIT:
	debug_puts("Reading packet type TRANSMIT");
	break;
    case RECV:
	debug_puts("Reading packet type RECEIVE");
	break;
    case ACK:
	debug_puts("Reading packet type ACK");
	break;
    case NAK:
 	debug_puts("Reading packet type NAK");
 	break;
    case MAX_PACKET_TYPE:
 	debug_puts("Reading packet type MAX_PACKET_TYPE");
 	break;
    default:
	debug_puts("Reading packet with undefined type");
	break;
    }
#endif

    /* Get characters until we encounter another FRAME character */
    for (i = 0; ((c = GetByte()) >= 0) && i < 2 * MAX_PKT_DATA; i++) {
	buf[i] = c;
    }
    if (c == -1) {
    	/*
    	 * A timeout occurred while in the middle of reading a
    	 * packet.  Return an error.
    	 */
#if DEBUG > 0
	debug_puts("timeout while reading packet");
#endif
        return FAIL;
    }
    else if (c == -2) {
	/*
	 * We've found the trailing FRAME character.  The last two bytes are
	 * the crc.  Compute the length, copy the buffer into pkt->data, and
	 * finally check the crcs.
	 */
	recv_crc = buf[--i];
	recv_crc += buf[--i] << 8;
	pkt->length = i;
	memcpy(pkt->data, buf, pkt->length);
	crc = ComputeCRC(pkt->type, pkt->data, pkt->length);
	if (crc != recv_crc) {
#if DEBUG > 1
	    debug_putc('%');
#endif
	    error(MSG_WARNING, "crc error in received packet");
	    return FAIL;
	}
	else {
#if DEBUG > 1
	    debug_putc('.');
#endif
	}
    }
    else {
	/*
	 * Something screwy happened -- perhaps a FRAME character was
	 * garbled.  Anyway, we've overflowed our buffer so we discard all
	 * data up to the next FRAME character we can find and return an
	 * error.
	 */
#if DEBUG > 0
	debug_puts("Packet overflow -- data discarded");
#endif
	SyncPkt();
	return FAIL;
    }

    return SUCCESS;
}

void
SendPkt(enum pkt_type type, UINT8 * data, int len)
{
    int             i, j;
    UINT16          crc;
    UINT8           buf[2 * MAX_PKT_DATA];

#if DEBUG > 0
    switch (type) {
    case DATA:
#if DEBUG > 1
	debug_putc('*');
#endif
	break;
    case EOT:
	debug_puts("Sending packet type EOT");
	break;
    case XMIT:
	debug_puts("Sending packet type TRANSMIT");
	break;
    case RECV:
	debug_puts("Sending packet type RECEIVE");
	break;
    case ACK:
	debug_puts("Sending packet type ACK");
	break;
    default:
	debug_puts("Sending packet with undefined type");
	break;
    }
#endif

    i = 0;
    buf[i++] = FRAME;
    buf[i++] = type;
    for (j = 0; j < len; j++)
	AddByte(buf, i, data[j]);
    crc = ComputeCRC((UINT8) type, data, len);
    AddByte(buf, i, (crc >> 8) & 0xff);	/* msb first */
    AddByte(buf, i, crc & 0xff);
    buf[i++] = FRAME;
    WritePort(buf, i);
}

void
SyncPkt(void)
{
    int             c;

    /*
     * Advance to beginning of next frame by discarding incoming characters
     * until a FRAME character is received.
     */
    while (IncomingPort()) {
	c = ReadPort(1);
	if (c == FRAME)
	    break;
    }
}

int
GetByte()
{
    int             c;

    switch (c = ReadPort(RP_TIMEOUT)) {
    case FRAME:
	c = -2;
	break;
    case ESCAPE:
	c = ReadPort(RP_TIMEOUT) ^ 0x20;
	break;
    default:
	break;
    }

    return c;
}
