/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: CColumnTable.cp,v 1.12 1999/05/26 10:16:25 wprice Exp $
____________________________________________________________________________*/
#include "CColumnTable.h"

#include <LDropFlag.h>
#include <LTableArrayStorage.h>
#include <LTableMultiGeometry.h>
#include <LTableSingleSelector.h>
#include <LTableMultiSelector.h>
#include <LStream.h>
#include <PP_KeyCodes.h>
#include <PP_Messages.h>
#include <UDrawingState.h>
#include <UDrawingUtils.h>
#include <UGraphicUtils.h>

const Int16		kDefaultColumnWidth		=	16;
const Int16		kRowHeightSlop			=	9;
const Int16		kRowHeightDefault		=	18;
const Int16		kLeftSelectBorder		=	20;
const Int16		kLeftIndent				=	2;
const Int16		kBottomBorder			=	5;
const Int16		kTextIndent				=	5;
const Int16		kLevelIndent			=	20;

CColumnTable::CColumnTable(
	LStream	*inStream)
		:	CPGPHierarchyTable(inStream)
{
	TableIndexT		columnIndex;
	Style			style;
	PGPByte			fontName[256];
	
	*inStream >> mNumColumns;
	for( columnIndex = 0; columnIndex < kColumnTableMaxColumns; columnIndex++ )
		*inStream >> mColumnWidths[ columnIndex ];
	*inStream >> mAllowSelect;
	*inStream >> mMultiSelect;
	*inStream >> mHierarchical;
	*inStream >> mStretchColumn;
	mHas85Themes = TRUE;
#if TARGET_RT_MAC_CFM
	/* Check for waek link failure */
	if( (void *) GetThemeFont ==
			(void *) kUnresolvedCFragSymbolAddress )
	{
		mHas85Themes = FALSE;
	}
#endif
	if( mHas85Themes )
	{
		GetThemeFont( kThemeViewsFont, 0, &fontName[0], &mRowHeight, &style );
		mRowHeight += kRowHeightSlop;
	}
	else
	{
		mRowHeight = kRowHeightDefault;
	}
	
	mSortColumn		= 1;

	SetTableGeometry(new LTableMultiGeometry(this,	kDefaultColumnWidth,
													mRowHeight));
	if( mMultiSelect )
		SetTableSelector(new LTableMultiSelector(this));
	else
		SetTableSelector(new LTableSingleSelector(this));
	SetTableStorage(new LTableArrayStorage(this, (UInt32)0 ));
}

void
CColumnTable::FinishCreateSelf()
{
	CPGPHierarchyTable::FinishCreateSelf();
	SetupColumns();
	SetCustomHilite( TRUE );
}

CColumnTable::~CColumnTable()
{
}

	void
CColumnTable::ResizeFrameBy(
	Int16		inWidthDelta,
	Int16		inHeightDelta,
	Boolean		inRefresh)
{
	LView::ResizeFrameBy(inWidthDelta, inHeightDelta, inRefresh);
	
	if( mStretchColumn )
	{
		UInt16	width;
		
		width = GetColWidth( mStretchColumn );
		SetColWidth( width + inWidthDelta, mStretchColumn, mStretchColumn );
	}
}

	Boolean
CColumnTable::ObeyCommand(CommandT inCommand, void *ioParam)
{
	Boolean		cmdHandled = true;
	
	switch( inCommand )
	{
		case msg_TabSelect:
			cmdHandled = IsEnabled();
			break;
		
		default:
			cmdHandled = LCommander::ObeyCommand( inCommand, ioParam );
			break;
	}
	
	return( cmdHandled );
}

	void
CColumnTable::SetupColumns()
{
	TableIndexT		columnIndex;

	RemoveAllCols(false);
	InsertCols( mNumColumns, 1, NULL, 0, false );
	for( columnIndex = 0; columnIndex < mNumColumns; columnIndex++ )
		SetColWidth( mColumnWidths[columnIndex],
					columnIndex + 1, columnIndex + 1 );
}

	void
CColumnTable::SetSortColumn(
	TableIndexT col )
{
	Rect frame;

	if( col <= mNumColumns )
		mSortColumn = col;
	
	CalcPortFrameRect( frame );
	InvalPortRect( &frame );
}

	void
CColumnTable::DrawSelf()
{
	StColorPenState	colorState;
	RgnHandle		localUpdateRgnH = GetLocalUpdateRgn();
	Rect			updateRect = (**localUpdateRgnH).rgnBBox;
	Rect			frame,
					sortFrame,
					noSortFrame,
					cellRect;
	STableCell		cell,
					topLeftCell,
					botRightCell;
	TableIndexT		columnIndex;
	Int16			hVal;
	
	DisposeRgn( localUpdateRgnH );
	CalcLocalFrameRect( frame );
	FetchIntersectingCells( updateRect, topLeftCell, botRightCell );
	noSortFrame = updateRect;
	sortFrame.left = sortFrame.right = 0;
	hVal = 0;
	for( columnIndex = 0; columnIndex < mNumColumns; columnIndex++ )
	{
		hVal += mColumnWidths[columnIndex];
		if( ( noSortFrame.left <= hVal ) && ( columnIndex + 1 == mSortColumn ) )
		{
			sortFrame.left = hVal - mColumnWidths[columnIndex];
			sortFrame.right = sortFrame.left + GetColWidth( mSortColumn );
		}
	}
	sortFrame.top = noSortFrame.top;
	sortFrame.bottom = noSortFrame.bottom;
	noSortFrame.left = sortFrame.right;
	
	colorState.Normalize();
	if( mHas85Themes )
		UseThemeFont( kThemeViewsFont, 0 );
	else
	{
		TextFont( kFontIDGeneva );
		TextSize( 9 );
		TextFace( 0 );
	}
	TextMode( srcOr );

	SetThemeBackground( kThemeListViewBackgroundBrush, GetBitDepth(),
						TRUE );
	EraseRect( &noSortFrame );
	if( sortFrame.right > 0 )
	{
		SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
							GetBitDepth(), TRUE );
		EraseRect( &sortFrame );
	}
	mLowVertUpdate = mHighVertUpdate = 0;

	for(cell.row = topLeftCell.row; cell.row <= botRightCell.row;
		cell.row++)
	{
		for(cell.col = topLeftCell.col; cell.col <= botRightCell.col;
			cell.col++)
		{
			GetLocalCellRect( cell, cellRect );
			DrawCell( cell, cellRect );
		}
	}
	if( mHighVertUpdate > 0 && mLowVertUpdate > 0 )
	{
		SetThemePen( kThemeListViewSeparatorBrush, GetBitDepth(),
					TRUE );
		do
		{
			MoveTo( frame.left, mLowVertUpdate );
			LineTo( frame.right - 1, mLowVertUpdate );
			mLowVertUpdate += mRowHeight;
		} while( mLowVertUpdate <= mHighVertUpdate );
	}

	HiliteSelection(IsActive(), true);
}

	Boolean
CColumnTable::DrawCellCustom(
	const STableCell		&,
	const Rect				&,
	Boolean						 )
{
	return FALSE;
}

	void
CColumnTable::DrawCell1(
	const STableCell	&inCell,
	const Rect			&inLocalRect,
	Boolean				hilite,
	Boolean				inactive )
{
	TableIndexT			woRow = mCollapsableTree->GetWideOpenIndex(inCell.row);
	StClipRgnState		clip;
	STableCell			woCell(woRow, inCell.col);
	Rect				hiliteRect;
	ResIDT				iconID = 0;
	Str255				str;
	Int16				indentLevel = 0,
						width,
						textLeft;
	StyleParameter		style = 0;
	Boolean				custom;

	str[0] = 0;
	clip.ClipToIntersection( inLocalRect );
	SetThemeTextColor( kThemeListViewTextColor, GetBitDepth(), TRUE );
	hiliteRect.left = 0;
	custom = GetCellDrawData( woCell, iconID, indentLevel, str, style );
	if( custom )
	{
		hiliteRect.left += 15;
		if( DrawCellCustom( woCell, inLocalRect, hilite ) )
			custom = FALSE;
	}
	if( !custom )
	{
		TextFace( style );
		width = StringWidth( str );
		if( ( inCell.col == 1 ) && ( iconID != 0 ) )
			hiliteRect.left += inLocalRect.left + kLeftSelectBorder +
				( ( mHierarchical && ( inCell.col == 1 ) ) ? kLevelIndent : 0 );
		else
			hiliteRect.left += inLocalRect.left + kTextIndent;
		if( iconID != 0 )
			DrawIcon( inCell, iconID, inLocalRect, indentLevel,
				hilite ? kTransformSelected : kTransformNone );

		textLeft = hiliteRect.left + kLeftIndent +
				indentLevel * kLevelIndent;
		MoveTo( textLeft, inLocalRect.bottom - kBottomBorder );
		if( hilite && !inactive )
			TextMode(srcCopy);
		if( width > inLocalRect.right - textLeft )
		{
			TextFace( condense + style );
			width = StringWidth( str );
			TruncString(	inLocalRect.right - textLeft,
							str, truncMiddle);
		}
		DrawString( str );
	}
	if( inCell.col == 1 )
	{
		if( hilite )
		{
			hiliteRect.right = hiliteRect.left + width + kLeftIndent +
								indentLevel * kLevelIndent;
			hiliteRect.top = inLocalRect.top;
			hiliteRect.bottom = inLocalRect.bottom - 1;
			if( inactive )
			{
				RGBColor	hiliteColor;
				
				LMGetHiliteRGB( &hiliteColor );
				RGBForeColor( &hiliteColor );
				FrameRect( &hiliteRect );
			}
			else
			{
				UDrawingUtils::SetHiliteModeOn();
				InvertRect(&hiliteRect);
			}
		}
		DrawDropFlag( inCell, woRow );
	}
	
	if( !mLowVertUpdate ||
		( mLowVertUpdate > inLocalRect.bottom - 1 ) )
		mLowVertUpdate = inLocalRect.bottom - 1;
	if( !mHighVertUpdate ||
		( mHighVertUpdate < inLocalRect.bottom - 1 ) )
		mHighVertUpdate = inLocalRect.bottom - 1;
}

	void
CColumnTable::DrawIcon(
	const STableCell	&inCell,
	ResIDT				iconID,
	Rect				cellRect,
	UInt32				indentLevel,
	IconTransformType	transform )
{
	Rect				iconRect;

	iconRect.left	=	cellRect.left + kLeftIndent +
						( indentLevel +
						( ( mHierarchical && ( inCell.col == 1 ) ) ?1:0 ) )
						* kLevelIndent;
	iconRect.top	=	cellRect.top + 1;
	iconRect.right 	= 	iconRect.left + 16;
	iconRect.bottom = 	iconRect.top + 16;
	
	PlotIconID(&iconRect, kAlignNone, transform, iconID);
}

	void
CColumnTable::DrawCell(
	const STableCell	&inCell,
	const Rect			&inLocalRect)
{
	DrawCell1( inCell, inLocalRect, false, false );
}

	void
CColumnTable::HiliteCellActively(
	const STableCell	&inCell,
	Boolean				inHilite )
{
	StColorPenState	saveColorPen;
	Rect			cellFrame,
					insideFrame;
	
	if( inCell.col != 1 )
		return;
	if( GetLocalCellRect(inCell, cellFrame) && FocusExposed() )
	{
		insideFrame = cellFrame;
		insideFrame.bottom--;
		if( mHas85Themes )
			UseThemeFont( kThemeViewsFont, 0 );
		else
		{
			TextFont( kFontIDGeneva );
			TextSize( 9 );
			TextFace( 0 );
		}
		TextMode( srcOr );
		if( !inHilite )
		{
			StDeviceLoop	devLoop(insideFrame);		
			SInt16			depth;

			while(devLoop.NextDepth(depth))
			{
				SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
									depth, ( depth > 1 ) );
				EraseRect( &insideFrame );
			}
		}
		DrawCell1( inCell, cellFrame, inHilite, false );
	}
}

	void
CColumnTable::HiliteCellInactively(
	const STableCell	&inCell,
	Boolean			 	inHilite )
{
	StColorPenState	saveColorPen;
	Rect			cellFrame,
					insideFrame;
	
	if( inCell.col != 1 )
		return;
	if( GetLocalCellRect(inCell, cellFrame) && FocusExposed() )
	{
		insideFrame = cellFrame;
		insideFrame.bottom--;
		if( mHas85Themes )
			UseThemeFont( kThemeViewsFont, 0 );
		else
		{
			TextFont( kFontIDGeneva );
			TextSize( 9 );
			TextFace( 0 );
		}
		TextMode( srcOr );
		if( !inHilite )
		{
			StDeviceLoop	devLoop(insideFrame);		
			SInt16			depth;

			while(devLoop.NextDepth(depth))
			{
				SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
									depth, ( depth > 1 ) );
				EraseRect( &insideFrame );
			}
		}
		DrawCell1( inCell, cellFrame, inHilite, true );
	}
}

	void
CColumnTable::ClickSelf(
	const SMouseDownEvent &inMouseDown)
{
	if( !IsTarget() )
		SwitchTarget( this );
	if( mAllowSelect )
	{
		STableCell	hitCell;
		SPoint32	imagePt;
		
		LocalToImagePoint(inMouseDown.whereLocal, imagePt);
		
		if (GetCellHitBy(imagePt, hitCell)) {
											// Click is inside hitCell
											// Check if click is inside DropFlag
			TableIndexT	woRow = mCollapsableTree->GetWideOpenIndex(hitCell.row);
			Rect	flagRect;
			CalcCellFlagRect(hitCell, flagRect);
			
			if ((hitCell.col == 1) &&
				mCollapsableTree->IsCollapsable(woRow) &&
				::MacPtInRect(inMouseDown.whereLocal, &flagRect))
			{
				SInt16	depth;
											// Click is inside DropFlag
				FocusDraw();
				if( mSortColumn == hitCell.col )
				{
					depth = (**((**GetMainDevice()).gdPMap)).pixelSize;
					SetThemeBackground( kThemeListViewSortColumnBackgroundBrush,
											depth, ( depth > 1 ) );
				}
				else
					ApplyForeAndBackColors();
				Boolean	expanded = mCollapsableTree->IsExpanded(woRow);
				if (LDropFlag::TrackClick(flagRect, inMouseDown.whereLocal,
										expanded)) {
											// Mouse released inside DropFlag
											//   so toggle the Row
					if (inMouseDown.macEvent.modifiers & optionKey) {
											// OptionKey down means to do
											//   a deep collapse/expand						
						if (expanded) {
							DeepCollapseRow(woRow);
						} else {
							DeepExpandRow(woRow);
						}
					
					} else {				// Shallow collapse/expand
						if (expanded) {
							CollapseRow(woRow);
						} else {
							ExpandRow(woRow);
						}
					}
				}
		
			}
			else
			{
				hitCell.col = 1;
				ClickSelect(hitCell, inMouseDown);
											// Click outside of the DropFlag
				ClickCell(hitCell, inMouseDown);
			}
			
		} else {							// Click is outside of any Cell
			UnselectAllCells();
		}
	}
}

	void
CColumnTable::ClickCell(
	const STableCell&		inCell,
	const SMouseDownEvent&	inMouseDown )
{
	if( inCell.col == 1 )
	{
		if( GetClickCount() == 2 )
		{
			STableCell cell = inCell;
			
			BroadcastMessage( coltable_DoubleClick, (void *)&cell );
		}
		else
		{
			CPGPHierarchyTable::ClickCell( inCell, inMouseDown );
		}
	}
}

	void
CColumnTable::SelectionChanged()
{
	BroadcastMessage( coltable_SelectChange, this );
}

	Boolean
CColumnTable::HandleKeyPress(const EventRecord&	inKeyEvent)
{
	Boolean		keyHandled	= true;
	Int16		theKey		= inKeyEvent.message & charCodeMask;
	STableCell	oCell, cell;
	TableIndexT	rows, cols;
	
	if( inKeyEvent.modifiers & cmdKey )
		keyHandled = LCommander::HandleKeyPress(inKeyEvent);
	else switch(theKey)
	{
		default:
			keyHandled = LCommander::HandleKeyPress(inKeyEvent);
			break;
		case char_UpArrow:
			oCell = cell = GetFirstSelectedCell();
			cell.col = 1;
			if(cell.row > 1)
				cell.row--;
			if(cell.row < 1)
				cell.row = 1;
			if( oCell != cell )
			{
				if( !mMultiSelect || ! ( inKeyEvent.modifiers & shiftKey ) )
					UnselectAllCells();
				ScrollCellIntoFrame(cell);
				SelectCell(cell);
			}
			break;
		case char_DownArrow:
			GetTableSize(rows, cols);
			cell.col = 1;
			for( cell.row = rows; ( cell.row > 0 ) && !CellIsSelected( cell );
				cell.row-- )
				;
			oCell = cell;
			cell.row++;
			if(cell.row > rows)
				cell.row = rows;
			if( oCell != cell )
			{
				if( !mMultiSelect || ! ( inKeyEvent.modifiers & shiftKey ) )
					UnselectAllCells();
				ScrollCellIntoFrame(cell);
				SelectCell(cell);
			}
			break;
		case char_Home:
			cell.row = cell.col = 1;
			ScrollCellIntoFrame(cell);
			break;
		case char_End:
			GetTableSize(rows, cols);
			cell.col = 1;
			cell.row = rows;
			ScrollCellIntoFrame(cell);
			break;
		case char_PageUp:
		{
			Rect	frameRect;
			
			CalcLocalFrameRect(frameRect);
			ScrollPinnedImageBy( 0,
				-UGraphicUtils::RectHeight(frameRect), true );
			break;
		}
		case char_PageDown:
		{
			Rect	frameRect;
			
			CalcLocalFrameRect(frameRect);
			ScrollPinnedImageBy( 0,
				UGraphicUtils::RectHeight(frameRect), true );
			break;
		}
	}
	return keyHandled;
}

