/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: CKeyInfoWindow.cp,v 1.67.2.1 1999/06/11 06:14:42 heller Exp $
____________________________________________________________________________*/

#include <LBevelButton.h>
#include <LCheckBox.h>
#include <LEditText.h>
#include <LGATabsControlImp.h>
#include <LLittleArrows.h>
#include <LMultiPanelView.h>
#include <LRadioButton.h>
#include <LTableArrayStorage.h>
#include <LTableMonoGeometry.h>
#include <LTableSingleSelector.h>
#include <LTabsControl.h>
#include <LTextGroupBox.h>
#include <LTextEditView.h>
#include <LProgressBar.h>
#include <LPushButton.h>
#include <LSlider.h>
#include <PP_Messages.h>
#include <UDesktop.h>
#include <UDrawingUtils.h>
#include <UGAColorRamp.h>
#include <UGraphicUtils.h>
#include <URegistrar.h>
#include <UTextTraits.h>

#include "CKeyInfoWindow.h"
#include "CKeyWindow.h"
#include "CColumnTable.h"
#include "CKeyTable.h"
#include "CPGPKeys.h"
#include "CPhotoUserID.h"
#include "CGADurationEditField.h"
#include "CWarningAlert.h"
#include "CPGPStDialogHandler.h"
#include "CGAProgressDialog.h"
#include "MacErrors.h"
#include "MacQuickDraw.h"
#include "MacSecureMemory.h"
#include "MacStrings.h"
#include "PGPSharedEncryptDecrypt.h"
#include "ResourceConstants.h"
#include "pgpClientLib.h"
#include "pgpClientPrefs.h"
#include "pgpAdminPrefs.h"
#include "pgpRandomPool.h"
#include "GetPassphrase.h"

#include "pgpMem.h"
#include "pgpErrors.h"
#include "pgpUtilities.h"

enum
{
	kNeverExpiresID			= 1,
	kRSATypeID,
	kDSATypeID,
	kTooManyWindowsID,
	kInvalidKeySetTrustID,
	kUnknownMRKID,
	kSubkeysStrIndex,
	kADKStrIndex,
	kRevokerStrIndex,
	kSubkeyWrongSizeMsgStrIndex,
	kBadSubkeyExpirationMsgStrIndex,
	kSubkeyAlreadyExpiredMsgStrIndex,
	kConfirmRemoveSubkeyStrIndex,
	kConfirmRemoveLastSubkeyStrIndex,
	kConfirmRevokeSubkeyStrIndex,
	kAdminKeySizeErrorStrIndex,
	kPassphraseChangedStrIndex,
	kEnterNewPhraseStrIndex,
	kJoinButtonStrIndex,
	kCreatingSubkeyStrIndex,
	kKeyRejoinedStrIndex,
	kChangePassphraseButtonStrIndex
};

const ResIDT		kKeyInfoWindowID		= 140;
const ResIDT		kChangePassDialogID		= 136;
const ResIDT		kNewSubkeyDialogResID	= 137;
const ResIDT		kKeyInfoStringListID	= 1010;

const short			kNumTrustValues			=	3;
const short			kNumValidityValues		=	3;
const Int16			kUnsupportedStringID	=	4;

const ResIDT		kGeneralPanelViewResID	= 142;
const ResIDT		kSubkeysPanelViewResID	= 143;
const ResIDT		kADKPanelViewResID		= 144;
const ResIDT		kRevokerPanelViewResID	= 148;


short			CKeyInfoWindow::sNumKeyInfoWindows	= 0;
CKeyInfoWindow	*CKeyInfoWindow::sKeyInfoWindows[kMaxKeyInfoWindows];


#pragma mark --- CKeyInfoPanelView ---

class CKeyInfoPanelView : 	public LView,
							public LCommander
{
public:
	
	enum { class_ID = 'vKIP' };
		
					CKeyInfoPanelView(LStream *inStream);
	virtual			~CKeyInfoPanelView();

	Boolean			HasChanged(void) { return( mChanged ); };
	virtual void	SavePanel(void);
	virtual void	SetInfo(PGPKeySetRef keySet, PGPKeyRef theKey,
							PGPBoolean keyCanBeModified);
	
protected:

	PGPKeySetRef	mKeySet;
	PGPKeyRef		mKey;
	Boolean			mKeyCanBeModified;
	Boolean			mChanged;
	PGPKeyIterRef	mKeyIter;
	PGPKeyListRef	mKeyList;
};

CKeyInfoPanelView::CKeyInfoPanelView(LStream *inStream)
	: LView( inStream )
{
	mKeySet 			= kInvalidPGPKeySetRef;
	mKey 				= kInvalidPGPKeyRef;
	mKeyCanBeModified	= TRUE;
	mChanged			= FALSE;
	mKeyIter 			= kInvalidPGPKeyIterRef;
	mKeyList			= kInvalidPGPKeyListRef;
}

CKeyInfoPanelView::~CKeyInfoPanelView()
{
	if( PGPKeyIterRefIsValid( mKeyIter ) )
	{
		PGPFreeKeyIter( mKeyIter );
		mKeyIter = kInvalidPGPKeyIterRef;
	}
	
	if( PGPKeyListRefIsValid( mKeyList ) )
	{
		PGPFreeKeyList( mKeyList );
		mKeyList = kInvalidPGPKeyListRef;
	}
}

	void
CKeyInfoPanelView::SetInfo(
	PGPKeySetRef 	keySet,
	PGPKeyRef 		theKey,
	PGPBoolean 		keyCanBeModified)
{
	PGPError	err;
	
	mKeySet				= keySet;
	mKey 				= theKey;
	mKeyCanBeModified	= keyCanBeModified;

	err = PGPOrderKeySet( keySet, kPGPAnyOrdering, &mKeyList );
	pgpAssertNoErr( err );
	if( IsntPGPError( err ) )
	{
		err = PGPNewKeyIter( mKeyList, &mKeyIter );
		pgpAssertNoErr( err );
		if( IsntPGPError( err ) )
		{
			(void) PGPKeyIterSeek( mKeyIter, theKey );
		}
	}
	
}

      void
CKeyInfoPanelView::SavePanel(void)
{
}

#pragma mark --- CKeyPanelGeneral ---

class CKeyPanelGeneral : 	public CKeyInfoPanelView,
							public LListener
{
public:

	enum { class_ID = 'vKIG' };

					CKeyPanelGeneral(LStream *inStream);
	virtual			~CKeyPanelGeneral(void);
	
	virtual void	ListenToMessage(MessageT inMessage, void *ioParam);
	virtual void	SavePanel(void);
	virtual void	SetInfo(PGPKeySetRef keySet, PGPKeyRef theKey,
							PGPBoolean keyCanBeModified);
	
protected:

	virtual	void	FinishCreateSelf(void);
	
private:
	virtual void	ChangePassphrase();

	enum
	{
		kKeyIDCaption			= 'eKID',
		kCreationCaption		= 'cCre',
		kValidityBox			= 'bVal',
		kTrustSlider			= 'sTru',
		kFingerprintCaption		= 'eFin',
		kFingerprintBorder		= 'fFin',
		kEnabledCheckbox		= 'xEna',
		kPassphraseButton		= 'bPas',
		kKeyTypeCaption			= 'cTyp',
		kExpiresCaption			= 'cExp',
		kSizeCaption			= 'cSiz',
		kCipherCaption			= 'cCyp',
		kAxiomaticCheckbox		= 'xTru',
		kPhotoUserID			= 'pPID',
		kNextPhotoButtonID		= 'Next',
		kPrevPhotoButtonID		= 'Prev',
		kHashWordsView			= 'vWor',
		kHashWords1Caption		= 'cWo1',
		kHashWords2Caption		= 'cWo2',
		kHashWords3Caption		= 'cWo3',
		kHashWords4Caption		= 'cWo4',
		kHexFingerCheckbox		= 'xHex'
	};

	PGPSharedKeyProperties		mKeyProperties;
	
	Boolean						mTrustChanged;
	
	LStaticText					*mCreationCaption,
								*mExpiresCaption,
								*mKeyTypeCaption,
								*mSizeCaption,
								*mCipherCaption;
	LTextEditView				*mFingerprintCaption,
								*mKeyIDCaption;
	LCheckBox					*mEnabledCheckbox,
								*mAxiomaticCheckbox,
								*mHexCheckbox;
	LSlider						*mTrustSlider;
	LProgressBar				*mValidityBox;
	LPushButton					*mPassphraseButton;
	CPhotoUserID				*mPhotoUserID;
	LBevelButton				*mNextPhotoButton;
	LBevelButton				*mPrevPhotoButton;
	PGPUInt32					mNumPhotoIDs;
	PGPUInt32					mPhotoIDIndex;
	
	void						SwitchPhoto(PGPUInt32 newPhotoIndex);
	void						UpdatePhotoStatus(PGPUserIDRef userID);
};


CKeyPanelGeneral::CKeyPanelGeneral(LStream *inStream)
	: CKeyInfoPanelView( inStream )
{
	mNumPhotoIDs	= 0;
	mPhotoIDIndex	= 1;
}

CKeyPanelGeneral::~CKeyPanelGeneral()
{
}

	void
CKeyPanelGeneral::FinishCreateSelf(void)
{
	CKeyInfoPanelView::FinishCreateSelf();
	
	mTrustChanged = FALSE;
	
	mKeyIDCaption =			(LTextEditView *)FindPaneByID(kKeyIDCaption);
	pgpAssertAddrValid(mKeyIDCaption,		VoidAlign);
	mKeyTypeCaption =		(LStaticText *)	FindPaneByID(kKeyTypeCaption);
	pgpAssertAddrValid(mKeyTypeCaption,		VoidAlign);
	mCreationCaption =		(LStaticText *)	FindPaneByID(kCreationCaption);
	pgpAssertAddrValid(mCreationCaption,	VoidAlign);
	mSizeCaption =			(LStaticText *)	FindPaneByID(kSizeCaption);
	pgpAssertAddrValid(mSizeCaption,		VoidAlign);
	mCipherCaption =		(LStaticText *)	FindPaneByID(kCipherCaption);
	pgpAssertAddrValid(mCipherCaption,		VoidAlign);
	mExpiresCaption =		(LStaticText *)	FindPaneByID(kExpiresCaption);
	pgpAssertAddrValid(mExpiresCaption,		VoidAlign);
	mEnabledCheckbox =		(LCheckBox *)	FindPaneByID(kEnabledCheckbox);
	pgpAssertAddrValid(mEnabledCheckbox,	VoidAlign);
	mTrustSlider =			(LSlider *)	FindPaneByID(kTrustSlider);
	pgpAssertAddrValid(mTrustSlider,		VoidAlign);
	mValidityBox =			(LProgressBar *)FindPaneByID(kValidityBox);
	pgpAssertAddrValid(mValidityBox,		VoidAlign);
	mPassphraseButton =		(LPushButton *)FindPaneByID(kPassphraseButton);
	pgpAssertAddrValid(mPassphraseButton,	VoidAlign);
	mAxiomaticCheckbox =	(LCheckBox *)	FindPaneByID(kAxiomaticCheckbox);
	pgpAssertAddrValid(mAxiomaticCheckbox,	VoidAlign);
	mHexCheckbox =			(LCheckBox *)	FindPaneByID(kHexFingerCheckbox);
	pgpAssertAddrValid(mHexCheckbox,	VoidAlign);
	mFingerprintCaption =	(LTextEditView *)FindPaneByID(kFingerprintCaption);
	pgpAssertAddrValid(mFingerprintCaption,	VoidAlign);
	mPhotoUserID = 			(CPhotoUserID *)FindPaneByID(kPhotoUserID);
	pgpAssertAddrValid(mPhotoUserID, 		VoidAlign);
	mNextPhotoButton = 		(LBevelButton *)FindPaneByID(kNextPhotoButtonID);
	pgpAssertAddrValid(mNextPhotoButton, 	VoidAlign);
	mPrevPhotoButton = 		(LBevelButton *)FindPaneByID(kPrevPhotoButtonID);
	pgpAssertAddrValid(mPrevPhotoButton, 	VoidAlign);

	mPhotoUserID->Lock();
	mNextPhotoButton->AddListener( this );
	mPrevPhotoButton->AddListener( this );
	mHexCheckbox->AddListener( this );
	SwitchTarget(mKeyIDCaption);
}

	void
CKeyPanelGeneral::SetInfo(
	PGPKeySetRef 	keySet,
	PGPKeyRef 		key,
	PGPBoolean 		keyCanBeModified)
{
	PGPError		err;
	Str255			pstr;
	Str32			str2;
	PGPSize			len;
	PGPInt32		keyStatus;
	Int16			trust;
	PGPKeyID		keyID;
	PGPTime			creation,
					expiration;
	PGPInt32		bits,
					algorithm;
	PGPBoolean		useHex;
	
	CKeyInfoPanelView::SetInfo( keySet, key, keyCanBeModified );
	
	err = PGPSharedGetKeyProperties( key, NULL, &mKeyProperties );
	pgpAssertNoErr( err );
	
	if( mKeyProperties.isExpired || mKeyProperties.isRevoked )
	{
		keyCanBeModified = FALSE;
	}
	
	// Get the KeyID
	err = PGPGetKeyIDFromKey( key, &keyID );
	pgpAssertNoErr( err );
	if( IsntPGPError( err ) )
	{
		char	keyIDString[ kPGPMaxKeyIDStringSize ];
		
		err = PGPGetKeyIDString( &keyID,
					kPGPKeyIDString_Abbreviated, keyIDString);
		pgpAssertNoErr(err);
		CToPString(keyIDString, pstr);
		if(pstr[0] > 10)
			pstr[0] = 10;
			
		mKeyIDCaption->SetDescriptor( pstr );
	}
	mKeyIDCaption->SelectAll();
	
	// Get the Fingerprint
	CKeyInfoWindow::GetFingerprintString( pstr, key );
	mFingerprintCaption->SetDescriptor( pstr );
	{
		PGPByte		fingerprint[ 32 ];
		PGPUInt32	wordsPerColumn,
					columnNum,
					wordNum,
					fingIndex = 0;
		PaneIDT		fingPane = kHashWords1Caption;
		
		err = PGPGetKeyPropertyBuffer( key, kPGPKeyPropFingerprint,
				sizeof( fingerprint ), fingerprint, &len);
		pgpAssertNoErr( err );
		wordsPerColumn = len / 4;
		for( columnNum = 1; columnNum <= 4; columnNum++ )
		{
			pstr[0] = '\0';
			for( wordNum = 0; wordNum < wordsPerColumn;wordNum++ )
			{
				fingIndex = columnNum + ( wordNum * 4 ) - 1;
				if( fingIndex % 2 )
					PGPGetHashStringOdd( fingerprint[fingIndex], str2 );
				else
					PGPGetHashStringEven( fingerprint[fingIndex], str2 );
				AppendPString( str2, pstr );
				AppendPString( "\p\r", pstr );
			}
			FindPaneByID(fingPane)->SetDescriptor( pstr );
			fingPane++;
		}
	}
	PGPGetPrefBoolean( gPrefRef, kPGPPrefUseHexFingerprint, &useHex );
	if( useHex )
		mHexCheckbox->SetValue(1);
	
	mEnabledCheckbox->SetValue( ! mKeyProperties.isDisabled );
	
	// Get the Trust
	err = PGPGetKeyNumber(key, kPGPKeyPropTrust, &keyStatus);
	pgpAssertNoErr( err );
	switch(keyStatus & kPGPKeyTrust_Mask)
	{
		case kPGPKeyTrust_Undefined:
		case kPGPKeyTrust_Unknown:
		case kPGPKeyTrust_Never:
		default:
			trust = 0;
			break;
		case kPGPKeyTrust_Marginal:
			trust = 1;
			break;
		case kPGPKeyTrust_Complete:
		case kPGPKeyTrust_Ultimate:
			trust = 2;
			break;
	}

	mTrustSlider->SetMaxValue(kNumTrustValues - 1);
	mTrustSlider->SetValue(trust);
	
	// Get the Validity
	err = PGPGetKeyNumber(key, kPGPKeyPropValidity, &keyStatus);
	pgpAssert(!err);
	switch(keyStatus)
	{
		case kPGPValidity_Unknown:
		case kPGPValidity_Invalid:
		default:
			trust = 0;
			break;
		case kPGPValidity_Marginal:
			trust = 1;
			break;
		case kPGPValidity_Complete:
			trust = 2;
			break;
	}

	mValidityBox->SetMaxValue(kNumValidityValues - 1);
	mValidityBox->SetValue(trust);
	
	mAxiomaticCheckbox->SetValue(mKeyProperties.isAxiomatic ? 1 : 0);
	if(mKeyProperties.isAxiomatic)
	{
		mTrustSlider->Disable();
	}
		
	// Get the Creation Date
	err = PGPGetKeyTime(key, kPGPKeyPropCreation, &creation);
	pgpAssert(!err);
	creation = PGPTimeToMacTime(creation);
	::DateString(creation, shortDate, pstr, NULL);
	mCreationCaption->SetDescriptor(pstr);
 		
	// Get the Expiration Date
	err = PGPGetKeyTime(key, kPGPKeyPropExpiration, &expiration);
	pgpAssert(!err);
	if(IsntErr(err) && expiration)
	{
		expiration = PGPTimeToMacTime(expiration);
		::DateString(expiration, shortDate, pstr, NULL);
	}
	else
		::GetIndString(pstr, kKeyInfoStringListID, kNeverExpiresID);
	mExpiresCaption->SetDescriptor(pstr);
	
	// Get the Algorithm ID
	err = PGPGetKeyNumber(key, kPGPKeyPropAlgID, &algorithm);
	pgpAssert(!err);
	switch(algorithm)
	{
		case kPGPPublicKeyAlgorithm_RSA:
		case kPGPPublicKeyAlgorithm_RSAEncryptOnly:
		case kPGPPublicKeyAlgorithm_RSASignOnly:
			::GetIndString(pstr, kKeyInfoStringListID, kRSATypeID);
			break;
		case kPGPPublicKeyAlgorithm_DSA:
		case kPGPPublicKeyAlgorithm_ElGamal:	// Technically not
			::GetIndString(pstr, kKeyInfoStringListID, kDSATypeID);
			break;
		default:
			pgpDebugPStr("\pUnknown algorithm");
			break;
	}
	mKeyTypeCaption->SetDescriptor(pstr);
	
	// Get the Key Size
	err = PGPGetKeyNumber(key, kPGPKeyPropBits, &bits);
	pgpAssertNoErr(err);
	::NumToString(bits, pstr);
	if(algorithm == kPGPPublicKeyAlgorithm_DSA)
	{
		PGPSubKeyRef	subKey;
		
		err = PGPKeyIterNextSubKey(mKeyIter, &subKey);
		if(IsntPGPError(err) && IsntNull(subKey))
		{
			err = PGPGetSubKeyNumber(subKey, kPGPKeyPropBits,
										&bits);
			pgpAssertNoErr(err);
			::NumToString(bits, str2);
			AppendPString("\p/", str2);
			AppendPString(pstr, str2);
			CopyPString(str2, pstr);
		}
	}
	mSizeCaption->SetDescriptor(pstr);
	
	// Get the Preferred Cipher
	{
		PGPByte		propBuff[128];	
		
		err = PGPGetKeyPropertyBuffer(key, kPGPKeyPropPreferredAlgorithms,
								sizeof( propBuff ), propBuff, &len);
		pgpAssertNoErr(err);
		if ( IsntPGPError( err ) )
		{
			if(len >= sizeof(PGPCipherAlgorithm))
			{
				UInt32	prefAlg = *(PGPCipherAlgorithm const *)&propBuff[0];
				
				pgpAssert((prefAlg >= kPGPCipherAlgorithm_None) &&
							(prefAlg <= kPGPCipherAlgorithm_CAST5));
				::GetIndString(pstr, kAlgorithmStringsID, prefAlg);
			}
			else
				::GetIndString(pstr, kAlgorithmStringsID, kIDEAString);
			mCipherCaption->SetDescriptor(pstr);
		}
	}

	if( mKeyProperties.isAxiomatic || ! keyCanBeModified )
	{
		mEnabledCheckbox->Hide();
		mTrustSlider->Disable();
	}

	// Set the Passphrase button status
	if( ! mKeyProperties.isSecret || ! keyCanBeModified )
	{
		mPassphraseButton->Hide();
		mAxiomaticCheckbox->Hide();
	}
	else if( mKeyProperties.isSplit )
	{
		::GetIndString(pstr, kKeyInfoStringListID, kJoinButtonStrIndex);
		mPassphraseButton->SetDescriptor( pstr );
	}
	
	// Count photo user ID's
	err = PGPKeyIterRewindUserID( mKeyIter );
	pgpAssertNoErr( err );
	
	while( IsntPGPError( err ) )
	{
		PGPUserIDRef	userID;
		
		err = PGPKeyIterNextUserID( mKeyIter, &userID );
		if( IsntPGPError( err ) )
		{
			PGPBoolean	isAttributeID;
			
			err = PGPGetUserIDBoolean( userID, kPGPUserIDPropIsAttribute,
							&isAttributeID );
			pgpAssertNoErr( err );
			
			if( IsntPGPError( err ) && isAttributeID )
			{
				PGPInt32	attributeType;
				
				err = PGPGetUserIDNumber( userID,
							kPGPUserIDPropAttributeType, &attributeType );
				pgpAssertNoErr( err );
				
				if( IsntPGPError( err ) &&
					isAttributeID == kPGPAttribute_Image )
				{
					mNumPhotoIDs++;
				}
			}
		}		
	}
	
	if( mNumPhotoIDs < 2 )
	{
		mNextPhotoButton->Hide();
		mPrevPhotoButton->Hide();
	}
	
	if( mNumPhotoIDs >= 1 )
	{
		SwitchPhoto( 1 );
	}
	else
	{
		PGPUserIDRef	userID;
		
		err = PGPGetPrimaryUserID( key, &userID );
		pgpAssertNoErr( err );
		
		UpdatePhotoStatus( userID );
	}

	mPassphraseButton->AddListener( this );
	mAxiomaticCheckbox->AddListener( this );
	mEnabledCheckbox->AddListener( this );
	mTrustSlider->AddListener(this);
}

	void
CKeyPanelGeneral::ChangePassphrase()
{
	PGPError			err = kPGPError_NoErr;
	PGPBoolean			shared = FALSE;
	CSecureCString256	oldPhrase;
	char				*newPhrase = NULL;
	PGPByte				*oldPassKey = NULL;
	PGPSize				oldPassKeySize;
	PGPInt32			algorithm;
	Str255				pstr;
	char				cstr[256];
#if PGP_BUSINESS_SECURITY
	PGPBoolean			enforceLength,
						enforceQuality;
	PGPUInt32			minLength,
						minQuality;
#endif
	
	err = GetPassForKey( mKey, &shared, oldPhrase,
						&oldPassKey, &oldPassKeySize );
	if( IsPGPError( err ) )
		goto done;
	
#if PGP_BUSINESS_SECURITY
	err = PGPGetPrefBoolean( gAdminPrefRef, kPGPPrefEnforceMinChars,
			&enforceLength);
	pgpAssertNoErr( err );
	err = PGPGetPrefBoolean( gAdminPrefRef, kPGPPrefEnforceMinQuality,
			&enforceQuality);
	pgpAssertNoErr( err );
	err = PGPGetPrefNumber( gAdminPrefRef, kPGPPrefMinChars,
			&minLength);
	pgpAssertNoErr( err );
	err = PGPGetPrefNumber( gAdminPrefRef, kPGPPrefMinQuality,
			&minQuality);
	pgpAssertNoErr( err );
#endif

	GetIndString( pstr, kKeyInfoStringListID, kEnterNewPhraseStrIndex );
	PToCString( pstr, cstr );
	err = PGPConfirmationPassphraseDialog( gPGPContext,
			PGPOUIDialogPrompt( gPGPContext, cstr ),
			PGPOUIOutputPassphrase( gPGPContext, &newPhrase ),
#if PGP_BUSINESS_SECURITY
			enforceLength ?
				PGPOUIMinimumPassphraseLength(
					gPGPContext, minLength ) :
				PGPONullOption( gPGPContext ),
			enforceQuality ?
				PGPOUIMinimumPassphraseQuality(
					gPGPContext, minQuality ) :
				PGPONullOption( gPGPContext ),
#endif
			PGPOLastOption( gPGPContext ) );
	
	if( IsPGPError( err ) )
		goto done;
	
	err = PGPChangePassphrase( mKey, shared ?
				PGPOPasskeyBuffer( gPGPContext, oldPassKey, oldPassKeySize ) :
				PGPOPassphrase( gPGPContext, oldPhrase ),
				PGPOPassphrase( gPGPContext, newPhrase ),
				PGPOLastOption( gPGPContext ) );
	if( IsPGPError( err ) )
		goto done;
	err = PGPGetKeyNumber( mKey, kPGPKeyPropAlgID, &algorithm );
	if( IsPGPError( err ) )
		goto done;
	if(algorithm == kPGPPublicKeyAlgorithm_DSA)
	{
		PGPKeyIterRef		keyIter;
		PGPSubKeyRef		subKey;
		
		err = PGPNewKeyIter( CPGPKeys::TheApp()->GetKeyList(), &keyIter );
		if( IsPGPError( err ) )
			goto done;
		PGPKeyIterSeek( keyIter, mKey );
		while( IsntPGPError ( err ) &&
			IsntPGPError( PGPKeyIterNextSubKey( keyIter, &subKey ) ) &&
			IsntNull( subKey ) )
		{
			err = PGPChangeSubKeyPassphrase( subKey, shared ?
				PGPOPasskeyBuffer( gPGPContext, oldPassKey, oldPassKeySize ) :
				PGPOPassphrase( gPGPContext, oldPhrase ),
				PGPOPassphrase( gPGPContext, newPhrase ),
				PGPOLastOption( gPGPContext ) );
		}
		PGPFreeKeyIter( keyIter );
	}
	if( IsntPGPError( err ) )
	{
		CWarningAlert::Display( kWACautionAlertType, kWAOKStyle,
						kKeyInfoStringListID,
						shared ?	kKeyRejoinedStrIndex	:
									kPassphraseChangedStrIndex );
	}

	mChanged = TRUE;
	mKeyProperties.isAxiomatic = TRUE;
	mAxiomaticCheckbox->SetValue(1);
	if( shared )
	{
		GetIndString( pstr, kKeyInfoStringListID, kChangePassphraseButtonStrIndex );
		mPassphraseButton->SetDescriptor( pstr );
	}
done:
	if( IsntNull( oldPassKey ) )
		PGPFreeData( oldPassKey );
	if( IsntNull( newPhrase ) )
		PGPFreeData( newPhrase );
	return;
}

	void
CKeyPanelGeneral::ListenToMessage(MessageT inMessage, void *ioParam)
{
	Int32 value;
	
	value = *(Int32 *)ioParam;
	switch(inMessage)
	{
		case kTrustSlider:
			mTrustChanged = TRUE;
			mChanged = TRUE;
			break;
		case kEnabledCheckbox:
			mChanged = TRUE;
			break;
		case kPassphraseButton:
			ChangePassphrase();
			break;
		
		case kAxiomaticCheckbox:
			mChanged = TRUE;
			if(value != 0)
			{
				mTrustSlider->Disable();
				mTrustSlider->SetValue(kNumTrustValues - 1);
				mValidityBox->SetValue(kNumValidityValues - 1);
				mEnabledCheckbox->Hide();
			}
			else
			{
				mTrustSlider->Enable();
				mTrustSlider->SetValue(0);
				mValidityBox->SetValue(0);
				mEnabledCheckbox->Show();
			}
			mTrustSlider->Draw(nil);
			mValidityBox->Draw(nil);
			break;
			
		case kHexFingerCheckbox:
			if( value != 0 )
			{
				FindPaneByID(kHashWordsView)->Hide();
				FindPaneByID(kFingerprintBorder)->Enable();
				FindPaneByID(kFingerprintBorder)->Show();
				PGPSetPrefBoolean( gPrefRef, kPGPPrefUseHexFingerprint, TRUE );
			}
			else
			{
				FindPaneByID(kFingerprintBorder)->Hide();
				FindPaneByID(kFingerprintBorder)->Disable();
				FindPaneByID(kHashWordsView)->Show();
				PGPSetPrefBoolean( gPrefRef, kPGPPrefUseHexFingerprint, FALSE );
			}
			break;
		
		case kNextPhotoButtonID:
			SwitchPhoto( mPhotoIDIndex + 1 );
			break;

		case kPrevPhotoButtonID:
			SwitchPhoto( mPhotoIDIndex - 1 );
			break;
	}
}

	void
CKeyPanelGeneral::UpdatePhotoStatus(PGPUserIDRef userID)
{
	PGPError				err = kPGPError_NoErr;
	PGPPhotoUserIDStatus	status;
	
	status = kPGPPhotoUserIDStatus_Normal;
	
	if( mKeyProperties.isRevoked )
	{
		status = kPGPPhotoUserIDStatus_Revoked;
	}
	else if( mKeyProperties.isExpired )
	{
		status = kPGPPhotoUserIDStatus_Expired;
	}
	else if( ! mKeyProperties.isAxiomatic )
	{
		PGPInt32	temp;
		PGPValidity	validity;
		PGPBoolean	marginalIsInvalid;
		
		(void) PGPGetUserIDNumber( userID, kPGPUserIDPropValidity, &temp );
		pgpAssertNoErr( err );
		
		validity = (PGPValidity) temp;
		
		if( validity == kPGPValidity_Complete )
		{
			status = kPGPPhotoUserIDStatus_Normal;
		}
		else if( validity == kPGPValidity_Marginal )
		{
			err = PGPGetPrefBoolean( gPrefRef, kPGPPrefMarginalIsInvalid,
						&marginalIsInvalid);
				pgpAssertNoErr( err );
			
			if( marginalIsInvalid )
			{
				status = kPGPPhotoUserIDStatus_NoValidity;
			}
		}
		else
		{
			status = kPGPPhotoUserIDStatus_NoValidity;
		}
	}
	
	mPhotoUserID->SetStatus( status );
}
	
	void
CKeyPanelGeneral::SwitchPhoto(PGPUInt32 newPhotoIndex)
{
	PGPError		err;
	PGPUInt32		curPhotoIndex = 0;
	PGPUserIDRef	userID = kInvalidPGPUserIDRef;
		
	if( newPhotoIndex < 1 || newPhotoIndex > mNumPhotoIDs )
		return;
		
	mPhotoUserID->ClearPicture();
	
	err = PGPKeyIterRewindUserID( mKeyIter );
	pgpAssertNoErr( err );
	
	while( IsntPGPError( err ) )
	{
		err = PGPKeyIterNextUserID( mKeyIter, &userID );
		if( IsntPGPError( err ) )
		{
			PGPBoolean	isAttributeID;
			
			err = PGPGetUserIDBoolean( userID, kPGPUserIDPropIsAttribute,
							&isAttributeID );
			pgpAssertNoErr( err );
			
			if( IsntPGPError( err ) && isAttributeID )
			{
				PGPInt32	attributeType;
				
				err = PGPGetUserIDNumber( userID,
							kPGPUserIDPropAttributeType, &attributeType );
				pgpAssertNoErr( err );
				
				if( IsntPGPError( err ) &&
					isAttributeID == kPGPAttribute_Image )
				{
					++curPhotoIndex;
					
					if( curPhotoIndex == newPhotoIndex )
					{
						break;
					}
				}
			}
		}		
	}

	if( IsntPGPError( err ) &&
		PGPUserIDRefIsValid( userID ) )
	{
		char	temp;
		PGPSize	bufferSize;
		
		// get length of photo id buffer
		(void) PGPGetUserIDStringBuffer( userID,
					kPGPUserIDPropAttributeData, 1, &temp, &bufferSize );

		if( bufferSize > 0 )
		{
			void	*buffer;
			
			buffer = PGPNewData(gPGPMemoryMgr, bufferSize, 0);
			if( IsntNull( buffer ) )
			{
				err = PGPGetUserIDStringBuffer( userID,
							kPGPUserIDPropAttributeData,
							bufferSize, (char *) buffer, &bufferSize );
				if( IsntPGPError( err ) )
				{
					err = MacErrorToPGPError( mPhotoUserID->SetPhotoData(
								gPGPContext, buffer, bufferSize, FALSE ) );
					pgpAssertNoErr( err );
					
					if( IsntPGPError( err ) )
					{
						UpdatePhotoStatus( userID );
					}
				}
				
				PGPFreeData( buffer );
			}
			else
			{
				err = kPGPError_OutOfMemory;
			}
			
			pgpAssertNoErr( err );
		}
	}
	
	mPhotoIDIndex = newPhotoIndex;
	
	if( mPhotoIDIndex == 1 )
	{
		mPrevPhotoButton->Disable();
	}
	else
	{
		mPrevPhotoButton->Enable();
	}
	
	if( mPhotoIDIndex == mNumPhotoIDs )
	{
		mNextPhotoButton->Disable();
	}
	else
	{
		mNextPhotoButton->Enable();
	}
}

	void
CKeyPanelGeneral::SavePanel(void)
{
	PGPError	err;
	uchar		trust;
	PGPValidity	validity;
	
	if( HasChanged() &&
		mKeyCanBeModified &&
		! mKeyProperties.isExpired &&
		! mKeyProperties.isRevoked)
	{
		PGPBoolean	axiomatic = ( mAxiomaticCheckbox->GetValue() != 0 );
		
		if(axiomatic && !mKeyProperties.isAxiomatic)
		{
			if( mKeyProperties.isDisabled )
			{
				/* Enable key before setting axiomatic trust */
				
				err = PGPEnableKey(mKey);
				pgpAssertNoErr(err);
			}
			
			err = PGPSetKeyAxiomatic(mKey, PGPOLastOption(
							PGPGetKeyContext( mKey ) ) );
			pgpAssertNoErr(err);
		}
		else
		{
			if(!axiomatic && mKeyProperties.isAxiomatic)
			{
				err = PGPUnsetKeyAxiomatic(mKey);
				pgpAssertNoErr(err);
			}
			else if(mTrustSlider->IsEnabled())
			{
				err = PGPGetPrimaryUserIDValidity(mKey, &validity);
				pgpAssert(!err);
				switch(mTrustSlider->GetValue())
				{
					default:
					case 0:
						trust = kPGPKeyTrust_Never;
						break;
					case 1:
						trust = kPGPKeyTrust_Marginal;
						break;
					case 2:
						trust = kPGPKeyTrust_Complete;
						break;
					case 3:
						trust = kPGPKeyTrust_Complete;
						break;
				}
				err = PGPSetKeyTrust(mKey, trust);
				pgpAssert(!err);
				if(mTrustChanged && ((validity == kPGPValidity_Invalid) ||
					(validity == kPGPValidity_Unknown)))
				{
					CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
									kKeyInfoStringListID, kInvalidKeySetTrustID);
				}
			}

			if( mEnabledCheckbox->GetValue() != 0 )
			{
				err = PGPEnableKey(mKey);
				pgpAssert(!err);
			}
			else
			{
				err = PGPDisableKey(mKey);
				pgpAssert(!err);
			}	
		}	
	}
}

#pragma mark --- CNewSubkeyDialog ---

class CNewSubkeyDialog : public LGADialog
{
public:

	enum { class_ID = 'NSub' };
	
					CNewSubkeyDialog(LStream *inStream);
	virtual			~CNewSubkeyDialog(void);

	void			GetParams(PGPTime *startDate, PGPTime *expirationDate,
							PGPInt32 *keyBits);
	virtual void	ListenToMessage(MessageT inMessage, void *ioParam);
	
protected:

	virtual void	FinishCreateSelf(void);

private:

	enum
	{
		kKeyBitsEditTextPaneID		= 'eBit',
		kStartDateEditFieldPaneID	= 'eStr',
		kStartDateArrowsPaneID		= 'aStr',
		kExpDateEditFieldPaneID		= 'eExp',
		kExpDateArrowsPaneID		= 'aExp',
		kNeverExpiresButtonPaneID	= 'bNvr',
		
		cmd_ChangeStartDate			= kStartDateArrowsPaneID,
		cmd_ChangeExpirationDate	= kExpDateArrowsPaneID
	};
	
	LEditText				*mKeyBitsEditField;
	CGADurationEditField	*mStartDateEditField;
	CGADurationEditField	*mExpDateEditField;
	Int32					mLastStartDateArrowsValue;
	Int32					mLastExpDateArrowsValue;
};

CNewSubkeyDialog::CNewSubkeyDialog(LStream *inStream)
		: LGADialog(inStream)
{
}

CNewSubkeyDialog::~CNewSubkeyDialog(void)
{
}

	void
CNewSubkeyDialog::FinishCreateSelf(void)
{
	UInt32			expiration;
	DateTimeRec		dtRec;
	LLittleArrows	*arrowsObj;
	UInt32			now;
	
	LGADialog::FinishCreateSelf();
	
	mKeyBitsEditField = (LEditText *) FindPaneByID( kKeyBitsEditTextPaneID );
	pgpAssertAddrValid( mKeyBitsEditField, VoidAlign );
	
	mKeyBitsEditField->SelectAll();

	mStartDateEditField = (CGADurationEditField *)
								FindPaneByID( kStartDateEditFieldPaneID );
	pgpAssertAddrValid( mStartDateEditField, VoidAlign );

	mExpDateEditField = (CGADurationEditField *)
								FindPaneByID( kExpDateEditFieldPaneID );
	pgpAssertAddrValid( mExpDateEditField, VoidAlign );
	
	GetDateTime( &now );

	mStartDateEditField->SetDurationType( kDateDurationType );					
	mStartDateEditField->SetDurationValue( now );					

	/* Set expiration to +1 year */
	SecondsToDate( now, &dtRec );
	dtRec.year++;
	DateToSeconds( &dtRec, &expiration );
	
	mExpDateEditField->SetDurationType( kDateDurationType );					
	mExpDateEditField->SetDurationValue( expiration );					

	arrowsObj = (LLittleArrows *) FindPaneByID( kStartDateArrowsPaneID );
	pgpAssertAddrValid( arrowsObj, VoidAlign );
	
	arrowsObj->AddListener( this );
	mLastStartDateArrowsValue = arrowsObj->GetValue();

	arrowsObj = (LLittleArrows *) FindPaneByID( kExpDateArrowsPaneID );
	pgpAssertAddrValid( arrowsObj, VoidAlign );
	
	arrowsObj->AddListener( this );
	mLastExpDateArrowsValue = arrowsObj->GetValue();
	
}

	void
CNewSubkeyDialog::ListenToMessage(
	MessageT 	inMessage,
	void 		*ioParam)
{
	Int32	newValue;
	
	switch( inMessage )
	{
		case cmd_ChangeStartDate:
			newValue = *(Int32 *) ioParam;
			mStartDateEditField->AdjustValue(
							newValue < mLastStartDateArrowsValue ?
							kDownAdjust : kUpAdjust );
			mLastStartDateArrowsValue = newValue;
			break;
			
		case cmd_ChangeExpirationDate:
			newValue = *(Int32 *) ioParam;
			mExpDateEditField->AdjustValue(
							newValue < mLastExpDateArrowsValue ?
							kDownAdjust : kUpAdjust );
			mLastExpDateArrowsValue = newValue;
			break;
	}
}

	void
CNewSubkeyDialog::GetParams(
	PGPTime		*startDate,
	PGPTime		*expirationDate,
	PGPInt32	*keyBits)
{
	LRadioButton	*radio;
	
	*keyBits 	= mKeyBitsEditField->GetValue();
	*startDate 	= PGPTimeFromMacTime(mStartDateEditField->GetDurationValue());
	
	radio = (LRadioButton *) FindPaneByID( kNeverExpiresButtonPaneID );
	if( radio->GetValue() != 0)
	{
		*expirationDate = 0;
	}
	else
	{
		*expirationDate = PGPTimeFromMacTime(
								mExpDateEditField->GetDurationValue() );
	}
}
	
#pragma mark --- CSubkeyTable ---

typedef struct SubkeyTableItem
{
	PGPSubKeyRef		subKeyRef;
	PGPInt32			keyBits;
	PGPTime				validFromDate;
	PGPTime				expirationDate;
	PGPBoolean			isExpired;
	PGPBoolean			isRevoked;
	
} SubkeyTableItem;

class	CSubkeyTable	:	public CColumnTable
{
public:
	enum { class_ID = 'SubK' };
						CSubkeyTable(LStream *inStream)
							: CColumnTable(inStream) { };
	virtual				~CSubkeyTable() { };
	virtual Boolean		GetCellDrawData(
								STableCell			inCell,
								ResIDT				&iconID,
								Int16				&indentLevel,
								Str255				data,
								StyleParameter		&style );
private:
};

	Boolean
CSubkeyTable::GetCellDrawData(
	STableCell			inCell,
	ResIDT				&iconID,
	Int16				&indentLevel,
	Str255				data,
	StyleParameter		&style )
{
	SubkeyTableItem		tableItem;
	TableIndexT			col;
	Uint32				dataSize = sizeof(SubkeyTableItem);
	
	indentLevel;	style;
	col = inCell.col;
	inCell.col = 1;
	GetCellData( inCell, &tableItem, dataSize );
	switch( col )
	{
		case 1:
			if( tableItem.isRevoked )
				iconID = kDSARevokedKeyID;
			else if( tableItem.isExpired )
				iconID = kDSAExpiredKeyID;
			else
				iconID = kDSAKeyIconID;
			DateString( PGPTimeToMacTime( tableItem.validFromDate ),
				shortDate, data, NULL );	
			break;
		case 2:
			if( tableItem.expirationDate != 0 )
				DateString( PGPTimeToMacTime( tableItem.expirationDate ),
					shortDate, data, NULL );
			else
				GetIndString( data, kKeyInfoStringListID,
					kNeverExpiresID );
			break;
		case 3:
			NumToString( tableItem.keyBits, data );
			break;
	}
	return FALSE;
}

#pragma mark --- CKeyPanelSubkeys ---

class CKeyPanelSubkeys : public CKeyInfoPanelView,
						 public LListener
{
public:

	enum { class_ID = 'vKIS' };

					CKeyPanelSubkeys(LStream *inStream);
	virtual			~CKeyPanelSubkeys(void);
	
	virtual void	ListenToMessage(MessageT inMessage, void *ioParam);
	virtual void	SetInfo(PGPKeySetRef keySet, PGPKeyRef theKey,
							PGPBoolean keyCanBeModified);
	virtual void	Show(void);

protected:

	virtual	void	FinishCreateSelf(void);
	
private:

	enum
	{
		kNewButtonPaneID		= 'bNew',
		kRemoveButtonPaneID		= 'bRem',
		kRevokeButtonPaneID		= 'bRev',
		kSubkeyTablePaneID		= 'tSub',
		
		cmd_NewSubkey			= kNewButtonPaneID,
		cmd_RevokeSubkey		= kRevokeButtonPaneID,
		cmd_RemoveSubkey		= kRemoveButtonPaneID,
		cmd_ChangedSubkey		= kSubkeyTablePaneID
	};
	
	LPushButton				*mNewButton;
	LPushButton				*mRevokeButton;
	LPushButton				*mRemoveButton;
	CSubkeyTable			*mSubkeyTable;
	PGPSharedKeyProperties	mKeyProperties;
	static ulong			sKeyGenStartTicks;
	
	void			AdjustButtons(void);
	void			NewSubKey(void);
	void			RemoveSubKey(void);
	void			RevokeSubKey(void);

	static PGPError	GenerateKeyCallback(PGPContextRef context,
							PGPEvent *event, PGPUserValue userValue);
};

ulong	CKeyPanelSubkeys::sKeyGenStartTicks = 0;

CKeyPanelSubkeys::CKeyPanelSubkeys(LStream *inStream)
	: CKeyInfoPanelView( inStream )
{
}

CKeyPanelSubkeys::~CKeyPanelSubkeys()
{
}

	void
CKeyPanelSubkeys::FinishCreateSelf(void)
{
	CKeyInfoPanelView::FinishCreateSelf();

	mNewButton = (LPushButton *) FindPaneByID( kNewButtonPaneID );
	pgpAssertAddrValid( mNewButton, VoidAlign );

	mRevokeButton = (LPushButton *) FindPaneByID( kRevokeButtonPaneID );
	pgpAssertAddrValid( mRevokeButton, VoidAlign );

	mRemoveButton = (LPushButton *) FindPaneByID( kRemoveButtonPaneID );
	pgpAssertAddrValid( mRemoveButton, VoidAlign );
	
	mSubkeyTable = (CSubkeyTable *) FindPaneByID( kSubkeyTablePaneID );
	pgpAssertAddrValid( mSubkeyTable, VoidAlign );

	mNewButton->AddListener( this );
	mRevokeButton->AddListener( this );
	mRemoveButton->AddListener( this );
	mSubkeyTable->AddListener( this );
}

	PGPError
CKeyPanelSubkeys::GenerateKeyCallback(
	PGPContextRef 	context,
	PGPEvent 		*event,
	PGPUserValue 	userValue)
{
	PGPError			err = kPGPError_NoErr;
	
	if( event->type == kPGPEvent_KeyGenEvent )
	{
		ulong				timeNow;
		CPGPStDialogHandler	*dialogHandler;
		CGAProgressDialog	*dialogObj;
		static ulong		lastEventTime;

		dialogHandler 	= (CPGPStDialogHandler *) userValue;
		dialogObj		= (CGAProgressDialog *) dialogHandler->GetDialog();
		
		timeNow = LMGetTicks();
		if( ! dialogObj->IsVisible() )
		{
			// Show progress dialog after 3 seconds
			if( timeNow - sKeyGenStartTicks > 60L * 3 )
			{
				dialogObj->Show();
			}
		}
		
		if( dialogObj->IsVisible() )
		{
			if( timeNow - lastEventTime > 10 )
			{
				MessageT	theMessage;
				
				lastEventTime = timeNow;
				
				theMessage = dialogHandler->DoDialog();
				if( theMessage == msg_Cancel )
					err = kPGPError_UserAbort;
			}
		}
	}
	
	return( err );
}

	void
CKeyPanelSubkeys::NewSubKey(void)
{
	CPGPStDialogHandler	dialogHandler( kNewSubkeyDialogResID, this );
	CNewSubkeyDialog	*theDialog;	
	MessageT			hitMessage;
	SubkeyTableItem		tableItem;
	
	mSubkeyTable->UnselectAllCells();
	
	theDialog = (CNewSubkeyDialog *) dialogHandler.GetDialog();
	theDialog->Show();
	
	do
	{
		hitMessage = dialogHandler.DoDialog();
		if( hitMessage == msg_OK )
		{
			Boolean	addSubKey = TRUE;
			
			pgpClearMemory( &tableItem, sizeof( tableItem ) );
			
			theDialog->GetParams( &tableItem.validFromDate,
						&tableItem.expirationDate, &tableItem.keyBits );
			
			if( tableItem.keyBits < 768 || tableItem.keyBits > 4096 )
			{
				SysBeep( 1 );
				CWarningAlert::Display( kWAStopAlertType, kWAOKStyle,
						kKeyInfoStringListID, kSubkeyWrongSizeMsgStrIndex );
						
				addSubKey = FALSE;
			}
			else
			{
		#if PGP_BUSINESS_SECURITY
				PGPUInt32	minAdminKeySize;
				PGPError	err;
				
				err = PGPGetPrefNumber( gAdminPrefRef,
									kPGPPrefMinimumKeySize, &minAdminKeySize);
				pgpAssertNoErr( err );
				if( tableItem.keyBits < minAdminKeySize )
				{
					Str255	adminKeySizeStr;
					
					::NumToString(minAdminKeySize, adminKeySizeStr);
					(void) CWarningAlert::Display(kWACautionAlertType,
									kWAOKStyle,
									kKeyInfoStringListID,
									kAdminKeySizeErrorStrIndex,
									adminKeySizeStr );
					addSubKey = FALSE;
				}
		#endif
			}
			
			if( addSubKey && tableItem.expirationDate != 0 )
			{
				if( tableItem.expirationDate <= tableItem.validFromDate )
				{
					SysBeep( 1 );
					CWarningAlert::Display( kWAStopAlertType, kWAOKStyle,
								kKeyInfoStringListID,
								kBadSubkeyExpirationMsgStrIndex );
							
					addSubKey = FALSE;
				}
				else if( tableItem.expirationDate <= PGPGetTime() )
				{
					SysBeep( 1 );
					if( CWarningAlert::Display( kWAStopAlertType,
								kWAOKCancelStyle,
								kKeyInfoStringListID,
								kSubkeyAlreadyExpiredMsgStrIndex ) != msg_OK )
					{
						addSubKey = FALSE;
					}
				}
			}
			
			if( ! addSubKey )
				hitMessage = msg_Cancel;
		}
		
	} while( hitMessage != msg_OK && hitMessage != msg_Cancel );
	
	theDialog->Hide();
	
	if( hitMessage == msg_OK )
	{
		PGPError			err;
		CSecureCString256	passphrase;
		PGPByte				*passKey = NULL;
		PGPSize				passKeySize;
		PGPBoolean			split = FALSE;
		
		err = GetPassForKey( mKey, &split, passphrase,
								&passKey, &passKeySize );

		if( IsntPGPError( err ) )
		{
			PGPUInt32	neededBits;
			PGPUInt32	currentBits;
			PGPBoolean	fastGen;
	
			err = PGPGetPrefBoolean(gPrefRef, kPGPPrefFastKeyGen, &fastGen);
			pgpAssertNoErr( err );

			neededBits = PGPGetKeyEntropyNeeded( gPGPContext,
								PGPOKeyGenParams( gPGPContext, 
									kPGPPublicKeyAlgorithm_ElGamal,
									tableItem.keyBits ),
								PGPOKeyGenFast( gPGPContext, fastGen ),
								PGPOLastOption( gPGPContext ) );
			
			currentBits = PGPGlobalRandomPoolGetEntropy();
			if( neededBits > currentBits )
			{
				// If we're displaying the dialog, get at lest 500 bits so
				// the dislog remains visible long enough to see it.
				neededBits = neededBits - currentBits;
				if( neededBits < 500 )
					neededBits = 500;
					
				UDesktop::Deactivate();
					err = PGPCollectRandomDataDialog( gPGPContext, neededBits,
								PGPOLastOption( gPGPContext ) );
				UDesktop::Activate();
			}
			
			if( IsntPGPError( err ) )
			{
				UInt32				expirationDays = 0;
				CPGPStDialogHandler	dialogHandler(kProgressDialogResID, this);
				CGAProgressDialog	*dialogObj;
				Str255				caption;
				
				sKeyGenStartTicks = LMGetTicks();
				
				dialogObj = (CGAProgressDialog *) dialogHandler.GetDialog();
				pgpAssertAddrValid( dialogObj, VoidAlign );
				
				GetIndString( caption, kKeyInfoStringListID,
						kCreatingSubkeyStrIndex );
						
				dialogObj->SetCaption( caption );
				dialogObj->SetIndeterminateFlag( TRUE );
				
				if( tableItem.expirationDate != 0 )
				{
					expirationDays = ( ( tableItem.expirationDate -
										tableItem.validFromDate ) / 86400 ) + 1;
				}
										
				err = PGPGenerateSubKey( gPGPContext, &tableItem.subKeyRef,
								PGPOKeyGenMasterKey( gPGPContext, mKey ),
								PGPOKeyGenParams( gPGPContext, 
									kPGPPublicKeyAlgorithm_ElGamal,
									tableItem.keyBits ),
								PGPOCreationDate( gPGPContext,
									tableItem.validFromDate ),
								PGPOExpiration( gPGPContext, expirationDays ),
								PGPOKeyGenFast( gPGPContext, fastGen ),
								split ?
									PGPOPasskeyBuffer( gPGPContext,
										passKey, passKeySize ) :
									PGPOPassphrase( gPGPContext, passphrase ),
								PGPOEventHandler(gPGPContext, 
									GenerateKeyCallback,
									(PGPUserValue) &dialogHandler ),
								PGPOLastOption( gPGPContext ) );
			}
							
			if( IsntPGPError( err ) )
			{
				mSubkeyTable->InsertRows( 1, LArray::index_Last,
					&tableItem, sizeof( tableItem ), TRUE );
				AdjustButtons();
				mChanged = TRUE;
			}
		}
		if( IsntNull( passKey ) )
			PGPFreeData( passKey );
	}
}

	void
CKeyPanelSubkeys::RemoveSubKey(void)
{
	STableCell	theCell;
	
	theCell = mSubkeyTable->GetFirstSelectedCell();
	if( ! theCell.IsNullCell() )
	{
		Boolean		doRemove = FALSE;
		TableIndexT	numRows;
		TableIndexT	numColumns;
		short		strIndex;
		
		mSubkeyTable->UnselectAllCells();
		mSubkeyTable->GetTableSize( numRows, numColumns );
		
		if( numRows == 1 )
			strIndex = kConfirmRemoveLastSubkeyStrIndex;
		else
			strIndex = kConfirmRemoveSubkeyStrIndex;
		
		if( CWarningAlert::Display( kWACautionAlertType,
					kWACancelOKStyle, kKeyInfoStringListID,
					strIndex ) == msg_OK )
		{
			SubkeyTableItem		tableItem;
			Uint32				dataSize = sizeof( tableItem );
			PGPError			err;
			
			mSubkeyTable->GetCellData( theCell, &tableItem, dataSize );
			
			err = PGPRemoveSubKey( tableItem.subKeyRef );
			if( IsntPGPError( err ) )
			{
				mSubkeyTable->RemoveRows( 1, theCell.row, TRUE );
				AdjustButtons();
				mChanged = TRUE;
			}
		}
	}
}

	void
CKeyPanelSubkeys::RevokeSubKey(void)
{
	STableCell	theCell;
	
	theCell = mSubkeyTable->GetFirstSelectedCell();
	if( ! theCell.IsNullCell() )
	{
		mSubkeyTable->UnselectAllCells();
		
		if( CWarningAlert::Display( kWACautionAlertType,
					kWACancelOKStyle, kKeyInfoStringListID,
					kConfirmRevokeSubkeyStrIndex ) == msg_OK )
		{
			SubkeyTableItem		tableItem;
			Uint32				dataSize = sizeof( tableItem );
			PGPBoolean			needsPassphrase = FALSE,
								split = FALSE;
			CSecureCString256	passphrase;
			PGPError			err;
			PGPByte				*passKey = NULL;
			PGPSize				passKeySize;
			
			mSubkeyTable->GetCellData( theCell, &tableItem, dataSize );
			
			err = GetPassForKey( mKey, &split, passphrase,
								&passKey, &passKeySize );			
			if( IsntPGPError( err ) )
			{
				CPGPKeys::TheApp()->GetMinimumRandomData();
				
				err = PGPRevokeSubKey( tableItem.subKeyRef, split ?
						PGPOPasskeyBuffer( gPGPContext, passKey, passKeySize ) :
						PGPOPassphrase( gPGPContext, passphrase ),
						PGPOLastOption( gPGPContext ) );
				pgpAssertNoErr( err );
				
				if( IsntPGPError( err ) )
				{
					tableItem.isRevoked = TRUE;
					
					mSubkeyTable->SetCellData( theCell, &tableItem, sizeof(tableItem));
					mSubkeyTable->Refresh();
					AdjustButtons();
					mChanged = TRUE;
				}
			}
			if( IsntNull( passKey ) )
				PGPFreeData( passKey );
		}
	}
}

	void
CKeyPanelSubkeys::ListenToMessage(
	MessageT 	inMessage,
	void 		*ioParam)
{
	switch( inMessage )
	{
		case cmd_NewSubkey:
			NewSubKey();
			break;
		
		case cmd_RevokeSubkey:
			RevokeSubKey();
			break;
			
		case cmd_RemoveSubkey:
			RemoveSubKey();
			break;
			
		case coltable_SelectChange:
			AdjustButtons();
			break;
	}
}

	void
CKeyPanelSubkeys::Show(void)
{
	mSubkeyTable->UnselectAllCells();
	
	CKeyInfoPanelView::Show();
}

	void
CKeyPanelSubkeys::AdjustButtons(void)
{
	Boolean		enableNewButton 	= FALSE;
	Boolean		enableRemoveButton 	= FALSE;
	Boolean		enableRevokeButton 	= FALSE;
	
	if( mKeyCanBeModified )
	{
		Boolean				haveSelection = FALSE;
		STableCell			theCell;
		SubkeyTableItem		tableItem;
		
		theCell = mSubkeyTable->GetFirstSelectedCell();
		if( ! theCell.IsNullCell() )
		{
			Uint32	dataSize = sizeof( tableItem );

			haveSelection = TRUE;
			
			mSubkeyTable->GetCellData( theCell, &tableItem, dataSize );
		}
		
		if( mKeyProperties.isSecret &&
			! mKeyProperties.isExpired &&
			! mKeyProperties.isRevoked )
		{
			enableNewButton = TRUE;
			
			if( haveSelection && ! tableItem.isRevoked )
			{
				enableRevokeButton = TRUE;
			}
		}
		
		enableRemoveButton = mKeyProperties.isSecret && haveSelection;
	}

#if PGP_BUSINESS_SECURITY

	PGPBoolean	allowKeyGen;
	PGPError	err;
	
	err = PGPGetPrefBoolean( gAdminPrefRef, kPGPPrefAllowKeyGen,
				&allowKeyGen );
	pgpAssertNoErr(err);
	if( IsntPGPError( err ) && ! allowKeyGen )
	{
		enableNewButton = FALSE;
	}
	
#endif	// PGP_BUSINESS_SECURITY

	if( enableNewButton )
	{
		mNewButton->Enable();
	}
	else
	{
		mNewButton->Disable();
	}

	if( enableRemoveButton )
	{
		mRemoveButton->Enable();
	}
	else
	{
		mRemoveButton->Disable();
	}

	if( enableRevokeButton )
	{
		mRevokeButton->Enable();
	}
	else
	{
		mRevokeButton->Disable();
	}
}

	void
CKeyPanelSubkeys::SetInfo(
	PGPKeySetRef 	keySet,
	PGPKeyRef 		key,
	PGPBoolean 		keyCanBeModified)
{
	PGPError		err;
	PGPSubKeyRef	subKeyRef;
	
	CKeyInfoPanelView::SetInfo( keySet, key, keyCanBeModified );
	
	err = PGPSharedGetKeyProperties( key, NULL, &mKeyProperties );
	pgpAssertNoErr( err );

	err = PGPKeyIterRewindSubKey( mKeyIter );
	pgpAssertNoErr( err );
	
	while( IsntPGPError( PGPKeyIterNextSubKey( mKeyIter, &subKeyRef ) ) )
	{
		SubkeyTableItem	tableItem;
		
		pgpClearMemory( &tableItem, sizeof( tableItem ) );
		
		tableItem.subKeyRef = subKeyRef;
		
		err = PGPGetSubKeyNumber(subKeyRef, kPGPKeyPropBits,
						&tableItem.keyBits);
		pgpAssertNoErr( err );
		
		err = PGPGetSubKeyTime( subKeyRef, kPGPKeyPropCreation,
						&tableItem.validFromDate );
		pgpAssertNoErr( err );

		err = PGPGetSubKeyTime( subKeyRef, kPGPKeyPropExpiration,
						&tableItem.expirationDate );
		pgpAssertNoErr( err );
		
		err = PGPGetSubKeyBoolean( subKeyRef, kPGPKeyPropIsRevoked,
						&tableItem.isRevoked );
		pgpAssertNoErr( err );

		err = PGPGetSubKeyBoolean( subKeyRef, kPGPKeyPropIsExpired,
						&tableItem.isExpired );
		pgpAssertNoErr( err );
						
		mSubkeyTable->InsertRows( 1, LArray::index_Last,
			&tableItem, sizeof( SubkeyTableItem ), TRUE );
	}
	
	if( ! mKeyProperties.isSecret )
	{
		LPane	*pane;
		
		// Hide explanatory text at the bottom of the panel.
		
		pane = FindPaneByID( 1 );
		pgpAssertAddrValid( pane, VoidAlign );
		
		pane->Hide();

		pane = FindPaneByID( 2 );
		pgpAssertAddrValid( pane, VoidAlign );
		
		pane->Hide();

		pane = FindPaneByID( 3 );
		pgpAssertAddrValid( pane, VoidAlign );
		
		pane->Hide();
	}
	
	AdjustButtons();
}

#pragma mark --- CADKTable ---

typedef struct ADKTableItem
{
	PGPKeyRef	keyRef;
	PGPKeyID	keyID;	/* Valid if keyRef is invalid */
	PGPByte		adkClass;
} ADKTableItem;

class	CADKTable	:	public CColumnTable
{
public:
	enum { class_ID = 'tADK' };
						CADKTable(LStream *inStream)
							: CColumnTable(inStream) { };
	virtual				~CADKTable() { };
	virtual Boolean		GetCellDrawData(
								STableCell			inCell,
								ResIDT				&iconID,
								Int16				&indentLevel,
								Str255				data,
								StyleParameter		&style );
private:
};

	Boolean
CADKTable::GetCellDrawData(
	STableCell			inCell,
	ResIDT				&iconID,
	Int16				&indentLevel,
	Str255				data,
	StyleParameter		&style )
{
	ADKTableItem		tableItem;
	TableIndexT			col;
	UInt32				dataSize = sizeof(ADKTableItem);
	PGPError			err;
	
	style;
	col = inCell.col;
	inCell.col = 1;
	GetCellData( inCell, &tableItem, dataSize );
	switch( col )
	{
		case 1:
			iconID = kDSAKeyIconID;
			if( PGPKeyRefIsValid( tableItem.keyRef ) )
			{
				PGPSize		len;
				PGPInt32	algorithm;
				
				err = PGPGetPrimaryUserIDNameBuffer( tableItem.keyRef,
					sizeof( Str255 ) - 1, (char *) &data[1], &len );
				pgpAssertNoErr( err );
				
				data[0] = len;
				
				PGPGetKeyNumber( tableItem.keyRef,
					kPGPKeyPropAlgID, &algorithm );
				if( algorithm != kPGPPublicKeyAlgorithm_DSA )
					iconID = kRSAKeyIconID;
			}
			else
			{
				Str255	tempStr;
				char	keyIDString[ kPGPMaxKeyIDStringSize ];
				
				err = PGPGetKeyIDString( &tableItem.keyID,
							kPGPKeyIDString_Abbreviated, keyIDString );
				pgpAssertNoErr( err );
				
				CToPString( keyIDString, tempStr );
				
				GetIndString( data, kKeyInfoStringListID, kUnknownMRKID );
				PrintPString( data, data, tempStr );
			}
			break;
		case 2:
			iconID = ( ( tableItem.adkClass & 0x80 ) != 0 ) ?
						kMRKActiveIconID : kKeyAttrOffIconID;
			indentLevel = 1;
			break;
	}
	return FALSE;
}

#pragma mark --- CKeyPanelADK ---

class CKeyPanelADK : public CKeyInfoPanelView
{
public:

	enum { class_ID = 'vKIA' };

					CKeyPanelADK(LStream *inStream);
	virtual			~CKeyPanelADK(void);
	
	virtual void	SetInfo(PGPKeySetRef keySet, PGPKeyRef theKey,
							PGPBoolean keyCanBeModified);

protected:

	virtual	void	FinishCreateSelf(void);
	
private:

	enum
	{
		kADKTablePaneID = 'ADKs'
	};
	
	CADKTable		*mADKTable;
};


CKeyPanelADK::CKeyPanelADK(LStream *inStream)
	: CKeyInfoPanelView( inStream )
{
}

CKeyPanelADK::~CKeyPanelADK()
{
}

	void
CKeyPanelADK::FinishCreateSelf(void)
{
	CKeyInfoPanelView::FinishCreateSelf();
	
	mADKTable = (CADKTable *) FindPaneByID( kADKTablePaneID );
	pgpAssertAddrValid( mADKTable, VoidAlign );
}

	void
CKeyPanelADK::SetInfo(
	PGPKeySetRef 	keySet,
	PGPKeyRef 		key,
	PGPBoolean 		keyCanBeModified)
{
	PGPError		err;
	PGPUInt32		numADKeys;
	
	CKeyInfoPanelView::SetInfo( keySet, key, keyCanBeModified );
	
	err = PGPCountAdditionalRecipientRequests( key, &numADKeys );
	if( IsntPGPError( err ) && numADKeys > 0 )
	{
		PGPUInt32		adkIndex;
		
		for( adkIndex = 0; adkIndex < numADKeys; adkIndex++ )
		{
			ADKTableItem	tableItem;
			Boolean			addTableItem = TRUE;
			
			pgpClearMemory( &tableItem, sizeof( tableItem ) );
			
			err = PGPGetIndexedAdditionalRecipientRequestKey( key, keySet,
						adkIndex, &tableItem.keyRef, &tableItem.keyID,
						&tableItem.adkClass );
			if( IsPGPError( err ) )
				addTableItem = FALSE;
			
			if( addTableItem )
			{
				mADKTable->InsertRows( 1, LArray::index_Last,
					&tableItem, sizeof( tableItem ), TRUE );
			}
		}
	}
}

#pragma mark --- CRevokerTable ---

typedef struct RevokerTableItem
{
	PGPKeyRef	keyRef;
	PGPKeyID	keyID;	/* Valid if keyRef is invalid */
	
} RevokerTableItem;

class	CRevokerTable	:	public CColumnTable
{
public:
	enum { class_ID = 'tRev' };
						CRevokerTable(LStream *inStream)
							: CColumnTable(inStream) { };
	virtual				~CRevokerTable() { };
	virtual Boolean		GetCellDrawData(
								STableCell			inCell,
								ResIDT				&iconID,
								Int16				&indentLevel,
								Str255				data,
								StyleParameter		&style );
private:
};

	Boolean
CRevokerTable::GetCellDrawData(
	STableCell			inCell,
	ResIDT				&iconID,
	Int16				&indentLevel,
	Str255				data,
	StyleParameter		&style )
{
	RevokerTableItem	tableItem;
	TableIndexT			col;
	Uint32				dataSize = sizeof(RevokerTableItem);
	PGPError			err;
	
	indentLevel;	style;
	col = inCell.col;
	inCell.col = 1;
	GetCellData( inCell, &tableItem, dataSize );
	switch( col )
	{
		case 1:
			iconID = kDSAKeyIconID;
			if( PGPKeyRefIsValid( tableItem.keyRef ) )
			{
				PGPSize		len;
				PGPInt32	algorithm;
				
				err = PGPGetPrimaryUserIDNameBuffer( tableItem.keyRef,
					sizeof( Str255 ) - 1, (char *) &data[1], &len );
				pgpAssertNoErr( err );
				
				data[0] = len;
				
				PGPGetKeyNumber( tableItem.keyRef, kPGPKeyPropAlgID,
											&algorithm );
				
				if( algorithm != kPGPPublicKeyAlgorithm_DSA )
					iconID = kRSAKeyIconID;
			}
			else
			{
				Str255	tempStr;
				char	keyIDString[ kPGPMaxKeyIDStringSize ];
				
				err = PGPGetKeyIDString( &tableItem.keyID,
							kPGPKeyIDString_Abbreviated, keyIDString );
				pgpAssertNoErr( err );
				
				CToPString( keyIDString, tempStr );
				
				GetIndString( data, kKeyInfoStringListID, kUnknownMRKID );
				PrintPString( data, data, tempStr );
			}
			break;
	}
	return FALSE;
}

#pragma mark --- CKeyPanelRevoker ---

class CKeyPanelRevoker : public CKeyInfoPanelView
{
public:

	enum { class_ID = 'vKIR' };

					CKeyPanelRevoker(LStream *inStream);
	virtual			~CKeyPanelRevoker(void);
	
	virtual void	SetInfo(PGPKeySetRef keySet, PGPKeyRef theKey,
							PGPBoolean keyCanBeModified);

protected:

	virtual	void	FinishCreateSelf(void);
	
private:

	enum
	{
		kRevokerTablePaneID = 'Rev '
	};
	
	CRevokerTable		*mRevokerTable;
};


CKeyPanelRevoker::CKeyPanelRevoker(LStream *inStream)
	: CKeyInfoPanelView( inStream )
{
}

CKeyPanelRevoker::~CKeyPanelRevoker()
{
}

	void
CKeyPanelRevoker::FinishCreateSelf(void)
{
	CKeyInfoPanelView::FinishCreateSelf();
	
	mRevokerTable = (CRevokerTable *) FindPaneByID( kRevokerTablePaneID );
	pgpAssertAddrValid( mRevokerTable, VoidAlign );
}

	void
CKeyPanelRevoker::SetInfo(
	PGPKeySetRef 	keySet,
	PGPKeyRef 		key,
	PGPBoolean 		keyCanBeModified)
{
	PGPError		err;
	PGPUInt32		numRevokerKeys;
	PGPKeySetRef	mainKeySet = CPGPKeys::TheApp()->GetKeySet();
	
	CKeyInfoPanelView::SetInfo( keySet, key, keyCanBeModified );
	
	err = PGPCountRevocationKeys( key, &numRevokerKeys );
	if( IsntPGPError( err ) && numRevokerKeys > 0 )
	{
		PGPUInt32		keyIndex;
		
		for( keyIndex = 0; keyIndex < numRevokerKeys; keyIndex++ )
		{
			RevokerTableItem	tableItem;
			
			pgpClearMemory( &tableItem, sizeof( tableItem ) );
			
			err = PGPGetIndexedRevocationKey( key, keySet,
						keyIndex, &tableItem.keyRef, &tableItem.keyID );
			if( IsntPGPError( err ) )
			{
				if( ! PGPKeyRefIsValid( tableItem.keyRef ) &&
					PGPKeySetRefIsValid( mainKeySet ) &&
					mainKeySet != keySet )
				{
					/*
					** Search the main key set for the revoker key. We would
					** never really do this when processing a revoker because
					** strict fingerprint checking is required, however is
					** it adequate to do so here for display purposes.
					*/
					
					err = PGPGetKeyByKeyID( mainKeySet, &tableItem.keyID,
								kPGPPublicKeyAlgorithm_DSA, &tableItem.keyRef );
					if( IsPGPError( err ) )
						tableItem.keyRef = kInvalidPGPKeyRef;
				}
					
				mRevokerTable->InsertRows( 1, LArray::index_Last,
						&tableItem, sizeof( tableItem ), TRUE );
			}
		}
	}
}

#pragma mark --- CKeyInfoWindow ---

/*
Pass NIL for the table to OpenKeyInfo to disable redrawing the
table on close and disable all editable control in the key info
window.  This is useful for editing keyserver search keysets
for instance where you can't modify the trust, etc...
*/
	Boolean
CKeyInfoWindow::OpenKeyInfo(PGPKeyRef key, CKeyTable *table)
{
	CKeyInfoWindow *keyInfoWindow;
	short kiIndex;
	
	if(sNumKeyInfoWindows < kMaxKeyInfoWindows)
	{
		for(kiIndex = 0;kiIndex < sNumKeyInfoWindows;kiIndex++)
		{
			if(sKeyInfoWindows[kiIndex]->GetKey() == key)
			{
				sKeyInfoWindows[kiIndex]->Select();
				return TRUE;
			}
		}

		RegisterClass_( CKeyInfoWindow );
		
		keyInfoWindow = (CKeyInfoWindow *)
			LWindow::CreateWindow( kKeyInfoWindowID, CPGPKeys::TheApp() );
		sKeyInfoWindows[sNumKeyInfoWindows] = keyInfoWindow;
		keyInfoWindow->SetInfo( table, key );
		sNumKeyInfoWindows++;
		return TRUE;
	}
	else
	{
		CWarningAlert::Display(kWACautionAlertType, kWAOKStyle,
						kKeyInfoStringListID, kTooManyWindowsID);
		return FALSE;
	}
}

	void
CKeyInfoWindow::CloseAll()
{	
	while(sNumKeyInfoWindows > 0)
	{
		pgpAssertAddrValid(sKeyInfoWindows[0], VoidAlign);
		sKeyInfoWindows[0]->DoClose();
	}
	pgpAssert(sNumKeyInfoWindows == 0);
}

	PGPKeyRef
CKeyInfoWindow::GetKey()
{
	return mWindowKey;
}

CKeyInfoWindow::CKeyInfoWindow(LStream *inStream)
	:	LWindow(inStream)
{
	RegisterClass_( CKeyPanelGeneral );
	RegisterClass_( CKeyPanelSubkeys );
	RegisterClass_( CKeyPanelADK );
	RegisterClass_( CKeyPanelRevoker );
	
	RegisterClass_( CADKTable );
	RegisterClass_( CRevokerTable );
	RegisterClass_( CSubkeyTable );
	RegisterClass_( CNewSubkeyDialog );
}

CKeyInfoWindow::~CKeyInfoWindow()
{
}

	void
CKeyInfoWindow::CleanupPanel(short panelIndex)
{
	CKeyInfoPanelView	*panel;
	
	panel = (CKeyInfoPanelView *) mMultiView->RemovePanel( panelIndex );
	panel->SavePanel();
	
	delete panel;
}

	Boolean
CKeyInfoWindow::HasChanged(void)
{
	UInt16	numPanels;
	Boolean	hasChanged = FALSE;
	
	numPanels = mMultiView->GetPanelCount();
	for( UInt16 panelIndex = 1; panelIndex <= numPanels; panelIndex++ )
	{
		CKeyInfoPanelView	*panel;
		
		panel = (CKeyInfoPanelView *) mMultiView->GetPanel( panelIndex );
		if( IsntNull( panel ) )
		{
			if( panel->HasChanged() )
			{
				hasChanged = TRUE;
				break;
			}
		}
	}

	return( hasChanged );
}

	void
CKeyInfoWindow::ProcessClose(void)
{
	short	windowIndex;
	short	windowID;
	Boolean	hasChanged;
	UInt16	numPanels;
	
	hasChanged = HasChanged();
	
	numPanels = mMultiView->GetPanelCount();
	for( UInt16 panelIndex = numPanels; panelIndex >= 1; panelIndex-- )
	{
		CleanupPanel( panelIndex );
	}
	
	Hide();
	
	// Remove this Key Info window from the list
	for(windowID = 0;windowID < sNumKeyInfoWindows;windowID++)
	{
		if(sKeyInfoWindows[windowID] == this)
		{
			sKeyInfoWindows[windowID] = NULL;
			break;
		}
	}
	
	for(windowIndex = windowID + 1;
		windowIndex < sNumKeyInfoWindows;
		windowIndex++)
		sKeyInfoWindows[windowIndex-1] = sKeyInfoWindows[windowIndex];
	sNumKeyInfoWindows--;

		
	pgpAssert((hasChanged && IsntNull(mParentTable)) || !hasChanged);
	if(hasChanged && mParentTable)
	{
		CPGPKeys::TheApp()->CommitDefaultKeyrings();
		mParentTable->RedrawTable();
	}
}

	void
CKeyInfoWindow::AttemptClose(void)
{
	ProcessClose();
	
	LWindow::AttemptClose();
}

	void
CKeyInfoWindow::DoClose(void)
{
	ProcessClose();
	
	LWindow::DoClose();
}

	void
CKeyInfoWindow::SetInfo(
	CKeyTable 	*table,
	PGPKeyRef 	theKey)
{
	UInt16				numPanels;
	UInt16				panelIndex;
	Str255				pstr;
	PGPError			err;
	PGPSize				len;
	LTabsControl		*tabsControl;
	PGPInt32			algorithm;
	PGPUInt32			numKeys;
	ControlTabInfoRec	tabInfo;
	
	mParentTable 	= table;
	mWindowKey		= theKey;
	panelIndex		= 2;
	
	mMultiView = (LMultiPanelView *) FindPaneByID( kMultiPanelViewPaneID );
	pgpAssertAddrValid( mMultiView, VoidAlign );
	
	tabsControl = (LTabsControl *) FindPaneByID( kTabsControlPaneID );
	pgpAssertAddrValid( tabsControl, VoidAlign );
	
	mMultiView->AddPanel( kGeneralPanelViewResID, NULL, 1 );
	
	pgpClearMemory( &tabInfo, sizeof( tabInfo ) );
	
	tabInfo.version = kControlTabInfoVersionZero;
	
	if( IsntPGPError( PGPGetKeyNumber( theKey, kPGPKeyPropAlgID,
				&algorithm ) ) && algorithm == kPGPPublicKeyAlgorithm_DSA )
	{
		// Add Subkeys panel
		mMultiView->AddPanel( kSubkeysPanelViewResID, NULL, panelIndex );
		
		GetIndString( tabInfo.name, kKeyInfoStringListID, kSubkeysStrIndex );
		tabsControl->SetDataTag( panelIndex, kControlTabInfoTag,
					sizeof( tabInfo ), &tabInfo );
		
		panelIndex++;
	}
	
	if( IsntPGPError( PGPCountAdditionalRecipientRequests( theKey,
			&numKeys ) ) && numKeys != 0 )
	{
		// Add ADK panel
		mMultiView->AddPanel( kADKPanelViewResID, NULL, panelIndex );

		GetIndString( tabInfo.name, kKeyInfoStringListID, kADKStrIndex );
		tabsControl->SetDataTag( panelIndex, kControlTabInfoTag,
					sizeof( tabInfo ), &tabInfo );
					
		panelIndex++;
	}
	
	if( IsntPGPError( PGPCountRevocationKeys( theKey, &numKeys ) ) &&
			numKeys != 0 )
	{
		// Add Revoker panel
		mMultiView->AddPanel( kRevokerPanelViewResID, NULL, panelIndex );

		GetIndString( tabInfo.name, kKeyInfoStringListID, kRevokerStrIndex );
		tabsControl->SetDataTag( panelIndex, kControlTabInfoTag,
					sizeof( tabInfo ), &tabInfo );
					
		panelIndex++;
	}
	
	// Force creation of all panels so we can control the LCommander hierarchy
	
	numPanels = mMultiView->GetPanelCount();
	tabsControl->SetMaxValue( numPanels );

	for( panelIndex = 1; panelIndex <= numPanels; panelIndex++ )
	{
		CKeyInfoPanelView	*panel;
		
		LCommander::SetDefaultCommander( mMultiView );
		LPane::SetDefaultView( mMultiView );
		
		panel = (CKeyInfoPanelView *) mMultiView->CreatePanel( panelIndex );
		if( IsntNull( panel ) )
		{
			panel->SetInfo( table->GetKeySet(), theKey, table->IsWritable() );
		}
	}

	mMultiView->SwitchToPanel( 1, FALSE );
	
	// Set the window totle
	err = PGPGetPrimaryUserIDNameBuffer( theKey, sizeof( pstr ) - 1,
					(char *) &pstr[1], &len);
	pgpAssertNoErr( err );
	pstr[0] = len;
	SetDescriptor( pstr );
	
	Show();
}

	void
CKeyInfoWindow::GetFingerprintString(Str255 pstr, PGPKeyRef key)
{
	static char hexDigit[] = "0123456789ABCDEF";
	PGPError	err;
	PGPSize		len;
	PGPByte		fingerprint[ 256 ];
	Int16		strIndex;
	uchar		*p;
	
	err = PGPGetKeyPropertyBuffer( key, kPGPKeyPropFingerprint,
			sizeof( fingerprint ), fingerprint, &len);
	pgpAssertNoErr(err);
	if ( IsntPGPError( err ) )
	{
		if(len == 20)
		{
			pstr[0] = 50;
			p = pstr + 1;
			for(strIndex = 0 ; strIndex < 20 ; strIndex++)
			{
				*p++ = hexDigit[fingerprint[strIndex]>>4];
				*p++ = hexDigit[fingerprint[strIndex]&0xF];
				if((strIndex == 1) || (strIndex == 3) || (strIndex == 5)
						|| (strIndex == 7) || (strIndex == 11) ||
						(strIndex == 13)
						|| (strIndex == 13) || (strIndex == 15) ||
						(strIndex == 17))
					*p++ = ' ';
				else if(strIndex == 9)
				{
					*p++ = ' ';
					*p++ = ' ';
				}
			}
		}
		else
		{
			pstr[0] = 40;
			p = pstr + 1;
			for(strIndex = 0 ; strIndex < 16 ; strIndex++)
			{
				*p++ = hexDigit[fingerprint[strIndex]>>4];
				*p++ = hexDigit[fingerprint[strIndex]&0xF];
				if((strIndex == 1) || (strIndex == 3) || (strIndex == 5)
					|| (strIndex == 9) || (strIndex == 11) || (strIndex == 13))
					*p++ = ' ';
				else if(strIndex == 7)
				{
					*p++ = ' ';
					*p++ = ' ';
				}
			}
		}
	}
}
