/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: PGPnetModule.cp,v 1.30 1999/05/05 09:54:19 wprice Exp $
____________________________________________________________________________*/

/*	Improvements
	* Use rotating pool of preallocated message blocks instead of allocating them on the fly
	* Adjust sdu based on previous sessions statistics
	* Use ICMP messages to adjust mtu
	* Optimize IP reassembly routine
	* Remove pullupmsgs
*/


#include <dlpi.h>
#include <LowMem.h>

#include "EthernetHandler.h"
#include "IPHandler.h"
#include "PGPnetModuleCommon.h"
#include "PGPnetModule.h"



namespace {
	const long			kMinPacketSize		=	46;
	const long			kMaxPacketSize		=	1500;
	const PGPUInt32		kSendStatsDelay		=	10;		// Number of seconds
	
	
	struct module_info	sModuleInfo			= {	'PG',
												"PGPnetModule",
												kMinPacketSize,
												kMaxPacketSize,
												10 * kMaxPacketSize,
												kMaxPacketSize };

	struct qinit		sReadQ				= {	ReadPut,
												NULL,
												ModuleOpen,
												ModuleClose,
												NULL,
												&sModuleInfo,
												NULL };
											
	struct qinit		sWriteQ				= {	WritePut,
												NULL,
												NULL,
												NULL,
												NULL,
												&sModuleInfo,
												NULL };

	struct streamtab	sStreamTab			= {	&sReadQ,
												&sWriteQ,
												NULL,
												NULL };

	install_info		sInstallInfo		= {	&sStreamTab,
												kOTModIsModule | kOTModUpperIsDLPI
													| kOTModLowerIsDLPI | kOTModIsFilter,
												SQLVL_QUEUE,
												NULL, 
												0, 
												0 };
	
	
	char *				sModuleListHead		=	NULL;
	PGPBoolean			sPGPnetOn			=	false;
	SGlobals			sGlobals			=	{ true, false, NULL, NULL, NULL, 0 };
//	PGPBoolean			sAllowUnconfigured	=	true;
//	PGPBoolean			sRequireSecure		=	false;
//	SSAList *			sSAList				=	NULL;
//	SHostEntryList *	sHostEntryList		=	NULL;
//	SPendingSAList *	sPendingSAList		=	NULL;
//	PGPUInt32			sBroadcastAddress	=	0;
	queue_t *			sControlWQ			=	NULL;
	PGPUInt32			sStatsTime			=	0;
}

PGPBoolean	ReceiveControlMessage(queue_t * inQ, mblk_t * inMessage);
void		UpdateModuleResource(queue_t * inQ, mblk_t * inMessage);
void		FreeGlobalLists();



// Enable/Disable packet logging
#define LOGGING 0
#if	LOGGING
	void	LogWriteDLPI(mblk_t * inMessage);
	void	LogReadDLPI(mblk_t * inMessage);
	void	LogDATA(mblk_t * inMessage);
#else
#	define	LogWriteDLPI(x)
#	define	LogReadDLPI(x)
#	define	LogDATA(x)
#endif


	install_info *
GetOTInstallInfo(void)
{
	DConLogNote("Getting the Install Info\n");
	
	return &sInstallInfo;
}



	Boolean
InitStreamModule(
	void * inUnused)
{
	(void) inUnused;
	
	DConLogNote("InitStreamModule\n");
	
	return true;
}

	void
TerminateStreamModule(void)
{
	DConLogNote("TerminateStreamModule\n");
	FreeGlobalLists();
}



	int
ModuleOpen(
	queue_t *	inQ,
	dev_t *		inDevice,
	int			inOpenFlag,
	int			inStreamFlag,
	cred_t *	inCredentials)
{
	int	err = noErr;
	
	DConLogNote("ModuleOpen\n");
	SPerStreamData	data = { false, kInvalidPGPIPsecContextRef, kInvalidPGPIPsecContextRef,
							 0, 0, 0, false, NULL, NULL };
	
	try {
		if (inStreamFlag != MODOPEN) {
			DConThrowOSError(ENXIO);
		}
		if (inQ->q_ptr == NULL) {
			PGPError		pgpErr;
			
			pgpErr = PGPNewIPsecContext(&data.readContext);
			DConThrowIfPGPError(pgpErr);
			pgpErr = PGPNewIPsecContext(&data.writeContext);
			DConThrowIfPGPError(pgpErr);
			err = mi_open_comm(&sModuleListHead, sizeof(SPerStreamData), inQ, inDevice, inOpenFlag,
					inStreamFlag, inCredentials);
			DConThrowIfOSError(err);
			
			*GetPerStreamData(inQ) = data;
		}
	}
	
	catch (OSStatus exception) {
		err = exception;
	}
	
	catch (...) {
		err = -1;
	}
	
	if (err != noErr) {
		if (PGPIPsecContextRefIsValid(data.readContext)) {
			PGPFreeIPsecContext(data.readContext);
		}
		if (PGPIPsecContextRefIsValid(data.writeContext)) {
			PGPFreeIPsecContext(data.writeContext);
		}
	}
	
	return err;
}



	int
ModuleClose(
	queue_t *	inQ,
	int			inFlag,
	cred_t *	inCredentials)
{
	(void) inFlag;
	(void) inCredentials;
	
	DConLogNote("ModuleClose\n");
	
	if (PGPIPsecContextRefIsValid(GetPerStreamData(inQ)->readContext)) {
		PGPFreeIPsecContext(GetPerStreamData(inQ)->readContext);
	}
	if (PGPIPsecContextRefIsValid(GetPerStreamData(inQ)->writeContext)) {
		PGPFreeIPsecContext(GetPerStreamData(inQ)->writeContext);
	}
	
	SIPReassemblyList *	curr = GetPerStreamData(inQ)->inReassemblyList;
	SIPReassemblyList *	temp;
	PGPBoolean			done = true;
	
	do {
		while (curr != NULL) {
			temp = curr;
			curr = curr->next;
			freeb(temp->header);
			freemsg(temp->packetMessage);
			OTFreeMem(temp);
		}
		curr = GetPerStreamData(inQ)->outReassemblyList;
		done = !done;
	} while (! done);
	mi_close_comm(&sModuleListHead, inQ);

	return noErr;
}



	int
ReadPut(
	queue_t *	inQ,
	mblk_t *	inMessage)
{
	PGPBoolean	putNext = true;
	
	switch (inMessage->b_datap->db_type) {
		case M_FLUSH:
		{
			DConLogNote("Read: M_FLUSH\n");
		}
		break;
		
		
		case M_IOCACK:
		{
			iocblk *	io = reinterpret_cast<iocblk *>(inMessage->b_rptr);

			DConLogNote("Read: M_IOCACK\n");
			if (io->ioc_count == TRANSPARENT) {
				DConLogNote("  TRANSPARENT\n");
			} else {
				switch (io->ioc_cmd) {
					case I_OTSetRawMode:
					{
						dl_recv_control_t * recvControl = reinterpret_cast<
															dl_recv_control_t *>(
															inMessage->b_cont->b_rptr);

						if (recvControl->dl_primitive == kOTSetRecvMode) {
							if ((recvControl->dl_flags & DL_NORMAL_STATUS)
							|| (recvControl->dl_flags & DL_ERROR_STATUS)) {
								DConLogNote("  SetRawRecv: true\n");
								GetPerStreamData(inQ)->rawRecv = true;
							} else {
								DConLogNote("  SetRawRecv: false\n");
								GetPerStreamData(inQ)->rawRecv = false;
							}
						}
					}
					break;


					default:
					{
						DConLogNote("    Command: 0x%X\n", io->ioc_cmd);
					}
					break;
				}
			}
		}
		break;
		
		
		case M_IOCNAK:
		{
			DConLogNote("Read: M_IOCNAK\n");
		}
		break;



		case M_PROTO:
		case M_PCPROTO:
		{
			DConLogNote("Read: %s\n",
				(inMessage->b_datap->db_type == M_PROTO) ? "M_PROTO" : "M_PCPROTO");
			if ((inMessage->b_wptr - inMessage->b_rptr) >= sizeof(UInt32)) {
				DL_primitives *	thePrimitive = reinterpret_cast<DL_primitives *>(
												inMessage->b_rptr);
				
				LogReadDLPI(inMessage);
				switch (thePrimitive->dl_primitive) {
					case DL_INFO_ACK:
					{
						DConLogNote("  DL_INFO_ACK: maxsdu = %lu, minsdu = %lu, mac = %lu\n",
							thePrimitive->info_ack.dl_max_sdu, thePrimitive->info_ack.dl_min_sdu,
							thePrimitive->info_ack.dl_mac_type);
						GetPerStreamData(inQ)->maxSDU = thePrimitive->info_ack.dl_max_sdu;
						GetPerStreamData(inQ)->minSDU = thePrimitive->info_ack.dl_min_sdu;
						GetPerStreamData(inQ)->macType = thePrimitive->info_ack.dl_mac_type;
						switch (thePrimitive->info_ack.dl_mac_type) {
							case DL_CSMACD:
							case DL_ETHER:
							{
								DConLogNote("  Ethernet media\n");
								GetPerStreamData(inQ)->streamOn = true;
							}
							break;
							
							default:
							{
								GetPerStreamData(inQ)->streamOn = true;
								DConLogError("ERROR: Unknown mac: %lu\n", thePrimitive->info_ack.dl_mac_type);
							}
							break;
						}
					}
					break;
					
					case DL_UNITDATA_IND:
					{
						DConLogNote("  DL_UNITDATA_IND\n");
						if (sPGPnetOn && GetPerStreamData(inQ)->streamOn) {
							putNext = IPProcessAndPutNext(true, inQ, inMessage, inMessage->b_cont,
										&sGlobals, &GetPerStreamData(inQ)->inReassemblyList);
//							putNext = IPDecodeAndPutNext(inQ, inMessage, inMessage->b_cont, &sGlobals);
						}
					}
					break;
				}
			} else {
				DConLogError("  Unknown PCPROTO: No primitive\n");
				DConDumpError("Primitive: ", inMessage->b_rptr, inMessage->b_wptr - inMessage->b_rptr);
			}
		}
		break;


		case M_DATA:
		{
			DConLogNote("Read: M_DATA\n");
			LogDATA(inMessage);
			if (OTHERQ(inQ) == sControlWQ) {
				DConLogNote("  Sending PGPnetModuleMsg\n");
			} else {
				if (sPGPnetOn && GetPerStreamData(inQ)->streamOn) {
					switch (GetPerStreamData(inQ)->macType) {
						case DL_CSMACD:
						case DL_ETHER:
						{
							putNext = EthernetHandleReadPutDATA(inQ, inMessage, &sGlobals);
						}
						break;
					}
				}
			}
		}
		break;


		default:
		{
			DConLogWarning("Read: Unknown Message type: 0%o\n",
				(unsigned int) inMessage->b_datap->db_type);
		}
		break;
	}	

	if (putNext) {
		putnext(inQ, inMessage);
	} else {
		if (inMessage != NULL) {
			freemsg(inMessage);
		}
	}
	
	return 0;
}



	int
WritePut(
	queue_t *	inQ,
	mblk_t *	inMessage)
{
	PGPBoolean	putNext = true;

	if (sControlWQ == inQ) {
		iocblk *	io = reinterpret_cast<iocblk *>(inMessage->b_rptr);
		
		if ((inMessage->b_datap->db_type == M_IOCTL) && (io->ioc_count != TRANSPARENT)
		&& (io->ioc_cmd == I_PGPnetModuleMsg)) {
			PGPBoolean	qReply = true;
			PGPUInt8	ackType = M_IOCACK;
			
			if (pullupmsg(inMessage->b_cont, -1)) {
				qReply = ReceiveControlMessage(inQ, inMessage);
			} else {
				ackType = M_IOCNAK;
			}
			if (qReply) {
				inMessage->b_datap->db_type = ackType;
				if (ackType == M_IOCNAK) {
					DConLogNote("  Reply: NAK\n");
					io->ioc_error = EAGAIN;
				} else {
					DConLogNote("  Reply: ACK\n");
				}
				qreply(inQ, inMessage);
			}
			putNext = false;
			inMessage = NULL;
		} else {
			DConLogWarning("WARNING: Received non PGP message on control stream\n");
		}
	} else {
		switch (inMessage->b_datap->db_type) {
			case M_FLUSH:
			{
				DConLogNote("Write: M_FLUSH\n");
			}
			break;
			
			
			case M_IOCTL:
			{
				iocblk *	io = reinterpret_cast<iocblk *>(inMessage->b_rptr);

				DConLogNote("Write: M_IOCTL\n");
				if (io->ioc_count == TRANSPARENT) {
					DConLogNote("  TRANSPARENT\n");
				} else {
					switch (io->ioc_cmd) {
						case I_PGPnetModuleMsg:
						{
							if (pullupmsg(inMessage->b_cont, -1)) {
								PGPnetModuleStartupMsg *	message = 
																reinterpret_cast<
																PGPnetModuleStartupMsg *>(
																inMessage->b_cont->b_rptr);

								DConLogNote("  I_PGPnetModule\n");
								if ((message->type == kPGPnetModuleStartupMsg)
								&& OTStrEqual(message->modName, kPGPnetModuleName)) {
									mps_become_writer(inQ, inMessage, UpdateModuleResource);
									putNext = false;
									inMessage = NULL;
								} else {
									DConLogWarning("  WARNING: Unknown I_PGPnetModule message on non"
												   "-control stream\n");								
								}
							}
						}
						break;


						default:
						{
							DConLogNote("    Command: 0x%X\n", io->ioc_cmd);
						}
						break;
					}
				}
			}
			break;


			case M_PROTO:
			case M_PCPROTO:
			{
				DConLogNote("Write: %s\n",
					(inMessage->b_datap->db_type == M_PROTO) ? "M_PROTO" : "M_PCPROTO");
				if ((inMessage->b_wptr - inMessage->b_rptr) >= sizeof(UInt32)) {
					DL_primitives *	thePrimitive = reinterpret_cast<DL_primitives *>(
													inMessage->b_rptr);

					LogReadDLPI(inMessage);
					switch (thePrimitive->dl_primitive) {
						case DL_UNITDATA_REQ:
						{
							DConLogNote("  DL_UNITDATA_REQ\n");
							if (sPGPnetOn && GetPerStreamData(inQ)->streamOn) {
								putNext = IPProcessAndPutNext(false, inQ, inMessage, inMessage->b_cont,
											&sGlobals, &GetPerStreamData(inQ)->outReassemblyList);
//								putNext = IPEncodeAndPutNext(inQ, inMessage, inMessage->b_cont,
//											&sGlobals);
							}
						}
						break;
					}
				} else {
					DConLogError("  Unknown (PC)PROTO: No primitive\n");
					DConDumpError("Primitive: ", inMessage->b_rptr, inMessage->b_wptr - inMessage->b_rptr);
				}
			}
			break;


			case M_DATA:
			{
				DConLogNote("Write: M_DATA\n");
				LogDATA(inMessage);
				if (sPGPnetOn && GetPerStreamData(inQ)->streamOn) {
					switch (GetPerStreamData(inQ)->macType) {
						case DL_CSMACD:
						case DL_ETHER:
						{

							putNext = EthernetHandleWritePutDATA(inQ, inMessage, &sGlobals);
						}
						break;
					}
				}
			}
			break;


			default:
			{
				DConLogWarning("Write: Unknown Message type: 0%o\n",
					(unsigned int) inMessage->b_datap->db_type);
			}
			break;
		}
	
		if (putNext) {
			putnext(inQ, inMessage);
		} else {
			if (inMessage != NULL) {
				freemsg(inMessage);
			}
		}
	}


	PGPUInt32	clock = OTGetClockTimeInSecs();
	
	if (clock > (sStatsTime + kSendStatsDelay)) {
		SendStatsToService();
		sStatsTime = clock;
	}

	return 0;
}



	void
UpdateModuleResource(
	queue_t *	inQ,
	mblk_t *	inMessage)
{
	PGPUInt8	ackType = M_IOCACK;
	
	DConLogNote("  UpdateModuleResource\n");
//	DConDumpNote("", inMessage->b_cont->b_rptr,
//		inMessage->b_cont->b_wptr - inMessage->b_cont->b_rptr);
	switch (*reinterpret_cast<PGPUInt16 *>(inMessage->b_cont->b_rptr)) {
		case kPGPnetModuleStartupMsg:
		{
			DConLogNote("  kPGPnetModuleStartup\n");
			FreeGlobalLists();
			sControlWQ = inQ;
		}
		break;
		
		
		case kPGPnetModuleNewSAMsg:
		{
			SSAList *				newSA = static_cast<SSAList *>(OTAllocMem(sizeof(SSAList)));
			PGPnetModuleNewSAMsg *	message = reinterpret_cast<PGPnetModuleNewSAMsg *>(
												inMessage->b_cont->b_rptr);
												
			DConLogNote("  kPGPnewModuleNewSAMsg\n");
			if (newSA == NULL) {
				ackType = M_IOCNAK;
			} else {
				SPendingSAList *		curr = sGlobals.pendingSAList;
				SPendingSAList **		previous = &sGlobals.pendingSAList;
		
				while (curr != NULL) {
					if( ( curr->ipAddress	== message->sa.ipAddress ) &&
						( curr->ipAddrStart	== message->sa.ipAddrStart ) &&
						( curr->ipMaskEnd	== message->sa.ipMaskEnd ) &&
						( curr->destIsRange	== message->sa.destIsRange ) ) {
						*previous = curr->next;
						OTFreeMem(curr);
						break;
					} else {
						previous = &curr->next;
						curr = curr->next;
					}
				}
				newSA->index = message->index;
				newSA->sa = message->sa;
				newSA->sa.birthTime = LMGetTime();	/* we don't know PGPTime */
				reinterpret_cast<PGPnetIKESAUserData *>(newSA->sa.userData)->
									sequenceWindowUpper = (sizeof(PGPUInt32) * 8) - 1;
				newSA->killed = false;
				newSA->next = sGlobals.saList;
				sGlobals.saList = newSA;
			}
		}
		break;
		
		
		case kPGPnetModuleNoSAMsg:
		{
			SPendingSAList *		curr = sGlobals.pendingSAList;
			SPendingSAList **		previous = &sGlobals.pendingSAList;
			PGPnetModuleNoSAMsg *	message = reinterpret_cast<PGPnetModuleNoSAMsg *>(
												inMessage->b_cont->b_rptr);
												
			DConLogNote("  kPGPnetModuleNoSAMsg\n");
			while (curr != NULL) {
				if( ( curr->ipAddress	== message->ipAddress ) &&
					( curr->ipAddrStart	== message->ipAddrStart ) &&
					( curr->ipMaskEnd	== message->ipMaskEnd ) &&
					( curr->destIsRange	== message->destIsRange ) ) {
					*previous = curr->next;
					OTFreeMem(curr);
					break;
				} else {
					previous = &curr->next;
					curr = curr->next;
				}
			}
		}
		break;
		
		
		case kPGPnetModuleUpdateSAMsg:
		{
			SSAList *					curr = sGlobals.saList;
			PGPnetModuleUpdateSAMsg *	message = reinterpret_cast<PGPnetModuleUpdateSAMsg *>(
													inMessage->b_cont->b_rptr);
												
			DConLogNote("  kPGPnetModuleUpdateSAMsg\n");
			while (curr != NULL) {
				if (curr->index == message->index) {
					PGPUInt32	oldBirth = curr->sa.birthTime;
					
					OTMemcpy(&curr->sa, &message->sa, sizeof(PGPikeSA) - kPGPike_UserDataSize);
					curr->sa.birthTime = oldBirth;
					break;
				} else {
					curr = curr->next;
				}
			}
		}
		break;


		case kPGPnetModuleKillSAMsg:
		{
			SSAList *				curr = sGlobals.saList;
			SSAList **				previous = &sGlobals.saList;
			PGPnetModuleKillSAMsg *	message = reinterpret_cast<PGPnetModuleKillSAMsg *>(
												inMessage->b_cont->b_rptr);
												
			DConLogNote("  kPGPnetModuleKillSAMsg\n");
			while (curr != NULL) {
				if (curr->index == message->index) {
					*previous = curr->next;
					OTFreeMem(curr);
					break;
				} else {
					previous = &curr->next;
					curr = curr->next;
				}
			}
		}
		break;


		case kPGPnetModuleHostsMsg:
		{
			PGPnetModuleHostsMsg *	message = reinterpret_cast<PGPnetModuleHostsMsg *>(
												inMessage->b_cont->b_rptr);
			SHostEntryList *		newList = static_cast<SHostEntryList *>(OTAllocMem(
												(message->numHosts * sizeof(PGPNetPrefHostEntry))
												+ sizeof(PGPUInt32)));

			DConLogNote("  kPGPnetModuleHostsMsg: numHosts = %lu, allocSize = %lu, result = %d\n",
				message->numHosts, static_cast<PGPUInt32>((message->numHosts * 
				sizeof(PGPNetPrefHostEntry)) + sizeof(PGPUInt32)), static_cast<int>(newList != NULL));
			if (newList == NULL) {
				ackType = M_IOCNAK;
			} else {
				if (sGlobals.hostEntryList != NULL) {
					OTFreeMem(sGlobals.hostEntryList);
				}
				sGlobals.hostEntryList = newList;
				sGlobals.hostEntryList->numHosts = message->numHosts;
				OTMemcpy(sGlobals.hostEntryList->host, message->host,
					message->numHosts * sizeof(PGPNetPrefHostEntry));
			}
		}
		break;
	}
	inMessage->b_datap->db_type = ackType;
	if (ackType == M_IOCNAK) {
		DConLogNote("  Reply: NAK\n");
		reinterpret_cast<iocblk *>(inMessage->b_rptr)->ioc_error = EAGAIN;
	} else {
		DConLogNote("  Reply: ACK\n");
	}
	qreply(inQ, inMessage);
}


	PGPBoolean
ReceiveControlMessage(
	queue_t *	inQ,
	mblk_t *	inMessage)
{
	PGPBoolean	result = true;
	
	DConLogNote("ReceiveControlMessage\n");
//	DConDumpNote("", inMessage->b_cont->b_rptr,
//		inMessage->b_cont->b_wptr - inMessage->b_cont->b_rptr);
	switch (*reinterpret_cast<PGPUInt16 *>(inMessage->b_cont->b_rptr)) {
		case kPGPnetModuleShutdownMsg:
		{
			DConLogNote("  kPGPnetModuleShutdown\n");
			sPGPnetOn = false;
			sControlWQ = NULL;
		}
		break;
		
		
		case kPGPnetModulePrefsMsg:
		{
			PGPnetModulePrefsMsg *	message = reinterpret_cast<PGPnetModulePrefsMsg *>(
												inMessage->b_cont->b_rptr);
			
			DConLogNote("  kPGPnetModulePrefsMsg\n"
						"    Enabled: %d\n"
						"    Allow Unconfigured: %d\n"
						"    Require Secure: %d\n", (int) message->enabled,
						(int) message->allowUnconfigured,
						(int) message->requireSecure);
			sPGPnetOn = message->enabled;
			sGlobals.allowUnconfigured = message->allowUnconfigured;
			sGlobals.requireSecure = message->requireSecure;
		}
		break;
		
		
		case kPGPnetModuleBroadcastMsg:
		{
			PGPnetModuleBroadcastMsg *	message = reinterpret_cast<PGPnetModuleBroadcastMsg *>(
													inMessage->b_cont->b_rptr);
													
			DConLogNote("  kPGPnetModuleBroadcastMsg\n"
						"    Address: %d.%d.%d.%d\n", (int) ((message->addr >> 24) & 0x000000FF),
						(int) ((message->addr >> 16) & 0x000000FF),
						(int) ((message->addr >> 8) & 0x000000FF),
						(int) (message->addr & 0x000000FF));
			sGlobals.broadcastAddress = message->addr;
		}
		break;


		case kPGPnetModuleStartupMsg:
		case kPGPnetModuleNewSAMsg:
		case kPGPnetModuleNoSAMsg:
		case kPGPnetModuleUpdateSAMsg:
		case kPGPnetModuleKillSAMsg:
		case kPGPnetModuleHostsMsg:
		{
			mps_become_writer(inQ, inMessage, UpdateModuleResource);
			result = false;
		}
		break;
		
		
		default:
		{
			DConLogNote("  Unknown: %hu\n", *reinterpret_cast<PGPUInt16 *>(inMessage->b_cont->b_rptr));
		}
		break;
	}
	
	return result;
}



	void
FreeGlobalLists()
{
	void *		temp;
	
	DConLogNote("  Freeing global lists\n");
	while (sGlobals.saList != NULL) {
		temp = sGlobals.saList;
		sGlobals.saList = sGlobals.saList->next;
		OTFreeMem(temp);
	}
	if (sGlobals.hostEntryList != NULL) {
		OTFreeMem(sGlobals.hostEntryList);
		sGlobals.hostEntryList = NULL;
	}
	while (sGlobals.pendingSAList != NULL) {
		temp = sGlobals.pendingSAList;
		sGlobals.pendingSAList = sGlobals.pendingSAList->next;
		OTFreeMem(temp);
	}
}


	PGPBoolean
AddIPAddressToPendingList(
	PGPUInt32	inAddress,
	PGPBoolean	destIsRange,
	PGPUInt32	ipAddrStart,
	PGPUInt32	ipMaskEnd )
{
	PGPBoolean			result;
	SPendingSAList *	newAddress = reinterpret_cast<SPendingSAList *>(OTAllocMem(
										sizeof(SPendingSAList)));
	
	if (newAddress != NULL) {
		PGPBoolean	done;
		
		newAddress->ipAddress		= inAddress;
		newAddress->destIsRange		= destIsRange;
		newAddress->ipAddrStart		= ipAddrStart;
		newAddress->ipMaskEnd		= ipMaskEnd;
		do {
			newAddress->next = sGlobals.pendingSAList;
			done = OTCompareAndSwapPtr(newAddress->next, newAddress, &sGlobals.pendingSAList);
		} while (! done);
		DConLogNote("  Added 0x%lX to pending list\n", inAddress);
		result = true;
	} else {
		result = false;
	}
	
	return result;
}



	PGPBoolean
SendKillSAToService(
	PGPUInt32	inIndex)
{
	PGPBoolean	result = false;
	queue_t	*	controlQ = sControlWQ;
	
	if (controlQ != NULL) {
		mblk_t *	message = allocb(sizeof(PGPnetModuleKillSAMsg), 0);
		
		if (message != NULL) {
			reinterpret_cast<PGPnetModuleKillSAMsg *>(message->b_rptr)->type = kPGPnetModuleKillSAMsg;
			reinterpret_cast<PGPnetModuleKillSAMsg *>(message->b_rptr)->index = inIndex;
			message->b_wptr += sizeof(PGPnetModuleKillSAMsg);
			DConLogNote("  SendKillSAToService index=%lu\n", inIndex);
			qreply(controlQ, message);
			result = true;
		}
	}
	
	return result;
}


	PGPBoolean
SendRekeySAToService(
	PGPUInt32	inIndex)
{
	PGPBoolean	result = false;
	queue_t	*	controlQ = sControlWQ;
	
	if (controlQ != NULL) {
		mblk_t *	message = allocb(sizeof(PGPnetModuleRekeySAMsg), 0);
		
		if (message != NULL) {
			reinterpret_cast<PGPnetModuleRekeySAMsg *>(message->b_rptr)->type = kPGPnetModuleRekeySAMsg;
			reinterpret_cast<PGPnetModuleRekeySAMsg *>(message->b_rptr)->index = inIndex;
			message->b_wptr += sizeof(PGPnetModuleRekeySAMsg);
			DConLogNote("  SendRekeySAToService index=%lu\n", inIndex);
			qreply(controlQ, message);
			result = true;
		}
	}
	
	return result;
}


	PGPBoolean
SendNeedSAToService(
	PGPUInt32		inAddress,
	PGPBoolean		destIsRange,
	PGPUInt32		ipAddrStart,
	PGPUInt32		ipMaskEnd )
{
	PGPBoolean		result = false;
	queue_t	*		controlQ = sControlWQ;
	
	if (controlQ != NULL) {
		mblk_t *	message = allocb(sizeof(PGPnetModuleNeedSAMsg), 0);
		
		if (message != NULL) {
			reinterpret_cast<PGPnetModuleNeedSAMsg *>(message->b_rptr)->type = kPGPnetModuleNeedSAMsg;
			reinterpret_cast<PGPnetModuleNeedSAMsg *>(message->b_rptr)->ipAddress	= inAddress;
			reinterpret_cast<PGPnetModuleNeedSAMsg *>(message->b_rptr)->ipAddrStart	= ipAddrStart;
			reinterpret_cast<PGPnetModuleNeedSAMsg *>(message->b_rptr)->ipMaskEnd	= ipMaskEnd;
			reinterpret_cast<PGPnetModuleNeedSAMsg *>(message->b_rptr)->destIsRange	= destIsRange;
			message->b_wptr += sizeof(PGPnetModuleNeedSAMsg);
			DConLogNote("  SendNeedSAToService address=0x%lX\n", inAddress);
			qreply(controlQ, message);
			result = true;
		}
	}
	
	return result;
}


	PGPBoolean
SendErrorToService(
	PGPError	inError,
	PGPUInt32	address )
{
	PGPBoolean	result = false;
	queue_t	*	controlQ = sControlWQ;
	
	if (controlQ != NULL) {
		mblk_t *	message = allocb(sizeof(PGPnetModuleErrorMsg), 0);
		
		if (message != NULL) {
			reinterpret_cast<PGPnetModuleErrorMsg *>(message->b_rptr)->type = kPGPnetModuleErrorMsg;
			reinterpret_cast<PGPnetModuleErrorMsg *>(message->b_rptr)->err	= inError;
			reinterpret_cast<PGPnetModuleErrorMsg *>(message->b_rptr)->addr = address;
			message->b_wptr += sizeof(PGPnetModuleErrorMsg);
			DConLogNote("  SendErrorToService error=%ld\n", inError);
			qreply(controlQ, message);
			result = true;
		}
	}
	
	return result;
}


	PGPBoolean
SendStatsToService()
{
	PGPBoolean	result = false;
	queue_t	*	controlQ = sControlWQ;
	
	if (controlQ != NULL) {
		SSAList *	curr = sGlobals.saList;
		mblk_t *	message;
		
		while (curr != NULL) {
			message = allocb(sizeof(PGPnetModuleSAStatsMsg), 0);
			if (message != NULL) {
				reinterpret_cast<PGPnetModuleSAStatsMsg *>(message->b_rptr)->type = kPGPnetModuleSAStatsMsg;
				reinterpret_cast<PGPnetModuleSAStatsMsg *>(message->b_rptr)->index = curr->index;
				OTMemcpy(	&reinterpret_cast<PGPnetModuleSAStatsMsg *>(message->b_rptr)->stats,
							curr->sa.userData,
							sizeof(PGPnetIKESAUserData));

				message->b_wptr += sizeof(PGPnetModuleSAStatsMsg);
				qreply(controlQ, message);
				result = true;
			}
			curr = curr->next;
		}
//		DConLogNote("  SendStatsToService\n", inError);
	}
	
	return result;
}


/***************************** Logging Calls ***********************************/
#if LOGGING

	void
LogWriteDLPI(
	mblk_t *	inMessage)
{
	DL_primitives *	thePrimitive = reinterpret_cast<DL_primitives *>(inMessage->b_rptr);
	
	switch (thePrimitive->dl_primitive) {
		case DL_INFO_REQ:
		{
			DConLogNote("  DL_INFO_REQ\n");
		}
		break;
		
		
		case DL_BIND_REQ:
		{
			dl_bind_req_t *	req = &thePrimitive->bind_req;

			DConLogNote("  DL_BIND_REQ\n");
			DConLogNote("             dl_sap: %lu\n", req->dl_sap);
			DConLogNote("      dl_max_conind: %lu\n", req->dl_max_conind);
			DConLogNote("    dl_service_mode: %lu\n", req->dl_service_mode);
			DConLogNote("       dl_conn_mgmt: 0x%lX\n", req->dl_conn_mgmt);
			DConLogNote("     dl_xidtest_flg: %lu\n", req->dl_xidtest_flg);
		}
		break;
		
		
		case DL_UNBIND_REQ:
		{
			DConLogNote("  DL_UNBIND_REQ\n");
		}
		break;


		case DL_SUBS_BIND_REQ:
		{
			DConLogNote("  DL_SUBS_BIND_REQ\n");
		}
		break;


		case DL_ENABMULTI_REQ:
		{
			dl_enabmulti_req_t *	req = &thePrimitive->enabmulti_req;

			DConLogNote("  DL_ENABMULTI_REQ\n");
			DConDumpNote("Address: ", inMessage->b_rptr + req->dl_addr_offset,
				req->dl_addr_length);
		}
		break;
		
		
		case DL_DISABMULTI_REQ:
		{
			dl_disabmulti_req_t *	req = &thePrimitive->disabmulti_req;

			DConLogNote("  DL_DISABMULTI_REQ\n");
			DConDumpNote("Address: ", inMessage->b_rptr + req->dl_addr_offset,
				req->dl_addr_length);
		}
		break;
		
		
		case DL_UNITDATA_REQ:
		{
			dl_unitdata_req_t *	req = &thePrimitive->unitdata_req;

			DConLogNote("  DL_UNITDATA_REQ\n");
			DConDumpNote("Address: ", inMessage->b_rptr + req->dl_dest_addr_offset,
				req->dl_dest_addr_length);
			DConLogNote("Data: ");
			LogDATA(inMessage->b_cont);
		}
		break;


		case DL_PHYS_ADDR_REQ:
		{
			dl_phys_addr_req_t *	req = &thePrimitive->physaddr_req;

			DConLogNote("  DL_PHYS_ADDR_REQ\n");
			DConLogNote("    dl_addr_type: %lu\n", req->dl_addr_type);
		}
		break;
		
		
		default:
		{
			DConLogWarning("  WARNING: Unknown primitive: 0x%lX\n", thePrimitive->dl_primitive);
		}
		break;
	}
}



	void
LogReadDLPI(
	mblk_t *	inMessage)
{
	DL_primitives *	thePrimitive = reinterpret_cast<DL_primitives *>(inMessage->b_rptr);
	
	switch (thePrimitive->dl_primitive) {
		case DL_INFO_ACK:
		{
			dl_info_ack_t *	ack = &thePrimitive->info_ack;
			
			DConLogNote("  DL_INFO_ACK\n");
			DConLogNote("               dl_max_sdu: %lu\n", ack->dl_max_sdu);
			DConLogNote("               dl_min_sdu: %lu\n", ack->dl_min_sdu);
			DConLogNote("           dl_addr_length: %lu\n", ack->dl_addr_length);
			DConLogNote("              dl_mac_type: 0x%lX\n", ack->dl_mac_type);
			DConLogNote("         dl_current_state: %lu\n", ack->dl_current_state);
			DConLogNote("            dl_sap_length: %ld\n", ack->dl_sap_length);
			DConLogNote("          dl_service_mode: 0x%lX\n", ack->dl_service_mode);
			DConLogNote("            dl_qos_length: %lu\n", ack->dl_qos_length);
			DConLogNote("            dl_qos_offset: %lu\n", ack->dl_qos_offset);
			DConLogNote("      dl_qos_range_length: %lu\n", ack->dl_qos_range_length);
			DConLogNote("      dl_qos_range_offset: %lu\n", ack->dl_qos_range_offset);
			DConLogNote("        dl_provider_style: 0x%lX\n", ack->dl_provider_style);
			DConLogNote("           dl_addr_offset: %lu\n", ack->dl_addr_offset);
			DConLogNote("               dl_version: %lu\n", ack->dl_version);
			DConLogNote("    dl_brdcst_addr_length: %lu\n", ack->dl_brdcst_addr_length);
			DConLogNote("    dl_brdcst_addr_offset: %lu\n", ack->dl_brdcst_addr_offset);
		}
		break;
		
		
		case DL_BIND_ACK:
		{
			dl_bind_ack_t *	ack = &thePrimitive->bind_ack;

			DConLogNote("  DL_BIND_ACK\n");
			DConLogNote("            dl_sap: %lu\n", ack->dl_sap);
			DConLogNote("    dl_addr_length: %lu\n", ack->dl_addr_length);
			DConLogNote("    dl_addr_offset: %lu\n", ack->dl_addr_offset);
			DConLogNote("     dl_max_conind: 0x%lX\n", ack->dl_max_conind);
			DConLogNote("    dl_xidtest_flg: %lu\n", ack->dl_xidtest_flg);
		}
		break;
		
		
		case DL_SUBS_BIND_ACK:
		{
			DConLogNote("  DL_SUBS_BIND_ACK\n");
		}
		break;


		case DL_OK_ACK:
		{
			dl_ok_ack_t *	ack = &thePrimitive->ok_ack;

			DConLogNote("  DL_OK_ACK\n");
			switch (ack->dl_correct_primitive) {
				case DL_UNBIND_REQ:
				{
					DConLogNote("    dl_correct_primitive: DL_UNBIND_REQ\n");
				}
				break;
				
				
				case DL_SUBS_UNBIND_REQ:
				{
					DConLogNote("    dl_correct_primitive: DL_SUBS_UNBIND_REQ\n");
				}
				break;
				
				
				case DL_ENABMULTI_REQ:
				{
					DConLogNote("    dl_correct_primitive: DL_ENABMULTI_REQ\n");
				}
				break;
				
				
				case DL_DISABMULTI_REQ:
				{
					DConLogNote("    dl_correct_primitive: DL_DISABMULTI_REQ\n");
				}
				break;
				
				
				default:
				{
					DConLogWarning("    WARNING: Unknown OK_ACK: 0x%lX\n", ack->dl_correct_primitive);
				}
				break;
			}
		}
		break;
		
		
		case DL_UDERROR_IND:
		{
			dl_uderror_ind_t *	ack = &thePrimitive->uderror_ind;

			DConLogNote("  DL_UDERROR_IND\n");
			DConLogNote("    dl_unix_errno: %lu\n", ack->dl_unix_errno);
			DConLogNote("         dl_errno: %lu\n", ack->dl_errno);
			DConDumpNote("Address: ", inMessage->b_rptr + ack->dl_dest_addr_offset,
				ack->dl_dest_addr_length);
		}
		break;
		
		
		case DL_UNITDATA_IND:
		{
			dl_unitdata_ind_t *	ack = &thePrimitive->unitdata_ind;

			DConLogNote("  DL_UNITDATA_IND\n");
			DConLogNote("    dl_group_address: %lu\n", ack->dl_group_address);
			DConDumpNote("Destination Address: ", inMessage->b_rptr + ack->dl_dest_addr_offset,
				ack->dl_dest_addr_length);
			DConDumpNote("Source Address: ", inMessage->b_rptr + ack->dl_src_addr_offset,
				ack->dl_src_addr_length);
			DConLogNote("Data: ");
			LogDATA(inMessage->b_cont);
		}
		break;
		
		
		case DL_PHYS_ADDR_ACK:
		{
			dl_phys_addr_ack_t *	ack = &thePrimitive->physaddr_ack;

			DConLogNote("  DL_PHYS_ADDR_ACK\n");
			DConDumpNote("Address: ", inMessage->b_rptr + ack->dl_addr_offset,
				ack->dl_addr_length);
		}
		break;
		
		
		default:
		{
			DConLogWarning("  WARNING: Unknown primitive: 0x%lX\n", thePrimitive->dl_primitive);
		}
		break;
	}
}



	void
LogDATA(
	mblk_t *	inMessage)
{
	while (inMessage != NULL) {
		if (inMessage->b_datap->db_type != M_DATA) {
			DConLogWarning("WARNING: LogDATA encountered a non-M_DATA message: 0%o\n",
				(unsigned int) inMessage->b_datap->db_type);
			inMessage = NULL;
		} else {
			DConDumpNote("", inMessage->b_rptr, inMessage->b_wptr - inMessage->b_rptr);
			inMessage = inMessage->b_cont;
		}
	}
}

#endif