/*____________________________________________________________________________
	Copyright (C) 1998 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPDiskFile.c,v 1.10 1999/03/15 17:43:41 heller Exp $
____________________________________________________________________________*/

#include "MacSecureMemory.h"
#include "MacStrings.h"
#include "pgpMacMemory.h"

#include "PGPDisk.h"
#include "PGPDiskEncryptDecrypt.h"
#include "PGPDiskFile.h"
#include "PGPDiskFileFormat.h"
#include "PGPDiskUtils.h"
#include "pgpErrors.h"
#include "pgpKeys.h"
#include "pgpPublicKey.h"
#include "pgpsdkPrefs.h"
#include "pgpUtilities.h"

#define MIN(a,b)		( (a) <= (b) ? (a) : (b) )

#define	kPGPDiskFileMagic	((OSType) 'DFIL')
#define	kPGPDiskKeyMagic	((OSType) 'DKEY')

typedef struct PGPDiskHeaderListItem
{
	struct PGPDiskHeaderListItem	*next;
	PGPDiskFileHeaderInfo			*header;
	
} PGPDiskHeaderListItem;

typedef struct PGPDiskFile
{
	OSType			magic;			// kPGPDiskFileMagic
	FSSpec			fileSpec;
	UInt32			keyCount;
	PGPContextRef	context;
	PGPKeySetRef	keySet;

	PGPDiskFileHeader		*mainHeader;
	PGPDiskHeaderListItem	*headerList;

} PGPDiskFile;
	
typedef struct PGPDiskKey
{
	OSType				magic;		// kPGPDiskKeyMagic
	
	Boolean				isPassphraseKey;
	Boolean				isMasterKey;
	Boolean				deleted;
	PGPDiskFileRef		diskFile;
	
	union
	{
		PassphraseKey			*passKey;
		long					dummy;
		PGPDiskPublicKeyHeader	*publicKey;
	};
	
} PGPDiskKey;

const MacMemoryAllocFlags kMemAllocationFlags = kMacMemory_UseApplicationHeap |
													kMacMemory_UseTempMem |
													kMacMemory_ClearBytes;

static OSStatus		GetIndKey(PGPDiskFileRef diskFileRef, UInt32 index,
							Boolean searchPassphraseKeys,
							Boolean searchPublicKeys,
							PGPDiskKeyRef *keyRefPtr);
	
	
	static OSStatus
ReadPGPDiskFileHeaderAtOffset(
	short 					fileRef,
	UInt32					offset,
	PGPDiskFileHeaderInfo	**header)
{
	OSStatus				err;
	PGPDiskFileHeaderInfo	info;
	
	AssertFileRefNumIsValid( fileRef, "ReadPGPDiskFileHeaderAtOffset" );
	pgpAssertAddrValid( header, PGPDiskFileHeaderInfo * );

	*header = NULL;
	
	err = FSReadFileUsingDriver( fileRef, offset, sizeof( info ), &info,
					NULL );
	if( IsntErr( err ) )
	{
		if( info.headerMagic == kPGPDiskHeaderMagic )
		{
			*header = (PGPDiskFileHeaderInfo *) pgpAllocMac( info.headerSize,
								kMemAllocationFlags );
			if( IsntNull( *header ) )
			{
				err = FSReadFileUsingDriver( fileRef, offset,
								info.headerSize, *header, NULL );
			}
			else
			{
				err = memFullErr;
			}
		}
		else
		{
			err = kPGPDiskInvalidFileHeaderError;
		}
	}
	
	if( IsErr( err ) && IsntNull( *header ) )
	{
		pgpFreeMac( *header );
		*header = NULL;
	}
	
	return( err );
}

	static CRC32
ComputePGPDiskFileHeaderCRC(const PGPDiskFileHeaderInfo *headerInfo)
{
	CRC32	saveHeaderCRC;
	CRC32	crc;
	
	pgpAssertAddrValid( headerInfo, PGPDiskFileHeaderInfo );

	// Always set CRC and next offset value to zero before computing.
	// Version 2.0 file format did not set nextHeaderOffset to zero
	// before computing CRC, but it was always zero so no problem.
	
	saveHeaderCRC 	= headerInfo->headerCRC;
	
	((PGPDiskFileHeaderInfo *) headerInfo)->headerCRC = 0;
	
	crc = ComputeCRC32( headerInfo, headerInfo->headerSize );
	
	((PGPDiskFileHeaderInfo *) headerInfo)->headerCRC = saveHeaderCRC;
	
	return( crc );
}

	static OSStatus
VerifyPGPDiskFileMainHeader(
	const PGPDiskFileHeader *fileHeader,
	UInt32					fileSize)
{
	if( fileHeader->headerInfo.headerType != kPGPDiskPrimaryHeaderType )
	{
		pgpDebugMsg( "VerifyPGPDiskFileMainHeader: Invalid header type" );
		return( kPGPDiskInvalidFileHeaderError );
	}

	if( fileHeader->majorVersion > kPGPDiskPrimaryHeaderMajorVersion )
	{
		pgpDebugMsg( "VerifyPGPDiskFileMainHeader: Newer header version" );
		return( kPGPDiskNewerFileVersionError );
	}

	if( fileHeader->algorithm != kCASTEncryptionAlgorithm )
	{
		pgpDebugMsg( "VerifyPGPDiskFileMainHeader: Unknown algorithm" );
		return( kPGPDiskUnknownAlgorithmError );
	}

	if( fileHeader->numFileBlocks * kDiskBlockSize != fileSize )
	{
		pgpDebugMsg( "VerifyPGPDiskFileMainHeader: Missing part of the file");
		return( eofErr );
	}

	if( fileHeader->numHeaderBlocks > fileHeader->numFileBlocks )
	{
		pgpDebugMsg( "VerifyPGPDiskFileMainHeader: Too many header blocks" );
		return( kPGPDiskInvalidFileHeaderError );
	}

	if( fileHeader->numDataBlocks > fileHeader->numFileBlocks )
	{
		pgpDebugMsg( "VerifyPGPDiskFileMainHeader: Too many data blocks" );
		return( kPGPDiskInvalidFileHeaderError );
	}

	if( fileHeader->numHeaderBlocks + fileHeader->numDataBlocks >
				fileHeader->numFileBlocks )
	{
		pgpDebugMsg( "VerifyPGPDiskFileMainHeader: Too many blocks" );
		return( kPGPDiskInvalidFileHeaderError );
	}
			
	return( noErr );
}

	static OSStatus
VerifyPGPDiskFileHeader(
	const PGPDiskFileHeaderInfo *fileHeader,
	UInt32						fileSize)
{
	OSStatus	err = noErr;
	
	pgpAssertAddrValid( fileHeader, PGPDiskFileHeaderInfo );
	pgpAssert( ( fileSize % kDiskBlockSize ) == 0 );
	pgpAssert( fileSize > 64 * 1024L );

	if( fileHeader->headerMagic != kPGPDiskHeaderMagic )
	{
		pgpDebugMsg( "VerifyPGPDiskFileHeader: Invalid header magic" );
		return( kPGPDiskInvalidFileHeaderError );
	}

	if( fileHeader->headerCRC != ComputePGPDiskFileHeaderCRC( fileHeader ) )
	{
		pgpDebugMsg( "VerifyPGPDiskFileHeader: Bad header CRC" );
		return( kPGPDiskInvalidFileHeaderError );
	}

	switch( fileHeader->headerType )
	{
		case kPGPDiskPrimaryHeaderType:
		{
			err = VerifyPGPDiskFileMainHeader(
							(PGPDiskFileHeader *) fileHeader, fileSize );
			break;
		}
		
		default:
			break;
	}
	
	return( err );
}

	static OSStatus
ReadFileHeaderItemAtOffset(
	short					fileRef,
	UInt32					offset,
	UInt32					fileSize,
	PGPDiskHeaderListItem	**headerItem)
{
	OSStatus	err;
	
	*headerItem = (PGPDiskHeaderListItem *) pgpAllocMac(
						sizeof( **headerItem ), kMemAllocationFlags );
	if( IsntNull( *headerItem ) )
	{
		(*headerItem)->next 	= NULL;
		(*headerItem)->header 	= NULL;
		
		/* Read and verify main header. */
		err = ReadPGPDiskFileHeaderAtOffset( fileRef, offset,
					&(*headerItem)->header );
		if( IsntErr( err ) )
		{
			err = VerifyPGPDiskFileHeader( (*headerItem)->header, fileSize );
			if( IsErr( err ) )
			{
				pgpFreeMac( (*headerItem)->header );
				(*headerItem)->header = NULL;
			}
		}
		
		if( IsErr( err ) )
		{
			pgpFreeMac( *headerItem );
			*headerItem = NULL;
		}
	}
	else
	{
		err = memFullErr;
	}
	
	return( err );
}

	OSStatus
OpenPGPDiskFile(
	PGPContextRef	context,
	const FSSpec	*fileSpec,
	PGPDiskFileRef	*diskFileRefPtr)
{
	OSStatus		err = noErr;
	PGPDiskFileRef	diskFileRef;
	
	pgpAssertAddrValid( fileSpec, FSSpec );
	pgpAssertAddrValid( diskFileRefPtr, PGPDiskFileRef );
	
	diskFileRef = (PGPDiskFileRef) pgpAllocMac( sizeof( *diskFileRef ),
						kMemAllocationFlags );
	if( PGPDiskFileRefIsValid( diskFileRef ) )
	{
		short	fileRef;
		
		diskFileRef->magic		= kPGPDiskFileMagic;
		diskFileRef->fileSpec	= *fileSpec;
		diskFileRef->context	= context;
		
		err = FSpOpenDF( fileSpec, fsRdPerm, &fileRef );
		if( IsntErr( err ) )
		{
			long	fileSize;
			
			err = GetEOF( fileRef, &fileSize );
			if( IsntErr( err ) )
			{
				PGPDiskHeaderListItem	*mainHeaderItem;
				PGPDiskHeaderListItem	*curHeaderItem;
				PGPDiskFileHeader		*mainHeader;
				UInt32					headerOffset;
				
				/* Read and verify main header. */

				err = ReadFileHeaderItemAtOffset( fileRef, 0, fileSize,
							&mainHeaderItem );
				if( IsErr( err ) )
				{
					/* If an error occured, try for alternate main header */

					err = ReadFileHeaderItemAtOffset( fileRef, fileSize -
									kPGPDiskAlternateHeaderSize, fileSize,
									&mainHeaderItem );
				}
				
				if( IsntErr( err ) )
				{
					mainHeader = (PGPDiskFileHeader *) mainHeaderItem->header;
					
					diskFileRef->mainHeader	= mainHeader;
					diskFileRef->headerList	= mainHeaderItem;

					// Fix up passphrase count value. Version 2.0 of the file
					// format did not have a valid value for this field.
					
					if( mainHeader->numPassphrases == 0 )
					{
						mainHeader->numPassphrases = kNumPassphrases;
					}
				}
				
				curHeaderItem = diskFileRef->headerList;
				
				headerOffset = mainHeader->headerInfo.nextHeaderOffset.lo;
				while( headerOffset != 0 && IsntErr( err ) )
				{
					PGPDiskHeaderListItem	*headerItem;
					
					err = ReadFileHeaderItemAtOffset( fileRef, headerOffset,
								fileSize, &headerItem );
					if( IsntErr( err ) )
					{
						// Link item at end of header list.
						// List has one guaranteed item
					
						headerItem->next		= NULL;
						curHeaderItem->next		= headerItem;
						curHeaderItem			= headerItem;

						headerOffset =
								headerItem->header->nextHeaderOffset.lo;
					}	
				}
			}
			
			(void) FSClose( fileRef );
		}

		if( IsErr( err ) )
		{
			ClosePGPDiskFile( diskFileRef );
			diskFileRef = kInvalidPGPDiskFileRef;
		}
	}
	else
	{
		err = memFullErr;
	}
		
	*diskFileRefPtr = diskFileRef;
	
	return( err );
}

	static OSStatus
VerifyPGPDiskFileRef(PGPDiskFileRef diskFileRef)
{
	pgpAssertAddrValid( diskFileRef, PGPDiskFile );

	if( PGPDiskFileRefIsValid( diskFileRef ) &&
		diskFileRef->magic == kPGPDiskFileMagic )
	{
		return( noErr );
	}
	
	return( paramErr );
}

	void
ClosePGPDiskFile(PGPDiskFileRef diskFileRef)
{
	if( IsntErr( VerifyPGPDiskFileRef( diskFileRef ) ) )
	{
		PGPDiskHeaderListItem	*curHeader;

		pgpAssert( diskFileRef->keyCount == 0 );
		
		curHeader = diskFileRef->headerList;
		while( IsntNull( curHeader ) )
		{
			PGPDiskHeaderListItem	*nextHeader;
			
			nextHeader = curHeader->next;
			
			if( IsntNull( curHeader->header ) )
				pgpFreeMac( curHeader->header );
			
			pgpFreeMac( curHeader );
			
			curHeader = nextHeader;
		}
		
		if( PGPKeySetRefIsValid( diskFileRef->keySet ) )
		{
			PGPFreeKeySet( diskFileRef->keySet );
			diskFileRef->keySet = kInvalidPGPKeySetRef;
		}
		
		pgpFreeMac( diskFileRef );
	}

}

	static OSStatus
WritePGPDiskFileHeaderAtOffset(
	short 						fileRef,
	UInt32						offset,
	const PGPDiskFileHeaderInfo	*header)
{
	OSStatus	err;
	
	AssertFileRefNumIsValid( fileRef, "ReadPGPDiskFileHeaderAtOffset" );
	pgpAssertAddrValid( header, PGPDiskFileHeaderInfo );

	err = FSWriteFileUsingDriver( fileRef, offset, header->headerSize,
				header, NULL );
	
	return( err );
}

	static OSStatus
CopyHeader(
	const PGPDiskFileHeaderInfo	*header,
	PGPDiskFileHeaderInfo		**headerCopy)
{
	OSStatus	err = noErr;
	
	*headerCopy = (PGPDiskFileHeaderInfo *) pgpAllocMac( header->headerSize,
							kMemAllocationFlags );
	if( IsntNull( *headerCopy ) )
	{
		pgpCopyMemory( header, *headerCopy, header->headerSize );
		
		(*headerCopy)->nextHeaderOffset.hi	= 0;
		(*headerCopy)->nextHeaderOffset.lo	= 0;
	}
	else
	{
		err = memFullErr;
	}

	return( err );
}

	static void
UpdateHeaderList(PGPDiskFileRef diskFileRef)
{
	PGPDiskHeaderListItem	*curHeaderItem;
	PGPDiskFileHeader		*mainHeader;
	UInt32					curOffset;
	UInt32					remainingBytes;
	UInt32					minFileSize;
	UInt32					newFileSize;
	
	curOffset		= 0;
	mainHeader 		= diskFileRef->mainHeader;
	remainingBytes	= mainHeader->numHeaderBlocks * kDiskBlockSize;
			
	// Compute offsets in file for each header
	// For each header, fill in the actual header's offset in
	// the next header offset field first.
	
	curHeaderItem = diskFileRef->headerList;
	while( IsntNull( curHeaderItem ) )
	{
		PGPDiskFileHeaderInfo	*curHeader = curHeaderItem->header;
		
		pgpAssert( IsntNull( curHeader ) );

		if( remainingBytes > 0 )
		{
			if( curHeader->headerSize < remainingBytes )
			{
				// OK to put at current offset
				// Decrement remaining bytes
				
				remainingBytes -= curHeader->headerSize;
			}
			else
			{
				// Exhausted bytes at the beginning of the file.
				// Moving offset to first byte after data blocks.
				
				remainingBytes	= 0;
				curOffset		= ( mainHeader->numHeaderBlocks +
									mainHeader->numDataBlocks ) *
										kDiskBlockSize;
			}
		}
		else
		{
			// Already after data blocks. OK to place header.
		}
				
		curHeader->nextHeaderOffset.hi = 0;
		curHeader->nextHeaderOffset.lo = curOffset;

		curOffset += curHeader->headerSize;
		
		curHeaderItem = curHeaderItem->next;
	}
	
	curHeaderItem = diskFileRef->headerList;
	while( IsntNull( curHeaderItem->next ) )
	{
		curHeaderItem = curHeaderItem->next;
	}
	
	minFileSize = ( mainHeader->numHeaderBlocks + mainHeader->numDataBlocks ) *
						kDiskBlockSize;
	newFileSize = curHeaderItem->header->nextHeaderOffset.lo +
						curHeaderItem->header->headerSize;
	
	if( newFileSize < minFileSize )
		newFileSize = minFileSize;
		
	mainHeader->numFileBlocks = ( ( newFileSize + kDiskBlockSize - 1 ) /
										kDiskBlockSize ) +
											kPGPDiskAlternateHeaderBlocks;

	// Store real values for next header offset in heach header and
	// recompute the CRC's for each header.
	
	curHeaderItem = diskFileRef->headerList;
	while( IsntNull( curHeaderItem ) )
	{
		UInt32	nextOffset;
		
		if( IsntNull( curHeaderItem->next ) )
		{
			nextOffset = curHeaderItem->next->header->nextHeaderOffset.lo;
		}
		else
		{
			nextOffset = 0;
		}
			
		curHeaderItem->header->nextHeaderOffset.lo = nextOffset;
		curHeaderItem->header->headerCRC =
					ComputePGPDiskFileHeaderCRC( curHeaderItem->header );
		
		curHeaderItem = curHeaderItem->next;
	}
}

	static OSErr
FSWipeBytesUsingDriver(
	short	fileRef,
	UInt32	startOffset,
	UInt32	numBytes,
	uchar	wipeChar )
{
	OSStatus	err	= noErr;
	
	err	= SetFPos( fileRef, fsFromStart, startOffset );
	if ( IsntErr( err ) )
	{
		void *		buffer	= nil;
		UInt32		bufferSize;
		uchar		emergencyBuffer[ 512 ];
		Boolean		disposeBuffer	= false;
		UInt32		remaining;
		UInt32		writeOffset;
		
		buffer = pgpNewPtrMost( numBytes, MIN( 512, numBytes ),
						kMemAllocationFlags, &bufferSize );
		if ( IsntNull( buffer ) )
		{
			disposeBuffer	= TRUE;
		}
		else
		{
			buffer		= &emergencyBuffer[ 0 ];
			bufferSize	= sizeof( emergencyBuffer );
		}
		
		
		pgpFillMemory( buffer, bufferSize, wipeChar );
		
		writeOffset	= startOffset;
		remaining	= numBytes;
		while( remaining != 0 )
		{
			long	count;
			
			count	= MIN( remaining, bufferSize );
			err	= FSWriteFileUsingDriver( fileRef, writeOffset, count,
						buffer, nil );
			if ( IsErr( err ) )
				break;
				
			remaining	-= count;
			writeOffset	+= count;
		}
		
		if ( disposeBuffer )
		{
			pgpFreeMac( buffer );
		}
	}
	
	return( err );
}

	OSStatus
SavePGPDiskFile(PGPDiskFileRef diskFileRef)
{
	OSStatus	err = noErr;
	
	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		short	fileRef;
		
		UpdateHeaderList( diskFileRef );
		
		(void) FlushVol( nil, diskFileRef->fileSpec.vRefNum );

		err = FSpOpenDF( &diskFileRef->fileSpec, fsRdWrPerm, &fileRef );
		if( IsntErr( err ) )
		{
			long					fileSize;
			const PGPDiskFileHeader	*mainHeader;
			
			mainHeader 	= diskFileRef->mainHeader;
			fileSize 	= mainHeader->numFileBlocks * kDiskBlockSize;
			
			err = SetEOF( fileRef, fileSize );
			if( IsntErr( err ) )
			{
				PGPDiskHeaderListItem	*curHeaderItem;
				UInt32					headerOffset;
				
				// wipe header blocks and trailer block so people don't
				// think there is a security issue
				
				err	= FSWipeBytesUsingDriver( fileRef, 0UL,
								mainHeader->numHeaderBlocks *
								kDiskBlockSize, 0 );
				if ( IsntErr( err ) )
				{
					PGPUInt32	trailingBytes;
					
					trailingBytes = ( mainHeader->numFileBlocks - 
										mainHeader->numHeaderBlocks -
											mainHeader->numDataBlocks ) *
												kDiskBlockSize;
												
					err	= FSWipeBytesUsingDriver( fileRef, fileSize -
								trailingBytes, trailingBytes, 0 );
				}

				headerOffset = 0;
				
				curHeaderItem = diskFileRef->headerList;
				while( IsntNull( curHeaderItem ) &&
						IsntErr( err ) )
				{
					PGPDiskFileHeaderInfo	*curHeader = curHeaderItem->header;
				
					err = VerifyPGPDiskFileHeader( curHeader, fileSize );
					if( IsntErr( err ) )
					{
						err = WritePGPDiskFileHeaderAtOffset( fileRef,
									headerOffset, curHeader );
					}
					
					headerOffset = curHeader->nextHeaderOffset.lo;
					
					curHeaderItem = curHeaderItem->next;
				}
			
				// Always write the alternate header.
				
				(void) WritePGPDiskFileHeaderAtOffset( fileRef,
								fileSize - kPGPDiskAlternateHeaderSize,
								&mainHeader->headerInfo );
			}
			
			(void) FSClose( fileRef );
			(void) FlushVol( nil, diskFileRef->fileSpec.vRefNum );
		}
	}
	
	return( err );
}

	static UInt32
CalcNumDataBlocks(
	short	vRefNum,
	UInt32	inDiskSizeInK
	)
{
	const UInt32	kOverheadBlocks = kPGPDiskReservedHeaderBlocks +
											kPGPDiskAlternateHeaderBlocks;
	
	UInt32			diskSizeInBlocks;
	UInt32			allocBlockSize;
	UInt32			blocksPerAllocBlock;
	UInt32			freeBlocks;
	
	diskSizeInBlocks	= ( inDiskSizeInK * 1024UL ) / kDiskBlockSize;
	allocBlockSize		= GetVolumeAllocBlockSize( vRefNum );
	blocksPerAllocBlock	= allocBlockSize / kDiskBlockSize;
	freeBlocks			= GetVolumeFreeAllocBlocks( vRefNum ) *
								blocksPerAllocBlock;
	
	if ( diskSizeInBlocks + kOverheadBlocks > freeBlocks )
	{
		diskSizeInBlocks	= freeBlocks - kOverheadBlocks;
	}
	
	return( diskSizeInBlocks );
}

	static OSStatus
DecryptPassphraseKey(
	const PassphraseKey		*passKey,
	const PassphraseSalt 	*salt,
	ConstStr255Param		passphrase,
	CASTKey					*decryptedKey)
{
	EncryptedCASTKey	tempKey;
	ExpandedCASTKey		expandedKey;
	Boolean				validPassphrase;
	CheckBytes			checkBytes;
	CMemoryWiper		tempKeyWiper( &tempKey, sizeof( tempKey ) );
	CMemoryWiper		expandedKeyWiper(&expandedKey, sizeof( expandedKey ));
	CMemoryWiper		checkBytesWiper( &checkBytes, sizeof( checkBytes ) );
	short				hashReps;
	OSStatus			err = noErr;
	
	// Hash the old passphrase and compute the salted version
	HashBuf( &passphrase[1], passphrase[0], &tempKey );

	hashReps = passKey->hashReps;
	SaltPassphrase( &tempKey, &tempKey, salt, &hashReps );

	// Expand the old key for decryption purposes.
	CAST5schedule( expandedKey.keyWords, tempKey.keyBytes );

	// Decrypt the encrypted key
	*decryptedKey = passKey->encryptedKey;
	CAST5decrypt( &decryptedKey->keyBytes[0], &decryptedKey->keyBytes[0],
			expandedKey.keyWords );
	CAST5decrypt( &decryptedKey->keyBytes[8], &decryptedKey->keyBytes[8],
			expandedKey.keyWords );

	// Get the ckeck bytes
	checkBytes = passKey->checkBytes;
	
	// Decrypt the check bytes using the expanded key
	CAST5decrypt( checkBytes.theBytes, checkBytes.theBytes,
			expandedKey.keyWords );
	
	validPassphrase = pgpMemoryEqual( &checkBytes.theBytes[ 0 ],
							&tempKey.keyBytes[ 0 ], sizeof( checkBytes ) );
	if( ! validPassphrase )
	{
		err = kPGPDiskIncorrectPassphraseError;
	}
	
	return( err );
}

	static void
EncryptPassphraseKey(
	CASTKey					*decryptedKey,
	const PassphraseSalt 	*salt,
	ConstStr255Param		passphrase,
	PassphraseKey			*passKey)
{
	EncryptedCASTKey	tempKey;
	ExpandedCASTKey		expandedKey;
	short				index;
	CMemoryWiper		tempKeyWiper( &tempKey, sizeof( tempKey ) );
	CMemoryWiper		expandedKeyWiper( &expandedKey, sizeof(expandedKey ));
	
	// Compute the hashed, salted key for the passphrase	
	passKey->hashReps = 0;
	HashBuf( &passphrase[1], passphrase[0], &tempKey );
	SaltPassphrase( &tempKey, &tempKey, salt, &passKey->hashReps );

	// Expand the key for encryption purposes
	CAST5schedule( expandedKey.keyWords, tempKey.keyBytes );

	// Encrypt the new salted passphrase
	passKey->encryptedKey = *decryptedKey;
	CAST5encrypt( &passKey->encryptedKey.keyBytes[0],
			&passKey->encryptedKey.keyBytes[0], expandedKey.keyWords );
	CAST5encrypt( &passKey->encryptedKey.keyBytes[8],
			&passKey->encryptedKey.keyBytes[8], expandedKey.keyWords );
	
	// Compute the checkbytes. The checkbytes are the encrypted hashed salted
	// user key.
	CAST5encrypt( &tempKey.keyBytes[0], &tempKey.keyBytes[0],
			expandedKey.keyWords );
	
	for( index = 0; index < sizeof( passKey->checkBytes ); index++ )
	{
		passKey->checkBytes.theBytes[ index ] = tempKey.keyBytes[ index ];
	}
}

	static OSStatus
InitPGPDiskFileHeader(
	PGPContextRef				context,
	short						vRefNum,
	const UInt32				inDiskSizeInK,
	PGPDiskEncryptionAlgorithm	algorithm,
	const uchar 				*passphrase,
	PGPDiskFileHeader			*fileHeader)
{
	EncryptedCASTKey	key;
	ExpandedCASTKey		expandedKey;
	ItemCount			index;
	CMemoryWiper		keyWiper( &key, sizeof( key ) );
	CMemoryWiper		expandedKeyWiper( &expandedKey, sizeof(expandedKey) );
	PassphraseKey		*masterPassphrase;
	PGPError			pgpErr;
	
	pgpAssertAddrValid( fileHeader, PGPDiskFileHeader );
	
	pgpClearMemory( fileHeader, sizeof( *fileHeader ) );
	
	fileHeader->headerInfo.headerMagic	= kPGPDiskHeaderMagic;
	fileHeader->headerInfo.headerType	= kPGPDiskPrimaryHeaderType;
	fileHeader->headerInfo.headerSize	= sizeof( *fileHeader );

	fileHeader->majorVersion		= kPGPDiskPrimaryHeaderMajorVersion;
	fileHeader->minorVersion 		= kPGPDiskPrimaryHeaderMinorVersion;
	
	fileHeader->algorithm 			= algorithm;
	fileHeader->numPassphrases		= kNumPassphrases;
	
	fileHeader->numHeaderBlocks	= kPGPDiskReservedHeaderBlocks;
	fileHeader->numDataBlocks	= CalcNumDataBlocks( vRefNum, inDiskSizeInK );
	fileHeader->numFileBlocks 	= kPGPDiskReservedHeaderBlocks +
										fileHeader->numDataBlocks + 
										kPGPDiskAlternateHeaderBlocks;

	pgpErr = PGPContextGetRandomBytes( context, fileHeader->salt.saltBytes,
					sizeof( fileHeader->salt ) );
	if( IsntPGPError( pgpErr ) )
	{
		masterPassphrase = &fileHeader->passphrases[kMasterPassphraseIndex];
		
		pgpErr = PGPContextGetRandomBytes( context, masterPassphrase->encryptedKey.keyBytes,
					sizeof( masterPassphrase->encryptedKey ) );
	}
	
	if( IsPGPError( pgpErr ) )
	{
		return( PGPErrorToMacError( pgpErr ) );
	}	

	// Hash and salt the passphrase	
	HashBuf( &passphrase[1], passphrase[0], &key );
	SaltPassphrase( &key, &key, &fileHeader->salt,
			&masterPassphrase->hashReps );

	// Expand the key for encrpytion purposes
	CAST5schedule( expandedKey.keyWords, key.keyBytes );
	
	// Encrypt the random key twice
	CAST5encrypt( &masterPassphrase->encryptedKey.keyBytes[0],
					&masterPassphrase->encryptedKey.keyBytes[0],
					expandedKey.keyWords );
	CAST5encrypt( &masterPassphrase->encryptedKey.keyBytes[8],
					&masterPassphrase->encryptedKey.keyBytes[8],
					expandedKey.keyWords );

	CAST5encrypt( &masterPassphrase->encryptedKey.keyBytes[0],
					&masterPassphrase->encryptedKey.keyBytes[0],
					expandedKey.keyWords );
	CAST5encrypt( &masterPassphrase->encryptedKey.keyBytes[8],
					&masterPassphrase->encryptedKey.keyBytes[8],
					expandedKey.keyWords );

	// Encrypt the key itself for checkbytes purposes
	CAST5encrypt( key.keyBytes, key.keyBytes, expandedKey.keyWords);

	for( index = 0; index < sizeof( masterPassphrase->checkBytes ); index++ )
	{
		masterPassphrase->checkBytes.theBytes[index] = key.keyBytes[index];
	}
	
	masterPassphrase->inUse = TRUE;
	
	fileHeader->headerInfo.headerCRC =
				ComputePGPDiskFileHeaderCRC( &fileHeader->headerInfo );
				
	return( noErr );
}

	OSStatus
NewPGPDiskFile(
	PGPContextRef				context,
	const FSSpec				*fileSpec,
	PGPUInt32					diskSizeInK,
	PGPDiskEncryptionAlgorithm	algorithm,
	ConstStr255Param			masterPassphrase)
{
	PGPDiskFileHeader	header;
	OSStatus			err;
	
	AssertSpecIsValid( fileSpec, nil );
	pgpAssertAddrValid( masterPassphrase, uchar );

	err = InitPGPDiskFileHeader( context, fileSpec->vRefNum,
				diskSizeInK, algorithm, masterPassphrase, &header );
	if( IsntErr( err ) )
	{
		(void) FSpDelete( fileSpec );
		
		err = FSpCreate( fileSpec, kPGPDiskFileCreator, kPGPDiskFileType,
								smSystemScript );
	}
	
	if( IsntErr( err ) )
	{
		short	fileRef;
		
		// Create and initialize the data fork	
		err = FSpOpenDF( fileSpec, fsRdWrPerm, &fileRef );
		if( IsntErr( err ) )
		{
			UInt32	allocSize;
			UInt32	fileSize;
			
			fileSize = header.numFileBlocks * kDiskBlockSize;

			// Try for contiguous allocation on the file. If this fails,
			// SetEOF will allocate the file using whatever policy is
			// available

			allocSize = fileSize;		
			(void) AllocContig( fileRef, (long *) &allocSize );
			
			err = SetEOF( fileRef, fileSize );
			if( IsntErr( err ) )
			{
				// wipe header blocks and trailer block so people don't
				// think there is a security issue
				err	= FSWipeBytesUsingDriver( fileRef, 0UL,
								kPGPDiskReservedHeaderBlocks *
								kDiskBlockSize, 0 );
				if ( IsntErr( err ) )
				{
					err	= FSWipeBytesUsingDriver( fileRef,
									fileSize - kPGPDiskAlternateHeaderSize,
									kPGPDiskAlternateHeaderSize, 0 );
				}
			}
			
			if( IsntErr( err ) )
			{
				err = WritePGPDiskFileHeaderAtOffset( fileRef, 0,
							&header.headerInfo );
				if( IsntErr( err ) )
				{
					err = WritePGPDiskFileHeaderAtOffset( fileRef,
								fileSize - kPGPDiskAlternateHeaderSize,
								&header.headerInfo );
				}
			}
			
			FSClose( fileRef );
			FlushVol( nil, fileSpec->vRefNum );
		}
	}
	
	if( IsErr( err ) )
		(void) FSpDelete( fileSpec );
		
	return( err );
}	

	static OSStatus
VerifyPGPDiskKeyRef(PGPDiskKeyRef keyRef)
{
	pgpAssertAddrValid( keyRef, PGPDiskKey );

	if( PGPDiskKeyRefIsValid( keyRef ) &&
		keyRef->magic == kPGPDiskKeyMagic )
	{
		return( noErr );
	}
	
	return( paramErr );
}	
	
	void
DisposePGPDiskKey(PGPDiskKeyRef keyRef)
{
	if( IsntErr( VerifyPGPDiskKeyRef( keyRef ) ) )
	{
		--keyRef->diskFile->keyCount;
		
		pgpClearMemory( keyRef, sizeof( keyRef ) );
		pgpFreeMac( keyRef );
	}
}

	static OSStatus
NewPassphrasePGPDiskKeyRef(
	PGPDiskFileRef	diskFileRef,
	PassphraseKey	*passphraseKey,
	Boolean			isMasterKey,
	PGPDiskKeyRef	*keyRefPtr)
{
	OSStatus		err = noErr;
	PGPDiskKeyRef	keyRef;

	keyRef = (PGPDiskKeyRef) pgpAllocMac( sizeof( *keyRef ),
					kMemAllocationFlags );
	if( PGPDiskKeyRefIsValid( keyRef ) )
	{
		keyRef->magic			= kPGPDiskKeyMagic;
		keyRef->isPassphraseKey	= TRUE;
		keyRef->isMasterKey		= isMasterKey;
		keyRef->diskFile		= diskFileRef;
		keyRef->passKey			= passphraseKey;
		
		++diskFileRef->keyCount;
	}
	else
	{
		err = memFullErr;
	}
	
	*keyRefPtr = keyRef;
	
	return( err );
}

	static OSStatus
VerifyPassphraseKey(
	const PassphraseKey *	userInfo,
	const PassphraseSalt *	salt,
	ConstStr255Param		passphrase )
{
	OSStatus			status = noErr;
	EncryptedCASTKey	key;
	ExpandedCASTKey		expandedKey;
	CheckBytes			checkBytes;
	short				hashReps;
	CMemoryWiper		keyWiper( &key, sizeof( key ) );
	CMemoryWiper		expandedKeyWiper( &expandedKey, sizeof(expandedKey) );
	CMemoryWiper		checkBytesWiper( &checkBytes, sizeof( checkBytes ) );
	
	pgpAssertAddrValid( userInfo, PassphraseKey );
	pgpAssertAddrValid( passphrase, uchar );
	pgpAssertAddrValid( salt, PassphraseSalt );
	
	pgpAssert( userInfo->inUse );
	
	hashReps = userInfo->hashReps;
	
	// Hash and salt the passphrase
	HashBuf( &passphrase[1], StrLength( passphrase ), &key );
	SaltPassphrase( &key, &key, salt, &hashReps );
	
	// Compute expanded key for decryption purposes.
	CAST5schedule( expandedKey.keyWords, key.keyBytes );
	
	// Get the ckeck bytes
	checkBytes = userInfo->checkBytes;
	
	// Decrypt the check bytes using the expanded key
	CAST5decrypt( checkBytes.theBytes, checkBytes.theBytes,
			expandedKey.keyWords );
	
	// Compare the check bytes against the key for equality.
	if ( ! pgpMemoryEqual( &checkBytes.theBytes[ 0 ], &key.keyBytes[ 0 ],
				sizeof( checkBytes ) )  )
	{
		status = kPGPDiskIncorrectPassphraseError;
	}

	return( status );
}

	OSStatus
FindPassphrasePGPDiskKey(
	PGPDiskFileRef		diskFileRef,
	ConstStr255Param	passphrase,
	PGPDiskKeyRef		*keyRefPtr)
{
	OSStatus		err 	= noErr;
	PGPDiskKeyRef	keyRef	= kInvalidPGPDiskKeyRef;
	
	pgpAssertAddrValid( keyRefPtr, PGPDiskKeyRef );
	
	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		UInt32	keyIndex;
		
		for( keyIndex = 0; keyIndex < diskFileRef->mainHeader->numPassphrases;
					++keyIndex )
		{
			PassphraseKey *	userInfo;
		
			userInfo = &diskFileRef->mainHeader->passphrases[keyIndex];
			if ( userInfo->inUse )
			{
				if ( IsntErr( VerifyPassphraseKey( userInfo,
							&diskFileRef->mainHeader->salt, passphrase ) ) )
				{
					err = NewPassphrasePGPDiskKeyRef( diskFileRef, userInfo,
								keyIndex == kMasterPassphraseIndex, &keyRef );
					break;
				}
			}
		}
		
		if( IsntErr( err ) && ! PGPDiskKeyRefIsValid( keyRef ) )
		{
			err = kPGPDiskIncorrectPassphraseError;
		}
	}

	*keyRefPtr = keyRef;
	
	return( err );
}

	OSStatus
GetMasterPGPDiskKey(
	PGPDiskFileRef	diskFileRef,
	PGPDiskKeyRef	*keyRefPtr)
{
	OSStatus		err 	= noErr;
	PGPDiskKeyRef	keyRef	= kInvalidPGPDiskKeyRef;
	
	pgpAssertAddrValid( keyRefPtr, PGPDiskKeyRef );
	
	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		PassphraseKey *	userInfo;
	
		userInfo =
			&diskFileRef->mainHeader->passphrases[kMasterPassphraseIndex];
		
		err = NewPassphrasePGPDiskKeyRef( diskFileRef, userInfo, TRUE,
					&keyRef );
	}

	*keyRefPtr = keyRef;
	
	return( noErr );
}

	OSStatus
VerifyPGPDiskKeyPassphrase(
	PGPDiskKeyRef		keyRef,
	ConstStr255Param	passphrase)
{
	OSStatus	err;
	
	pgpAssertAddrValid( passphrase, uchar );
	
	err = VerifyPGPDiskKeyRef( keyRef );
	if( IsntErr( err ) )
	{
		err = VerifyPassphraseKey( keyRef->passKey,
					&keyRef->diskFile->mainHeader->salt, passphrase );
	}

	return( err );
}

	OSStatus
ChangePGPDiskKeyPassphrase(
	PGPDiskKeyRef		keyRef,
	ConstStr255Param	oldPassphrase,
	ConstStr255Param	newPassphrase)
{
	OSStatus	err;
	
	err = VerifyPGPDiskKeyRef( keyRef );
	if( IsntErr( err ) )
	{
		if( keyRef->isPassphraseKey )
		{
			err = VerifyPGPDiskKeyPassphrase( keyRef, oldPassphrase );
			if( IsntErr( err ) )
			{
				CASTKey			decryptedKey;
				CMemoryWiper	decryptedKeyWiper( &decryptedKey,
										sizeof( decryptedKey ) );

				DecryptPassphraseKey( keyRef->passKey,
						&keyRef->diskFile->mainHeader->salt, oldPassphrase,
						&decryptedKey );
				EncryptPassphraseKey( &decryptedKey,
						&keyRef->diskFile->mainHeader->salt, newPassphrase,
						keyRef->passKey );
			}
		}
		else
		{
			pgpDebugMsg( "Cannot change passphrase on public key" );
			
			err = paramErr;
		}
	}
	
	return( err );
}

	OSStatus
GetPGPDiskKeyBoolean(
	PGPDiskKeyRef		keyRef,
	PGPDiskKeyProperty	property,
	Boolean				*value)
{
	OSStatus	err;
	
	err = VerifyPGPDiskKeyRef( keyRef );
	if( IsntErr( err ) )
	{
		switch( property )
		{
			case kPGPDiskKeyProperty_MountReadOnly:
			{
				if( keyRef->isPassphraseKey )
				{
					*value = keyRef->passKey->readOnly;
				}
				else
				{
					*value = keyRef->publicKey->readOnly;
				}
			
				break;
			}
			
			case kPGPDiskKeyProperty_CanRemove:
			{
				if( keyRef->isMasterKey )
				{
					*value = FALSE;
				}
				else if( keyRef->isPassphraseKey )
				{
					*value = TRUE;
				}
			#if PGP_BUSINESS_SECURITY
				else
				{
					*value = ! keyRef->publicKey->locked;
				}
			#endif
				
				break;
			}
			
			case kPGPDiskKeyProperty_IsMasterKey:
				*value = keyRef->isMasterKey;
				break;
				
			default:
				pgpDebugMsg( "GetPGPDiskKeyBoolean(): Unknown property" );
				err = paramErr;
				break;
		}
	}
	
	return( err );	
}

	OSStatus
RemovePGPDiskKey(PGPDiskKeyRef keyRef)
{
	OSStatus	err;
	
	err = VerifyPGPDiskKeyRef( keyRef );
	if( IsntErr( err ) )
	{
		if( ! keyRef->isMasterKey )
		{
			keyRef->deleted = TRUE;
			
			if( keyRef->isPassphraseKey )
			{
				pgpClearMemory( keyRef->passKey, sizeof( *keyRef->passKey ) );
				keyRef->passKey = NULL;
			}
			else
			{
				PGPDiskHeaderListItem	*curHeaderItem;
				PGPDiskHeaderListItem	*prevHeaderItem = NULL;
				
				curHeaderItem = keyRef->diskFile->headerList;
				while( IsntNull( curHeaderItem ) )
				{
					if( curHeaderItem->header ==
							(PGPDiskFileHeaderInfo *) keyRef->publicKey )
					{
						// Remove from linked list
						
						if( prevHeaderItem == NULL )
						{
							keyRef->diskFile->headerList =
								curHeaderItem->next;
						}
						else
						{
							prevHeaderItem->next = curHeaderItem->next;
						}
						
						pgpFreeMac( curHeaderItem->header );
						pgpFreeMac( curHeaderItem );
						
						break;
					}
				
					prevHeaderItem 	= curHeaderItem;
					curHeaderItem 	= curHeaderItem->next;
				}
				
				keyRef->publicKey = NULL;
			}
		}
		else
		{
			pgpDebugMsg( "Cannot remove master key" );
			err = paramErr;
		}
	}
	
	return( err );
}

	OSStatus
AddPassphrasePGPDiskKey(
	PGPDiskFileRef		diskFileRef,
	ConstStr255Param	masterPassphrase,
	ConstStr255Param	keyPassphrase,
	Boolean				readOnly)
{
	OSStatus	err;

	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		Boolean	didAdd = FALSE;
		UInt32	index;
		
		for( index = 1; index < diskFileRef->mainHeader->numPassphrases;
					++index )
		{
			PassphraseKey *userInfo;
			
			userInfo = &diskFileRef->mainHeader->passphrases[index];
			if ( ! userInfo->inUse )
			{
				PassphraseKey *masterInfo;
				CASTKey			decryptedKey;
				CMemoryWiper	decryptedKeyWiper( &decryptedKey,
													sizeof( decryptedKey ) );
				
				userInfo->inUse 	= TRUE;
				userInfo->readOnly	= readOnly;
			
				masterInfo = &diskFileRef->mainHeader->
									passphrases[kMasterPassphraseIndex];
				
				// Decrypt the master key
				err = DecryptPassphraseKey( masterInfo,
						&diskFileRef->mainHeader->salt, masterPassphrase,
						&decryptedKey );
				if( IsntErr( err ) )
				{
					// Now, encrypt the decrypted key again using the
					// user's passphrase
					EncryptPassphraseKey( &decryptedKey,
							&diskFileRef->mainHeader->salt, keyPassphrase,
							userInfo );

					didAdd = TRUE;
				}
				
				break;
			}
		}
		
		if( IsntErr( err ) && ! didAdd )
		{
			err = kNoMorePassphrasesAvailableError;
		}
	}
	
	return( err );
}

	const PGPDiskFileHeader *
PeekPGPDiskFileHeader(PGPDiskFileRef diskFileRef)
{
	pgpAssert( IsntErr( VerifyPGPDiskFileRef( diskFileRef ) ) );
	
	return( diskFileRef->mainHeader );
}

	UInt32
GetPublicPGPDiskKeyCount(PGPDiskFileRef diskFileRef)
{
	UInt32	numPublicKeys = 0;

	if( IsntErr( VerifyPGPDiskFileRef( diskFileRef ) ) )
	{
		PGPDiskHeaderListItem	*curHeaderItem;
		
		curHeaderItem = diskFileRef->headerList;
		while( IsntNull( curHeaderItem ) )
		{
			if( curHeaderItem->header->headerType ==
						kPGPDiskPublicKeyHeaderType )
			{
				++numPublicKeys;
			}
		
			curHeaderItem = curHeaderItem->next;
		}
	}
	
	return( numPublicKeys );
}

	static PGPError
LoadKeyRings(PGPDiskFileRef diskFileRef)
{
	PGPError	err = kPGPError_NoErr;
	
	if( ! PGPKeySetRefIsValid( diskFileRef->keySet ) )
	{
		err = PGPOpenDefaultKeyRings( diskFileRef->context, 0,
						&diskFileRef->keySet );
	}

	return( err );
}

	static PGPError
DecryptPublicKey(
	PGPKeyRef			theKey,
	ConstStr255Param	passphrase,
	const void			*encryptedData,
	PGPSize				encryptedDataSize,
	const CheckBytes	*checkBytes,
	CASTKey				*decryptedKey)
{
	PGPError				err;
	char					cPassphrase[256];
	PGPPrivateKeyContextRef	privContext;
	PGPContextRef			context;
	
	PToCString( passphrase, cPassphrase );
	
	context = PGPGetKeyContext( theKey );
	
	err = PGPNewPrivateKeyContext( theKey, kPGPPublicKeyMessageFormat_PGP,
					&privContext, PGPOPassphrase( context, cPassphrase ),
					PGPOLastOption( context ) );
	if( IsntPGPError( err ) )
	{
		PGPSize	maxDecryptedBufferSize;
		PGPSize	maxEncryptedBufferSize;
		PGPSize	maxSignatureSize;
		
		err = PGPGetPrivateKeyOperationSizes( privContext,
						&maxDecryptedBufferSize,
						&maxEncryptedBufferSize,
						&maxSignatureSize );
		if( IsntPGPError( err ) )
		{
			PGPByte	fingerPrint[128];
			PGPSize	fingerPrintSize;
			
			pgpClearMemory( fingerPrint, sizeof( fingerPrint ) );
			
			err = PGPGetKeyPropertyBuffer( theKey, kPGPKeyPropFingerprint,
							sizeof( fingerPrint ), fingerPrint, 
							&fingerPrintSize );
			if( IsntPGPError( err ) )
			{
				PGPByte	*data;
				
				data = (PGPByte *) pgpAllocMac( maxDecryptedBufferSize,
							kMemAllocationFlags );
				if( IsntNull( data ) )
				{
					PGPSize	decryptedDataSize;
					
					err = PGPPrivateKeyDecrypt( privContext, encryptedData,
								encryptedDataSize, data, &decryptedDataSize );
					if( IsntPGPError( err ) )
					{
						if( decryptedDataSize == sizeof( *decryptedKey ) )
						{
							ExpandedCASTKey	expandedKey;
							CMemoryWiper	expandedKeyWiper( &expandedKey,
													sizeof( expandedKey ) );
							CheckBytes		decryptedCheckBytes;
							
							pgpCopyMemory( data, decryptedKey,
									sizeof( *decryptedKey ) );
									
							// Decrypt the check bytes. We encrypted the first]
							// eight bytes of the fingerprint.
							
							CAST5schedule( expandedKey.keyWords,
									decryptedKey->keyBytes );
							CAST5decrypt( &checkBytes->theBytes[0],
									&decryptedCheckBytes.theBytes[0],
									expandedKey.keyWords );

							if( ! pgpMemoryEqual( &decryptedCheckBytes,
										fingerPrint,
										sizeof( decryptedCheckBytes ) ) )
							{
								// Should never get here.
								err = kPGPError_KeyUnusableForDecryption;
							}						
						}
						else
						{
							err = kPGPError_OutputBufferTooSmall;
						}
					}
							
					pgpFreeMac( data );
				}
				else
				{
					err = kPGPError_OutOfMemory;
				}	
			}
		}
		
		PGPFreePrivateKeyContext( privContext );
	}
	
	return( err );
}

	static PGPError
EncryptPublicKey(
	const CASTKey	*decryptedKey,
	PGPKeyRef		theKey,
	void			**encryptedData,
	UInt32			*encryptedDataSize,
	CheckBytes		*checkBytes)
{
	PGPError				err;
	PGPPublicKeyContextRef	pubContext;
	PGPByte					*data;
	PGPSize					dataSize;
	
	data 		= NULL;
	dataSize	= 0;
	
	err = PGPNewPublicKeyContext( theKey,
						kPGPPublicKeyMessageFormat_PGP,
						&pubContext );
	if( IsntPGPError( err ) )
	{
		PGPSize	maxDecryptedBufferSize;
		PGPSize	maxEncryptedBufferSize;
		PGPSize	maxSignatureSize;
		
		err = PGPGetPublicKeyOperationSizes( pubContext,
						&maxDecryptedBufferSize,
						&maxEncryptedBufferSize,
						&maxSignatureSize );
		if( IsntPGPError( err ) )
		{
			PGPByte	fingerPrint[128];
			PGPSize	fingerPrintSize;
			
			pgpClearMemory( fingerPrint, sizeof( fingerPrint ) );
			
			err = PGPGetKeyPropertyBuffer( theKey, kPGPKeyPropFingerprint,
							sizeof( fingerPrint ), fingerPrint, 
							&fingerPrintSize );
			if( IsntPGPError( err ) )
			{
				ExpandedCASTKey		expandedKey;
				CMemoryWiper		expandedKeyWiper( &expandedKey,
										sizeof( expandedKey ) );
				
				// Compute the check bytes. We encrypt the first eight
				// bytes of the fingerprint.
				
				CAST5schedule( expandedKey.keyWords, decryptedKey->keyBytes );
				CAST5encrypt( fingerPrint, &checkBytes->theBytes[0],
						expandedKey.keyWords );
				
				data = (PGPByte *) pgpAllocMac( maxEncryptedBufferSize,
							kMemAllocationFlags );
				if( IsntNull( data ) )
				{
					err = PGPPublicKeyEncrypt( pubContext, decryptedKey,
							sizeof( *decryptedKey ), data, &dataSize );
				}
				else
				{
					err = kPGPError_OutOfMemory;
				}	
			}			
		}
		
		(void) PGPFreePublicKeyContext( pubContext );
	}
	
	*encryptedData 		= data;
	*encryptedDataSize	= dataSize;
	
	return( err );
}

	static PGPError
NewPublicPGPDiskKeyRef(
	PGPDiskFileRef			diskFileRef,
	PGPDiskPublicKeyHeader	*header,
	PGPDiskKeyRef			*keyRefPtr)
{
	PGPError		err = kPGPError_NoErr;
	PGPDiskKeyRef	keyRef;

	keyRef = (PGPDiskKeyRef) pgpAllocMac( sizeof( *keyRef ),
					kMemAllocationFlags );
	if( PGPDiskKeyRefIsValid( keyRef ) )
	{
		keyRef->magic			= kPGPDiskKeyMagic;
		keyRef->isPassphraseKey	= FALSE;
		keyRef->isMasterKey		= FALSE;
		keyRef->diskFile		= diskFileRef;
		keyRef->publicKey		= header;
		
		++diskFileRef->keyCount;
	}
	else
	{
		err = kPGPError_OutOfMemory;
	}
	
	*keyRefPtr = keyRef;
	
	return( err );
}

	static CComboError
FindPublicPGPDiskKey(
	PGPDiskFileRef		diskFileRef,
	ConstStr255Param	passphrase,
	PGPDiskKeyRef		*keyRefPtr)
{
	CComboError		err;
	PGPDiskKeyRef	keyRef = kInvalidPGPDiskKeyRef;
	
	err.pgpErr = LoadKeyRings( diskFileRef );
	if( err.IsntError() )
	{
		PGPDiskHeaderListItem	*curHeaderItem;
		char					cPassphrase[256];

		PToCString( passphrase, cPassphrase );
		
		curHeaderItem = diskFileRef->headerList;
		while( IsntNull( curHeaderItem ) && err.IsntError() )
		{
			if( curHeaderItem->header->headerType ==
					kPGPDiskPublicKeyHeaderType )
			{
				PGPDiskPublicKeyHeader	*pubKeyHeader;
				PGPByte					*exportedKeyID;
				PGPKeyID				keyID;
			
				pubKeyHeader 	= (PGPDiskPublicKeyHeader *)
											curHeaderItem->header;
				exportedKeyID	= (PGPByte *) pubKeyHeader +
											pubKeyHeader->keyIDOffset;
				
				err.pgpErr = PGPImportKeyID( exportedKeyID, &keyID );
				if( err.IsntError() )
				{
					PGPKeyRef	theKey;
					
					if( IsntPGPError( PGPGetKeyByKeyID( diskFileRef->keySet,
											&keyID, pubKeyHeader->algorithm,
											&theKey ) ) )
					{
						PGPBoolean	canDecrypt = FALSE;
						
						(void) PGPGetKeyBoolean( theKey,
									kPGPKeyPropCanDecrypt, &canDecrypt );
									
						if( canDecrypt &&
							PGPPassphraseIsValid( theKey,
									PGPOPassphrase( diskFileRef->context, 
										cPassphrase ),
									PGPOLastOption( diskFileRef->context ) ) )
						{
							err.pgpErr = NewPublicPGPDiskKeyRef( diskFileRef,
											pubKeyHeader, &keyRef );
							break;
						}
					}
					else
					{
						err.err = kPGPDiskIncorrectPassphraseKeyNotFoundError;
						err.pgpErr	= kPGPError_NoErr;
					}
				}
			}
		
			curHeaderItem = curHeaderItem->next;
		}
	}
	else
	{
		err.err 	= kPGPDiskIncorrectPassphraseNoKeyringsError;
		err.pgpErr	= kPGPError_NoErr;
	}
	
	if( PGPDiskKeyRefIsValid( keyRef ) )
	{
		*keyRefPtr = keyRef;
	}
	else if( err.IsntError() )
	{
		err.err = kPGPDiskIncorrectPassphraseError;
	}
	
	return( err );
}

	static PGPDiskPublicKeyHeader *
LocatePublicPGPDiskKeyID(
	PGPDiskFileRef		diskFileRef,
	const PGPKeyID		*searchKeyID)
{
	PGPDiskPublicKeyHeader	*result = NULL;
	PGPDiskHeaderListItem	*curHeaderItem;
	
	curHeaderItem = diskFileRef->headerList;
	while( IsntNull( curHeaderItem ) )
	{
		if( curHeaderItem->header->headerType == kPGPDiskPublicKeyHeaderType )
		{
			PGPDiskPublicKeyHeader	*pubKeyHeader;
			PGPByte					*exportedKeyID;
			PGPKeyID				keyID;
		
			pubKeyHeader 	= (PGPDiskPublicKeyHeader *) curHeaderItem->header;
			exportedKeyID	= (PGPByte *) pubKeyHeader +
										pubKeyHeader->keyIDOffset;
			
			if( IsntPGPError( PGPImportKeyID( exportedKeyID, &keyID ) ) )
			{
				if( PGPCompareKeyIDs( searchKeyID, &keyID ) == 0 )
				{
					result = pubKeyHeader;
					break;
				}
			}
		}
	
		curHeaderItem = curHeaderItem->next;
	}
	
	return( result );
}

	static PGPError
FindPublicPGPDiskKeyID(
	PGPDiskFileRef		diskFileRef,
	const PGPKeyID		*searchKeyID,
	PGPDiskKeyRef		*keyRefPtr)
{
	PGPDiskPublicKeyHeader	*header;
	PGPError				err;
	
	*keyRefPtr = kInvalidPGPDiskKeyRef;
	
	header = LocatePublicPGPDiskKeyID( diskFileRef, searchKeyID );
	if( IsntNull( header ) )
	{
		err = NewPublicPGPDiskKeyRef( diskFileRef, header, keyRefPtr );
	}
	else
	{
		err = kPGPError_ItemNotFound;
	}
	
	return( err );
}

	static CComboError
AddPublicKey(
	PGPDiskFileRef	diskFileRef,
	PGPKeyRef		theKey,
	const CASTKey	*decryptedKey,
	Boolean			readOnly,
	Boolean			locked)
{
	CComboError	err;
	PGPKeyID	keyID;
	
	err.pgpErr = PGPGetKeyIDFromKey( theKey, &keyID );
	if( err.IsntError() )
	{
		void					*encryptedKey 		= NULL;
		UInt32					encryptedKeySize	= 0;
		PGPByte 				exportedKeyID[kPGPMaxExportedKeyIDSize];
		PGPSize					exportedKeyIDSize;
		CheckBytes				checkBytes;
		PGPPublicKeyAlgorithm	algorithm;
		
		err.pgpErr = EncryptPublicKey( decryptedKey, theKey,
						&encryptedKey, &encryptedKeySize, &checkBytes );
		if( err.IsntError() )
		{
			PGPKeyID	keyID;
			
			err.pgpErr = PGPGetKeyIDFromKey( theKey, &keyID );
			if( err.IsntError() )
			{
				err.pgpErr = PGPExportKeyID( &keyID, exportedKeyID,
								&exportedKeyIDSize );
				if( err.IsntError() )
				{
					PGPInt32	alg;
					
					err.pgpErr = PGPGetKeyNumber( theKey, kPGPKeyPropAlgID,
										&alg );
					
					algorithm = (PGPPublicKeyAlgorithm) alg;
				}
			}
		}
		
		if( err.IsntError() )
		{
			PGPDiskHeaderListItem	*headerItem;
			
			headerItem = (PGPDiskHeaderListItem *) pgpAllocMac(
							sizeof( *headerItem ), kMemAllocationFlags );
			if( IsntNull( headerItem ) )
			{
				UInt32					headerSize;
				PGPDiskPublicKeyHeader	*header;
				
				// Allocate public key header and link into list.
				
				headerSize = sizeof( *header ) + exportedKeyIDSize + 
									encryptedKeySize;
				
				header = (PGPDiskPublicKeyHeader *) pgpAllocMac(
									headerSize, kMemAllocationFlags );
				if( IsntNull( header ) )
				{
					PGPByte					*data;
					PGPDiskHeaderListItem	*curHeader;
					
					headerItem->next 	= NULL;
					headerItem->header 	= &header->headerInfo;
					
					header->headerInfo.headerMagic = kPGPDiskHeaderMagic;
					header->headerInfo.headerType =
											kPGPDiskPublicKeyHeaderType;
					header->headerInfo.headerSize = headerSize;

					header->majorVersion =
								kPGPDiskPublicKeyHeaderMajorVersion;
					header->minorVersion =
								kPGPDiskPublicKeyHeaderMinorVersion;
												
					header->readOnly 			= readOnly;
					header->locked 				= locked;
					header->keyIDSize			= exportedKeyIDSize;
					header->encryptedKeySize	= encryptedKeySize;
					header->checkBytes			= checkBytes;
					header->algorithm			= algorithm;
					
					data = (PGPByte *) header + sizeof( *header );
					pgpCopyMemory( exportedKeyID, data, exportedKeyIDSize );
					
					header->keyIDOffset = data - (PGPByte *) header;
					
					data += exportedKeyIDSize;
					pgpCopyMemory( encryptedKey, data, encryptedKeySize );
					
					header->encryptedKeyOffset = data -
								(PGPByte *) header;

					// Link header at end of main header list.
					
					curHeader = diskFileRef->headerList;
					while( IsntNull( curHeader->next ) )
					{
						curHeader = curHeader->next;
					}
					
					curHeader->next = headerItem;
				}
				else
				{
					pgpFreeMac( headerItem );
					
					err.pgpErr = kPGPError_OutOfMemory;
				}
			}
			else
			{
				err.pgpErr = kPGPError_OutOfMemory;
			}
		}
		
		if( IsntNull( encryptedKey ) )
			pgpFreeMac( encryptedKey );
	}
	
	return( err );
}

	CComboError
AddPublicPGPDiskKey(
	PGPDiskFileRef		diskFileRef,
	PGPKeyRef			theKey,
	ConstStr255Param	masterPassphrase,
	Boolean				readOnly,
	Boolean				locked)
{
	CComboError	err;

	err.err = VerifyPGPDiskFileRef( diskFileRef );
	if( err.IsntError() )
	{
		PGPKeyID	keyID;
		
		err.pgpErr = PGPGetKeyIDFromKey( theKey, &keyID );
		if( err.IsntError() &&
			IsNull( LocatePublicPGPDiskKeyID( diskFileRef, &keyID ) ) )
		{
			PassphraseKey 	*masterInfo;
			CASTKey			decryptedKey;
			CMemoryWiper	decryptedKeyWiper( &decryptedKey,
													sizeof( decryptedKey ) );
			
			masterInfo = &diskFileRef->mainHeader->
								passphrases[kMasterPassphraseIndex];
			
			// Decrypt the master key
			err.err = DecryptPassphraseKey( masterInfo,
					&diskFileRef->mainHeader->salt, masterPassphrase,
					&decryptedKey );
			if( err.IsntError() )
			{
				err = AddPublicKey( diskFileRef, theKey,
								&decryptedKey, readOnly, locked );
			}
		}
	}
	
	return( err );
}

	OSStatus
GetIndPublicPGPDiskKey(
	PGPDiskFileRef	diskFileRef,
	UInt32			index,
	PGPDiskKeyRef	*keyRefPtr)
{
	OSStatus	err;

	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		err = GetIndKey( diskFileRef, index, FALSE, TRUE, keyRefPtr );
	}
	
	return( err );
}

	CComboError
GetPublicPGPDiskKeyInfo(
	PGPDiskKeyRef			keyRef,
	PGPKeyID				*keyID,
	PGPPublicKeyAlgorithm	*algorithm,
	PGPBoolean				*locked)
{
	CComboError	err;

	err.err = VerifyPGPDiskKeyRef( keyRef );
	if( err.IsntError() )
	{
		if( keyRef->isPassphraseKey )
		{
			err.err = paramErr;
		}
		else
		{
			PGPByte	*exportedKeyID;
		
			exportedKeyID = (PGPByte *) keyRef->publicKey +
										keyRef->publicKey->keyIDOffset;
			
			*algorithm 	= keyRef->publicKey->algorithm;
		
		#if PGP_BUSINESS_SECURITY
			*locked = keyRef->publicKey->locked;
		#else
			*locked	= FALSE;
		#endif
			
			err.pgpErr = PGPImportKeyID( exportedKeyID, keyID );
		}
	}
	
	return( err );
}

	CComboError
FindPGPDiskKey(
	PGPDiskFileRef		diskFileRef,
	ConstStr255Param	passphrase,
	PGPDiskKeyRef		*keyRefPtr)
{
	CComboError	err;
	
	err.err = FindPassphrasePGPDiskKey( diskFileRef, passphrase, keyRefPtr );
	if( err.err == kPGPDiskIncorrectPassphraseError &&
		GetPublicPGPDiskKeyCount( diskFileRef ) > 0 )
	{
		err = FindPublicPGPDiskKey( diskFileRef, passphrase, keyRefPtr );
	}
	
	return( err );
}

	CComboError
DecryptPGPDiskKey(
	PGPDiskKeyRef		keyRef,
	ConstStr255Param	passphrase,
	CASTKey				*decryptedKey)
{
	CComboError	err;
	
	err.err = VerifyPGPDiskKeyRef( keyRef );
	if( err.IsntError() )
	{
		if( keyRef->isPassphraseKey )
		{
			err.err = DecryptPassphraseKey( keyRef->passKey,
						&keyRef->diskFile->mainHeader->salt, passphrase,
						decryptedKey );
		}
		else
		{
			err.pgpErr = LoadKeyRings( keyRef->diskFile );
			if( err.IsntError() )
			{
				PGPByte		*exportedKeyID;
				PGPKeyID	keyID;
			
				exportedKeyID = (PGPByte *) keyRef->publicKey +
											keyRef->publicKey->keyIDOffset;
				
				err.pgpErr = PGPImportKeyID( exportedKeyID, &keyID );
				if( err.IsntError() )
				{
					PGPKeyRef	theKey;
					
					err.pgpErr = PGPGetKeyByKeyID( keyRef->diskFile->keySet,
										&keyID, keyRef->publicKey->algorithm,
										&theKey );
					if( err.IsntError() )
					{
						PGPByte		*encryptedKey;

						encryptedKey = (PGPByte *) keyRef->publicKey +
										keyRef->publicKey->encryptedKeyOffset;

						err.pgpErr = DecryptPublicKey( theKey, passphrase,
										encryptedKey,
										keyRef->publicKey->encryptedKeySize,
										&keyRef->publicKey->checkBytes,
										decryptedKey );
					}
				}
			}
		}
	}
	
	return( err );
}

	static OSStatus
GetIndKey(
	PGPDiskFileRef	diskFileRef,
	UInt32			index,
	Boolean			searchPassphraseKeys,
	Boolean			searchPublicKeys,
	PGPDiskKeyRef	*keyRefPtr)
{
	PGPDiskKeyRef	keyRef = kInvalidPGPDiskKeyRef;
	OSStatus		err;
	
	(void) searchPublicKeys;
	
	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		UInt32	foundKeys = 0;
		
		if( searchPassphraseKeys )
		{
			UInt32	keyIndex;

			for( keyIndex = 0;
					keyIndex < diskFileRef->mainHeader->numPassphrases;
						++keyIndex )
			{
				PassphraseKey *	userInfo = nil;
			
				userInfo = &diskFileRef->mainHeader->
										passphrases[keyIndex];
				
				if ( userInfo->inUse )
				{
					if( foundKeys == index )
					{
						err = NewPassphrasePGPDiskKeyRef( diskFileRef,
									userInfo,
									keyIndex == kMasterPassphraseIndex,
									&keyRef );
						break;
					}
					else
					{
						++foundKeys;
					}
				}
			}
		}
		
		if( IsntErr( err ) &&
			! PGPDiskKeyRefIsValid( keyRef ) &&
			searchPublicKeys )
		{
			PGPDiskHeaderListItem	*curHeaderItem;
			
			curHeaderItem = diskFileRef->headerList;
			while( IsntNull( curHeaderItem ) )
			{
				if( curHeaderItem->header->headerType ==
							kPGPDiskPublicKeyHeaderType )
				{
					if( foundKeys == index )
					{
						PGPDiskPublicKeyHeader	*pubKeyHeader;
					
						pubKeyHeader = (PGPDiskPublicKeyHeader *)
													curHeaderItem->header;

						err = NewPublicPGPDiskKeyRef( diskFileRef,
									pubKeyHeader, &keyRef );
						break;
					}
					else
					{
						++foundKeys;
					}
				}
			
				curHeaderItem = curHeaderItem->next;
			}
		}
		
		if( IsntErr( err ) &&
			! PGPDiskKeyRefIsValid( keyRef ) )
		{
			err = kEndOfKeyIterationError;
		}
	}
	
	*keyRefPtr = keyRef;
	
	return( err );
}

	OSStatus
GetIndPassphrasePGPDiskKey(
	PGPDiskFileRef	diskFileRef,
	UInt32			index,
	PGPDiskKeyRef	*keyRefPtr)
{
	OSStatus	err;
	
	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		err = GetIndKey( diskFileRef, index, TRUE, FALSE, keyRefPtr );
	}
	
	return( err );
}

	OSStatus
GetIndPGPDiskKey(
	PGPDiskFileRef	diskFileRef,
	UInt32			index,
	PGPDiskKeyRef	*keyRefPtr)
{
	OSStatus	err;

	err = VerifyPGPDiskFileRef( diskFileRef );
	if( IsntErr( err ) )
	{
		err = GetIndKey( diskFileRef, index, TRUE, TRUE, keyRefPtr );
	}
	
	return( err );
}


