#include <linux/module.h>
#include <linux/sched.h>
#include <linux/tcfs_fs.h>
#include <linux/tcfsiod.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/locks.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>

#include <asm/system.h>
#include <asm/segment.h>

/* This is for kernel_thread */
#define __KERNEL_SYSCALLS__
#include <linux/unistd.h>
#include "hash.h"

extern int close_fp(struct file *filp);

static int tcfs_notify_change(struct inode *, struct iattr *);
static void tcfs_put_inode(struct inode *);
static void tcfs_put_super(struct super_block *);
static void tcfs_read_inode(struct inode *);
static void tcfs_statfs(struct super_block *, struct statfs *, int bufsiz);

static struct super_operations tcfs_sops = { 
	tcfs_read_inode,		/* read inode */
	tcfs_notify_change,	/* notify change */
	NULL,			/* write inode */
	tcfs_put_inode,		/* put inode */
	tcfs_put_super,		/* put superblock */
	NULL,			/* write superblock */
	tcfs_statfs,		/* stat filesystem */
	NULL
};

/*
 * The "read_inode" function doesn't actually do anything:
 * the real data is filled in later in tcfs_fhget. Here we
 * just mark the cache times invalid, and zero out i_mode
 * (the latter makes "tcfs_refresh_inode" do the right thing
 * wrt pipe inodes)
 */
static void tcfs_read_inode(struct inode * inode)
{
	int rsize = inode->i_sb->u.tcfs_sb.s_server.rsize;
	int size = inode->i_sb->u.tcfs_sb.s_server.wsize;

	if (rsize > size)
		size = rsize;
	inode->i_blksize = size;
	inode->i_mode = 0;
	inode->u.tcfs_i.tcfs_fl.cflag=0;
	strcpy(inode->u.tcfs_i.pathname,"");
	inode->i_op = NULL;
	TCFS_CACHEINV(inode);
}

static void tcfs_put_inode(struct inode * inode)
{
	if (TCFS_RENAMED_DIR(inode))
		tcfs_sillyrename_cleanup(inode);
	if (inode->i_pipe)
		clear_inode(inode);
}

void tcfs_put_super(struct super_block *sb)
{
	close_fp(sb->u.tcfs_sb.s_server.file);
	tcrpc_closesock(sb->u.tcfs_sb.s_server.rsock);
	tcrpc_closesock(sb->u.tcfs_sb.x_server.rsock);
	lock_super(sb);
	sb->s_dev = 0;
	unlock_super(sb);
	MOD_DEC_USE_COUNT;
}

/*
 * The way this works is that the mount process passes a structure
 * in the data argument which contains an open socket to the TCFS
 * server and the root file handle obtained from the server's mount
 * daemon.  We stash these away in the private superblock fields.
 * Later we can add other mount parameters like caching values.
 */

struct super_block *tcfs_read_super(struct super_block *sb, void *raw_data,
				   int silent)
{
	struct tcfs_mount_data *data = (struct tcfs_mount_data *) raw_data;
	struct tcfs_server *server;
	struct tcfs_server *xserver;
	unsigned int fd,xasock;
	struct file *filp, *xafilp;

	kdev_t dev = sb->s_dev;

	MOD_INC_USE_COUNT;
	if (!data) {
		printk("tcfs_read_super: missing data argument\n");
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	fd = data->fd;
	xasock = data->xasock;
	if (data->version != TCFS_MOUNT_VERSION) {
		printk("tcfs warning: mount version %s than kernel\n",
			data->version < TCFS_MOUNT_VERSION ? "older" : "newer");
	}
	if (fd >= NR_OPEN || !(filp = current->files->fd[fd])) {
		printk("tcfs_read_super: invalid file descriptor\n");
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	if (xasock >= NR_OPEN || !(xafilp = current->files->fd[xasock])) {
		printk("tcfs_read_super: invalid file descriptor for xasock\n");
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	if (!S_ISSOCK(filp->f_inode->i_mode)) {
		printk("tcfs_read_super: not a socket\n");
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	filp->f_count++;
	if (!S_ISSOCK(xafilp->f_inode->i_mode)) {
		printk("tcfs_read_super: xasock not a socket\n");
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	xafilp->f_count++;
	lock_super(sb);

	sb->s_blocksize = 1024; /* XXX */
	sb->s_blocksize_bits = 10;
	sb->s_magic = TCFS_SUPER_MAGIC;
	sb->s_dev = dev;
	sb->s_op = &tcfs_sops;
	server = &sb->u.tcfs_sb.s_server;
	xserver = &sb->u.tcfs_sb.x_server;
	server->file = filp;
	server->lock = 0;
	server->wait = NULL;
	server->flags = data->flags;
	server->rsize = data->rsize;
	if (server->rsize <= 0)
		server->rsize = TCFS_DEF_FILE_IO_BUFFER_SIZE;
	else if (server->rsize >= TCFS_MAX_FILE_IO_BUFFER_SIZE)
		server->rsize = TCFS_MAX_FILE_IO_BUFFER_SIZE;
	server->wsize = data->wsize;
	if (server->wsize <= 0)
		server->wsize = TCFS_DEF_FILE_IO_BUFFER_SIZE;
	else if (server->wsize >= TCFS_MAX_FILE_IO_BUFFER_SIZE)
		server->wsize = TCFS_MAX_FILE_IO_BUFFER_SIZE;
	server->timeo = data->timeo*HZ/10;
	server->retrans = data->retrans;
	server->acregmin = data->acregmin*HZ;
	server->acregmax = data->acregmax*HZ;
	server->acdirmin = data->acdirmin*HZ;
	server->acdirmax = data->acdirmax*HZ;
	strcpy(server->hostname, data->hostname);

	/* Add same infos in the x_server for xattrd */
	xserver->file = xafilp;
	xserver->lock = 0;
	xserver->wait = NULL;
	xserver->flags = data->flags;
	xserver->rsize = data->rsize;
	if (xserver->rsize <= 0)
		xserver->rsize = TCFS_DEF_FILE_IO_BUFFER_SIZE;
	else if (xserver->rsize >= TCFS_MAX_FILE_IO_BUFFER_SIZE)
		xserver->rsize = TCFS_MAX_FILE_IO_BUFFER_SIZE;
	xserver->wsize = data->wsize;
	if (xserver->wsize <= 0)
		xserver->wsize = TCFS_DEF_FILE_IO_BUFFER_SIZE;
	else if (xserver->wsize >= TCFS_MAX_FILE_IO_BUFFER_SIZE)
		xserver->wsize = TCFS_MAX_FILE_IO_BUFFER_SIZE;
	xserver->timeo = data->timeo*HZ/10;
	xserver->retrans = data->retrans;
	xserver->acregmin = data->acregmin*HZ;
	xserver->acregmax = data->acregmax*HZ;
	xserver->acdirmin = data->acdirmin*HZ;
	xserver->acdirmax = data->acdirmax*HZ;
	strcpy(xserver->hostname, data->hostname);
	/* End xserver configuration */
	
	/* Start of JSP TCFS patch */
	/* Check if passed address in data->addr */
	if (data->addr.sin_addr.s_addr == INADDR_ANY) {  /* No address passed */
	  if (((struct sockaddr_in *)(&server->toaddr))->sin_addr.s_addr == INADDR_ANY) {
	    printk("TCFS: Error passed unconnected socket and no address\n") ;
	    MOD_DEC_USE_COUNT;
	    return NULL ;
	  } else {
	    /* Need access to socket internals  JSP */
	    struct socket *sock, *xasock1;
	    int dummylen ;

	 /*   printk("TCFS: using socket address\n") ;*/

	    sock = &((filp->f_inode)->u.socket_i);

	    /* extract the other end of the socket into server->toaddr */
	    sock->ops->getname(sock, &(server->toaddr), &dummylen, 1) ;
	    
	    xasock1 = &((xafilp->f_inode)->u.socket_i);

	    /* extract the other end of the socket into server->toaddr */
	    xasock1->ops->getname(xasock1, &(xserver->toaddr), &dummylen, 1) ;
	  }
	} else {
	/*  printk("TCFS: copying passed addr to server->toaddr\n") ;*/
	  memcpy((char *)&(server->toaddr),(char *)(&data->addr),sizeof(server->toaddr));
	  memcpy((char *)&(xserver->toaddr),(char *)(&data->xattraddr),sizeof(xserver->toaddr));
	}
	/* End of JSP TCFS patch */

	if ((server->rsock = tcrpc_makesock(filp)) == NULL) {
		printk("TCFS: cannot create RPC socket.\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	if ((xserver->rsock = tcrpc_makesock(xafilp)) == NULL) {
		printk("TCFS: cannot create RPC socket.\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}   
	sb->u.tcfs_sb.s_root = data->root;
	server->os=&sb->u.tcfs_sb.x_server; /* ugly but needed to know */
	xserver->os=&sb->u.tcfs_sb.s_server;  /* where xserver is in i/o functions */
	unlock_super(sb);
	if (!(sb->s_mounted = tcfs_fhget(sb, &data->root, NULL))) {
		sb->s_dev = 0;
		printk("tcfs_read_super: get root inode failed\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	strcpy(sb->s_mounted->u.tcfs_i.pathname,data->remote);
	up(&(sb->s_mounted->u.tcfs_i.wrt));
	up(&(sb->s_mounted->u.tcfs_i.mutex));
	return sb;
}

void tcfs_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
	int error;
	struct tcfs_fsinfo res;
	struct statfs tmp;

	error = tcfs_proc_statfs(&sb->u.tcfs_sb.s_server, &sb->u.tcfs_sb.s_root,
		&res);
	if (error) {
		printk("tcfs_statfs: statfs error = %d\n", -error);
		res.bsize = res.blocks = res.bfree = res.bavail = 0;
	}
	tmp.f_type = TCFS_SUPER_MAGIC;
	tmp.f_bsize = res.bsize;
	tmp.f_blocks = res.blocks;
	tmp.f_bfree = res.bfree;
	tmp.f_bavail = res.bavail;
	tmp.f_files = 0;
	tmp.f_ffree = 0;
	tmp.f_namelen = NAME_MAX;
	memcpy_tofs(buf, &tmp, bufsiz);
}

/*
 * This is our own version of iget that looks up inodes by file handle
 * instead of inode number.  We use this technique instead of using
 * the vfs read_inode function because there is no way to pass the
 * file handle or current attributes into the read_inode function.
 * We just have to be careful not to subvert iget's special handling
 * of mount points.
 */

struct inode *tcfs_fhget(struct super_block *sb, struct tcfs_fh *fhandle,
			struct tcfs_fattr *fattr)
{
	struct tcfs_fattr newfattr;
	int error;
	struct inode *inode;

	if (!sb) {
		printk("tcfs_fhget: super block is NULL\n");
		return NULL;
	}
	if (!fattr) {
		error = tcfs_proc_getattr(&sb->u.tcfs_sb.s_server, fhandle,
			&newfattr);
		if (error) {
			printk("tcfs_fhget: getattr error = %d\n", -error);
			return NULL;
		}
		fattr = &newfattr;
	}
	if (!(inode = iget(sb, fattr->fileid))) {
		printk("tcfs_fhget: iget failed\n");
		return NULL;
	}
	if (inode->i_dev == sb->s_dev) {
		if (inode->i_ino != fattr->fileid) {
			printk("tcfs_fhget: unexpected inode from iget\n");
			return inode;
		}
		*TCFS_FH(inode) = *fhandle;
		tcfs_refresh_inode(inode, fattr);
	}
	return inode;
}

int tcfs_notify_change(struct inode *inode, struct iattr *attr)
{
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	int error;

	sattr.mode = (unsigned) -1;
	if (attr->ia_valid & ATTR_MODE) 
		sattr.mode = attr->ia_mode;

	sattr.uid = (unsigned) -1;
	if (attr->ia_valid & ATTR_UID)
		sattr.uid = attr->ia_uid;

	sattr.gid = (unsigned) -1;
	if (attr->ia_valid & ATTR_GID)
		sattr.gid = attr->ia_gid;


	sattr.size = (unsigned) -1;
	if (attr->ia_valid & ATTR_SIZE)
		sattr.size = S_ISREG(inode->i_mode) ? attr->ia_size : -1;

	sattr.mtime.seconds = sattr.mtime.useconds = (unsigned) -1;
	if (attr->ia_valid & ATTR_MTIME) {
		sattr.mtime.seconds = attr->ia_mtime;
		sattr.mtime.useconds = 0;
	}

	sattr.atime.seconds = sattr.atime.useconds = (unsigned) -1;
	if (attr->ia_valid & ATTR_ATIME) {
		sattr.atime.seconds = attr->ia_atime;
		sattr.atime.useconds = 0;
	}

	error = tcfs_proc_setattr(TCFS_SERVER(inode), TCFS_FH(inode),
		&sattr, &fattr);
	if (!error)
		tcfs_refresh_inode(inode, &fattr);
	inode->i_dirt = 0;
	return error;
}

/* Every kernel module contains stuff like this. */

static struct file_system_type tcfs_fs_type = {
	tcfs_read_super, "tcfs", 0, NULL
};

/*
 * Start up an tcfsiod process. This is an awful hack, because when running
 * as a module, we will keep insmod's memory. Besides, the current->comm
 * hack won't work in this case
 * The best would be to have a syscall for tcfs client control that (among
 * other things) forks biod's.
 * Alternatively, we might want to have the idle task spawn biod's on demand.
 */
static int run_tcfsiod(void *dummy)
{
	int	ret;

#ifdef __SMP__
	lock_kernel();
	syscall_count++;
#endif

	MOD_INC_USE_COUNT;
	exit_mm(current);
	current->session = 1;
	current->pgrp = 1;
	sprintf(current->comm, "tcfsiod");
	ret = tcfsiod();
	MOD_DEC_USE_COUNT;
	return ret;
}

int init_tcfs_fs(void)
{
	/* Fork four biod's */
	kernel_thread(run_tcfsiod, NULL, 0);
	kernel_thread(run_tcfsiod, NULL, 0);
	kernel_thread(run_tcfsiod, NULL, 0);
	kernel_thread(run_tcfsiod, NULL, 0);
	init_hash(); /* Init hash tables */
/*	doinverse(); /* Init inverse table for mod 257 */
	return register_filesystem(&tcfs_fs_type);
}

#ifdef MODULE
int init_module(void)
{
	int status;

	if ((status = init_tcfs_fs()) == 0)
		register_symtab(0);
	printk("TCFS 2.2a registered\n");
	return status;
}

void cleanup_module(void)
{
	unregister_filesystem(&tcfs_fs_type);
	tcfs_kfree_cache();
	cleanup_hash();
	printk("TCFS 2.2a removed\n");
}

#endif
