/*
 * TCFS 2.0.1 
 *
 *      	   This  program  handles  RPC  "NFS"  data  requests
 *              adopting a secure transfer protocol.
 *                 This   is  an  unsecure   and  unchecked  version,
 *              use at your own risk.
 *
 *              Please, report Bugs to: <tcfs@edu-gw.dia.unisa.it>
 *
 * Authors:	Giuseppe Cattaneo, <cattaneo@udsab.dia.unisa.it>
 *		Giuseppe Persiano, <giuper@udsab.dia.unisa.it>
 *		Andrea Cozzolino, <andcoz@edu-gw.dia.unisa.it>
 *		Angelo Celentano, <angcel@edu-gw.dia.unisa.it>
 *		Aniello Del Sorbo, <anidel@edu-gw.dia.unisa.it>
 *		Ermelindo Mauriello, <ermmau@edu-gw.dia.unisa.it>
 *		Raffaele Pisapia, <rafpis@edu-gw.dia.unisa.it>
 *
 *   Permission to  use, copy, and modify  this software  without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies  of  any  software  which  is  or  includes a  copy  or
 * modification of this software and in all copies  of the supporting
 * documentation for such software.
 *
 *   This  software is  distribuited  under  the  GNU General  Public
 * License  (version  2, June  1991). Check  the  file  'COPYING'  for
 * more  infos. Some  parts of  this  software  derive  from the  NFS
 * implementation in the Linux kernel 2.0.x.
 *
 * This software  maybe be used  for any  purpose provided  the above
 * copyright  notice  is retained.  It  is  supplied  as is,  with no
 * warranty expressed or implied.
 *
 */

/* -+-_== */

#include <linux/mm.h>
#include <linux/malloc.h>
/* On some kernel versions this is wait.h */
#include <asm/semaphore.h>

#include <linux/tcfs_fs.h>
#include <linux/kdes.h>

struct hash_entry *hash_table[HASH_SIZE];

/* Since tcfsiod makes async calls to these function we must avoid 
   conflicts during hash functions .... this means hash funcs have to
   be atomic in their exectution and mutually exclusive one-to-one.
   So we use a semaphore to syncronize theme */
struct semaphore tcfs_hash_sem = MUTEX;
   
/* Here we use the division method as hash function */
#define hashpos(uid)	(uid % HASH_SIZE)
void hash_debug(void);

void init_hash(void)
{
	int i;
#ifdef HASH_DEBUG
	printk("TCFS: init_hash\n");
#endif
	for (i=0;i<HASH_SIZE;i++)
		hash_table[i]=NULL;
}

void cleanup_hash(void)
{
	int i;
	struct hash_entry *htmp, *htmp1;
#ifdef HASH_DEBUG
	printk("TCFS: cleanup_hash\n");
#endif
	down(&tcfs_hash_sem);
	for (i=0;i<HASH_SIZE;i++) {
		htmp=hash_table[i];
		while (htmp!=NULL) {
			htmp1=htmp->next;
			if (htmp->ks!=NULL)
				kfree(htmp->ks);
			kfree(htmp);
			htmp=htmp1;
		}
	}
	up(&tcfs_hash_sem);
}

struct hash_entry * hash_lookup(int uid)
{
	struct hash_entry * htmp;
#ifdef HASH_DEBUG
	printk("TCFS: hash_lookup %d\n",uid);
#endif
	down(&tcfs_hash_sem);
	htmp=hash_table[hashpos(uid)];
	while (htmp!=NULL && htmp->uid!=uid)
		htmp=htmp->next;
	up(&tcfs_hash_sem);
	return htmp;
}
struct hash_entry * hash_add(int uid,char *deskey, void *ks)
{
	int pos;
	struct hash_entry * htmp;
#ifdef HASH_DEBUG
	printk("TCFS: hash_add\n");
#endif
	htmp=hash_lookup(uid);
	down(&tcfs_hash_sem);
	if (htmp!=NULL) {

		if(!strncmp(deskey,htmp->deskey,64))
			htmp->count++;
		up(&tcfs_hash_sem);
		return NULL;

		/* BAGFIXED if deskey is different from the key 
		   already present, then count is not incremented 
			Thu Jul  3 18:55:24 MET DST 1997
								*/
	}

	pos=hashpos(uid);
	htmp=(struct hash_entry *)kmalloc(sizeof(struct hash_entry),GFP_KERNEL);
	if (htmp==NULL) {
		printk("TCFS: Unable to get a free page\n");
		up(&tcfs_hash_sem);
	}

	htmp->uid=uid;
	htmp->count=1;
	htmp->permanent=0;
	strncpy(htmp->deskey,deskey,64);
	htmp->ks=ks;
	htmp->next=hash_table[pos];
	hash_table[pos]=htmp;
#ifdef HASH_DEBUG
	hash_debug();
#endif
	up(&tcfs_hash_sem);
	return htmp;
}

void hash_rem_count (int uid)
{
  struct hash_entry * htmp1, * htmp2;

#ifdef HASH_DEBUG
  printk("TCFS: hash_rem_count\n");
#endif

  down(&tcfs_hash_sem);
  htmp2=NULL;
  htmp1=hash_table[hashpos(uid)];

  while(htmp1!=NULL && htmp1->uid!=uid)
    {
      htmp2=htmp1;
      htmp1=htmp1->next;
    }

  if (htmp1==NULL) /* Element not present into the hash table */
    { 
      up(&tcfs_hash_sem);
      return;
    }

  if (htmp1->count>0)
    {
      htmp1->count--;
    }

  if (htmp1->permanent==0 && htmp1->count==0) /* Key must be destroyed */
    {
      if (htmp2==NULL) /* This is the first element */
	{
	  hash_table[hashpos(uid)]=htmp1->next;
	  kfree(htmp1->ks);
	  kfree(htmp1);

#ifdef HASH_DEBUG
	  hash_debug();
#endif

	  up(&tcfs_hash_sem);
	  return;
	}
      
      htmp2->next=htmp1->next;
      kfree(htmp1->ks);
      kfree(htmp1);
    }

#ifdef HASH_DEBUG
  hash_debug();
#endif

  up(&tcfs_hash_sem);
}

void hash_rem_all (int uid)
{
  struct hash_entry *htmp1, *htmp2;

#ifdef HASH_DEBUG
  printk("TCFS: hash_rem_count\n");
#endif

  down(&tcfs_hash_sem);
  htmp2=NULL;
  htmp1=hash_table[hashpos(uid)];

  while(htmp1!=NULL && htmp1->uid!=uid)
    {
      htmp2=htmp1;
      htmp1=htmp1->next;
    }

  if (htmp1==NULL) /* Element not present into the hash table */
    { 
      up(&tcfs_hash_sem);
      return;
    }

  htmp1->count=0;

  if (htmp1->permanent==0 && htmp1->count==0) /* Key must be destroyed */
    {
      if (htmp2==NULL) /* This is the first element */
	{
	  hash_table[hashpos(uid)]=htmp1->next;
	  kfree(htmp1->ks);
	  kfree(htmp1);

#ifdef HASH_DEBUG
	  hash_debug();
#endif

	  up(&tcfs_hash_sem);
	  return;
	}
      
      htmp2->next=htmp1->next;
      kfree(htmp1->ks);
      kfree(htmp1);
    }

#ifdef HASH_DEBUG
  hash_debug();
#endif

  up(&tcfs_hash_sem);
}

void hash_rem_permanent (int uid)
{
  struct hash_entry * htmp1, * htmp2;

#ifdef HASH_DEBUG
  printk("TCFS: hash_rem_permanent\n");
#endif

  down(&tcfs_hash_sem);
  htmp2=NULL;
  htmp1=hash_table[hashpos(uid)];

  while(htmp1!=NULL && htmp1->uid!=uid)
    {
      htmp2=htmp1;
      htmp1=htmp1->next;
    }

  if (htmp1==NULL) /* Element not present into the hash table */
    { 
      up(&tcfs_hash_sem);
      return;
    }

  htmp1->permanent=0;

  if (htmp1->count==0) /* Key must be destroyed */
    {
      if (htmp2==NULL) /* This is the first element */
	{
	  hash_table[hashpos(uid)]=htmp1->next;
	  kfree(htmp1->ks);
	  kfree(htmp1);

#ifdef HASH_DEBUG
	  hash_debug();
#endif

	  up(&tcfs_hash_sem);
	  return;
	}
      
      htmp2->next=htmp1->next;
      kfree(htmp1->ks);
      kfree(htmp1);
    }

#ifdef HASH_DEBUG
  hash_debug();
#endif

  up(&tcfs_hash_sem);
}

#ifdef HASH_DEBUG
void hash_debug(void)
{
	int i;
	struct hash_entry * htmp;
	for (i=0;i<HASH_SIZE;i++) {
		printk("%d:",i);
		htmp=hash_table[i];
		while (htmp!=NULL) {
			printk("%d-->",htmp->uid);
			htmp=htmp->next;
		}
		printk("\n");
	}
}
#endif
