/*
 * T.C.F.S. 2.0 Alpha 1 
 *
 *      	   This  program  handles  RPC  "NFS"  data  requests
 *              adopting a secure transfer protocol.
 *                 This   is  an  unsecure   and  unchecked  version,
 *              use at your own risk.
 *
 *              Please, report Bugs to: <tcfs@edu-gw.dia.unisa.it>
 *
 * Authors:	Giuseppe Cattaneo, <cattaneo@udsab.dia.unisa.it>
 *		Giuseppe Persiano, <giuper@udsab.dia.unisa.it>
 *		Andrea Cozzolino, <andcoz@edu-gw.dia.unisa.it>
 *		Angelo Celentano, <angcel@edu-gw.dia.unisa.it>
 *		Aniello Del Sorbo, <anidel@edu-gw.dia.unisa.it>
 *		Ermelindo Mauriello, <ermmau@edu-gw.dia.unisa.it>
 *		Raffaele Pisapia, <rafpis@edu-gw.dia.unisa.it>
 *
 *   Permission to  use, copy, and modify  this software  without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies  of  any  software  which  is  or  includes a  copy  or
 * modification of this software and in all copies  of the supporting
 * documentation for such software.
 *
 *   This  software is  distribuited  under  the  GNU General  Public
 * License  (version  2, June  1991). Check  the  file  'COPYING'  for
 * more  infos. Some  parts of  this  software  derive  from the  NFS
 * implementation in the Linux kernel 2.0.x.
 *
 * This software  maybe be used  for any  purpose provided  the above
 * copyright  notice  is retained.  It  is  supplied  as is,  with no
 * warranty expressed or implied.
 *
 */

/* -+-_== */

/*
 * Defining TCFS_PROC_DEBUG causes a lookup of a file named
 * "xyzzy" to toggle debugging.  Just cd to an TCFS-mounted
 * filesystem and type 'ls xyzzy' to turn on debugging.
 */

#if 0
#define TCFS_PROC_DEBUG
#endif

#include <linux/param.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/tcfs_fs_sb.h>
#include <linux/tcfs_fs.h>
#include <linux/utsname.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/in.h>
#include <linux/pagemap.h>

#include "xattr_client.h"
#include "cypitfc.h"
#include <asm/segment.h>

#ifdef TCFS_PROC_DEBUG

static int proc_debug = 0;
#define PRINTK(format, args...) \
	do {						\
		if (proc_debug)				\
			printk(format , ## args);	\
	} while (0)

#else /* !TCFS_PROC_DEBUG */

#define PRINTK(format, args...) do ; while (0)

#endif /* !TCFS_PROC_DEBUG */

/* Mapping from TCFS error code to "errno" error code. */
#define errno_TCFSERR_IO EIO

static int *tcfs_tcrpc_header(int *p, int procedure, int ruid);
static int *tcfs_tcrpc_verify(int *p);
static int tcfs_stat_to_errno(int stat);

/* These are the function needed to handle keys hash table */
extern struct hash_entry ** hash_table;
extern void init_hash(void);
extern struct hash_entry * hash_add(int uid,char *deskey, void *other_data);
extern struct hash_entry * hash_lookup(int uid);
extern void hash_rem(int uid);

static int *xattr_tcrpc_header(int *p, int procedure, int ruid);
int xattr_tcrpc_call(struct tcfs_server *server, int *start, int *end, int size);
/*
 * Our memory allocation and release functions.
 */
 
#define TCFS_SLACK_SPACE		1024	/* Total overkill */ 
/* !!! Be careful, this constant is now also used in sock.c...
   We should easily convert to not using it anymore for most cases... */

static inline int *tcfs_tcrpc_alloc(int size)
{
	int *i;

	while (!(i = (int *)kmalloc(size+TCFS_SLACK_SPACE,GFP_NFS))) {
		schedule();
	}
	return i;
}

static inline void tcfs_tcrpc_free(int *p)
{
	kfree((void *)p);
}

/*
 * Here are a bunch of xdr encode/decode functions that convert
 * between machine dependent and xdr data formats.
 */

#define QUADLEN(len) (((len) + 3) >> 2)

static inline int *xdr_encode_fhandle(int *p, struct tcfs_fh *fhandle)
{
	*((struct tcfs_fh *) p) = *fhandle;
	return p + QUADLEN(sizeof(*fhandle));
}

static inline int *xdr_decode_fhandle(int *p, struct tcfs_fh *fhandle)
{
	*fhandle = *((struct tcfs_fh *) p);
	return p + QUADLEN(sizeof(*fhandle));
}

static inline int *xdr_encode_string(int *p, const char *string)
{
	int len = strlen(string);
	int quadlen = QUADLEN(len);

	p[quadlen] = 0;
	*p++ = htonl(len);
	memcpy(p, string, len);
	return p + quadlen;
}

static inline int *xdr_decode_string(int *p, char *string, unsigned int maxlen)
{
	unsigned int len = ntohl(*p++);
	if (len > maxlen)
		return NULL;
	memcpy(string, p, len);
	string[len] = '\0';
	return p + QUADLEN(len);
}

static inline int *xdr_decode_string2(int *p, char **string, unsigned int *len,
			unsigned int maxlen)
{
	*len = ntohl(*p++);
	if (*len > maxlen)
		return NULL;
	*string = (char *) p;
	return p + QUADLEN(*len);
}


static inline int *xdr_encode_data(int *p, const char *data, int len)
{	
	int quadlen = QUADLEN(len);
	
	p[quadlen] = 0;
	*p++ = htonl(len);
	memcpy_fromfs(p, data, len);
	return p + quadlen;
}

static inline int *xdr_encode_kdata(int *p, const char *data, int len)
{	
	int quadlen = QUADLEN(len);
	
	p[quadlen] = 0;
	*p++ = htonl(len);
	memcpy((char *)p, (char *)data, len);
	return p + quadlen;
}

static inline int *xdr_decode_data(int *p, char *data, int *lenp, int maxlen)
{
	unsigned len = *lenp = ntohl(*p++);
	if (len > maxlen)
		return NULL;
	memcpy(data, p, len);
	return p + QUADLEN(len);
}

static int *xdr_decode_eattr(int *p, unsigned int *mode)
{
	long tmp;
	tmp=ntohl(*p++);
	*mode=(int)tmp;
	return p;
}

static int *xdr_encode_eattr(int *p, unsigned int *mode)
{
	unsigned long tmp;
	tmp=(unsigned long)*mode;
	*p++ = htonl(tmp);
	return p;
}

static int *xdr_encode_lattr(int *p, unsigned long *mode)
{
	*p++ = htonl(*mode);
	return p;
}

static int *xdr_decode_fattr(int *p, struct tcfs_fattr *fattr)
{
	fattr->type = (enum tcfs_ftype) ntohl(*p++);
	fattr->mode = ntohl(*p++);
	fattr->nlink = ntohl(*p++);
	fattr->uid = ntohl(*p++);
	fattr->gid = ntohl(*p++);
	fattr->size = ntohl(*p++);
	fattr->blocksize = ntohl(*p++);
	fattr->rdev = ntohl(*p++);
	fattr->blocks = ntohl(*p++);
	fattr->fsid = ntohl(*p++);
	fattr->fileid = ntohl(*p++);
	fattr->atime.seconds = ntohl(*p++);
	fattr->atime.useconds = ntohl(*p++);
	fattr->mtime.seconds = ntohl(*p++);
	fattr->mtime.useconds = ntohl(*p++);
	fattr->ctime.seconds = ntohl(*p++);
	fattr->ctime.useconds = ntohl(*p++);
	return p;
}

static int *xdr_encode_sattr(int *p, struct tcfs_sattr *sattr)
{
	*p++ = htonl(sattr->mode);
	*p++ = htonl(sattr->uid);
	*p++ = htonl(sattr->gid);
	*p++ = htonl(sattr->size);
	*p++ = htonl(sattr->atime.seconds);
	*p++ = htonl(sattr->atime.useconds);
	*p++ = htonl(sattr->mtime.seconds);
	*p++ = htonl(sattr->mtime.useconds);
	return p;
}

static int *xdr_decode_entry(int *p, struct tcfs_entry *entry)
{
	entry->fileid = ntohl(*p++);
	if (!(p = xdr_decode_string(p, entry->name, TCFS_MAXNAMLEN)))
		return NULL;
	entry->cookie = ntohl(*p++);
	entry->eof = 0;
	return p;
}

static int *xdr_decode_fsinfo(int *p, struct tcfs_fsinfo *res)
{
	res->tsize = ntohl(*p++);
	res->bsize = ntohl(*p++);
	res->blocks = ntohl(*p++);
	res->bfree = ntohl(*p++);
	res->bavail = ntohl(*p++);
	return p;
}

/*
 * One function for each procedure in the TCFS protocol.
 */
int tcfs_proc_kwrite(struct inode * inode, unsigned long offset,
		   int count, const char *data, struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;
	void * kdata;	/* address of kernel copy */
	struct tcfs_server * server = TCFS_SERVER(inode);
	struct tcfs_fh *fhandle = TCFS_FH(inode);

	PRINTK("TCFS call  write %d @ %d\n", count, offset);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_WRITE, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	*p++ = htonl(offset); /* traditional, could be any value */
	*p++ = htonl(offset);
	*p++ = htonl(count); /* traditional, could be any value */
	kdata = (void *) (p+1);	/* start of data in RPC buffer */
	p = xdr_encode_kdata(p, data, count);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		update_vm_cache(inode, offset, kdata, count);
		p = xdr_decode_fattr(p, fattr);
		PRINTK("TCFS reply write\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply write failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}
static struct tcfs_entry *cc_entry = NULL;
extern int tcfs_lookup(struct inode *dir, const char *__name, int len,struct inode **result);

int tcfs_proc_cryptdir(struct inode *inode, char mode,struct file *filp,struct hash_entry *htmp)
{
	static kdev_t c_dev = 0;
	static int c_ino;
	static int c_size;

	int result,files=0;
	int i, index = 0,ret;
	struct tcfs_entry *entry;
	char newname[TCFS_MAXPATHLEN+1];
	char *namebuffer,*head,*tmp;

	
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_proc_cryptdir\n");
#endif
	namebuffer=kmalloc(inode->i_size,GFP_KERNEL);
	if (namebuffer==NULL) {
#ifdef DEBUG_TCFS
		printk("TCFS: no MEMORY for cryptdir\n");
#endif
		return -ENOMEM;
	}
	head=tmp=namebuffer;
	
	if (!inode || !S_ISDIR(inode->i_mode)) {
		printk("tcfs_readdir: inode is NULL or not a directory\n");
		kfree(namebuffer);
		return -EBADF;
	}
	/* initialize cache memory if it hasn't been used before */
	if (cc_entry == NULL) {
		i = sizeof (struct tcfs_entry)*TCFS_READDIR_CACHE_SIZE;
		cc_entry = (struct tcfs_entry *) kmalloc(i, GFP_KERNEL);
		if (cc_entry == NULL) {
			printk("tcfs_readdir: no MEMORY for cache\n");
			kfree(namebuffer);
			return -ENOMEM;
		}
		for (i = 0; i < TCFS_READDIR_CACHE_SIZE; i++) {
			cc_entry[i].name = (char *) kmalloc(TCFS_MAXNAMLEN + 1,
				GFP_KERNEL);
			if (cc_entry[i].name == NULL) {
				printk("tcfs_readdir: no MEMORY for cache\n");
				while (--i>=0)
					kfree(cc_entry[i].name);
				kfree(cc_entry);
				cc_entry = NULL;
			kfree(namebuffer);
			return -ENOMEM;
			}
			memset(cc_entry[i].name,0,TCFS_MAXNAMLEN+1);
		}
	} else {
		for (i = 0; i < TCFS_READDIR_CACHE_SIZE; i++) {
			memset(cc_entry[i].name,'\0',TCFS_MAXNAMLEN+1);
		}
	}
	entry = NULL;

	/* Count how many files are into the directory and load names into a buffer for rename */
	do {
		schedule();
		result = tcfs_proc_readdir(TCFS_SERVER(inode), TCFS_FH(inode),
			(int)(filp->f_pos), TCFS_READDIR_CACHE_SIZE, cc_entry);
		if (result > 0) {
			c_dev = inode->i_dev;
			c_ino = inode->i_ino;
			c_size = result;
			entry = cc_entry + (index = 0);
		}
		if (result>0) {
			files+=result;
			index+=result-1;
			filp->f_pos=entry[result-1].cookie;
		}
		/* Copy names into the buffer */
		if (result>0) {
			for (i=0;i<result;i++) {
				strcpy(tmp,cc_entry[i].name);
				tmp+=strlen(cc_entry[i].name)+1;
			}
		}
	} while (result>0);
	index=0;
	tmp=head;
	for (i=0;i<files;i++) {
		schedule();
		if (strcmp(tmp,".")!=0 && strcmp(tmp,"..")!=0) {
			memset(newname,'\0',TCFS_MAXPATHLEN+1);
			strcpy(newname,tmp);
			index=strlen(newname)+1;
			if (mode!=0) { /* Removing secure flag */
				kencrypt(newname,(index &0xfff8)+8,htmp->ks);
				tcfsencode(newname,(index &0xfff8)+8);
			}
			else {
				index=tcfsdecode(newname,index);
				kdecrypt(newname,(index & 0xfff8)+8,htmp->ks);
			}
			ret=tcfs_proc_rename(TCFS_SERVER(inode),TCFS_FH(inode),tmp,TCFS_FH(inode),newname,0);
			if (ret<0) {
#ifdef DEBUG
				printk("TCFS: tcfs_proc_rename returned %d\n",ret);
#endif
				kfree(namebuffer);
				return ret;
			}
		}  
		tmp+=strlen(tmp)+1;
	}
	kfree(namebuffer);
	return 0;
}

int tcfs_truncate(char *path,unsigned long lenght,struct tcfs_server *server)
{
	int *p, *p0;
	int status;
	int ruid=0;

	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
	p= xattr_tcrpc_header(p0, 3L, ruid);
	p= xdr_encode_string(p, path);
	p= xdr_encode_lattr(p, &lenght);
	if ((status=tcfs_tcrpc_call(server, p0, p, server->rsize))<0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status=-errno_TCFSERR_IO;
	p = xdr_decode_eattr( p, &status);
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_endes(struct inode *inode,int mode,struct file *filp)
{
	struct tcfs_server *server=NULL;
	struct tcfs_fh *fhandle;
	struct tcfs_fattr fattr;
	int read,write;
	unsigned long off,size;
	char *data=NULL;
	struct hash_entry *htmp;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_endes\n");
#endif
	htmp=hash_lookup(current->uid);
	if (htmp==NULL) {
#ifdef DEBUG_TCFS
	printk("TCFS: hash_lookup returned NULL\n");
#endif
		return -EACCES;
	}
	if (S_ISDIR(inode->i_mode)) {
		return tcfs_proc_cryptdir(inode,mode,filp,htmp);
	} 
	if (S_ISLNK(inode->i_mode)) {
		return 0;
	}
	if (S_ISCHR(inode->i_mode) || S_ISBLK(inode->i_mode)) {
		return 0;
	}
	if (S_ISFIFO(inode->i_mode) || S_ISSOCK(inode->i_mode)) {
		return -EINVAL;
	}
	if (S_ISREG(inode->i_mode)) {
		while(data==NULL) {
			data=kmalloc(1024,GFP_KERNEL);
			schedule();
		}
		server=&(inode->i_sb->u.tcfs_sb.s_server);
		fhandle=&(inode->u.tcfs_i.fhandle);
		size=inode->i_size;
		if (size>0) {
			for (off=0;off<size;off+=1024) {
				schedule();
				memset(data,'\0',1024);
				read=tcfs_proc_read(server,fhandle,(int)off,(int)1024,data,&fattr);
				if (read>0) {
					if (mode!=0) { /* Setting Secure Flag */
						if (read % 8!=0) {
							read+=8-(read %8);
						}
						mkencrypt(data,read,htmp->ks);
					} else { /* Removing Secure Flag */
						mkdecrypt(data,read,htmp->ks);
						if (off+1024>=size) { 
							/* We are at the end of file so we remove spure bytes */
							if (inode->u.tcfs_i.tcfs_fl.bf.spure>0)
								read-=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
						}
					}
				} else {
#ifdef DEBUG_TCFS
printk("TCFS: read error %d in tcfs_endes\n",read);
#endif
					kfree(data);
					return read;
				}
				write=tcfs_proc_kwrite(inode,off,read,data,&fattr);
				if (write!=0)
					printk("TCFS: write error in tcfs_endes\n");
			}
			if (mode == 0 ) {
				int status;
				loff_t lenght;
				lenght= inode->i_size;
				status=tcfs_truncate(inode->u.tcfs_i.pathname,lenght,&(inode->i_sb->u.tcfs_sb.x_server));
				if (status<0) {
					kfree(data);
					return status;
				}
				else {
					kfree(data);
					return 0;
				}
			}
			kfree(data);
			return (int)(size % 8);
		}
	}
	kfree(data);
	return 0; 
}
 

int tcfs_proc_geteattr(struct tcfs_server *server, char *filename, unsigned int *cflag)
{
	int *p, *p0;
	int status;
	int ruid=0;
	int mode;
	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
	p= xattr_tcrpc_header(p0,1L,ruid);
	p= xdr_encode_string(p,filename);
	if ((status=tcfs_tcrpc_call(server, p0, p, server->rsize))<0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	p = xdr_decode_eattr(p, &mode);
	status = 0; 
	tcfs_tcrpc_free(p0);
	*cflag=mode;
	if (mode<0) {
		return mode;
	}
	return status;
}

int tcfs_proc_seteattr(struct inode *inode,struct tcfs_server *server, char *filename, unsigned int cflag,struct file *filp)
{
	int *p, *p0;
	int status;
	int ruid=0,error=0;
	unsigned int oldflags;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_proc_seteattr\n");
#endif 
	if (inode!=NULL && 
		(TCFS_IS_SECURE(cflag)!=TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag))) /* CFLAG changed */
		error=tcfs_endes(inode,TCFS_IS_SECURE(cflag),filp);
	if (error <0)
		return error;
	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
	if (inode!=NULL) {
		oldflags=inode->u.tcfs_i.tcfs_fl.cflag;
		inode->u.tcfs_i.tcfs_fl.cflag=cflag;
		inode->u.tcfs_i.tcfs_fl.bf.spure=error;
		cflag=inode->u.tcfs_i.tcfs_fl.cflag;
		inode->u.tcfs_i.tcfs_fl.cflag=oldflags;
	}
	p= xattr_tcrpc_header(p0,2L,ruid);
	p= xdr_encode_string(p,filename);
	p= xdr_encode_eattr(p, &cflag);
	if ((status=tcfs_tcrpc_call(server, p0, p, server->rsize))<0) {
		tcfs_tcrpc_free(p0);
		return status;
	} 
	if (!(p = tcfs_tcrpc_verify(p0)))
	status = -errno_TCFSERR_IO;
	p = xdr_decode_eattr(p, &status); 
	if (inode!=NULL) {
		inode->u.tcfs_i.tcfs_fl.cflag=cflag;
	}
	tcfs_tcrpc_free(p0);
	return status;
}    

int tcfs_proc_getattr(struct tcfs_server *server, struct tcfs_fh *fhandle,
		     struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  getattr\n");
	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_GETATTR, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->rsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fattr(p, fattr);
		PRINTK("TCFS reply getattr\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply getattr failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_setattr(struct tcfs_server *server, struct tcfs_fh *fhandle,
		     struct tcfs_sattr *sattr, struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  setattr\n");
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_SETATTR, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	p = xdr_encode_sattr(p, sattr);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fattr(p, fattr);
		PRINTK("TCFS reply setattr\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply setattr failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_lookup(struct tcfs_server *server, struct tcfs_fh *dir, const char *name,
		    struct tcfs_fh *fhandle, struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  lookup %s\n", name);
#ifdef TCFS_PROC_DEBUG
	if (!strcmp(name, "xyzzy"))
		proc_debug = 1 - proc_debug;
#endif
	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_LOOKUP, ruid);
	p = xdr_encode_fhandle(p, dir);
	p = xdr_encode_string(p, name);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->rsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fhandle(p, fhandle);
		p = xdr_decode_fattr(p, fattr);
		PRINTK("TCFS reply lookup\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply lookup failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_readlink(struct tcfs_server *server, struct tcfs_fh *fhandle,
		int **p0, char **string, unsigned int *len, unsigned int maxlen)
{
	int *p;
	int status, ruid = 0;

	PRINTK("TCFS call  readlink\n");
	if (!(*p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(*p0, TCFSPROC_READLINK, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	if ((status = tcfs_tcrpc_call(server, *p0, p, server->rsize)) < 0)
		return status;
	if (!(p = tcfs_tcrpc_verify(*p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		if (!(p = xdr_decode_string2(p, string, len, maxlen))) {
			printk("tcfs_proc_readlink: giant pathname\n");
			status = -errno_TCFSERR_IO;
		}
		else	/* status = 0, */
			PRINTK("TCFS reply readlink\n");
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply readlink failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	return status;
}

int tcfs_proc_read(struct tcfs_server *server, struct tcfs_fh *fhandle,
	  int offset, int count, char *data, struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;
	int len;

	PRINTK("TCFS call  read %d @ %d\n", count, offset);
	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_READ, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	*p++ = htonl(offset);
	*p++ = htonl(count);
	*p++ = htonl(count); /* traditional, could be any value */
	if ((status = tcfs_tcrpc_call(server, p0, p, server->rsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fattr(p, fattr);
		if (!(p = xdr_decode_data(p, data, &len, count))) {
			printk("tcfs_proc_read: giant data size\n"); 
			status = -errno_TCFSERR_IO;
		}
		else {
			status = len;
			PRINTK("TCFS reply read %d\n", len);
		}
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply read failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int
tcfs_proc_read_request(struct tcrpc_ioreq *req, struct tcfs_server *server,
			struct tcfs_fh *fh, unsigned long offset,
			unsigned long count, __u32 *buf)
{
	__u32	*p, *p0;
	int	len;

	PRINTK("TCFS reqst read %ld @ %ld\n", count, offset);
	if (!(p0 = tcfs_tcrpc_alloc(TCFS_SLACK_SPACE)))
		return -EIO;

	p = tcfs_tcrpc_header(p0, TCFSPROC_READ, 0);
	p = xdr_encode_fhandle(p, fh);
	*p++ = htonl(offset);
	*p++ = htonl(count);
	*p++ = htonl(count); /* traditional, could be any value */
	req->rq_svec[0].iov_base = p0;
	req->rq_svec[0].iov_len  = (p - p0) << 2;
	req->rq_slen = (p - p0) << 2;
	req->rq_snr = 1;

	len = (6 + 1 + 17 + 1);		/* standard READ reply header */
	req->rq_rvec[0].iov_base = p0;
	req->rq_rvec[0].iov_len  = len << 2;
	req->rq_rvec[1].iov_base = buf;
	req->rq_rvec[1].iov_len  = count;
	req->rq_rvec[2].iov_base = p0 + len;		/* spill buffer */
	req->rq_rvec[2].iov_len  = (TCFS_SLACK_SPACE - len) << 2;
	req->rq_rlen = count + TCFS_SLACK_SPACE;
	req->rq_rnr = 3;

	req->rq_addr = &server->toaddr;
	req->rq_alen = sizeof(server->toaddr);

	return tcrpc_transmit(server->rsock, req);
}

int
tcfs_proc_read_reply(struct tcrpc_ioreq *req, struct tcfs_fattr *fattr)
{
	int		status;
	__u32		*p0, *p;
	int		count;

	p0 = (__u32 *) req->rq_rvec[0].iov_base;

	if (!(p = tcfs_tcrpc_verify(p0))) {
		/* Tell the upper layers to retry */
		status = -EAGAIN;
		/* status = -errno_TCFSERR_IO; */
	} else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fattr(p, fattr);
		count = ntohl(*p++);
		if (p != req->rq_rvec[2].iov_base) {
			/* unexpected RPC reply header size. punt.
			 * fixme: move iovec contents to align data
			 * on page boundary and adjust RPC header size
			 * guess. */
			status = -errno_TCFSERR_IO;
			PRINTK("TCFS reply read odd header size %d\n",
					(p - p0) << 2);
		} else {
			status = count;
			PRINTK("TCFS reply read %d\n", count);
		}
	}
	else {
		PRINTK("TCFS reply read failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_write(struct inode * inode, int offset,
		   int count, const char *data, struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;
	void * kdata;	/* address of kernel copy */
	struct tcfs_server * server = TCFS_SERVER(inode);
	struct tcfs_fh *fhandle = TCFS_FH(inode);

	PRINTK("TCFS call  write %d @ %d\n", count, offset);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_WRITE, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	*p++ = htonl(offset); /* traditional, could be any value */
	*p++ = htonl(offset);
	*p++ = htonl(count); /* traditional, could be any value */
	kdata = (void *) (p+1);	/* start of data in RPC buffer */
	p = xdr_encode_data(p, data, count);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		update_vm_cache(inode, offset, kdata, count);
		p = xdr_decode_fattr(p, fattr);
		PRINTK("TCFS reply write\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply write failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_create(struct tcfs_server *server, struct tcfs_fh *dir,
		    const char *name, struct tcfs_sattr *sattr,
		    struct tcfs_fh *fhandle, struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  create %s\n", name);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_CREATE, ruid);
	p = xdr_encode_fhandle(p, dir);
	p = xdr_encode_string(p, name);
	p = xdr_encode_sattr(p, sattr);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fhandle(p, fhandle);
		p = xdr_decode_fattr(p, fattr);
		PRINTK("TCFS reply create\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply create failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_remove(struct tcfs_server *server, struct tcfs_fh *dir, const char *name)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  remove %s\n", name);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_REMOVE, ruid);
	p = xdr_encode_fhandle(p, dir);
	p = xdr_encode_string(p, name);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		PRINTK("TCFS reply remove\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply remove failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_rename(struct tcfs_server *server,
		    struct tcfs_fh *old_dir, const char *old_name,
		    struct tcfs_fh *new_dir, const char *new_name,
		    int must_be_dir)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	/*
	 * Disallow "rename()" with trailing slashes over TCFS: getting
	 * POSIX.1 behaviour is just too unlikely.
	 */
	if (must_be_dir)
		return -EINVAL;
	PRINTK("TCFS call  rename %s -> %s\n", old_name, new_name);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_RENAME, ruid);
	p = xdr_encode_fhandle(p, old_dir);
	p = xdr_encode_string(p, old_name);
	p = xdr_encode_fhandle(p, new_dir);
	p = xdr_encode_string(p, new_name);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		PRINTK("TCFS reply rename\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply rename failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_link(struct tcfs_server *server, struct tcfs_fh *fhandle,
		  struct tcfs_fh *dir, const char *name)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  link %s\n", name);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_LINK, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	p = xdr_encode_fhandle(p, dir);
	p = xdr_encode_string(p, name);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		PRINTK("TCFS reply link\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply link failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_symlink(struct tcfs_server *server, struct tcfs_fh *dir,
		     const char *name, const char *path, struct tcfs_sattr *sattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  symlink %s -> %s\n", name, path);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_SYMLINK, ruid);
	p = xdr_encode_fhandle(p, dir);
	p = xdr_encode_string(p, name);
	p = xdr_encode_string(p, path);
	p = xdr_encode_sattr(p, sattr);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		PRINTK("TCFS reply symlink\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply symlink failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_mkdir(struct tcfs_server *server, struct tcfs_fh *dir,
		   const char *name, struct tcfs_sattr *sattr,
		   struct tcfs_fh *fhandle, struct tcfs_fattr *fattr)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  mkdir %s\n", name);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_MKDIR, ruid);
	p = xdr_encode_fhandle(p, dir);
	p = xdr_encode_string(p, name);
	p = xdr_encode_sattr(p, sattr);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fhandle(p, fhandle);
		p = xdr_decode_fattr(p, fattr);
		PRINTK("TCFS reply mkdir\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply mkdir failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_rmdir(struct tcfs_server *server, struct tcfs_fh *dir, const char *name)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  rmdir %s\n", name);
	if (!(p0 = tcfs_tcrpc_alloc(server->wsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_RMDIR, ruid);
	p = xdr_encode_fhandle(p, dir);
	p = xdr_encode_string(p, name);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->wsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		PRINTK("TCFS reply rmdir\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply rmdir failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_readdir(struct tcfs_server *server, struct tcfs_fh *fhandle,
		     int cookie, int count, struct tcfs_entry *entry)
{
	int *p, *p0;
	int status;
	int ruid = 0;
	int i;
	int size;
	int eof;

	PRINTK("TCFS call  readdir %d @ %d\n", count, cookie);
	size = server->rsize;
	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_READDIR, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	*p++ = htonl(cookie);
	*p++ = htonl(size);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->rsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		for (i = 0; i < count && *p++; i++) {
			if (!(p = xdr_decode_entry(p, entry++)))
				break;
		}
		if (!p) {
			printk("tcfs_proc_readdir: giant filename\n");
			status = -errno_TCFSERR_IO;
		}
		else {
			eof = (i == count && !*p++ && *p++)
			      || (i < count && *p++);
			if (eof && i)
				entry[-1].eof = 1;
			PRINTK("TCFS reply readdir %d %s\n", i,
			       eof ? "eof" : "");
			status = i;
		}
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply readdir failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

int tcfs_proc_statfs(struct tcfs_server *server, struct tcfs_fh *fhandle,
		    struct tcfs_fsinfo *res)
{
	int *p, *p0;
	int status;
	int ruid = 0;

	PRINTK("TCFS call  statfs\n");
	if (!(p0 = tcfs_tcrpc_alloc(server->rsize)))
		return -EIO;
retry:
	p = tcfs_tcrpc_header(p0, TCFSPROC_STATFS, ruid);
	p = xdr_encode_fhandle(p, fhandle);
	if ((status = tcfs_tcrpc_call(server, p0, p, server->rsize)) < 0) {
		tcfs_tcrpc_free(p0);
		return status;
	}
	if (!(p = tcfs_tcrpc_verify(p0)))
		status = -errno_TCFSERR_IO;
	else if ((status = ntohl(*p++)) == TCFS_OK) {
		p = xdr_decode_fsinfo(p, res);
		PRINTK("TCFS reply statfs\n");
		/* status = 0; */
	}
	else {
		if (!ruid && current->fsuid == 0 && current->uid != 0) {
			ruid = 1;
			goto retry;
		}
		PRINTK("TCFS reply statfs failed = %d\n", status);
		status = -tcfs_stat_to_errno(status);
	}
	tcfs_tcrpc_free(p0);
	return status;
}

/*
 * Here are a few RPC-assist functions.
 */

int *tcrpc_header(int *p, int procedure, int program, int version,
					int uid, int gid, int *groups)
{
	int *p1, *p2;
	int i;
	static int xid = 0;
	unsigned char *sys = (unsigned char *) system_utsname.nodename;

	if (xid == 0) {
		xid = CURRENT_TIME;
		xid ^= (sys[3]<<24) | (sys[2]<<16) | (sys[1]<<8) | sys[0];
	}
	*p++ = htonl(++xid);
	*p++ = htonl(TCRPC_CALL);
	*p++ = htonl(TCRPC_VERSION);
	*p++ = htonl(program);
	*p++ = htonl(version);
	*p++ = htonl(procedure);
	*p++ = htonl(TCRPC_AUTH_UNIX);
	p1 = p++;
	*p++ = htonl(CURRENT_TIME); /* traditional, could be anything */
	p = xdr_encode_string(p, (char *) sys);
	*p++ = htonl(uid);
	*p++ = htonl(gid);
	p2 = p++;
	for (i = 0; i < 16 && i < NGROUPS && groups[i] != NOGROUP; i++)
		*p++ = htonl(groups[i]);
	*p2 = htonl(i);
	*p1 = htonl((p - (p1 + 1)) << 2);
	*p++ = htonl(TCRPC_AUTH_NULL);
	*p++ = htonl(0);
	return p;
}


static int *tcfs_tcrpc_header(int *p, int procedure, int ruid)
{
	return tcrpc_header(p, procedure, TCFS_PROGRAM, TCFS_VERSION,
			(ruid ? current->uid : current->fsuid),
			current->egid, current->groups);
}

static int *xattr_tcrpc_header(int *p, int procedure, int ruid)
{
	return tcrpc_header(p, procedure, XATTR_PROGRAM, XATTR_VERSION,
			(ruid ? current->uid : current->fsuid),
			current->egid, current->groups);
}

int *tcrpc_verify(int *p)
{
	unsigned int n;

	p++;
	if ((n = ntohl(*p++)) != TCRPC_REPLY) {
		printk("tcfs_tcrpc_verify: not an RPC reply: %x\n", n);
		return NULL;
	}
	if ((n = ntohl(*p++)) != TCRPC_MSG_ACCEPTED) {
		printk("tcfs_tcrpc_verify: RPC call rejected: %d\n", n);
		return NULL;
	}
	switch (n = ntohl(*p++)) {
	case TCRPC_AUTH_NULL: case TCRPC_AUTH_UNIX: case TCRPC_AUTH_SHORT:
		break;
	default:
		printk("tcfs_tcrpc_verify: bad RPC authentication type: %d\n", n);
		return NULL;
	}
	if ((n = ntohl(*p++)) > 400) {
		printk("tcfs_tcrpc_verify: giant auth size\n");
		return NULL;
	}
	p += QUADLEN(n);
	if ((n = ntohl(*p++)) != TCRPC_SUCCESS) {
		printk("tcfs_tcrpc_verify: RPC call failed: %d\n", n);
		return NULL;
	}
	return p;
}


static int *tcfs_tcrpc_verify(int *p)
{
	return tcrpc_verify(p);
}


/*
 * We need to translate between tcfs status return values and
 * the local errno values which may not be the same.
 */

static struct {
	int stat;
	int errno;
} tcfs_errtbl[] = {
	{ TCFS_OK,		0		},
	{ TCFSERR_PERM,		EPERM		},
	{ TCFSERR_NOENT,		ENOENT		},
	{ TCFSERR_IO,		errno_TCFSERR_IO	},
	{ TCFSERR_NXIO,		ENXIO		},
	{ TCFSERR_EAGAIN,	EAGAIN		},
	{ TCFSERR_ACCES,		EACCES		},
	{ TCFSERR_EXIST,		EEXIST		},
	{ TCFSERR_NODEV,		ENODEV		},
	{ TCFSERR_NOTDIR,	ENOTDIR		},
	{ TCFSERR_ISDIR,		EISDIR		},
	{ TCFSERR_INVAL,		EINVAL		},
	{ TCFSERR_FBIG,		EFBIG		},
	{ TCFSERR_NOSPC,		ENOSPC		},
	{ TCFSERR_ROFS,		EROFS		},
	{ TCFSERR_NAMETOOLONG,	ENAMETOOLONG	},
	{ TCFSERR_NOTEMPTY,	ENOTEMPTY	},
	{ TCFSERR_DQUOT,		EDQUOT		},
	{ TCFSERR_STALE,		ESTALE		},
#ifdef EWFLUSH
	{ TCFSERR_WFLUSH,	EWFLUSH		},
#endif
	{ -1,			EIO		}
};

static int tcfs_stat_to_errno(int stat)
{
	int i;

	for (i = 0; tcfs_errtbl[i].stat != -1; i++) {
		if (tcfs_errtbl[i].stat == stat)
			return tcfs_errtbl[i].errno;
	}
	printk("tcfs_stat_to_errno: bad tcfs status return value: %d\n", stat);
	return tcfs_errtbl[i].errno;
}

